/*********************************************************************************************************
DVR, Digital Video Recorder - a tool to record movies (audio/video), using realtime compression

It uses libavifile (see http://divx.euro.ru) and some code from kwintv (see wenk@mathematik.uni-kl.de)

This program is free software; you can redistribute it and/or modify it under the terms of the GNU General
Public License as published by the Free Software Foundation, etc.
This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY, etc.

copyright(C) february 2001, Pierre Hbert (pierre.hebert@netcourrier.com)
*********************************************************************************************************/
#include <stdlib.h>
#include <iostream>
#include <qpixmap.h>
#include <qframe.h>
#include <qapp.h>
// hum... need to include qt headers before X11 headers...
#include <X11/Xlib.h>
#include <X11/extensions/Xvlib.h>
#include "framepool.h"
#include "v4l.h"
#include "dvr.h"
#include "qv4lwindow.h"

#include "tv.xpm"

QV4LWindow::QV4LWindow(DVR *_dvr, QWidget *w, char *n) :
  QFrame(w, n), 
  dvr(_dvr),
  v4l(_dvr->getV4l()),
  video_preview_enabled(false),
  record_mode(false)
{
  unsigned int width, height;
  v4l->getCaptureSize(width, height);
	setWFlags(getWFlags() | Qt::WType_TopLevel);
	setAutoAdjustSize(true);
	resize(width, height);
	setCaption("DVR : "+QString(v4l->deviceName()));
	setIcon(QPixmap(tv));

  unsigned int version, release, request_base, event_base, error_base;
  has_xv_extension=(XvQueryExtension(this->x11Display(), &version, &release, &request_base, &event_base, &error_base) == Success);
}

QV4LWindow::~QV4LWindow() {
  stopVideoPreview();
}

void QV4LWindow::startVideoPreview() {
  if(video_preview_enabled) {
    return;
  }
  
  if(v4l->hasVideoPreview()) {
	  v4l->enableVideoPreview(true);
  } else {
    video_preview_enabled=true;
    start();
  }
}

void QV4LWindow::stopVideoPreview() {
  if(v4l->hasVideoPreview()) {
	  v4l->enableVideoPreview(false);
    return;
  }

  if(!video_preview_enabled) {
    return;
  }

  video_preview_enabled=false;
  if(qApp->locked()) {
    qApp->unlock();
    wait();
    qApp->lock();
  } else {
    wait();
  }
}

void QV4LWindow::hideEvent(QHideEvent *) {
  stopVideoPreview();
}	

void QV4LWindow::showEvent(QShowEvent *e) {
  if(v4l->hasVideoPreview()) {
	  v4l->enableVideoPreview(true);
    return;
  }

  if(e->spontaneous()) {
    startVideoPreview();
  }
}	

void QV4LWindow::moveEvent(QMoveEvent *) {
  if(v4l->hasVideoPreview()) {
    v4l->setWindowPositionSize(geometry().x(), geometry().y(), width(), height());
  }
}	
		
void QV4LWindow::resizeEvent(QResizeEvent *) {
  if(v4l->hasVideoPreview()) {
	  v4l->enableVideoPreview(true);
    v4l->setWindowPositionSize(geometry().x(), geometry().y(), width(), height());
    return;
  }
  
  stopVideoPreview();
  startVideoPreview();
}	

void QV4LWindow::setAutoAdjustSize(bool aa) {
  if(v4l->hasVideoPreview()) {
    if(aa) {
      int minx, miny, maxx, maxy;
      v4l->getMinSize(minx, miny);
      v4l->getMaxSize(maxx, maxy);
      setMinimumSize(minx, miny);
      setMaximumSize(maxx, maxy);
    } else {
      setFixedSize(width(), height());
    }
	}
}

void QV4LWindow::run() {
  if(has_xv_extension) {
    xvVideoPreview();
  } else {
    x11VideoPreview();
  }
}

void QV4LWindow::xvVideoPreview() {
  qApp->lock();
  Display *dpy=x11Display();

  unsigned int nb_adaptators;
  XvAdaptorInfo *adaptators;
  if(XvQueryAdaptors(dpy, DefaultRootWindow(dpy), &nb_adaptators, &adaptators) != Success) {
    cerr<<"no adaptator available"<<endl;
    XvFreeAdaptorInfo(adaptators);
    qApp->unlock();
    return;
  }

  int format_looked;
  switch(v4l->pixelFormat()) {
    case V4L::PX_YUV420P : 
      format_looked=0x30323449;
      break;

    default:
      cerr<<"format "<<v4l->pixelFormat()<<" not handled. Send a mail to the author in order him to handle your case."<<endl;
      XvFreeAdaptorInfo(adaptators);
      qApp->unlock();
      return;
  }

  int port_found=0;
  int format_found=0;
  int nb_formats;
  XvImageFormatValues *formats=NULL;
  for(unsigned int i=0; i<nb_adaptators; i++) {
    if(adaptators[i].type & XvImageMask) {
      formats=XvListImageFormats(dpy, adaptators[i].base_id, &nb_formats);

      for(int j=0; j<nb_formats; j++) {
        if(formats[j].id==format_looked) {
          format_found=j;
          port_found=i;
          break;
        }
      }

      if(format_found) break;
    }
  }

  if(!format_found) {
    cerr<<"no Xv format found corresponding to input format ("<<format_looked<<")"<<endl;
    XvFreeAdaptorInfo(adaptators);
    qApp->unlock();
    return;
  }
  
  unsigned int window_width, window_height;
  unsigned int capture_width, capture_height;
  
  v4l->getCaptureSize(capture_width, capture_height);
  window_width=width();
  window_height=height();

  char *data=new char[capture_width*capture_height*4];
  XvImage *xv_image=XvCreateImage(dpy, adaptators[port_found].base_id, formats[format_found].id, data, capture_width, capture_height);
  
  Window win=winId();
  GC gc=XCreateGC(dpy, win, 0, NULL);
  qApp->unlock();
  
  unsigned char *frame;
  while(video_preview_enabled) {
    if(record_mode) {
      FramePool *external_fp=dvr->externalFramepool();
      if(external_fp) {
        frame=external_fp->getFrame();
      } else {
        frame=NULL;
      }
    } else {
      frame=v4l->captureFrame();
    }
    
    if(frame) {
      memcpy(xv_image->data, frame, capture_width*capture_height*3);
    } else {
      break;
    }

    if(record_mode) {
      dvr->externalFramepool()->removeFrame();
    }

    qApp->lock();
    window_width=width();
    window_height=height();
    if(frame) {
      XvPutImage(dpy, adaptators[port_found].base_id, win, gc, xv_image, 0, 0, capture_width, capture_height, 0, 0, window_width, window_height);
    }
    qApp->unlock();
  }

  qApp->lock();
  XFreeGC(dpy, gc);
  XFree(xv_image);
  delete []data;
  XvFreeAdaptorInfo(adaptators);
  qApp->unlock();

  video_preview_enabled=false;
  exit();
}

void QV4LWindow::x11VideoPreview() {
  // TODO if necessary
  while(video_preview_enabled) {
    sleep(1);
  }
  video_preview_enabled=false;
}


void QV4LWindow::setRecordMode(bool on) {
  if(v4l->hasVideoPreview()) {
    return;
  }
  record_mode=on;
}
