// ------------------------------------------------------------------------
// qeevent.cpp: Virtual base for classes representing libecasound
//              processing events
// Copyright (C) 1999-2002 Kai Vehmanen (kai.vehmanen@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <iostream>
#include <string>

#include <qprogressdialog.h>
#include <qmessagebox.h>

#include <kvu_numtostr.h>
#include <eca-error.h>

#include "qeevent.h"

using std::cerr;
using std::endl;
using std::string;

QEEvent::QEEvent(ECA_CONTROL* ctrl) 
  : info_string_rep("Processing..."),
    ectrl(ctrl) 
{
}

QEEvent::~QEEvent(void) { 
  if (ectrl->is_running() == true) ectrl->stop();
  if (ectrl->is_connected() == true) ectrl->disconnect_chainsetup();
  if (initialized_cs_rep.empty() != true) {
    ectrl->select_chainsetup(initialized_cs_rep);
    if (ectrl->selected_chainsetup() == initialized_cs_rep) ectrl->remove_chainsetup();
  }
}

void QEEvent::init(const string& chainsetup, const string& chain) {
  // --------
  DBC_REQUIRE(ectrl != 0);
  // --------

  triggered_rep = false;
  result_rep = true;
  input_object_repp = 0;
  output_object_repp = 0;

  if (ectrl->is_running() == true) ectrl->stop();
  if (ectrl->is_connected() == true) ectrl->disconnect_chainsetup();

  if (initialized_cs_rep.empty() != true) {
    ectrl->select_chainsetup(initialized_cs_rep);
    if (ectrl->selected_chainsetup() == initialized_cs_rep) ectrl->remove_chainsetup();
  }
  if (ectrl->selected_chainsetup() != chainsetup) ectrl->add_chainsetup(chainsetup);
  else {
    ectrl->remove_chainsetup();
    ectrl->add_chainsetup(chainsetup);
  }
  ectrl->clear_chains();
  if (chain != "") ectrl->add_chain(chain);
  initialized_cs_rep = chainsetup;

  // --------
  DBC_ENSURE(ectrl->selected_chainsetup() == initialized_cs_rep);
  // --------
}

void QEEvent::blocking_start(void) {
  // --------
  DBC_REQUIRE(ectrl->is_valid() == true);
  DBC_REQUIRE(ectrl->is_selected() == true);
  DBC_REQUIRE(is_triggered() == false);
  // --------

  try {
    ectrl->connect_chainsetup();
    if (ectrl->is_connected() != true) {
      QMessageBox* mbox = new QMessageBox(0, "mbox");
      mbox->information(0, "ecawave", QString("Error during operation!"), 0);
      toggle_result(false);
      return;
    }
    
    set_audio_object_params();

    ectrl->start();
    toggle_triggered_state(true);

    struct timespec sleepcount;
    sleepcount.tv_sec = 0;
    sleepcount.tv_nsec = 20000000;
    
    int progress_length = static_cast<int>(ectrl->length_in_seconds_exact() * 10.0);
    if (progress_length == 0 && input_object_repp != 0) 
      progress_length = static_cast<int>(input_object_repp->length_in_seconds_exact() * 10.0);
    if (progress_length == 0) progress_length = (1 << 23);
      
    double progress_start = ectrl->position_in_seconds_exact();

    QProgressDialog progress (status_info().c_str(), 0,
			      static_cast<int>(progress_length - progress_start * 10), 0, 0, true);

    progress.setProgress(0);
    progress.show();
    while(ectrl->is_finished() == false) {
      nanosleep(&sleepcount, NULL);
      progress.setProgress(static_cast<int>((ectrl->position_in_seconds_exact() - progress_start) * 10.0));
    }
    toggle_triggered_state(false);
    toggle_result(true);
  }
  catch(ECA_ERROR& e) {
    cerr << "---\nlibecasound error while processing event: [" << e.error_section() << "] : \"" << e.error_message() << "\"\n\n";
    toggle_result(false);
  }

  // --------
  DBC_ENSURE(is_triggered() == false);
  // --------
}

void QEEvent::nonblocking_start(void) {
  // --------
  DBC_REQUIRE(ectrl->is_valid() == true);
  DBC_REQUIRE(ectrl->is_selected() == true);
  DBC_REQUIRE(is_triggered() == false);
  // --------

  ectrl->connect_chainsetup();
  if (ectrl->is_connected() != true) {
    QMessageBox* mbox = new QMessageBox(0, "mbox");
    mbox->information(0, "ecawave", QString("Error during operation!"), 0);
    toggle_result(false);
    return;
  }

  set_audio_object_params();

  ectrl->start();
  toggle_triggered_state(true);
  toggle_result(true);

  // --------
  DBC_ENSURE(is_triggered() == true || ectrl->is_running() == false);
  // --------
}

void QEEvent::set_default_audio_format(const string& name) {
  // --------
  DBC_REQUIRE(name.empty() == false);
  // --------
  try {
    ectrl->select_audio_input(name);
    if (ectrl->get_audio_input() != 0)
      ectrl->set_default_audio_format_to_selected_input();
    else {
      ectrl->select_audio_output(name);
      if (ectrl->get_audio_output() != 0)
	ectrl->set_default_audio_format_to_selected_output();
    }

    ECA_AUDIO_FORMAT aio_params = ectrl->default_audio_format();

    // cerr << "Setting sample rate of: " << aio_params.samples_per_second() << ".\n";
    ectrl->set_chainsetup_parameter("-sr:" + kvu_numtostr(aio_params.samples_per_second()));
  }
  catch(ECA_ERROR& e) {
    cerr << "---\nlibecasound error while setting event input: [" << e.error_section() << "] : \"" << e.error_message() << "\"\n\n";
  }
}

void QEEvent::set_audio_object_params(void)
{
 // --------
  DBC_REQUIRE(ectrl->is_running() == false);
  DBC_REQUIRE(input_startpos_rep >= 0);
  DBC_REQUIRE(output_startpos_rep >= 0);
  // --------

  ectrl->select_audio_input(input_name());
  input_object_repp = ectrl->get_audio_input();
  DBC_CHECK(input_object_repp != 0);
  ectrl->command("ai-set-position-samples " + kvu_numtostr(input_startpos_rep));

  ectrl->select_audio_output(output_name());
  output_object_repp = ectrl->get_audio_output();
  DBC_CHECK(output_object_repp != 0);
  ectrl->command("ao-set-position-samples " + kvu_numtostr(output_startpos_rep));

  // --------
  DBC_ENSURE(input_startpos_rep == input_object_repp->position_in_samples());
  DBC_ENSURE(output_startpos_rep == output_object_repp->position_in_samples() ||
	     output_object_repp->supports_seeking() != true);
  // --------
}

void QEEvent::set_input(const string& name) {
  // --------
  DBC_REQUIRE(name.empty() == false);
  // --------
  try {
    ectrl->add_audio_input(name);
    input_object_repp = ectrl->get_audio_input();
    DBC_CHECK(input_object_repp != 0);
    input_name_rep = input_object_repp->label();
  }
  catch(ECA_ERROR& e) {
    cerr << "---\nlibecasound error while setting event input: [" << e.error_section() << "] : \"" << e.error_message() << "\"\n\n";
  }

  // --------
  DBC_ENSURE(input_object_repp != 0);
  // --------
}

void QEEvent::set_input_position(long int pos)
{
  // --------
  DBC_REQUIRE(pos >= 0);
  // --------
  input_startpos_rep = pos;
}

void QEEvent::set_output(const string& name) {
  // --------
  DBC_REQUIRE(name.empty() == false);
  // --------
  try {
    ectrl->add_audio_output(name);
    output_object_repp = ectrl->get_audio_output();
    DBC_CHECK(output_object_repp != 0);
    output_name_rep = output_object_repp->label();
  }
  catch(ECA_ERROR& e) {
    cerr << "---\nlibecasound error while setting event output: [" << e.error_section() << "] : \"" << e.error_message() << "\"\n\n";
  }
}

void QEEvent::set_output_position(long int pos) {
  // --------
  DBC_REQUIRE(pos >= 0);
  // --------
  output_startpos_rep = pos;
}

void QEEvent::set_length(long int pos) {
  // --------
  DBC_REQUIRE(ectrl->is_running() == false);
  // --------
  ectrl->set_chainsetup_processing_length_in_samples(pos);
}
