/*
    Eiciel - GNOME editor of ACL file permissions.
    Copyright (C) 2004-2005 Roger Ferrer Ibez

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef GESTOR_ACL_HPP
#define GESTOR_ACL_HPP

#include <config.hpp>

#include <string>
#include <vector>
#include <iostream>
#include <gettext.h>

#include <glibmm.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/acl.h>
#ifdef HAVE_ACL_LIBACL_H
 #include <acl/libacl.h>
#endif
#include <unistd.h>
#include <errno.h>
#include <vector>
#include <iostream>
#include <pwd.h>
#include <grp.h>
#include <sstream>
#include <algorithm>
#include <gettext.h>

using namespace std;

#ifdef HAVE_ACL_GET_PERM
   #define ACL_GET_PERM acl_get_perm
#else 
 #ifdef HAVE_ACL_GET_PERM_NP
   #define ACL_GET_PERM acl_get_perm_np 
 #else
   #error "An acl_get_perm-like function is needed" 
 #endif
#endif
  


class GestorACLException
{
	public:
		GestorACLException(Glib::ustring missatge) : missatge(missatge) 
		{
		}

		Glib::ustring getMessage() const
		{ 
			return missatge; 
		}
	private:
		Glib::ustring missatge;
};

struct permisos_t 
{
	bool lectura;
	bool escriptura;
	bool execucio;
	permisos_t(char c)
	{
		lectura = (c & 04);
		escriptura = (c & 02);
		execucio = (c & 01);
	}
	permisos_t(bool lect, bool escr, bool execu)
		: lectura(lect), escriptura(escr), execucio(execu) {}
	permisos_t() : 
		lectura(false), escriptura(false), execucio(false) {}
};


struct entrada_acl : permisos_t
{
	int qualificador; // El grup o l'usuari
	string nom; // El nom simbolic del qualificador
	bool nomValid;
};

class GestorACL
{
	private:
		string nomFitxer;
		bool esDir;
		uid_t uidPropietari;
		string nomPropietari;
		permisos_t permisosPropietari;
		string nomGrup;
		permisos_t permisosGrup;
		permisos_t permisosAltres;
		bool hiHaMascara;
		permisos_t mascaraACL;
		vector<entrada_acl> aclUsuari;
		vector<entrada_acl> aclGrup;
		vector<entrada_acl> defaultACLUsuari;
		vector<entrada_acl> defaultACLGrup;
		permisos_t defaultUsuari; // Camps default
		bool hiHaDefaultUsuari;
		permisos_t defaultGrup;
		bool hiHaDefaultGrup;
		permisos_t defaultAltres;
		bool hiHaDefaultAltres;
		permisos_t defaultMascara;
		bool hiHaDefaultMascara;
		string textACLAccess;
		string textACLDefault;

		void obtenirPermisosUGO();
		void obtenirACLAccess();
		void obtenirACLDefault();
		void generarRepresentacioTextual();
		string permisStr(permisos_t& p);
		string escriureNom(entrada_acl& eacl);
		void modificarACLGenerica(const string& nom, vector<entrada_acl>& llistACL, const permisos_t& permisos);
		void eliminarACLGenerica(const string& nom, vector<entrada_acl>& llistaACL);

		void aplicarCanvisAlFitxer();
		void calculaMascaraAccess();
		void actualitzarCanvisACLAccess();
		void actualitzarCanvisACLDefault();
		void emplenaACLDefaultNecessaries();

		class EquivalenciaACL
		{
			private:
				string qualificador;
			public:
				EquivalenciaACL(const string& qualif)
					: qualificador(qualif) {}
				bool operator ()(entrada_acl& a)
				{
					return (a.nomValid && (a.nom == qualificador));
				}
		};
	public:
		const static int PERMIS_LECTURA = 0;
		const static int PERMIS_ESCRIPTURA = 1;
		const static int PERMIS_EXECUCIO = 2;

		GestorACL(const string& nomFitxer) throw (GestorACLException);
		
		string getACLAccess() const { return textACLAccess; }
		string getACLDefault() const { return textACLDefault; }
		
		bool esDirectori() const { return esDir; }
		
		void modificarACLUsuari(const string& nomUsuari, const permisos_t& permisos);
		void modificarACLGrup(const string& nomGrup, const permisos_t& permisos);
		
		void modificarDefaultACLUsuari(const string& nomUsuari, const permisos_t& permisos);
		void modificarDefaultACLGrup(const string& nomGrup, const permisos_t& permisos);

		void modificarPermisosPropietari(permisos_t& p);
		void modificarPermisosGrup(permisos_t& p);
		void modificarPermisosAltres(permisos_t& p);
		void modificarMascara(permisos_t& p);

		void modificarPermisosDefaultPropietari(permisos_t& p);
		void modificarPermisosDefaultGrup(permisos_t& p);
		void modificarPermisosDefaultAltres(permisos_t& p);
		void modificarMascaraDefault(permisos_t& p);
		
		void eliminarACLUsuari(const string& nomUsuari);
		void eliminarACLGrup(const string& nomGrup);
		void eliminarDefaultACLUsuari(const string& nomUsuari);
		void eliminarDefaultACLGrup(const string& nomGrup);

		void buidarTotesLesACL();
		void buidarACLDefecte();

		void creaACLDefecte();

		vector<entrada_acl> donaACLAccessUsuari() const { return aclUsuari; }
		vector<entrada_acl> donaACLAccessGrup() const { return aclGrup; }
		vector<entrada_acl> donaACLDefaultUsuari() const { return defaultACLUsuari; }
		vector<entrada_acl> donaACLDefaultGrup() const { return defaultACLGrup; }
		permisos_t donaMascara() { return mascaraACL; }

		permisos_t donaUsuari() { return permisosPropietari; }
		permisos_t donaGrup() { return permisosGrup; }
		permisos_t donaAltres() { return permisosAltres; }
		
		permisos_t donaDefaultUsuari() { return defaultUsuari; }
		permisos_t donaDefaultGrup() { return defaultGrup; }
		permisos_t donaDefaultAltres() { return defaultAltres; }
		permisos_t donaDefaultMascara() { return defaultMascara; }

		string donaNomPropietari() { return nomPropietari; }
		string donaNomGrup() { return nomGrup; }
		uid_t donaUidPropietari() { return uidPropietari; }

		bool teMascara() const { return hiHaMascara; }
		bool teDefaultUsuari() const { return hiHaDefaultUsuari; }
		bool teDefaultGrup() const { return hiHaDefaultGrup; }
		bool teDefaultAltres() const { return hiHaDefaultAltres; }
		bool teDefaultMascara() const { return hiHaDefaultMascara; }

};

#endif
