/* BTP Emcast - Banana Tree Protocol Emcast Interface
 * Copyright (C) 2001  The Regents of the University of Michigan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <emcast-protocol.h>

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <limits.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <time.h>
#include <glib.h>

#include "btp.h"
#include "btp_debug.h"


static gboolean peer_dump (gpointer data);
static gboolean in_func (GIOChannel* source, GIOCondition condition, gpointer data);

void packet_func (Btp* btp, const void* buffer, guint length, gpointer user_data);
void error_func (Btp* btp, gpointer user_data);

static int emjoin (char* url);
static int emleave (void);
static int emsend (char* buf, unsigned short len);
static int emgetopt (char* optname, void* optval, unsigned short* optlen);
static int emsetopt (char* optname, void* optval, unsigned short optlen);

static void sig_pipe_cb (int ignore);

static Emfuncs emfuncs = { NULL,
			   emjoin,
			   emleave,
			   emsend,
			   emgetopt,	
			   emsetopt, 1, 1};


static BPeer*     bpeer;
static Btp*       btp;
static int        fd_fifo;


int
main (int argc, char* argv[])
{
  GIOChannel* in;
  guint in_watch;
  GMainLoop* main_loop;

  /* Seed RNG */
  srand(time(NULL) + getpid());

  /* Abort on sigpipe */
  if (0)
    signal (SIGPIPE, sig_pipe_cb);

  /* Print peer occasionally */
  g_timeout_add (60*1000, peer_dump, NULL);

  /* Watch STDIN */
  in = g_io_channel_unix_new (STDIN_FILENO);
  g_assert (in);

  in_watch = g_io_add_watch (in, G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
			     in_func, NULL);

  main_loop = g_main_new(FALSE);
  g_main_run (main_loop);

  exit (EXIT_SUCCESS);
  return 0;
}



/* **************************************** */

static gboolean
peer_dump (gpointer data)
{
  time_t t;
  char* time_str;

  if (!(btp_debug_flags & (1<<8)))
    return FALSE;

  t = time (NULL);
  t = mktime (gmtime(&t));
  time_str = ctime(&t);         /* don't free string */
  time_str[strlen(time_str) - 1] = '\0';

  fprintf (stderr, "time: %s UTC\n", time_str);

  if (bpeer)
    btp_print (stderr, bpeer);

  return TRUE;
}



gboolean
in_func (GIOChannel* source, GIOCondition condition, gpointer data)
{
  int rv;

  if (condition & (G_IO_ERR | G_IO_HUP | G_IO_NVAL))
    exit (EXIT_SUCCESS);
      
  rv = emcast_loop_once (&emfuncs, STDIN_FILENO, STDOUT_FILENO, &fd_fifo);
  if (rv != 0)
    exit (EXIT_SUCCESS);

  return TRUE;
}


/* ******************** */

void
packet_func (Btp* btp, const void* buffer, guint length, gpointer user_data)
{
  int rv;

  rv = emcast_handler_recv (fd_fifo, buffer, length, NULL, 0);

  if (rv != length)
    exit (EXIT_FAILURE);
}


void
error_func (Btp* btp, gpointer user_data)
{
  fprintf (stderr, "btp-emcast: group failure\n");
  exit (EXIT_FAILURE);
}


/* ******************** */


static int
emjoin (char* url)
{
  GURL*   	gurl;
  GInetAddr*	iface;
  
  if (btp)
    return 1;

  gurl = gnet_url_new (url);
  if (!gurl)
    return 1;

  /* Attempt to create a peer named by the URL.  If it's local and the
     port isn't in use, it will succeed.  Otherwise, it will fail and
     we will attempt to join the group. */
  iface = gnet_inetaddr_new (gurl->hostname, gurl->port? gurl->port: BTP_PORT);
  if (!iface)
    return 1;
  bpeer = b_peer_new (gurl->hostname, iface, TRUE);
  gnet_inetaddr_delete (iface);

  /* Create */
  if (bpeer)
    {
      btp = btp_create (bpeer, gurl->resource);
    }
  /* Join */
  else
    {
      gchar*     my_hostname;
      GInetAddr* my_iface;
      gchar*     my_port_str;
      gint       my_port;
      gboolean	 my_force_port;

      /* Get my interface */
      my_iface = gnet_inetaddr_autodetect_internet_interface ();
      if (!my_iface)
	{
	  fprintf (stderr, "btp-emcast: Could not find Internet interface\n");
	  return 1;
	}

      /* Get my hostname */
      my_hostname = g_getenv ("BTP_DOMAINNAME");
      if (!my_hostname)
	my_hostname = g_getenv ("DOMAINNAME");
      if (!my_hostname)
	{
	  my_hostname = gnet_inetaddr_get_name (my_iface);
	  if (!my_hostname || !gnet_inetaddr_is_internet_domainname (my_hostname))
	    {
	      g_free (my_hostname);
	      my_hostname = gnet_inetaddr_get_canonical_name (my_iface);
	    }
	}
      if (!my_hostname)
	{
	  gnet_inetaddr_delete (my_iface);
	  fprintf (stderr, "btp-emcast: Could not get hostname\n");
	  return 1;
	}

      /* Get my port */
      my_port = 0;
      my_force_port = FALSE;
      my_port_str = g_getenv ("BTP_PORT");
      if (my_port_str && (my_port = atoi(my_port_str)) != 0)
	{
	  gnet_inetaddr_set_port (my_iface, my_port);
	  my_force_port = TRUE;
	}

      /* Create bpeer again.  Use any port. */
      bpeer = b_peer_new (my_hostname, my_iface, my_force_port);
      g_free (my_hostname);
      gnet_inetaddr_delete (my_iface);
      if (!bpeer)
	{
	  fprintf (stderr, "btp-emcast: Could not create peer\n");
	  return 1;
	}

      btp = btp_join (bpeer, gurl);
    }

  gnet_url_delete (gurl);
  if (btp == NULL)
    return 1;
  btp->packet_func = packet_func;
  btp->error_func =  error_func;

  return 0;
}


static int
emleave (void)
{
  if (!btp)
    return 1;

  btp_leave (btp);
  btp = NULL;

  b_peer_delete (bpeer);
  bpeer = NULL;

  return 0;
}


static int
emsend (char* buf, unsigned short len)
{
  if (!btp)
    return 1;

  if (len > 16096)
    return 1;

  btp_send (btp, buf, len);
  
  return 0;
}


static int
emgetopt (char* optname, void* optval, unsigned short* optlen)
{
  if (!strcmp(optname, "btp_debug_flags"))
    {
      (*(int*) optval) = g_htonl(btp_debug_flags);
      *optlen = sizeof(int);

      return 0;
    }
  else if (!strcmp(optname, "btp_max_degree"))
    {
      (*(int*) optval) = g_htonl(btp_get_max_degree (btp));
      *optlen = sizeof(int);
    }
  else if (!strcmp(optname, "btp_follow_nodes"))
    {
      (*(int*) optval) = g_htonl(btp_get_follow_nodes(btp));
      *optlen = sizeof(int);
    }
  else if (!strcmp(optname, "btp_use_shortcuts"))
    {
      (*(int*) optval) = g_htonl(btp_get_use_shortcuts(btp));
      *optlen = sizeof(int);
    }

  return 1;
}


static int
emsetopt (char* optname, void* optval, unsigned short optlen)
{
  if (!strcmp(optname, "btp_debug_flags") && optlen == sizeof(int))
    {
      btp_debug_flags = g_ntohl(*(int*)optval);
      return 0;
    }
  else if (!strcmp(optname, "btp_max_degree") && optlen == sizeof(int))
    {
      btp_set_max_degree (btp, g_ntohl(*(int*)optval));
      return 0;
    }
  else if (!strcmp(optname, "btp_follow_nodes") && optlen == sizeof(int))
    {
      gboolean follow = FALSE;
      if (*(int*)optval)
	follow = TRUE;

      btp_set_follow_nodes (btp, follow);

      return 0;
    }
  else if (!strcmp(optname, "btp_use_shortcuts") && optlen == sizeof(int))
    {
      gboolean use_shortcuts = FALSE;
      if (*(int*)optval)
	use_shortcuts = TRUE;

      btp_set_use_shortcuts (btp, use_shortcuts);

      return 0;
    }


  return 1;
}


/* ******************** */

static void
sig_pipe_cb (int ignore)
{
  fprintf (stderr, "btp-emcast: Caught SIGPIPE signal. Exiting.\n");
  exit (EXIT_FAILURE);
}
