#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <time.h>
#include <utime.h>
#include <signal.h>
#include <sys/stat.h>
#ifdef HAVE_LIBZIP
# include <zip.h>
#endif
#include <gtk/gtk.h>
#include <unistd.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"

#include "cdialog.h"
#include "progressdialog.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_archive_obj.h"
#include "edv_archive_extract.h"
#include "edv_archive_extract_zip.h"
#include "endeavour2.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"


gint EDVArchExtractPKZip(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive,
	gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define UNLINK(p)	(((p) != NULL) ? (gint)unlink((const char *)(p)) : -1)
#define INTERRUPT(i)	(((i) > 0) ? (gint)kill((int)(i), SIGINT) : -1)

#define ISCR(c)		(((c) == '\n') || ((c) == '\r'))


/*
 *	Extract object from a PKZip archive.
 *
 *	Inputs assumed valid.
 */
gint EDVArchExtractPKZip(
	edv_core_struct *core,
	const gchar *arch_path,
	GList *objs_list, const gint nobjs,
	const gboolean extract_all,
	const gchar *dest_path,
	GList **new_paths_list_rtn,
	const gchar *password,
	GtkWidget *toplevel,
	const gboolean show_progress, const gboolean interactive, gboolean *yes_to_all,
	const gboolean preserve_directories,
	const gboolean preserve_timestamps
)
{
#ifdef HAVE_LIBZIP
	struct stat stat_buf;
	struct zip *archive;
	gint status, zip_error_code;
	gulong cur_size, total_size;
	gchar *src_path, *tar_path, *parent;
	GList *glist;
	edv_object_type type;
	edv_archive_object_struct *obj;

	/* Open the PKZip archive for reading */
	archive = zip_open(arch_path, 0, &zip_error_code);
	if(archive == NULL)
	{
	    const gint sys_error_code = (gint)errno;
	    gchar *msg, err_msg[1024];
	    zip_error_to_str(
		err_msg, sizeof(err_msg),
		zip_error_code, sys_error_code
	    );
	    msg = g_strdup_printf(
"Unable to open the PKZip Archive for reading:\n\
\n\
    %s\n\
\n\
%s.",
		arch_path, err_msg
	    );
	    EDVArchExtractCopyErrorMessage(core, msg);
	    g_free(msg);
	    return(-1);
	}

	/* Iterate through the list of objects to extract in order
	 * to calculate the total uncompressed size
	 */
	total_size = 0l;
	for(glist = objs_list; glist != NULL; glist = g_list_next(glist))
	{
	    obj = EDV_ARCHIVE_OBJECT(glist->data);
	    if(obj == NULL)
		continue;

	    /* Count only the sizes of files and links */
	    type = obj->type;
	    if((type == EDV_OBJECT_TYPE_FILE) ||
	       (type == EDV_OBJECT_TYPE_LINK)
	    )
		total_size += obj->size;
	}

	/* Iterate through the list of objects to extract and
	 * extract each one
	 */
	cur_size = 0l;
	status = 0;
	for(glist = objs_list; glist != NULL; glist = g_list_next(glist))
	{
	    obj = EDV_ARCHIVE_OBJECT(glist->data);
	    if(obj == NULL)
		continue;

	    /* Get the path of the object in the archive */
	    src_path = STRDUP(obj->full_path);
	    if(STRISEMPTY(src_path))
	    {
		g_free(src_path);
		continue;
	    }

	    /* Format the path of the object to extract */
	    if(preserve_directories)
		tar_path = STRDUP(PrefixPaths(
		    (const char *)dest_path,
		    (const char *)src_path
		));
	    else
		tar_path = STRDUP(PrefixPaths(
		    (const char *)dest_path,
		    (const char *)g_basename(src_path)
		));
	    if(STRISEMPTY(tar_path))
	    {
		g_free(src_path);
		g_free(tar_path);
		core->archive_last_error =
"Unable to generate the extracted object's path.";
		status = -1;
		break;
	    }

	    EDVSimplifyPath(tar_path);

	    /* Update the progress dialog message to display the
	     * current object being extracted
	     */
	    if(show_progress)
	    {
		gchar	*p1 = EDVShortenPath(
		    src_path,
		    EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
		),
			*p2 = EDVShortenPath(
		    tar_path,
		    EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
		),
			*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Extraer:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Extraire:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Extrahieren:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Estrarre:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onttrekken:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Extrair:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Trekking Ut:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Extracting:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
		    , p1, p2
		);
		g_free(p1);
		g_free(p2);
		EDVArchExtractMapProgressDialog(
		    msg,
		    (total_size > 0l) ?
			((gfloat)cur_size / (gfloat)total_size) : -1.0f,
		    toplevel, FALSE
		);
		g_free(msg);

		if(ProgressDialogStopCount() > 0)
		{
		    g_free(src_path);
		    g_free(tar_path);
		    status = -4;
		    break;
		}
	    }

	    /* Need to create the parent directory(ies)? */
	    parent = g_dirname(tar_path);
	    if(parent != NULL)
	    {
		const gint	stat_status = lstat((const char *)parent, &stat_buf),
				error_code = errno;
		if((stat_status != 0) && (error_code == ENOENT))
		{
		    /* Create each parent directory(ies) and add
		     * them to the list of extracted objects
		     */
		    if(EDVArchExtractRMkDir(parent, new_paths_list_rtn))
		    {
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to create the parent directories to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
			    tar_path, arch_path, g_strerror(error_code)
			);
			EDVArchExtractCopyErrorMessage(core, msg);
			g_free(msg);
			g_free(parent);
			g_free(src_path);
			g_free(tar_path);
			status = -1;
			break;
		    }
		}
		g_free(parent);
	    }

	    /* Report the progress? */
	    if(show_progress && ProgressDialogIsQuery())
	    {
		ProgressDialogUpdate(
		    NULL, NULL, NULL, NULL,
		    (total_size > 0l) ?
			((gfloat)cur_size / (gfloat)total_size) : -1.0f,
		    EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
		{
		    g_free(src_path);
		    g_free(tar_path);
		    status = -4;
		    break;
		}
	    }

#define QUERY_CONTINUE_EXTRACT	{			\
 /* Count this object even after a failed extraction */	\
 if(new_paths_list_rtn != NULL)				\
  *new_paths_list_rtn = g_list_append(			\
   *new_paths_list_rtn, STRDUP(tar_path)		\
  );							\
							\
 /* Need to query the user? */				\
 if(interactive && !(*yes_to_all)) {			\
  gint response;					\
  gchar *msg = g_strdup_printf(				\
"An error occured while extracting the object:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n\
\n\
Continue extracting subsequent objects?",		\
   src_path,						\
   tar_path						\
  );							\
  EDVPlaySoundError(core);				\
  CDialogSetTransientFor(toplevel);			\
  response = CDialogGetResponse(			\
   "Extract Failed",					\
   msg,							\
   NULL,						\
   CDIALOG_ICON_ERROR,					\
   CDIALOG_BTNFLAG_YES |				\
    CDIALOG_BTNFLAG_YES_TO_ALL |			\
   CDIALOG_BTNFLAG_NO,					\
   CDIALOG_BTNFLAG_YES					\
  );							\
  g_free(msg);						\
  CDialogSetTransientFor(NULL);				\
							\
  /* Stop extracting? */				\
  if((response == CDIALOG_RESPONSE_NO) ||		\
     (response == CDIALOG_RESPONSE_CANCEL) ||		\
     (response == CDIALOG_RESPONSE_NOT_AVAILABLE)	\
  )							\
  {							\
   g_free(src_path);					\
   g_free(tar_path);					\
   break;						\
  }							\
							\
  /* Yes to all? */					\
  if(response == CDIALOG_RESPONSE_YES_TO_ALL)		\
   *yes_to_all = TRUE;					\
 }							\
							\
 g_free(src_path);					\
 g_free(tar_path);					\
 continue;						\
}

	    /* Directory? */
	    if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
	    {
		/* Extract the directory by creating it as needed */

		/* Does the directory not already exist? */
		if(stat((const char *)tar_path, &stat_buf))
		{
		    /* Directory does not exist, create it */
		    const guint m = EDVGetUMask();

		    /* Create the directory */
		    if(mkdir(
			(const char *)tar_path,
			(~m) &
			(S_IRUSR | S_IWUSR | S_IXUSR |
			 S_IRGRP | S_IWGRP | S_IXGRP |
			 S_IROTH | S_IWOTH | S_IXOTH)
		    ))
		    {
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
			    tar_path, arch_path, g_strerror(error_code)
			);
			EDVArchExtractCopyErrorMessage(core, msg);
			g_free(msg);
			status = -1;
			QUERY_CONTINUE_EXTRACT
		    }

		    /* Append the target path to the list of extracted objects */
		    if(new_paths_list_rtn != NULL)
			*new_paths_list_rtn = g_list_append(
			    *new_paths_list_rtn, STRDUP(tar_path)
			);
		}
		else
		{
		    /* Object exists, check if it is not a directory */
#ifdef S_ISDIR
		    if(!S_ISDIR(stat_buf.st_mode))
		    {
			/* Object exists and is not a directory, remove
			 * the object and create the directory
			 */
			const guint m = EDVGetUMask();

			/* Remove the existing object */
			if(EDVArchExtractRemove(tar_path))
			{
			    const gint error_code = (gint)errno;
			    gchar *msg = g_strdup_printf(
"Unable to remove the existing object:\n\
\n\
    %s\n\
\n\
To extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
				tar_path,
				src_path,
				arch_path,
				g_strerror(error_code)
			    );
			    EDVArchExtractCopyErrorMessage(core, msg);
			    g_free(msg);
			    status = -1;
			    QUERY_CONTINUE_EXTRACT
			}

			/* Create the directory */
			if(mkdir(
			    (const char *)tar_path,
			    (~m) &
			    (S_IRUSR | S_IWUSR | S_IXUSR |
			     S_IRGRP | S_IWGRP | S_IXGRP |
			     S_IROTH | S_IWOTH | S_IXOTH)
			))
			{
			    const gint error_code = (gint)errno;
			    gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
				tar_path, arch_path, g_strerror(error_code)
			    );
			    EDVArchExtractCopyErrorMessage(core, msg);
			    g_free(msg);
			    status = -1;
			    QUERY_CONTINUE_EXTRACT
			}

			/* Append the target path to the list of extracted objects */
			if(new_paths_list_rtn != NULL)
			    *new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn, STRDUP(tar_path)
			    );
		    }
#endif	/* S_ISDIR */
		}
	    }
	    /* Link? */
 	    else if(obj->type == EDV_OBJECT_TYPE_LINK)
	    {
		struct zip_file *src_fp;
		gint bytes_read;
		const gulong value_len = MAX(obj->size, 0l);
		gchar *value, *src_path_pc;

		/* Open the source file in the PKZip archive */
		src_path_pc = g_strconcat(src_path, "@", NULL);
		src_fp = zip_fopen(archive, src_path_pc, 0);
		g_free(src_path_pc);
		if(src_fp == NULL)
		{
		    gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
			src_path, arch_path, zip_strerror(archive)
		    );
		    EDVArchExtractCopyErrorMessage(core, msg);
		    g_free(msg);
		    status = -1;
		    QUERY_CONTINUE_EXTRACT
		}

		/* Allocate the link's target value */
		value = (gchar *)g_malloc((value_len + 1) * sizeof(gchar));
		if(value == NULL)
		{
		    core->archive_last_error = "Memory allocation error.";
		    zip_fclose(src_fp);
		    g_free(src_path);
		    g_free(tar_path);
		    status = -3;
		    break;
		}

		/* Read the link's target value from the source
		 * file in the PKZip archive
		 */
		if(value_len > 0)
		{
		    bytes_read = (gint)zip_fread(src_fp, value, value_len);
		    if(bytes_read < 0)
		    {
			gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
			    src_path, arch_path, zip_file_strerror(src_fp)
			);
			EDVArchExtractCopyErrorMessage(core, msg);
			g_free(msg);
			g_free(value);
			zip_fclose(src_fp);
			status = -1;
			QUERY_CONTINUE_EXTRACT
		    }
		}
		else
		{
		    bytes_read = 0;
		}

		if((gulong)bytes_read < value_len)
		    value[bytes_read] = '\0';
		else
		    value[value_len] = '\0';

		/* Close the source and target files */
		zip_fclose(src_fp);

		/* Remove the target object in case it exists */
		if(EDVArchExtractRemove(tar_path))
		{
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"Unable to remove the existing object:\n\
\n\
    %s\n\
\n\
To extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
			tar_path,
			src_path,
			arch_path,
			g_strerror(error_code)
		    );
		    EDVArchExtractCopyErrorMessage(core, msg);
		    g_free(msg);
		    g_free(value);
		    status = -1;
		    QUERY_CONTINUE_EXTRACT
		}

		/* Create the link */
		if(symlink(
		    (const char *)value,
		    (const char *)tar_path
		) != 0)
		{
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
			tar_path, arch_path, g_strerror(error_code)
		    );
		    EDVArchExtractCopyErrorMessage(core, msg);
		    g_free(msg);
		    g_free(value);
		    status = -1;
		    QUERY_CONTINUE_EXTRACT
		}

		/* Delete the link's target value */
		g_free(value);

		cur_size += (gulong)bytes_read;

		/* Append the target path to the list of extracted objects */
		if(new_paths_list_rtn != NULL)
		    *new_paths_list_rtn = g_list_append(
			*new_paths_list_rtn, STRDUP(tar_path)
		    );
	    }
	    /* All else extract the file */
 	    else
	    {
		FILE *tar_fp;
		struct zip_file *src_fp;
		gint tar_fd;
		const guint m = EDVGetUMask();
		gulong block_size;

		/* Open the source file in the PKZip archive */
		src_fp = zip_fopen(archive, src_path, 0);
		if(src_fp == NULL)
		{
		    gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
			src_path,
			arch_path,
			zip_strerror(archive)
		    );
		    EDVArchExtractCopyErrorMessage(core, msg);
		    g_free(msg);
		    status = -1;
		    QUERY_CONTINUE_EXTRACT
		}

		/* Remove the target object in case it exists */
		if(EDVArchExtractRemove(tar_path))
		{
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"Unable to remove the existing object:\n\
\n\
    %s\n\
\n\
To extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
			tar_path,
			src_path,
			arch_path,
			g_strerror(error_code)
		    );
		    EDVArchExtractCopyErrorMessage(core, msg);
		    g_free(msg);
		    zip_fclose(src_fp);
		    status = -1;
		    QUERY_CONTINUE_EXTRACT
		}

		/* Open the target file for writing */
		tar_fp = fopen((const char *)tar_path, "wb");
		if(tar_fp == NULL)
		{
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
			tar_path,
			arch_path,
			g_strerror(error_code)
		    );
		    EDVArchExtractCopyErrorMessage(core, msg);
		    g_free(msg);
		    zip_fclose(src_fp);
		    status = -1;
		    QUERY_CONTINUE_EXTRACT
		}

		/* Append the target path to the list of extracted objects */
		if(new_paths_list_rtn != NULL)
		    *new_paths_list_rtn = g_list_append(
			*new_paths_list_rtn, STRDUP(tar_path)
		    );

		tar_fd = (gint)fileno(tar_fp);

		/* Set the file permissions */
		fchmod(
		    (int)tar_fd,
		    (~m) &
		    (S_IRUSR | S_IWUSR |
		     S_IRGRP | S_IWGRP |
		     S_IROTH | S_IWOTH)
		);

		/* Calculate the block size */
		if(fstat((int)tar_fd, &stat_buf))
		{
		    block_size = 0l;
		}
		else
		{
#if defined(_WIN32)
		    block_size = 1024l;
#else
		    block_size = (gulong)stat_buf.st_blksize;
#endif
		}
		if(block_size > 0l)
		{
		    /* Extract one block size at a time */
		    gint bytes_read, bytes_written;
		    const gint read_buf_len = (gint)block_size;
		    guint8 *read_buf = (guint8 *)g_malloc(
			block_size * sizeof(guint8)
		    );
		    if(read_buf == NULL)
		    {
			core->archive_last_error = "Memory allocation error.";
			zip_fclose(src_fp);
			fclose(tar_fp);
			g_free(src_path);
			g_free(tar_path);
			status = -3;
			break;
		    }

		    while(TRUE)
		    {
			/* Read one block from the source file in the archive */
			bytes_read = (gint)zip_fread(
			    src_fp,
			    read_buf, (int)read_buf_len
			);
			if(bytes_read <= 0)
			{
			    if(bytes_read < 0)
			    {
				gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
				    src_path,
				    arch_path,
				    zip_file_strerror(src_fp)
				);
				EDVArchExtractCopyErrorMessage(core, msg);
				g_free(msg);
				status = -1;
			    }
			    break;
			}

			/* Write the block to the file */
			bytes_written = (gint)fwrite(
			    read_buf, sizeof(guint8), (size_t)bytes_read, tar_fp
			);

			/* Add to the current size */
			if(bytes_written > 0)
			    cur_size += (gulong)bytes_written;

			/* Write error? */
			if(bytes_written != bytes_read)
			{
			    const gint error_code = (gint)errno;
			    gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
				tar_path,
				arch_path,
				g_strerror(error_code)
			    );
			    EDVArchExtractCopyErrorMessage(core, msg);
			    g_free(msg);
			    status = -1;
			    break;
			}

			/* Update progress */
			if(show_progress && ProgressDialogIsQuery())
			{
			    ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				(total_size > 0l) ?
		      ((gfloat)cur_size / (gfloat)total_size) : -1.0f,
				EDV_DEF_PROGRESS_BAR_TICKS, TRUE
			    );
			    if(ProgressDialogStopCount() > 0)
			    {
				status = -4;
				break;
			    }
			}
		    }

		    /* Delete the read buffer */
		    g_free(read_buf);
		}
		else
		{
		    /* File system IO block size not available,
		     * extract one character at a time
		     */
		    gint bytes_read, bytes_written;
		    const gint buf_len = 1;
		    guint8 buf[1];

		    while(TRUE)
		    {
			/* Read one character from the source file in the archive */
			bytes_read = (gint)zip_fread(src_fp, buf, (int)buf_len);
			if(bytes_read <= 0)
			{
			    if(bytes_read < 0)
			    {
				gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
				    src_path, arch_path, zip_file_strerror(src_fp)
				);
				EDVArchExtractCopyErrorMessage(core, msg);
				g_free(msg);
				status = -1;
			    }
			    break;
			}

			/* Write the character to the file */
			if(fputc((int)buf[0], tar_fp) != EOF)
			    bytes_written = 1;
			else
			    bytes_written = 0;

			/* Add to the current size */
			if(bytes_written > 0)
			    cur_size += (gulong)bytes_written;

			/* Write error? */
			if(bytes_written != bytes_read)
			{
			    const gint error_code = (gint)errno;
			    gchar *msg = g_strdup_printf(
"Unable to extract:\n\
\n\
    %s\n\
\n\
From:\n\
\n\
    %s\n\
\n\
%s.",
				tar_path, arch_path, g_strerror(error_code)
			    );
			    EDVArchExtractCopyErrorMessage(core, msg);
			    g_free(msg);
			    status = -1;
			    break;
			}

			/* Update progress */
			if(show_progress && ProgressDialogIsQuery())
			{
			    ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				(total_size > 0l) ?
		      ((gfloat)cur_size / (gfloat)total_size) : -1.0f,
				EDV_DEF_PROGRESS_BAR_TICKS, TRUE
			    );
			    if(ProgressDialogStopCount() > 0)
			    {
				status = -4;
				break;
			    }
			}
		    }
		}

		/* Close the source and target files */
		zip_fclose(src_fp);
		fclose(tar_fp);

		/* Did an error occure or user aborted while writing
		 * the extracted object?
		 */
		if(status != 0)
		{
		    if(status == -4)
		    {
			break;
		    }
		    else
		    {
			QUERY_CONTINUE_EXTRACT
		    }
		}
	    }

	    /* Report the progress? */
	    if(show_progress && ProgressDialogIsQuery())
	    {
		ProgressDialogUpdate(
		    NULL, NULL, NULL, NULL,
		    (total_size > 0l) ?
			((gfloat)cur_size / (gfloat)total_size) : -1.0f,
		    EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
		{
		    g_free(src_path);
		    g_free(tar_path);
		    status = -4;
		    break;
		}
	    }

	    /* Preserve timestamps? */
	    if(preserve_timestamps &&
	       (obj->type != EDV_OBJECT_TYPE_LINK)
	    )
	    {
		struct utimbuf utime_buf;
		utime_buf.actime = (time_t)obj->access_time;
		utime_buf.modtime = (time_t)obj->modify_time;
		utime((const char *)tar_path, &utime_buf);
	    }

	    /* Report the final progress for this object? */
	    if(show_progress && ProgressDialogIsQuery())
	    {
		ProgressDialogUpdate(
		    NULL, NULL, NULL, NULL,
		    (total_size > 0l) ?
			((gfloat)cur_size / (gfloat)total_size) : -1.0f,
		    EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		);
		if(ProgressDialogStopCount() > 0)
		{
		    g_free(src_path);
		    g_free(tar_path);
		    status = -4;
		    break;
		}
	    }

	    g_free(src_path);
	    g_free(tar_path);

#undef QUERY_CONTINUE_EXTRACT
	}

	/* Close the PKZip archive */
	if(zip_close(archive))
	{
	    if((status == 0) || (status == -4))
	    {
		gchar *msg = g_strdup_printf(
"Unable to close the PKZip Archive:\n\
\n\
    %s\n\
\n\
%s.",
		    arch_path, zip_strerror(archive)
		);
		EDVArchExtractCopyErrorMessage(core, msg);
		g_free(msg);
		status = -1;
	    }
	}

	return(status);
#else
	const gchar *prog_unzip = CFGItemListGetValueS(
	    core->cfg_list, EDV_CFG_PARM_PROG_UNZIP
	);
	FILE *fp;
	gint status, p, nobjs_extracted;
	gchar	*cmd = NULL,
		*pwd = NULL,
		*stdout_path = NULL,
		*stderr_path = NULL;
	const gchar *src_path;
	GList *glist;
	edv_archive_object_struct *obj;

#define CLEANUP_RETURN(_v_)	{		\
 g_free(cmd);					\
 g_free(stdout_path);				\
 g_free(stderr_path);				\
						\
 /* Restore the previous working dir */		\
 if(pwd != NULL) {				\
  EDVSetCWD(pwd);				\
  g_free(pwd);					\
 }						\
						\
 return(_v_);					\
}

	/* Record previous working dir and set new working dir */
	pwd = EDVGetCWD();
	if(EDVSetCWD(dest_path))
	{
	    core->archive_last_error =
"Unable to change working directory to the destination directory.";
	    CLEANUP_RETURN(-1);
	}

	/* Format extract object from archive command */
	cmd = g_strdup_printf(
	    "\"%s\" -o -X \"%s\"",
	    prog_unzip, arch_path
	);
	if(cmd == NULL)
	{
	    core->archive_last_error = "Unable to generate the extract command.";
	    CLEANUP_RETURN(-1);
	}
	/* Append the objects to extract to the command string
	 * only if not extracting all the objects
	 */
	if(!extract_all)
	{
	    gchar *s;

	    for(glist = objs_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		obj = EDV_ARCHIVE_OBJECT(glist->data);
		if(obj == NULL)
		    continue;

		src_path = obj->full_path;
		if(STRISEMPTY(src_path))
		    continue;

		if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
		{
		    const gint len = STRLEN(src_path);

		    s = g_strconcat(cmd, " \"", src_path, NULL);
		    g_free(cmd);
		    cmd = s;

		    /* If directory does not have a tailing
		     * deliminator then we must append one or else
		     * it will not get matched
		     */
		    if(len > 1)
		    {
			if(src_path[len - 1] != G_DIR_SEPARATOR)
			{
			    s = g_strconcat(cmd, G_DIR_SEPARATOR_S, NULL);
			    g_free(cmd);
			    cmd = s;
			}
		    }
		    s = g_strconcat(cmd, "\"", NULL);
		    g_free(cmd);
		    cmd = s;
		}
		else
		{
		    s = g_strconcat(cmd, " \"", src_path, "\"", NULL);
		    g_free(cmd);
		    cmd = s;
		}
	    }
	}
	if(cmd == NULL)
	{
	    core->archive_last_error = "Unable to generate the extract command.";
	    CLEANUP_RETURN(-1);
	}


	/* Generate output file paths */
	stdout_path = EDVTmpName(NULL);
	stderr_path = EDVTmpName(NULL);

	/* Begin extracting */

	status = 0;
	nobjs_extracted = 0;

	/* Execute the extract objects from archive command */
	p = (gint)ExecOE(
	    (const char *)cmd,
	    (const char *)stdout_path,
	    (const char *)stderr_path
	);
	if(p <= 0)
	{
	    core->archive_last_error = "Unable to execute the extract command.";
	    CLEANUP_RETURN(-1);
	}

	g_free(cmd);
	cmd = NULL;

	/* Open the output file for reading */
	fp = fopen((const char *)stdout_path, "rb");
	if(fp != NULL)
	{
	    gint buf_pos = 0, line_count = 0;
	    gfloat progress = 0.0f;
	    gchar buf[10000];
	    gboolean need_break = FALSE;

	    /* Begin reading the output file */
	    while(TRUE)
	    {
		/* Update progress? */
		if(show_progress && ProgressDialogIsQuery())
		{
		    ProgressDialogUpdate(
			NULL, NULL, NULL, NULL,
			progress, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
		    );
		    if(ProgressDialogStopCount() > 0)
		    {
			INTERRUPT(p);
			p = 0;
			status = -4;
			break;
		    }
		}

		/* Check if there is new data to be read from the output
		 * file
		 */
		if(FPending(fp))
		{
		    gint c;
		    gboolean got_complete_line = FALSE;

		    /* Copy all available data from the current output
		     * file position to its end to the line buffer buf
		     */
		    while(TRUE)
		    {
			c = (gint)fgetc(fp);
			if((int)c == EOF)
			{
			    clearerr(fp);
			    break;
			}

			if(ISCR(c))
			{
			    got_complete_line = TRUE;
			    line_count++;

			    if(buf_pos < sizeof(buf))
				buf[buf_pos] = '\0';
			    else
				buf[sizeof(buf) - 1] = '\0';
			    buf_pos = 0;

			    break;
			}

			if(buf_pos < sizeof(buf))
			{
			    buf[buf_pos] = c;
			    buf_pos++;
			}
		    }

		    /* Got a complete line from the output file and the
		     * progress dialog is mapped?
		     */
		    if(got_complete_line)
		    {
			gchar *s = buf, *s2, *extracted_path;

			/* Seek past spaces */
			while(ISBLANK(*s))
			    s++;

			/* Skip lines that do not contain prefixes
			 * that we are looking for
			 */
			if(!strcasepfx(s, "creating:") &&
			   !strcasepfx(s, "updating:") &&
			   !strcasepfx(s, "inflating:") &&
			   !strcasepfx(s, "extracting:") &&
			   !strcasepfx(s, "linking:")
			)
			    continue;

			/* Seek s past the prefix to the path value */
			while(!ISBLANK(*s) && (*s != '\0'))
			    s++;
			while(ISBLANK(*s))
			    s++;

			/* Cap the first blank character after the path */
			for(s2 = s; *s2 != '\0'; s2++)
			{
			    if(ISSPACE(*s2))
			    {
				*s2 = '\0';
				break;
			    }
			}

			extracted_path = STRDUP(PrefixPaths(dest_path, s));
			StripPath(extracted_path);

			/* Append this path to the list of paths
			 * extracted from the archive
			 */
			if(new_paths_list_rtn != NULL)
			    *new_paths_list_rtn = g_list_append(
				*new_paths_list_rtn, STRDUP(extracted_path)
			    );

			/* Update the progress dialog's label? */
			if(show_progress && !STRISEMPTY(extracted_path))
			{
			    gchar	*p1 = EDVShortenPath(
				s, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*p2 = EDVShortenPath(
				extracted_path, EDV_DEF_PROGRESS_BAR_PATH_DISPLAY_MAX
			    ),
					*msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Extraer:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Extraire:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Extrahieren:\n\
\n\
    %s\n\
\n\
Zu:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Estrarre:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Onttrekken:\n\
\n\
    %s\n\
\n\
Te:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Extrair:\n\
\n\
    %s\n\
\n\
A:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Trekking Ut:\n\
\n\
    %s\n\
\n\
Til:\n\
\n\
    %s\n"
#else
"Extracting:\n\
\n\
    %s\n\
\n\
To:\n\
\n\
    %s\n"
#endif
				, p1, p2
			    );
			    EDVArchExtractMapProgressDialog(
				msg, progress, toplevel, FALSE
			    );
			    g_free(msg);
			    g_free(p1);
			    g_free(p2);
			}

			nobjs_extracted++;
			progress = (nobjs > 0) ?
			    ((gfloat)nobjs_extracted / (gfloat)nobjs) : 0.0f;

			g_free(extracted_path);

			continue;
		    }
		}

		if(need_break)
		    break;

		/* Check if the extract process has exited, if it has
		 * then we set need_break to TRUE. Which will be
		 * tested on the next loop if there is still no more
		 * data to be read
		 */
		if(!ExecProcessExists(p))
		    need_break = TRUE;

		usleep(8000);
	    }

	    fclose(fp);
	}

	/* Remove output files */
	UNLINK(stdout_path);
	UNLINK(stderr_path);

	/* Report the final progress? */
	if(show_progress && (status == 0) &&
	   ProgressDialogIsQuery()
	)
	{
	    ProgressDialogUpdate(
		NULL, NULL, NULL, NULL,
		1.0f, EDV_DEF_PROGRESS_BAR_TICKS, TRUE
	    );
	    if(ProgressDialogStopCount() > 0)
		status = -4;
	}

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
#endif	/* !HAVE_LIBZIP */
}
