#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "pdialog.h"

#include "menucfg_list.h"

#include "config.h"

#include "images/icon_add_20x20.xpm"
#include "images/icon_edit_20x20.xpm"
#include "images/icon_remove_20x20.xpm"


/* Callbacks */
static void MenuCfgListCListDestroyCB(gpointer data);

static gint MenuCfgListCListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void MenuCfgListCListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void MenuCfgListCListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void MenuCfgListCListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);
static void MenuCfgListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void MenuCfgListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);

static void MenuCfgListAddCB(GtkWidget *widget, gpointer data);
static void MenuCfgListEditCB(GtkWidget *widget, gpointer data);
static void MenuCfgListRemoveCB(GtkWidget *widget, gpointer data);
static void MenuCfgListShiftUpCB(GtkWidget *widget, gpointer data);
static void MenuCfgListShiftDownCB(GtkWidget *widget, gpointer data);

static gchar *MenuCfgPDialogBrowseSelectProgramCB(
	gpointer p, gpointer data, gint prompt_num
);
static gchar *MenuCfgPDialogBrowseSelectIconCB(
	gpointer p, gpointer data, gint prompt_num
);


/* Items */
menucfg_item_struct *MenuCfgItemNew(
	const gchar *label,
	const gchar *command,
	const gchar *icon_file,
	const gchar *description
);
void MenuCfgItemSetCustomData(
        menucfg_item_struct *item, GList *custom_datas_list
);
const gchar *MenuCfgItemGetCustomDataValue(
        menucfg_item_struct *item, const gint i
);
menucfg_item_struct *MenuCfgItemCopy(menucfg_item_struct *item);
void MenuCfgItemDelete(menucfg_item_struct *item);


/* MenuCfgList */
void MenuCfgListAdd(menucfg_list_struct *list);
gboolean MenuCfgListEdit(menucfg_list_struct *list);
void MenuCfgListRemove(menucfg_list_struct *list, const gboolean confirm);
void MenuCfgListShiftUp(menucfg_list_struct *list);
void MenuCfgListShiftDown(menucfg_list_struct *list);

menucfg_list_struct *MenuCfgListNew(
	GtkWidget *parent,
	const GtkOrientation orientation
);
void MenuCfgListSetCustomData(
	menucfg_list_struct *list,
	GList *custom_names_list
);
gint MeuCfgListGetTotalCustomDatas(menucfg_list_struct *list);
void MenuCfgListSetListStyle(
	menucfg_list_struct *list, const menucfg_list_style list_style
);
void MenuCfgListSetEditable(
	menucfg_list_struct *list, const gboolean editable
);
void MenuCfgListSetChangedCB(
	menucfg_list_struct *list,
	void (*cb)(menucfg_list_struct *, gpointer),
	gpointer data
);
void MenuCfgListSetItemAddedCB(
	menucfg_list_struct *list,
	void (*cb)(
		menucfg_list_struct *,
		gint,
		menucfg_item_struct *,
		gpointer
	),
	gpointer data
);
void MenuCfgListSetItemChangedCB(
	menucfg_list_struct *list,
	void (*cb)(
		menucfg_list_struct *,
		gint,
		menucfg_item_struct *,
		gpointer
	),
	gpointer data
);
void MenuCfgListSetItemRemovedCB(
	menucfg_list_struct *list,
	void (*cb)(
		menucfg_list_struct *,
		gint,
		gpointer
	),
	gpointer data
);
gint MenuCfgListInsert(
	menucfg_list_struct *list, const gint row,
	menucfg_item_struct *item               /* Transfered */
);
gint MenuCfgListAppend(
	menucfg_list_struct *list,
	menucfg_item_struct *item               /* Transfered */
);
void MenuCfgListSet(
	menucfg_list_struct *list, const gint row,
	menucfg_item_struct *item               /* Transfered */
);
void MenuCfgListClear(menucfg_list_struct *list);
menucfg_item_struct *MenuCfgListGet(
	menucfg_list_struct *list, const gint row
);
void MenuCfgListUpdateWidgets(menucfg_list_struct *list);
void MenuCfgListMap(menucfg_list_struct *list);
void MenuCfgListUnmap(menucfg_list_struct *list);
void MenuCfgListDelete(menucfg_list_struct *list);


#define MENUCFG_LIST_ROW_SPACING	20


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	GtkCList item destroy callback.
 */
static void MenuCfgListCListDestroyCB(gpointer data)
{
	MenuCfgItemDelete(MENUCFG_ITEM(data));
}

/*
 *	GtkCList event signal callback.
 */
static gint MenuCfgListCListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean key_press;
	gint x, y, row, column;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	GtkCList *clist;
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if((widget == NULL) || (event == NULL) || (list == NULL))
	    return(status);

	if(list->freeze_count > 0)
	    return(status);

	switch((gint)event->type)
	{
	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    key_press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    switch(key->keyval)
	    {
	      case GDK_Return:
		if(list->flags & MENUCFG_LIST_EDITABLE)
		{
		    if(key_press)
			MenuCfgListEdit(list);
		    status = TRUE;
		}
		break;

	      case GDK_Delete:
		if(list->flags & MENUCFG_LIST_EDITABLE)
		{
		    if(key_press)
			MenuCfgListRemove(list, TRUE);
		    status = TRUE;
		}
		break;
	    }
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    list->button = button->button;
	    switch(button->button)
	    {
	      case GDK_BUTTON2:
		gdk_pointer_grab(
		    button->window,
		    FALSE,
		    GDK_BUTTON_PRESS_MASK |
		    GDK_BUTTON_RELEASE_MASK |
		    GDK_POINTER_MOTION_MASK |
		    GDK_POINTER_MOTION_HINT_MASK,
		    NULL,
		    list->translate_cur,
		    button->time
		);
		list->drag_last_x = (gint)button->x;
		list->drag_last_y = (gint)button->y;
		status = TRUE;
		break;

	      case GDK_BUTTON3:
		/* Get the row and column */
		x = (gint)button->x;
		y = (gint)button->y;
		clist = GTK_CLIST(list->clist);
		if(!gtk_clist_get_selection_info(
		    clist, x, y, &row, &column
		))
		{ 
		    row = -1;
		    column = 0;
		}

		/* Select the row */
		gtk_clist_freeze(clist);
		if(!(button->state & GDK_CONTROL_MASK) &&
		   !(button->state & GDK_SHIFT_MASK)
		)
		    gtk_clist_unselect_all(clist);
		gtk_clist_select_row(clist, row, column);
		gtk_clist_thaw(clist);

		MenuCfgListUpdateWidgets(list);

		/* Map the right-click popup menu */
		gtk_menu_popup(
		    GTK_MENU(list->menu), NULL, NULL,
		    NULL, NULL,
		    button->button, button->time
		);
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON2:
		if(list->button > 0)
		    status = TRUE;
		if(gdk_pointer_is_grabbed())
		    gdk_pointer_ungrab(button->time);
		break;
	    }
	    list->button = 0;
	    list->drag_last_x = 0;
	    list->drag_last_y = 0;
	    break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;
	    if(list->button == GDK_BUTTON2)
	    {
		gint dx, dy;
		GtkCList *clist = GTK_CLIST(list->clist);
		GtkAdjustment *hadj = clist->hadjustment,
			      *vadj = clist->vadjustment;
		if(motion->is_hint)
		{
		    gint x, y;
		    GdkModifierType mask;
		    gdk_window_get_pointer(
			motion->window, &x, &y, &mask
		    );
		    dx = x - list->drag_last_x;
		    dy = y - list->drag_last_y;
		    list->drag_last_x = x;
		    list->drag_last_y = y;
		}
		else
		{
		    dx = (gint)(motion->x - list->drag_last_x);
		    list->drag_last_x = (gint)motion->x;
		    dy = (gint)(motion->y - list->drag_last_y);
		    list->drag_last_y = (gint)motion->y;
		}
#define DO_SCROLL(adj,d)	{			\
 if((d) != 0) {						\
  if(((adj)->upper - (adj)->page_size) > (adj)->lower) {\
   gtk_adjustment_set_value(				\
    adj,						\
    CLIP(						\
     (adj)->value - (d),				\
     (adj)->lower, (adj)->upper - (adj)->page_size	\
    )							\
   );							\
  }							\
 }							\
}
		DO_SCROLL(hadj, dx);
		DO_SCROLL(vadj, dy);

#undef DO_SCROLL
		status = TRUE;
	    }
	    break;
	}

	return(status);
}

/*
 *	DND "drag_data_get" signal callback.
 */
static void MenuCfgListCListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint buf_len;
	guint8 *buf;
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if((widget == NULL) || (dc == NULL) || (list == NULL))
	    return;

	if(list->freeze_count > 0)
	    return;

	/* Respond with the source widget */
	buf_len = sizeof(GtkWidget *);
	buf = (guint8 *)g_malloc(buf_len + 1);
	if(buf == NULL)
	    return;

	*((GtkWidget **)buf) = widget;
	buf[buf_len] = '\0';

	gtk_selection_data_set(
	    selection_data,
	    GDK_SELECTION_TYPE_STRING,
	    8,				/* Bits Per Character */
	    buf,			/* Data */
	    buf_len			/* Length */
	);

	g_free(buf);
}

/*
 *	DND "drag_data_received" signal callback.
 */
static void MenuCfgListCListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if((widget == NULL) || (dc == NULL) || (list == NULL))
	    return;

	if(list->freeze_count > 0)
	    return;

	if(selection_data == NULL)
	    return;

	if((selection_data->data == NULL) ||
	   (selection_data->length <= 0)
	)
	    return;

	if(widget == list->clist)
	{
	    GtkCList *clist = GTK_CLIST(widget);

	    if(info == MENUCFG_LIST_ITEM_INFO)
	    {
		GtkWidget *src_w = (selection_data->length >= sizeof(GtkWidget *)) ?
		    *((GtkWidget **)selection_data->data) : NULL;
		gint row = -1, column = 0, insert_row, new_row;
		GList *glist, *sel_list;
		menucfg_item_struct *item;

		if(src_w != widget)
		    return;

		/* Find the dropped-on row */
		gtk_clist_get_selection_info(
		    clist,
		    x,
		    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
			clist->column_title_area.height +
			    clist->column_title_area.y : 0),
		    &row, &column
		);
		insert_row = row;

		/* Create a list of selected items */
		sel_list = NULL;
		for(glist = clist->selection;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		    sel_list = g_list_append(
			sel_list,
			gtk_clist_get_row_data(clist, (gint)glist->data)
		    );

		gtk_clist_freeze(clist);

		/* Insert the selected items */
		for(glist = sel_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    item = MENUCFG_ITEM(glist->data);
		    if(item == NULL)
			continue;

		    if(insert_row > -1)
		    {
			new_row = MenuCfgListInsert(
			    list, insert_row,
			    MenuCfgItemCopy(item)
			);
			insert_row++;
		    }
		    else
		    {
			new_row = MenuCfgListAppend(
			    list,
			    MenuCfgItemCopy(item)
			);
		    }
		}
		g_list_free(sel_list);

		/* Remove all the selected rows */
		while(clist->selection != NULL)
		    gtk_clist_remove(
			clist,
			(gint)clist->selection->data
		    );

		gtk_clist_columns_autosize(clist);

		gtk_clist_thaw(clist);

		MenuCfgListUpdateWidgets(list);

		/* Call the changed callback */
		if(list->changed_cb != NULL)
		    list->changed_cb(
			list,
			list->changed_data
		    );
	    }
	}
}

/*
 *	DND "drag_data_delete" signal callback.
 */
static void MenuCfgListCListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if((widget == NULL) || (dc == NULL) || (list == NULL))
	    return;

	if(list->freeze_count > 0)
	    return;
}

/*
 *	GtkCList "select_row" signal callback.
 */
static void MenuCfgListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	GdkBitmap *mask = NULL;
	GdkPixmap *pixmap = NULL;
	GtkCellType cell_type;
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if((clist == NULL) || (event == NULL) || (list == NULL))
	    return;

	if(list->freeze_count > 0)
	    return;

	/* Scroll to the row if it is not visible */
	if(gtk_clist_row_is_visible(clist, row) !=
	   GTK_VISIBILITY_FULL
	)
	    gtk_clist_moveto(
		clist,
		row, -1,	/* Row, column */
		0.5f, 0.0f	/* Row, column */
	    );

	/* Set the DND drag icon */
	cell_type = gtk_clist_get_cell_type(clist, row, 0);
	if(cell_type == GTK_CELL_PIXTEXT)
	{
	    gchar *text;
	    guint8 spacing;
	    gtk_clist_get_pixtext(
		clist,
		row, 0,
		&text,
		&spacing,
		&pixmap, &mask
	    );
	}     
	else if(cell_type == GTK_CELL_PIXMAP)
	{
	    gtk_clist_get_pixmap(
		clist,
		row, 0,
		&pixmap, &mask   
	    );
	}
	if(pixmap != NULL)
	{  
	    gint w = 15, h = 15;
	    gdk_window_get_size(pixmap, &w, &h);
	    GUIDNDSetDragIcon(
		pixmap, mask,
		w / 2, h / 2
	    );
	}
	else  
	{     
	    GUIDNDSetDragIcon(NULL, NULL, 0, 0);
	}

	/* Double click? */
	if(event->type == GDK_2BUTTON_PRESS)
	{
	    GdkEventButton *button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON1:
		if(list->flags & MENUCFG_LIST_EDITABLE)
		    MenuCfgListEdit(list);
		break;
	    }
	}

	MenuCfgListUpdateWidgets(list);
}

/*
 *	GtkCList "unselect_row" signal callback.
 */
static void MenuCfgListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if((clist == NULL) || (event == NULL) || (list == NULL))
	    return;

	if(list->freeze_count > 0)
	    return;

	MenuCfgListUpdateWidgets(list);
}


/*
 *	Add callback.
 */
static void MenuCfgListAddCB(GtkWidget *widget, gpointer data)
{
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if(list == NULL)
	    return;

	if(list->freeze_count > 0)
	    return;

	MenuCfgListAdd(list);
}

/*
 *	Edit callback.
 */
static void MenuCfgListEditCB(GtkWidget *widget, gpointer data)
{
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if(list == NULL)
	    return;

	if(list->freeze_count > 0)
	    return;

	MenuCfgListEdit(list);
}

/*
 *	Remove callback.
 */
static void MenuCfgListRemoveCB(GtkWidget *widget, gpointer data)
{
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if(list == NULL)
	    return;

	if(list->freeze_count > 0)
	    return;

	MenuCfgListRemove(list, TRUE);
}

/*
 *	Shift up callback.
 */
static void MenuCfgListShiftUpCB(GtkWidget *widget, gpointer data)
{
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if(list == NULL)
	    return;

	if(list->freeze_count > 0)
	    return;

	MenuCfgListShiftUp(list);
}

/*
 *	Shift down callback.
 */
static void MenuCfgListShiftDownCB(GtkWidget *widget, gpointer data)
{
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if(list == NULL)
	    return;

	if(list->freeze_count > 0)
	    return;

	MenuCfgListShiftDown(list);
}


/*
 *	Prompt Dialog select program callback.
 */
static gchar *MenuCfgPDialogBrowseSelectProgramCB(
	gpointer p, gpointer data, gint prompt_num
)
{
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if((list == NULL) || FileBrowserIsQuery())
	    return(NULL);

	if(list->freeze_count > 0)
	    return(NULL);

	g_free(list->browse_path_rtn);
	list->browse_path_rtn = NULL;

	toplevel = PDialogGetToplevel();

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);
	  
	/* Query the user for the new location */
	list->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Program",
	    "Select", "Cancel",
	    PDialogGetPromptValue(prompt_num),
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	list->freeze_count--;

	/* Got user response? */
	if(response)
	{         
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		g_free(list->browse_path_rtn);
		list->browse_path_rtn = STRDUP(path);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	return(list->browse_path_rtn);
}

/*
 *	Prompt Dialog select icon callback.
 */
static gchar *MenuCfgPDialogBrowseSelectIconCB(
	gpointer p, gpointer data, gint prompt_num
)
{
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	menucfg_list_struct *list = MENUCFG_LIST(data);
	if((list == NULL) || FileBrowserIsQuery())
	    return(NULL);

	if(list->freeze_count > 0)
	    return(NULL);

	g_free(list->browse_path_rtn);
	list->browse_path_rtn = NULL;

	toplevel = PDialogGetToplevel();

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    ".xpm", "X Pixmap"
	);
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);
	  
	/* Query the user for the new location */
	list->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Icon",
	    "Select", "Cancel",
	    PDialogGetPromptValue(prompt_num),
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	list->freeze_count--;

	/* Got user response? */
	if(response)
	{         
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		g_free(list->browse_path_rtn);
		list->browse_path_rtn = STRDUP(path);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	return(list->browse_path_rtn);
}


/*
 *	Creates a new Item.
 */
menucfg_item_struct *MenuCfgItemNew(
	const gchar *label,
	const gchar *command,
	const gchar *icon_file,
	const gchar *description
)
{
	menucfg_item_struct *item = MENUCFG_ITEM(g_malloc0(
	    sizeof(menucfg_item_struct)
	));
	if(item == NULL)
	    return(NULL);

	item->label = STRDUP(label);
	item->command = STRDUP(command);
	item->icon_file = STRDUP(icon_file);
	item->description = STRDUP(description);
	item->custom_datas_list = NULL;

	return(item);
}

/*
 *	Sets the Item's custom datas list.
 *
 *	The custom_datas_list specifies a GList of gchar * strings
 *	describing each custom data value. The custom_datas_list will
 *	be transfered and must not be referenced again after this call.
 */
void MenuCfgItemSetCustomData(
        menucfg_item_struct *item, GList *custom_datas_list
)
{
	if(item == NULL)
	{
	    if(custom_datas_list != NULL)
	    {
		g_list_foreach(
		    custom_datas_list, (GFunc)g_free, NULL
		);
		g_list_free(custom_datas_list);
	    }
	    return;
	}

	if(item->custom_datas_list != NULL)
	{
	    g_list_foreach(
		item->custom_datas_list, (GFunc)g_free, NULL
	    );
	    g_list_free(item->custom_datas_list);
	}

	item->custom_datas_list = custom_datas_list;
}

/*
 *	Gets the custom data value.
 *
 *	The i specifies the custom data index.
 */
const gchar *MenuCfgItemGetCustomDataValue(
        menucfg_item_struct *item, const gint i
)
{
	GList *glist;

	if(item == NULL)
	    return(NULL);

	glist = g_list_nth(item->custom_datas_list, i);
	if(glist == NULL)
	    return(NULL);

	return((const gchar *)glist->data);
}

/*
 *	Coppies the Item.
 */
menucfg_item_struct *MenuCfgItemCopy(menucfg_item_struct *item)
{
	const gchar *custom_data;
	GList *glist;
	menucfg_item_struct *item2;

	if(item == NULL)
	    return(NULL);

	/* Copy the Item */
	item2 = MenuCfgItemNew(
	    item->label,
	    item->command,
	    item->icon_file,
	    item->description
	);
	if(item2 == NULL)
	    return(NULL);

	/* Copy the Item's custom datas list */
	for(glist = item->custom_datas_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    custom_data = (gchar *)glist->data;
	    item2->custom_datas_list = g_list_append(
		item2->custom_datas_list,
		STRDUP(custom_data)
	    );
	}

	return(item2);
}

/*
 *	Deletes the Item.
 */
void MenuCfgItemDelete(menucfg_item_struct *item)
{
	if(item == NULL)
	    return;

	g_free(item->label);
	g_free(item->command);
	g_free(item->icon_file);
	g_free(item->description);
	if(item->custom_datas_list != NULL)
	{
	    g_list_foreach(item->custom_datas_list, (GFunc)g_free, NULL);
	    g_list_free(item->custom_datas_list);
	}
	g_free(item);
}


/*
 *	Add.
 */
void MenuCfgListAdd(menucfg_list_struct *list)
{
	gint row, new_row;
	GList *glist;
	GtkCList *clist;

	if(list == NULL)
	    return;

	clist = GTK_CLIST(list->clist);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	new_row = MenuCfgListInsert(
	    list, row,
	    MenuCfgItemNew(
		"New Item",
		NULL,
		NULL,
		NULL
	    )
	);
	if(new_row < 0)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_select_row(clist, new_row, 0);
	gtk_clist_thaw(clist);

	if(list->item_added_cb != NULL)
	    list->item_added_cb(
		list,
		new_row,
		MENUCFG_ITEM(gtk_clist_get_row_data(clist, new_row)),
		list->item_added_data
	    );
	if(list->changed_cb != NULL)
	    list->changed_cb(
		list,
		list->changed_data
	    );

	/* Edit the new item */
	if(!MenuCfgListEdit(list))
	{
	    MenuCfgListRemove(list, FALSE);
	}
}

/*
 *	Edit.
 *
 *	Returns TRUE if an item was changed.
 */
gboolean MenuCfgListEdit(menucfg_list_struct *list)
{
	gboolean status;
	gint row, strc, item_num;
	gchar **strv;
	GList *glist, *glist_nth;
	GtkWidget *toplevel;
	GtkCList *clist;
	menucfg_item_struct *item;

	if(list == NULL)
	    return(FALSE);

	clist = GTK_CLIST(list->clist);
	toplevel = gtk_widget_get_toplevel(list->toplevel);

	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;
	if(row < 0)
	    return(FALSE);

	item_num = row;
	item = MENUCFG_ITEM(gtk_clist_get_row_data(clist, row));
	if(item == NULL)
	    return(FALSE);

	/* Set up the Prompt Dialog */
	PDialogDeleteAllPrompts();
	PDialogSetSize(350, -1);
	PDialogAddPrompt(NULL, "Label:", item->label);
	PDialogSetPromptTip(
	     -1,
"Enter the name of this item"
	);
	PDialogAddPromptWithBrowse(
	    NULL, "Command:", item->command,
	    list, MenuCfgPDialogBrowseSelectProgramCB
	);
	PDialogSetPromptCompletePath(-1);
	PDialogSetPromptTip(
	     -1,
"Enter the command that is to be executed for this item (or leave this blank for none)"
	);
	PDialogAddPromptWithBrowse(
	    NULL, "Icon File:", item->icon_file,
	    list, MenuCfgPDialogBrowseSelectIconCB
	);
	PDialogSetPromptCompletePath(-1);
	PDialogSetPromptTip(
	     -1,
"Enter the icon file for this item (or leave this blank for none)"
	);
	PDialogAddPrompt(NULL, "Description:", item->description);
	PDialogSetPromptTip(
	     -1,
"Enter the description of this item (or leave this blank for none)"
	);
	if(list->custom_data_names != NULL)
	{
	    gint i;
	    gchar *label;
	    const gchar *name, *value;

	    for(glist = list->custom_data_names,
		i = 0;
		glist != NULL;
		glist = g_list_next(glist),
		i++
	    )
	    {
		name = (const gchar *)glist->data;
		if(name != NULL)
		    label = g_strconcat(name, ":", NULL);
		else
		    label = g_strdup_printf("Value #%i:", i + 1);

		glist_nth = g_list_nth(item->custom_datas_list, i);
		value = (glist_nth != NULL) ?
                    (const gchar *)glist_nth->data : NULL;

		PDialogAddPrompt(NULL, label, value);

		g_free(label);
	    }
	}
	gtk_window_set_transient_for(
	    GTK_WINDOW(PDialogGetToplevel()), GTK_WINDOW(toplevel)
	);
/*      PDialogSetTransientFor(toplevel); */
	strv = PDialogGetResponse(            
	    "Item Properties",
	    NULL, NULL,
	    PDIALOG_ICON_FILE_PROPERTIES,
	    "Set", "Cancel",
	    PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
	    PDIALOG_BTNFLAG_SUBMIT,
	    &strc
	);
	gtk_window_set_transient_for(
	    GTK_WINDOW(PDialogGetToplevel()), NULL
	);
/*      PDialogSetTransientFor(NULL); */
	if((strv != NULL) && (strc >= 4))
	{
	    /* Create a new item with the new values */
	    menucfg_item_struct *item = MenuCfgItemNew(
		strv[0],
		strv[1],
		strv[2],
		strv[3]
	    );
	    if(item != NULL)
	    {
		gint i;

		/* Add the subsequent values to the item's custom
		 * datas list starting from value 4
		 */
		for(i = 4; i < strc; i++)
		    item->custom_datas_list = g_list_append(
			item->custom_datas_list,
			STRDUP(strv[i])
		    );

		/* Replace the existing item with the new item */
		MenuCfgListSet(
		    list, item_num,
		    item			/* Transfered */
		);
/*		item = NULL; */
	    }

	    MenuCfgListUpdateWidgets(list);

	    if(list->item_changed_cb != NULL)
		list->item_changed_cb(
		    list,
		    item_num,
		    item,
		    list->item_changed_data
		);
	    if(list->changed_cb != NULL)
		list->changed_cb(
		    list,
		    list->changed_data
		);

	    status = TRUE;
	}
	else
	{
	    status = FALSE;
	}

	PDialogDeleteAllPrompts();

	return(status);
}

/*
 *	Remove.
 */
void MenuCfgListRemove(menucfg_list_struct *list, const gboolean confirm)
{
	gint row;
	GList *glist, *items_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	menucfg_item_struct *item;

	if(list == NULL)
	    return;

	toplevel = gtk_widget_get_toplevel(list->toplevel);
	clist = GTK_CLIST(list->clist);

	/* Create a list of selected items */
	items_list = NULL;
	glist = clist->selection;
	while(glist != NULL)
	{
	    row = (gint)glist->data;
	    item = MENUCFG_ITEM(gtk_clist_get_row_data(clist, row));
	    if(item != NULL)
		items_list = g_list_append(items_list, item);
	    glist = g_list_next(glist);
	}

	/* Nothing to remove? */
	if(items_list == NULL)
	    return;

	/* Confirm remove? */
	if(confirm)
	{
	    gint nitems = g_list_length(items_list);
	    gint response;
	    gchar *msg;
	    if(nitems == 1)
	    {
		item = MENUCFG_ITEM(items_list->data);
		msg = g_strdup_printf(
"Remove item \"%s\"?",
		    (item != NULL) ? item->label : "(null)"
		);
	    }
	    else
	    {
		msg = g_strdup_printf(
"Remove %i selected items?",
		    nitems
		);
	    }
	    CDialogSetTransientFor(toplevel);
	    response = CDialogGetResponse(
		"Confirm Remove",
		msg,
		NULL,
		CDIALOG_ICON_QUESTION,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	    );
	    g_free(msg);
	    CDialogSetTransientFor(NULL); 

	    if(response != CDIALOG_RESPONSE_YES)
	    {
		g_list_free(items_list);
		return;
	    }
	}

	/* Begin removing the selected items */
	gtk_clist_freeze(clist);
	for(glist = items_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    item = MENUCFG_ITEM(glist->data);
	    row = gtk_clist_find_row_from_data(clist, item);
	    if(row < 0)
		continue;

	    if(list->item_removed_cb != NULL)
		list->item_removed_cb(
		    list,
		    row,
		    list->item_removed_data
		);

	    gtk_clist_remove(clist, row);
	}
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);

	g_list_free(items_list);

	MenuCfgListUpdateWidgets(list);

	if(list->changed_cb != NULL)
	    list->changed_cb(
		list,
		list->changed_data
	    );
}

/*
 *	Shift up.
 */
void MenuCfgListShiftUp(menucfg_list_struct *list)
{
	gint i, n, lead_row;
	GtkCList *clist;

	if(list == NULL)
	    return;

	clist = GTK_CLIST(list->clist);
	n = clist->rows;

	/* Nothing to shift? */
	if((clist->selection == NULL) || (n < 2))
	    return;

	/* Already at the top? */
	i = 0;
	if(g_list_find(clist->selection, (gpointer)i) != NULL)
	    return;

	/* Shift each selected row up */
	gtk_clist_freeze(clist);
	lead_row = -1;
	for(i = 1; i < n; i++)
	{
	    if(g_list_find(clist->selection, (gpointer)i) != NULL)
	    {
		if(lead_row < 0)
		    lead_row = i - 1;

		gtk_clist_swap_rows(clist, i, i - 1);

		if(list->item_changed_cb != NULL)
		{
		    list->item_changed_cb(
			list,
			i - 1,
			MENUCFG_ITEM(gtk_clist_get_row_data(clist, i - 1)),
			list->item_changed_data
		    );
		    list->item_changed_cb(
			list,
			i,
			MENUCFG_ITEM(gtk_clist_get_row_data(clist, i)),
			list->item_changed_data
		    );
		}
	    }
	}
	gtk_clist_thaw(clist);

	/* Scroll to the lead row if it is not visible */
	if(lead_row > -1)
	{
	    if(gtk_clist_row_is_visible(clist, lead_row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    lead_row, -1,	/* Row, column */
		    0.5f, 0.0f		/* Row, column */
		);
	}

	MenuCfgListUpdateWidgets(list);

	if(list->changed_cb != NULL)
	    list->changed_cb(
		list,
		list->changed_data
	    );
}

/*
 *	Shift down.
 */
void MenuCfgListShiftDown(menucfg_list_struct *list)
{
	gint i, n, lead_row;
	GtkCList *clist;

	if(list == NULL)
	    return;

	clist = GTK_CLIST(list->clist);
	n = clist->rows;

	/* Nothing to shift? */
	if((clist->selection == NULL) || (n < 2))
	    return;

	/* Already at the bottom? */
	i = n - 1;
	if(g_list_find(clist->selection, (gpointer)i) != NULL)
	    return;

	/* Shift each selected row down */
	gtk_clist_freeze(clist);
	lead_row = -1;
	for(i = n - 2; i >= 0; i--)
	{
	    if(g_list_find(clist->selection, (gpointer)i) != NULL)
	    {
		if(lead_row < 0)
		    lead_row = i + 1;

		gtk_clist_swap_rows(clist, i, i + 1);

		if(list->item_changed_cb != NULL)
		{
		    list->item_changed_cb(
			list,
			i,
			MENUCFG_ITEM(gtk_clist_get_row_data(clist, i)),
			list->item_changed_data
		    );
		    list->item_changed_cb(
			list,
			i + 1,
			MENUCFG_ITEM(gtk_clist_get_row_data(clist, i + 1)),
			list->item_changed_data
		    );
		}
	    }
	}
	gtk_clist_thaw(clist);

	/* Scroll to the lead row if it is not visible */
	if(lead_row > -1)
	{
	    if(gtk_clist_row_is_visible(clist, lead_row) !=
		GTK_VISIBILITY_FULL
	    )
		gtk_clist_moveto(
		    clist,
		    lead_row, -1,	/* Row, column */
		    0.5f, 0.0f		/* Row, column */
		);
	}

	MenuCfgListUpdateWidgets(list);

	if(list->changed_cb != NULL)
	    list->changed_cb(
		list,
		list->changed_data
	    );
}

/*
 *	Creates a new MenuCfg List.
 */
menucfg_list_struct *MenuCfgListNew(
	GtkWidget *parent,
	const GtkOrientation orientation
)
{
	const gint	border_major = 5,
			border_minor = 2;
	gchar *heading[2];
	GtkWidget *w, *parent2, *parent3;
	GtkCList *clist;
	menucfg_list_struct *list = MENUCFG_LIST(g_malloc0(
	    sizeof(menucfg_list_struct)
	));
	if(list == NULL)
	    return(NULL);

	list->freeze_count = 0;
	list->flags = MENUCFG_LIST_EDITABLE;
	list->orientation = orientation;
	list->list_style = MENUCFG_LIST_STYLE_SINGLE;
	list->translate_cur = gdk_cursor_new(GDK_FLEUR);
	list->button = 0;
	list->drag_last_x = 0;
	list->drag_last_y = 0;
	list->browse_path_rtn = NULL;
	list->changed_cb = NULL;
	list->changed_data = NULL;
	list->item_changed_cb = NULL;
	list->item_changed_data = NULL;
	list->custom_data_names = NULL;

	/* Create toplevel GtkBox */
	list->toplevel = w = (orientation == GTK_ORIENTATION_VERTICAL) ?
	    gtk_vbox_new(FALSE, border_major) :
	    gtk_hbox_new(FALSE, border_major);
	if(parent != NULL)
	{
	    if(GTK_IS_BOX(parent))
		gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	    else if(GTK_IS_CONTAINER(parent))
		gtk_container_add(GTK_CONTAINER(parent), w);
	}
	parent = w;

	/* Create the GtkScrolledWindow for the GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Create the GtkCList */
	heading[0] = "Name";
	heading[1] = "Command";
	list->clist = w = gtk_clist_new_with_titles(
	    sizeof(heading) / sizeof(gchar *),
	    heading
	);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event", 
	    GTK_SIGNAL_FUNC(MenuCfgListCListEventCB), list
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(MenuCfgListCListEventCB), list
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(MenuCfgListCListEventCB), list
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(MenuCfgListCListEventCB), list
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(MenuCfgListCListEventCB), list
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(MenuCfgListSelectRowCB), list
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(MenuCfgListUnselectRowCB), list
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, TRUE);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 1, TRUE);
	gtk_clist_set_column_resizeable(clist, 1, TRUE);
	gtk_clist_set_column_justification(
	    clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	switch(list->list_style)
	{
	  case MENUCFG_LIST_STYLE_SINGLE:
	    gtk_clist_column_titles_hide(clist);
	    gtk_clist_column_titles_active(clist);
	    gtk_clist_set_column_visibility(clist, 0, TRUE);
	    gtk_clist_set_column_visibility(clist, 1, FALSE);
	    break;
	  case MENUCFG_LIST_STYLE_DETAILS:
	    gtk_clist_column_titles_show(clist);
	    gtk_clist_column_titles_passive(clist);
	    gtk_clist_set_column_visibility(clist, 0, TRUE);
	    gtk_clist_set_column_visibility(clist, 1, TRUE);
	    break;
	}
	gtk_clist_set_row_height(clist, MENUCFG_LIST_ROW_SPACING);
	gtk_widget_realize(w);
	if(w != NULL)
	{
	    const GtkTargetEntry dnd_tar_types[] = {
{MENUCFG_LIST_ITEM_TARGET_NAME,	GTK_TARGET_SAME_APP,	MENUCFG_LIST_ITEM_INFO}
	    };
	    const GtkTargetEntry dnd_src_types[] = {
{MENUCFG_LIST_ITEM_TARGET_NAME,	GTK_TARGET_SAME_APP,	MENUCFG_LIST_ITEM_INFO}
	    };
	    GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_MOVE,		/* Actions */
		GDK_BUTTON1_MASK,		/* Buttons */
		NULL,
		MenuCfgListCListDragDataGetCB,
		MenuCfgListCListDragDataDeleteCB,
		NULL,
		list
	    );
	    GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_MOVE,		/* Actions */
		GDK_ACTION_MOVE,		/* Default action if same */
		GDK_ACTION_MOVE,		/* Default action */
		MenuCfgListCListDragDataReceivedCB,
		list
	    );
	}
	gtk_widget_show(w);


	if(orientation == GTK_ORIENTATION_VERTICAL)
	{
	    /* GtkBox for the buttons */
	    list->buttons_box = w = gtk_hbox_new(TRUE, 0);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;

	}
	else
	{
	    /* GtkBox for the buttons */
	    list->buttons_box = w = gtk_vbox_new(FALSE, 2 * border_major);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	}

	/* Add button */
	list->add_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_add_20x20_xpm,
	    "Add",
	    NULL
	);
	gtk_widget_set_usize(
	    w, GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(MenuCfgListAddCB), list
	);
	GUISetWidgetTip(
	    w,
	    "Add a new item"
	);
	gtk_widget_show(w);

	/* Edit button */
	list->edit_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_edit_20x20_xpm,
	    "Edit",
	    NULL
	);
	gtk_widget_set_usize(
	    w, GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(MenuCfgListEditCB), list
	);
	GUISetWidgetTip(
	    w,
	    "Edit the selected item"
	);
	gtk_widget_show(w);

	/* Remove button */
	list->remove_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_remove_20x20_xpm,
	    "Remove",
	    NULL
	);
	gtk_widget_set_usize(
	    w, GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(MenuCfgListRemoveCB), list
	);
	GUISetWidgetTip(
	    w,
	    "Remove the selected item(s)"
	);
	gtk_widget_show(w);

	if(orientation == GTK_ORIENTATION_HORIZONTAL)
	{
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	}

	/* Shift up button */
	list->shift_up_btn = w = GUIButtonArrowLabelH(
	    GTK_ARROW_UP, 20, 20,
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Arriba"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer Haut"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Op"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Cima"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Opp"
#else
"Shift Up"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(MenuCfgListShiftUpCB), list
	);
	GUISetWidgetTip(
	    w,
	    "Shift the selected item(s) up"
	);
	gtk_widget_show(w);

	/* Shift down button */
	list->shift_down_btn = w = GUIButtonArrowLabelH(
	    GTK_ARROW_DOWN, 20, 20,
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Abajo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer Bas"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Beneden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Baixo"      
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Ned"
#else       
"Shift Down"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(MenuCfgListShiftDownCB), list
	);
	GUISetWidgetTip(
	    w,
	    "Shift the selected item(s) down"
	);
	gtk_widget_show(w);


	/* Create the right-click menu */
#define DO_ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
  icon, label, accel_key, accel_mods, NULL,	\
  menu_d, func_cb				\
 );						\
}
#define DO_ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}
	list->menu = w = GUIMenuCreate();
	if(w != NULL)
	{
	    GtkAccelGroup *accelgrp = NULL;
	    GtkWidget *menu = w;
	    guint8 **icon;
	    const gchar *label;
	    guint accel_key, accel_mods;
	    void (*func_cb)(GtkWidget *w, gpointer);
	    gpointer menu_d = list;

	    icon = (guint8 **)icon_add_20x20_xpm;
	    label = "Add...";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = MenuCfgListAddCB;
	    DO_ADD_MENU_ITEM_LABEL   
	    list->menu_add_mi = w;

	    icon = (guint8 **)icon_edit_20x20_xpm;
	    label = "Edit...";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = MenuCfgListEditCB;
	    DO_ADD_MENU_ITEM_LABEL
	    list->menu_edit_mi = w;

	    icon = (guint8 **)icon_remove_20x20_xpm;
	    label = "Remove";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = MenuCfgListRemoveCB;
	    DO_ADD_MENU_ITEM_LABEL
	    list->menu_remove_mi = w;

	    DO_ADD_MENU_SEP

	    icon = NULL;
	    label = "Shift Up";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = MenuCfgListShiftUpCB;
	    DO_ADD_MENU_ITEM_LABEL
	    list->menu_shift_up_mi = w;

	    icon = NULL;
	    label = "Shift Down";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = MenuCfgListShiftDownCB;
	    DO_ADD_MENU_ITEM_LABEL
	    list->menu_shift_down_mi = w;
	}
#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP


	MenuCfgListUpdateWidgets(list);

	return(list);
}

/*
 *	Sets the MenuCfg List custom data list.
 *
 *	The custom_names_list specifies a GList of gchar * strings
 *	describing the names of each custom data. If custom_names_list
 *	is NULL then the custom data list will be cleared.
 */
void MenuCfgListSetCustomData(
	menucfg_list_struct *list,
	GList *custom_names_list
)
{
	const gchar *name;
	GList *glist;

	if(list == NULL)
	    return;

	if(list->custom_data_names != NULL)
	{
	    g_list_foreach(list->custom_data_names, (GFunc)g_free, NULL);
	    g_list_free(list->custom_data_names);
	    list->custom_data_names = NULL;
	}

	for(glist = custom_names_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    name = (const gchar *)glist->data;
	    list->custom_data_names = g_list_append(
		list->custom_data_names,
		STRDUP(name)
	    );
	}

	MenuCfgListUpdateWidgets(list);
}

/*
 *	Returns the total number of custom datas.
 */
gint MeuCfgListGetTotalCustomDatas(menucfg_list_struct *list)
{
	if(list == NULL)
	    return(0);

	return(g_list_length(list->custom_data_names));
}

/*
 *	Sets the MenuCfg List list style.
 */
void MenuCfgListSetListStyle(
	menucfg_list_struct *list, const menucfg_list_style list_style
)
{
	if(list == NULL)
	    return;

	if(list_style != list->list_style)
	{
	    GtkCList *clist = GTK_CLIST(list->clist);

	    list->list_style = list_style;

	    gtk_clist_freeze(clist);
	    switch(list_style)
	    {
	      case MENUCFG_LIST_STYLE_SINGLE:
		gtk_clist_column_titles_hide(clist);
		gtk_clist_column_titles_active(clist);
		gtk_clist_set_column_visibility(clist, 0, TRUE);
		gtk_clist_set_column_visibility(clist, 1, FALSE);
		break;
	      case MENUCFG_LIST_STYLE_DETAILS:
		gtk_clist_column_titles_show(clist);
		gtk_clist_column_titles_passive(clist);
		gtk_clist_set_column_visibility(clist, 0, TRUE);
		gtk_clist_set_column_visibility(clist, 1, TRUE);
		break;
	    }
	    gtk_clist_columns_autosize(clist);
	    gtk_clist_thaw(clist);

	    MenuCfgListUpdateWidgets(list);
	}
}

/*
 *	Sets the MenuCfg List editable.
 */
void MenuCfgListSetEditable(
	menucfg_list_struct *list, const gboolean editable  
)
{
	if(list == NULL)
	    return;

	if(editable && !(list->flags & MENUCFG_LIST_EDITABLE))
	{
	    list->flags |= MENUCFG_LIST_EDITABLE;
	    gtk_widget_show(list->buttons_box);
	    MenuCfgListUpdateWidgets(list);
	}
	else if(!editable && (list->flags & MENUCFG_LIST_EDITABLE))
	{
	    list->flags &= ~MENUCFG_LIST_EDITABLE;
	    gtk_widget_hide(list->buttons_box);
	    MenuCfgListUpdateWidgets(list);
	}
}

/*
 *	Sets the changed callback.
 */
void MenuCfgListSetChangedCB(
	menucfg_list_struct *list,
	void (*cb)(menucfg_list_struct *, gpointer),
	gpointer data
)
{
	if(list == NULL)
	    return;

	list->changed_cb = cb;
	list->changed_data = data;

	MenuCfgListUpdateWidgets(list);
}

/*
 *	Sets the item added callback.
 */
void MenuCfgListSetItemAddedCB(
	menucfg_list_struct *list,
	void (*cb)(
		menucfg_list_struct *,
		gint,
		menucfg_item_struct *,
		gpointer
	),
	gpointer data
)
{
	if(list == NULL)
	    return;

	list->item_added_cb = cb;
	list->item_added_data = data;

	MenuCfgListUpdateWidgets(list);
}

/*
 *	Sets the item changed callback.
 */
void MenuCfgListSetItemChangedCB(
	menucfg_list_struct *list,
	void (*cb)(
		menucfg_list_struct *,
		gint,
		menucfg_item_struct *,
		gpointer
	),
	gpointer data
)
{
	if(list == NULL)
	    return;

	list->item_changed_cb = cb;
	list->item_changed_data = data;

	MenuCfgListUpdateWidgets(list);
}

/*
 *	Sets the item removed callback.
 */
void MenuCfgListSetItemRemovedCB(
	menucfg_list_struct *list,
	void (*cb)(
		menucfg_list_struct *,
		gint,
		gpointer
	),
	gpointer data
)
{
	if(list == NULL)
	    return;

	list->item_removed_cb = cb;
	list->item_removed_data = data;

	MenuCfgListUpdateWidgets(list);
}

/*
 *	Inserts an item.
 */
gint MenuCfgListInsert(
	menucfg_list_struct *list, const gint row,
	menucfg_item_struct *item               /* Transfered */
)
{
	gint i, new_row, ncolumns;
	gchar **strv;
	GtkCList *clist;

	if(list == NULL)
	{
	    MenuCfgItemDelete(item);
	    return(-1);
	}

	clist = GTK_CLIST(list->clist);
	ncolumns = clist->columns;

	gtk_clist_freeze(clist);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	/* Insert or append row */
	if(row > -1)
	    new_row = gtk_clist_insert(clist, row, strv);
	else
	    new_row = gtk_clist_append(clist, strv);

	/* Delete the row cell values */
	g_free(strv);

	/* Failed to create the new row? */
	if(new_row < 0)
	{
	    gtk_clist_thaw(clist);
	    MenuCfgItemDelete(item);
	    return(-1);
	}

	/* Set the row values and transfer the item to the GtkCList
	 * row data
	 */
	MenuCfgListSet(list, new_row, item);

	gtk_clist_thaw(clist);

	MenuCfgListUpdateWidgets(list);

	return(new_row);
}

/*
 *	Appends an item.
 */
gint MenuCfgListAppend(
	menucfg_list_struct *list,
	menucfg_item_struct *item               /* Transfered */
)
{
	return(MenuCfgListInsert(list, -1, item));
}

/*
 *	Sets the row values based on the specified item.
 */
void MenuCfgListSet(
	menucfg_list_struct *list, const gint row,
	menucfg_item_struct *item               /* Transfered */
)
{
	gint column, ncolumns;
	GtkWidget *w;
	GtkCList *clist;

	if(list == NULL)
	{
	    MenuCfgItemDelete(item);
	    return;
	}

	w = list->clist;
	clist = GTK_CLIST(w);
	ncolumns = clist->columns;

	if((row < 0) || (row >= clist->rows))
	{
	    MenuCfgItemDelete(item);
	    return;
	}

	gtk_clist_freeze(clist);

	/* Name */
	column = 0;
	if(column < ncolumns)
	{
	    if((item != NULL) ? (item->icon_file != NULL) : FALSE)
	    {
		GdkBitmap *mask;
		GdkPixmap *pixmap;
		GtkStyle *style = gtk_widget_get_style(w);

		pixmap = gdk_pixmap_create_from_xpm(
		    w->window, &mask,
		    (style != NULL) ? &style->bg[GTK_STATE_NORMAL] : NULL,
		    item->icon_file
		);
		if(pixmap != NULL)
		{
		    gtk_clist_set_pixtext(
			clist, row, column,
			(item->label != NULL) ? item->label : "",
			2,
			pixmap,
			mask
		    );
		    GDK_PIXMAP_UNREF(pixmap);
		    GDK_BITMAP_UNREF(mask);
		}
		else
		{
		    gtk_clist_set_text(
			clist, row, column,
			(item->label != NULL) ? item->label : ""
		    );
		}
	    }
	    else
	    {
		if(item != NULL)
		    gtk_clist_set_text(
			clist, row, column,
			(item->label != NULL) ? item->label : ""
		    );
		else
		    gtk_clist_set_text(
			clist, row, column, ""
		    );
	    }
	}

	/* Command */
	column = 1;
	if(column < ncolumns)
	{
	    if(item != NULL)
		gtk_clist_set_text(
		    clist, row, column,
		    (item->command != NULL) ? item->command : ""
		);
	    else
		gtk_clist_set_text(
		    clist, row, column, ""
		);
	}

	/* Set the this row's data sa the Menu Item */
	gtk_clist_set_row_data_full(
	    clist, row,
	    item, MenuCfgListCListDestroyCB
	);

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);
}


/*
 *	Clears the MenuCfg List.
 */
void MenuCfgListClear(menucfg_list_struct *list)
{
	GtkCList *clist;

	if(list == NULL)
	    return;

	clist = GTK_CLIST(list->clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);

	MenuCfgListUpdateWidgets(list);
}

/*
 *	Returns the Keymap Key for the specified Keymap Key List item.
 */
menucfg_item_struct *MenuCfgListGet(
	menucfg_list_struct *list, gint row
)
{
	GtkCList *clist;

	if(list == NULL)
	    return(NULL);

	clist = GTK_CLIST(list->clist);
	return(MENUCFG_ITEM(
	    gtk_clist_get_row_data(clist, row)
	));
}

/*
 *	Updates the MenuCfg List's widgets.
 */
void MenuCfgListUpdateWidgets(menucfg_list_struct *list)
{
	gboolean b;
	gint nrows, sel_row, nselected;
	GList *glist;
	GtkCList *clist;

	if(list == NULL)
	    return;

	list->freeze_count++;

	clist = GTK_CLIST(list->clist);
	nrows = clist->rows;
	nselected = g_list_length(clist->selection);
	glist = clist->selection_end;
	sel_row = (glist != NULL) ? (gint)glist->data : -1;

	b = (nselected <= 1) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(list->add_btn, b);
	GTK_WIDGET_SET_SENSITIVE(list->menu_add_mi, b);

	b = (nselected == 1) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(list->edit_btn, b);
	GTK_WIDGET_SET_SENSITIVE(list->menu_edit_mi, b);

	b = (nselected >= 1) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(list->remove_btn, b);
	GTK_WIDGET_SET_SENSITIVE(list->menu_remove_mi, b);

	b = ((g_list_find(clist->selection, (gpointer)0) == NULL) &&
	    (nselected > 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(list->shift_up_btn, b);
	GTK_WIDGET_SET_SENSITIVE(list->menu_shift_up_mi, b);

	b = ((g_list_find(clist->selection, (gpointer)(nrows - 1)) == NULL) &&
	    (nselected > 0)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(list->shift_down_btn, b);
	GTK_WIDGET_SET_SENSITIVE(list->menu_shift_down_mi, b);

	list->freeze_count--;
}

/*
 *	Maps the MenuCfg List.
 */
void MenuCfgListMap(menucfg_list_struct *list)
{
	GtkWidget *w;

	if(list == NULL)
	    return;

	w = list->toplevel;
	if(w != NULL)
	    gtk_widget_show(w);
}

/*
 *	Unmaps the MenuCfg List.
 */
void MenuCfgListUnmap(menucfg_list_struct *list)
{
	GtkWidget *w;

	if(list == NULL)
	    return;

	w = list->toplevel;
	if(w != NULL)
	    gtk_widget_hide(w);
}

/*
 *	Deletes the MenuCfg List.
 */
void MenuCfgListDelete(menucfg_list_struct *list)
{
	if(list == NULL)
	    return;

	MenuCfgListClear(list);

	MenuCfgListUnmap(list);

	list->freeze_count++;

	GTK_WIDGET_DESTROY(list->menu);
	GTK_WIDGET_DESTROY(list->toplevel);

	GDK_CURSOR_DESTROY(list->translate_cur);

	g_free(list->browse_path_rtn);

	if(list->custom_data_names != NULL)
	{
	    g_list_foreach(list->custom_data_names, (GFunc)g_free, NULL);
	    g_list_free(list->custom_data_names);
	}

	list->freeze_count--;

	g_free(list);
}
