#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>
#include <time.h>
#include <utime.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <unistd.h>

#include "guiutils.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_date.h"
#include "edv_id.h"
#include "edv_obj.h"
#include "edv_recycled_obj.h"
#include "edv_archive_obj.h"
#include "edv_mime_type.h"
#include "edv_device.h"
#include "edv_recbin_index.h"
#include "edv_recbin_stat.h"
#include "obj_op_dlg.h"
#include "prop_dlg.h"
#include "prop_dlg_op.h"
#include "endeavour2.h"
#include "edv_obj_op.h"
#include "edv_recycled_obj_op.h"
#include "edv_cb.h"
#include "edv_op.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"


void EDVPropDlgDateTouch(edv_propdlg_struct *pd);
void EDVPropDlgDateTouchSpecific(edv_propdlg_struct *pd);

static void EDVPropDlgApplyVFS(edv_propdlg_struct *pd);
static void EDVPropDlgApplyRecBin(edv_propdlg_struct *pd);
void EDVPropDlgApply(edv_propdlg_struct *pd);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Sets the object's time stamps to the current time.
 */
void EDVPropDlgDateTouch(edv_propdlg_struct *pd)
{
	gboolean yes_to_all = FALSE;
	const gulong cur_time = (gulong)time(NULL);
	GtkWidget *toplevel;
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	toplevel = pd->toplevel;
	core = pd->core;
	cfg_list = core->cfg_list;

	EDVPropDlgSetBusy(pd, TRUE);

	/* Handle by the location type */
	switch(pd->location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    if(!STRISEMPTY(pd->path))
	    {
		gint status = 0;
		const gchar	*path = pd->path,
				*error_msg;
		GList *paths_list, *modified_paths_list;
		const edv_object_type type = pd->type;

		/* If this is a link then warn about changing
		 * the time stamps of links
		 */
		if(type == EDV_OBJECT_TYPE_LINK)
		{
		    gint response;
		    EDVPlaySoundWarning(core);
		    CDialogSetTransientFor(toplevel);
		    response = CDialogGetResponse(
"Changing Time Stamps Warning",
"Changing the time stamps on a link will effectively\n\
change the time stamps of its target object.\n\
\n\
Are you sure you want to change the time stamps\n\
on the target object?",
"Links do not have time stamps, instead, their time\n\
stamps are determined by the time stamps of the\n\
target object. So changing the time stamps on a link\n\
will effectively change the time stamps of the target\n\
object.",
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO |
			    CDIALOG_BTNFLAG_HELP,
			CDIALOG_BTNFLAG_YES
		    );
		    CDialogSetTransientFor(NULL);
		    if(response != CDIALOG_RESPONSE_YES)
		    {
			EDVPropDlgSetBusy(pd, FALSE);
			return;
		    }
		}

		/* Create the paths list */
		paths_list = NULL;
		paths_list = g_list_append(paths_list, STRDUP(path));

		/* Set the time stamps to the current time */
		status = EDVObjectOPChTime(
		    core,
		    paths_list,
		    cur_time,			/* Access time */
		    cur_time,			/* Modify time */
		    &modified_paths_list,
		    toplevel,
		    FALSE,			/* Do not show progress */
		    TRUE,			/* Interactive */
		    &yes_to_all,
		    FALSE,			/* Not recursive */
		    TRUE			/* Archive */
		);

		/* Unmap the progress dialog */
		ProgressDialogBreakQuery(FALSE);
		ProgressDialogSetTransientFor(NULL);

		/* Get the error message (if any) */
		error_msg = EDVObjectOPGetError(core);
		if(!STRISEMPTY(error_msg))
		{
		    /* Print the error message */
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Change Time Stamps Error",
			error_msg,
			NULL,
			toplevel
		    );
		}

		/* Notify about the object being modified */
		if(modified_paths_list != NULL)
		{
		    struct stat lstat_buf;
		    const gchar *path;
		    GList *glist;

		    for(glist = modified_paths_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			path = (const gchar *)glist->data;
			if(path == NULL)
			    continue;

			if(!lstat((const char *)path, &lstat_buf))
			    EDVObjectModifiedEmit(
				core,
				path,
				path,
				&lstat_buf
			    );
		    }

		    /* Delete the modified paths list */
		    g_list_foreach(modified_paths_list, (GFunc)g_free, NULL);
		    g_list_free(modified_paths_list);
		}

		/* Play the "completed" sound on success */
		if(status == 0)
		    EDVPlaySoundCompleted(core);

		/* Delete the paths list */
		if(paths_list != NULL)
		{
		    g_list_foreach(paths_list, (GFunc)g_free, NULL);
		    g_list_free(paths_list);
		}
 	    }
	    break;

	  case EDV_LOCATION_TYPE_RECBIN:
	    if(pd->recycled_obj_index > 0)
	    {
		gint status = 0;
		const guint index = pd->recycled_obj_index;
		const gchar *error_msg;
		GList	*indicies_list = NULL,
			*modified_indicies_list;

		/* Create the indicies list */
		indicies_list = g_list_append(
		    indicies_list,
		    (gpointer)index
		);

		/* Set the time stamps to the current time */
		(void)EDVRecycledObjectOPChTime(
		    core,
		    indicies_list,
		    cur_time,			/* Access time */
		    cur_time,			/* Modify time */
		    cur_time,			/* Delete time */
		    &modified_indicies_list,
		    toplevel,
		    FALSE,			/* Do not show progress */
		    TRUE,			/* Interactive */
		    &yes_to_all
		);

		/* Unmap the progress dialog */
		ProgressDialogBreakQuery(FALSE);
		ProgressDialogSetTransientFor(NULL);

		/* Get the error message (if any) */
		error_msg = EDVObjectOPGetError(core);
		if(!STRISEMPTY(error_msg))
		{
		    /* Print the error message */
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Change Time Stamps Error",
			error_msg,
			NULL,
			toplevel
		    );
		}

		/* Notify about the recycled object being modified */
		if(modified_indicies_list != NULL)
		{
		    guint index;
		    GList *glist;

		    for(glist = modified_indicies_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			index = (guint)glist->data;
			if(index == 0)
			    continue;

			EDVRecycledObjectModifiedEmit(core, index);
		    }

		    /* Delete the modified indicies list */
		    g_list_free(modified_indicies_list);
		}

		/* Play the "completed" sound on success */
		if(status == 0)
		    EDVPlaySoundCompleted(core);

		/* Delete the indicies list */
		g_list_free(indicies_list);
	    }
	    break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
"Changing Time Stamps Warning",
"The time stamps of an object in an archive can not be modified.",
		NULL,
		toplevel
	    );
	    break;
	}

	EDVPropDlgSetBusy(pd, FALSE);
}

/*
 *	Queries the user for the new time stamps to set on the object
 *	and sets the object's time stamps.
 */
void EDVPropDlgDateTouchSpecific(edv_propdlg_struct *pd)
{
	GtkWidget *toplevel;
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	toplevel = pd->toplevel;
	core = pd->core;
	cfg_list = core->cfg_list;

	EDVPropDlgSetBusy(pd, TRUE);

	/* Handle by the location type */
	switch(pd->location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    if(!STRISEMPTY(pd->path))
	    {
		const gchar *path = pd->path;
		GList *objs_list = NULL;

		/* Create the objects list */
		edv_object_struct *obj = EDVObjectNew();
		if(obj != NULL)
		{
		    struct stat lstat_buf;

		    /* Set the object's stats */
		    if(!lstat((const char *)path, &lstat_buf))
		    {
			EDVObjectSetPath(obj, path);
			EDVObjectSetStat(obj, &lstat_buf);
			EDVObjectUpdateLinkFlags(obj);
		    }

		    objs_list = g_list_append(
			objs_list,
			obj
		    );
		}

		/* Map the object operations dialog to change time stamps */
		EDVObjOpDlgMapValues(
		    EDVGetObjOpDlg(core),
		    EDV_OBJ_OP_DLG_OP_CHTIME,
		    EDV_LOCATION_TYPE_VFS,
		    objs_list,
		    NULL,
		    toplevel
		);

		/* Delete the objects list */
		if(objs_list != NULL)
		{
		    g_list_foreach(objs_list, (GFunc)EDVObjectDelete, NULL);
		    g_list_free(objs_list);
		}
	    }
	    break;

	  case EDV_LOCATION_TYPE_RECBIN:
	    if(pd->recycled_obj_index > 0)
	    {
		const guint index = pd->recycled_obj_index;
		GList *objs_list = NULL;

		/* Create the recycled objects list */
		edv_recycled_object_struct *obj = EDVRecBinObjectStat(
		    EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX),
		    index
		);
		if(obj != NULL)
		{
		    objs_list = g_list_append(
			objs_list,
			obj
		    );
		}

		/* Map the object operations dialog to change time stamps */
		EDVObjOpDlgMapValues(
		    EDVGetObjOpDlg(core),
		    EDV_OBJ_OP_DLG_OP_CHTIME,
		    EDV_LOCATION_TYPE_RECBIN,
		    objs_list,
		    NULL,
		    toplevel
		);

		/* Delete the recycled objects list */
		if(objs_list != NULL)
		{
		    g_list_foreach(objs_list, (GFunc)EDVRecycledObjectDelete, NULL);
		    g_list_free(objs_list);
		}
	    }
	    break;
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
"Changing Time Stamps Warning",
"The time stamps of an object in an archive can not be modified.",
		NULL,
		toplevel
	    );
	    break;
	}



	EDVPropDlgSetBusy(pd, FALSE);
}


/*
 *	Applies the values from the widgets to the object.
 */
static void EDVPropDlgApplyVFS(edv_propdlg_struct *pd)
{
	struct stat lstat_buf;
	gboolean yes_to_all = FALSE;
	GtkWidget *w, *toplevel = pd->toplevel;
	const gchar *path = pd->path;
	edv_core_struct *core = pd->core;

	if(STRISEMPTY(path))
	    return;

	/* Begin applying values from each page that exists */

	/* Device Node Page */
	w = pd->dev_node_toplevel;
	if(w != NULL)
	{
	    /* Device Numbers */
	    struct stat lstat_buf;
	    gint new_major = 0, new_minor = 0;

	    /* Get the new major & minor numbers */
	    w = pd->dev_node_major_entry;
	    if(w != NULL)
		new_major = ATOI(gtk_entry_get_text(GTK_ENTRY(w)));

	    w = pd->dev_node_minor_entry;
	    if(w != NULL)
	        new_minor = ATOI(gtk_entry_get_text(GTK_ENTRY(w)));

	    /* Get the device node's original statistics */
	    if(!lstat((const char *)path, &lstat_buf))
	    {
		gint major = 0, minor = 0;

		/* Get the current major & minor numbers */
		EDVGetDeviceNumbers(lstat_buf.st_rdev, &major, &minor);

		/* Original device node numbers different from the
		 * new device node numbers?
		 */
		if((new_major != major) || (new_minor != minor))
		{
		    /* Set the device number by recreating the device
		     * node
		     */
		    const gint rdev = EDVFormatDeviceNumbers(
			new_major, new_minor
		    );

		    /* Remove the existing device */
		    if(unlink((const char *)path))
		    {
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to change the device numbers.\n\
\n\
%s.",
			    g_strerror(error_code)
			);
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Change Device Numbers Error",
			    msg,
			    NULL,
			    toplevel
			);
			g_free(msg);
		    }
		    /* Recreate the device */
		    else if(mknod(
			(const char *)path,
			lstat_buf.st_mode,
			(dev_t)rdev
		    ))
		    {
			const gint error_code = (gint)errno;
			gchar *msg = g_strdup_printf(
"Unable to change the device numbers.\n\
\n\
%s.",
			    g_strerror(error_code)
			);
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Change Device Numbers Error",
			    msg,
			    NULL,
			    toplevel
			);
			g_free(msg);
		    }
		    else
		    {
			struct utimbuf utime_buf;

			/* Restore the ownership */
			lchown(
			    (const char *)path,
			    lstat_buf.st_uid, lstat_buf.st_gid
			);

			/* Restore the time stamps */
			utime_buf.actime = lstat_buf.st_atime;
			utime_buf.modtime = lstat_buf.st_mtime;
			utime((const char *)path, &utime_buf);
		    }
		}
	    }
	    else
	    {
		/* Unable to get the device node's statistics */
		const gint error_code = (gint)errno;
		gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
		    g_strerror(error_code), path
		);
		EDVPlaySoundError(core);
		EDVMessageError(
		    "Change Device Numbers Error",
		    msg,
		    NULL,
		    toplevel
		);
		g_free(msg);
	    }
	}

	/* Link Page */
	w = pd->link_toplevel;
	if(w != NULL)
	{
	    /* Link Target */
	    w = pd->link_target_entry;
	    if(w != NULL)
	    {
		gchar	*target = STRDUP(gtk_entry_get_text(GTK_ENTRY(w))),
			*cur_target = EDVGetLinkTarget(path);

		/* Links may not have empty target values */
		if(STRISEMPTY(target))
		{
		    /* Set the empty target value to "undefined" */
		    g_free(target);
		    target = STRDUP("undefined");
		}

		/* Is the new target value different? */
		if(((target != NULL) && (cur_target != NULL)) ?
		    strcmp((const char *)target, (const char *)cur_target) : TRUE
		)
		{
		    /* Set the new target value */
		    const gchar *error_msg;
		    GList *modified_paths_list;

		    /* Relink */
		    EDVObjectOPRelink(
			core,
			path,			/* Link */
			target,			/* Target */
			&modified_paths_list,
			toplevel,
			TRUE, TRUE,
			&yes_to_all
		    );

		    /* Unmap the progress dialog */
		    ProgressDialogBreakQuery(FALSE);
		    ProgressDialogSetTransientFor(NULL);

		    /* Check for errors */
		    error_msg = EDVObjectOPGetError(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			EDVPlaySoundError(core);
			EDVMessageError(
			    "Change Link Target Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Delete the modified paths list */
		    if(modified_paths_list != NULL)
		    {
			g_list_foreach(
			    modified_paths_list, (GFunc)g_free, NULL
			);
			g_list_free(modified_paths_list);
		    }
		}

		g_free(target);
		g_free(cur_target);
	    }
	}

	/* General Page */
	w = pd->gen_toplevel;
	if(w != NULL)
	{
	    /* Permissions */
	    w = pd->gen_permissions_frame;
	    if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
	    {
		const gchar *error_msg;
		GList *paths_list, *modified_paths_list;
		edv_permission_flags permissions = 0x00000000;

		/* Get the current permission values */
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_ur_check))
		    permissions |= EDV_PERMISSION_UREAD;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_uw_check))
		    permissions |= EDV_PERMISSION_UWRITE;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_ux_check))
		    permissions |= EDV_PERMISSION_UEXECUTE;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_gr_check))
		    permissions |= EDV_PERMISSION_GREAD;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_gw_check))
		    permissions |= EDV_PERMISSION_GWRITE;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_gx_check))
		    permissions |= EDV_PERMISSION_GEXECUTE;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_or_check))
		    permissions |= EDV_PERMISSION_AREAD;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_ow_check))
		    permissions |= EDV_PERMISSION_AWRITE;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_ox_check))
		    permissions |= EDV_PERMISSION_AEXECUTE;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_suid_check))
		    permissions |= EDV_PERMISSION_SETUID;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_sgid_check))
		    permissions |= EDV_PERMISSION_SETGID;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_sticky_check))
		    permissions |= EDV_PERMISSION_STICKY;

		/* Create the paths list */
		paths_list = NULL;
		paths_list = g_list_append(paths_list, STRDUP(path));

		/* Change the permissions */
		EDVObjectOPChMod(
		    core,
		    paths_list,
		    permissions,
		    &modified_paths_list,
		    toplevel,
		    FALSE,		/* Do not show progress */
		    TRUE,		/* Interactive */
		    &yes_to_all,
		    FALSE,		/* Not recursive */
		    TRUE		/* Archive */
		);

		/* Unmap the progress dialog just in case */
		ProgressDialogBreakQuery(FALSE);
		ProgressDialogSetTransientFor(NULL);

		/* Check for errors */
		error_msg = EDVObjectOPGetError(core);
		if(!STRISEMPTY(error_msg))
		{
		    /* Report the error */
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Change Permissions Error",
			error_msg,
			NULL,
			toplevel
		    );
		}

		/* Delete the modified paths list */
		if(modified_paths_list != NULL)
		{
		    g_list_foreach(
			modified_paths_list, (GFunc)g_free, NULL
		    );
		    g_list_free(modified_paths_list);
		}

		if(paths_list != NULL)
		{
		    g_list_foreach(paths_list, (GFunc)g_free, NULL);
		    g_list_free(paths_list);
		}
	    }

	    /* Owner & Group */
	    w = pd->gen_ownership_frame;
	    if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
	    {
		struct stat lstat_buf;
		gint owner_id = 0, group_id = 0;
		const gchar *error_msg;

		/* Get user id from user entry */
		w = pd->gen_owner_entry;
		if(w != NULL)
		    owner_id = EDVUIDNameToUID(
			core->uids_list,
			gtk_entry_get_text(GTK_ENTRY(w))
		    );

		/* Get group id from group entry */
		w = pd->gen_group_entry;
		if(w != NULL)
		    group_id = EDVGIDNameToGID(
			core->gids_list,
			gtk_entry_get_text(GTK_ENTRY(w))
		    );

		/* Get the object's original statistics */
		if(!lstat((const char *)path, &lstat_buf))
		{
		    /* Are the new owner or group different? */
		    if((owner_id != (gint)lstat_buf.st_uid) ||
		       (group_id != (gint)lstat_buf.st_gid)
		    )
		    {
			GList *paths_list, *modified_paths_list;

			/* Create the paths list */
			paths_list = NULL;
			paths_list = g_list_append(paths_list, STRDUP(path));

			/* Change the ownership */
			EDVObjectOPChOwn(
			    core,
			    paths_list,
			    owner_id,
			    group_id,
			    &modified_paths_list,
			    toplevel,
			    FALSE,	/* Do not show progress */
			    TRUE,	/* Interactive */
			    &yes_to_all,
			    FALSE,	/* Not recursive */
			    TRUE	/* Archive */
			);

			/* Unmap the progress dialog */
			ProgressDialogBreakQuery(FALSE);
			ProgressDialogSetTransientFor(NULL);

			/* Check for errors */
			error_msg = EDVObjectOPGetError(core);
			if(!STRISEMPTY(error_msg))
			{
			    /* Report the error */
			    EDVPlaySoundError(core);
			    EDVMessageError(
				"Change Ownership Error",
				error_msg,
				NULL,
				toplevel
			    );
			}

			/* Delete the modified paths list */
			if(modified_paths_list != NULL)
			{
			    g_list_foreach(
				modified_paths_list, (GFunc)g_free, NULL
			    );
			    g_list_free(modified_paths_list);
			}

			if(paths_list != NULL)
			{
			    g_list_foreach(paths_list, (GFunc)g_free, NULL);
			    g_list_free(paths_list);
			}
		    }
		}
		else
		{
		    /* Unable to get the object's statistics */
		    const gint error_code = (gint)errno;
		    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
			g_strerror(error_code),
			path
		    );
		    EDVPlaySoundError(core);
		    EDVMessageError(
			"Change Ownership Error",
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
	    }	/* Owner & Group */
	}

	/* Notify about this object being modified */
	if(lstat((const char *)path, &lstat_buf))
	{
	    const gint error_code = (gint)errno;
#ifdef ENOENT
	    if(error_code == ENOENT)
		EDVObjectRemovedEmit(core, path);
#endif
	}
	else
	{
	    EDVObjectModifiedEmit(
		core, path, path, &lstat_buf
	    );
	}
}

/*
 *	Applies the values from the widgets to the recycled object.
 */
static void EDVPropDlgApplyRecBin(edv_propdlg_struct *pd)
{
	gboolean yes_to_all = FALSE;
	GtkWidget *w, *toplevel = pd->toplevel;
	const gchar *index_file;
	const gchar *path = pd->path;
	const guint index = pd->recycled_obj_index;
	edv_recycled_object_struct *obj;
	edv_core_struct *core = pd->core;
	cfg_item_struct *cfg_list = core->cfg_list;

	if(STRISEMPTY(path) || (index == 0))
	    return;

	/* Get the recycled objects index file */
	index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);

	/* Get the current recycled object's statistics */
	obj = EDVRecBinObjectStat(index_file, index);
	if(obj == NULL)
	{
	    /* Unable to get the object's statistics */
	    gchar *msg = g_strdup_printf(
"%s:\n\
\n\
    %s",
		EDVRecBinIndexGetError(),
		path
	    );
	    EDVPlaySoundError(core);
	    EDVMessageError(
		"Apply Failed",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	    return;
	}

	/* Link Page */
	w = pd->link_toplevel;
	if(w != NULL)
	{
	    /* Link Target */
	    w = pd->link_target_entry;
	    if(w != NULL)
	    {
		gboolean need_set_new_target;
		const gchar *cur_target = obj->link_target;
		gchar *new_target = STRDUP(gtk_entry_get_text(GTK_ENTRY(w)));

		/* Check if there is a change in the target value? */
		if((cur_target != NULL) && (new_target != NULL))
		{
		    if(strcmp((const char *)cur_target, (const char *)new_target))
			need_set_new_target = TRUE;
		    else
			need_set_new_target = FALSE;
		}
		else
		{
		    need_set_new_target = TRUE;
		}
		if(need_set_new_target)
		{
		    const gchar *error_msg;
		    GList *modified_indicies_list;

		    /* Set the new target */
		    (void *)EDVRecycledObjectOPRelink(
			core,
			obj->index,
			new_target,
			&modified_indicies_list,
			toplevel,
			FALSE,			/* Do not show progress */
			TRUE,			/* Interactive */
			&yes_to_all
		    );

		    /* Unmap the progress dialog */
		    ProgressDialogBreakQuery(FALSE);
		    ProgressDialogSetTransientFor(NULL);

                    /* Check for errors */
                    error_msg = EDVRecycledObjectOPGetError(core);
                    if(!STRISEMPTY(error_msg))
                    {
                        /* Report the error */
                        EDVPlaySoundError(core);
                        EDVMessageError(
                            "Change Link Target Error",
                            error_msg,
                            NULL,
                            toplevel
                        );
                    }

		    g_list_free(modified_indicies_list);
		}

		g_free(new_target);
	    }
	}

	/* General Page */
	w = pd->gen_toplevel;
	if(w != NULL)
	{
	    /* Permissions */
	    w = pd->gen_permissions_frame;
	    if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
	    {
		edv_permission_flags permissions = 0x00000000;

		/* Get the current permission values */
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_ur_check))
		    permissions |= EDV_PERMISSION_UREAD;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_uw_check))
		    permissions |= EDV_PERMISSION_UWRITE;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_ux_check))
		    permissions |= EDV_PERMISSION_UEXECUTE;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_gr_check))
		    permissions |= EDV_PERMISSION_GREAD;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_gw_check))
		    permissions |= EDV_PERMISSION_GWRITE;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_gx_check))
		    permissions |= EDV_PERMISSION_GEXECUTE;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_or_check))
		    permissions |= EDV_PERMISSION_AREAD;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_ow_check))
		    permissions |= EDV_PERMISSION_AWRITE;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_ox_check))
		    permissions |= EDV_PERMISSION_AEXECUTE;

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_suid_check))
		    permissions |= EDV_PERMISSION_SETUID;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_sgid_check))
		    permissions |= EDV_PERMISSION_SETGID;
		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(pd->gen_sticky_check))
		    permissions |= EDV_PERMISSION_STICKY;

		/* Check if there is a change in the permissions */
		if(obj->permissions != permissions)
		{
		    const gchar *error_msg;
		    GList	*indicies_list = NULL,
				*modified_indicies_list;

		    indicies_list = g_list_append(
			indicies_list,
			(gpointer)obj->index
		    );

		    /* Set the new permissions */
		    obj->permissions = permissions;
		    (void *)EDVRecycledObjectOPChMod(
			core,
			indicies_list,
			obj->permissions,
			&modified_indicies_list,
			toplevel,
			FALSE,			/* Do not show progress */
			TRUE,			/* Interactive */
			&yes_to_all
		    );

		    /* Unmap the progress dialog */
		    ProgressDialogBreakQuery(FALSE);
		    ProgressDialogSetTransientFor(NULL);

                    /* Check for errors */
                    error_msg = EDVRecycledObjectOPGetError(core);
                    if(!STRISEMPTY(error_msg))
                    {
                        /* Report the error */
                        EDVPlaySoundError(core);
                        EDVMessageError(
                            "Change Permissions Error",
                            error_msg,
                            NULL,
                            toplevel
                        );
                    }

		    g_list_free(modified_indicies_list);
		    g_list_free(indicies_list);
		}
	    }

	    /* Owner & Group */
	    w = pd->gen_ownership_frame;
	    if((w != NULL) ? GTK_WIDGET_SENSITIVE(w) : FALSE)
	    {
		const gint	cur_owner_id = obj->owner_id,
				cur_group_id = obj->group_id;

		/* Get the user id from user entry */
		w = pd->gen_owner_entry;
		if(w != NULL)
		    obj->owner_id = EDVUIDNameToUID(
			core->uids_list,
			gtk_entry_get_text(GTK_ENTRY(w))
		    );

		/* Get the group id from group entry */
		w = pd->gen_group_entry;
		if(w != NULL)
		    obj->group_id = EDVGIDNameToGID(
			core->gids_list,
			gtk_entry_get_text(GTK_ENTRY(w))
		    );

		/* Check if there is a change in the owner or group */
		if((cur_owner_id != obj->owner_id) ||
		   (cur_group_id != obj->group_id)
		)
		{
		    const gchar *error_msg;
		    GList	*indicies_list = NULL,
				*modified_indicies_list;

		    indicies_list = g_list_append(
			indicies_list,
			(gpointer)obj->index
		    );

		    /* Set the new ownership */
		    (void *)EDVRecycledObjectOPChOwn(
			core,
			indicies_list,
			obj->owner_id, obj->group_id,
			&modified_indicies_list,
			toplevel,
			FALSE,			/* Do not show progress */
			TRUE,			/* Interactive */
			&yes_to_all
		    );

		    /* Unmap the progress dialog */
		    ProgressDialogBreakQuery(FALSE);
		    ProgressDialogSetTransientFor(NULL);

                    /* Check for errors */
                    error_msg = EDVRecycledObjectOPGetError(core);
                    if(!STRISEMPTY(error_msg))
                    {
                        /* Report the error */
                        EDVPlaySoundError(core);
                        EDVMessageError(
                            "Change Ownership Error",
                            error_msg,
                            NULL,
                            toplevel
                        );
                    }

		    g_list_free(modified_indicies_list);
		    g_list_free(indicies_list);
		}
	    }
	}

	/* Delete the recycled object's stats */
	EDVRecycledObjectDelete(obj);

	/* Notify about this recycled object being modified */
	EDVRecycledObjectModifiedEmit(core, index);
}

/*
 *	Applies the values from the widgets to the object.
 */
void EDVPropDlgApply(edv_propdlg_struct *pd)
{
	GtkWidget *toplevel;
	edv_core_struct *core;

	if(pd == NULL)
	    return;

	toplevel = pd->toplevel;
	core = pd->core;

	/* Check if the global write protect is on */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	    return;

	/* Check for objects on locations that you can not touch */
	switch(pd->location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    EDVPropDlgApplyVFS(pd);
	    break;
	  case EDV_LOCATION_TYPE_RECBIN:
	    EDVPropDlgApplyRecBin(pd);
	    break;
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    EDVPlaySoundWarning(core);
	    EDVMessageWarning(
"Apply Failed",
"Unable to modify the properties of an object in an archive.",
		NULL,
		toplevel
	    );
	    break;
	}
}
