#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
#if defined(HAVE_REGEX)
# include <regex.h>
#else
# include <fnmatch.h>
#endif
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "fprompt.h"
#include "cdialog.h"
#include "progressdialog.h"

#include "cfg.h"
#include "edv_types.h"
#include "edv_date.h"
#include "edv_id.h"
#include "edv_obj.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_recycled_obj.h"
#include "edv_recbin_index.h"
#include "edv_recbin_stat.h"
#include "edv_mime_type.h"
#include "recbin.h"
#include "recbin_cb.h"
#include "recbin_contents_list.h"
#include "endeavour2.h"
#include "edv_cb.h"
#include "edv_recycled_obj_op.h"
#include "edv_utils.h"
#include "edv_utils_gtk.h"

#include "edv_cfg_list.h"
#include "config.h"


typedef struct _EDVRecBinContentsListFPromptData	EDVRecBinContentsListFPromptData;
#define EDV_RECBIN_CONTENTS_LIST_FPROMPT_DATA(p)	((EDVRecBinContentsListFPromptData *)(p))


static edv_recycled_object_struct *EDVRecBinNewErrorObject(
	const guint index,
	const gchar *name
);

/* Columns */
static void EDVRecBinContentsResetColumns(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist
);
void EDVRecBinContentsResizeColumnOptimul(
	edv_recbin_struct *recbin,
	const gint column_num
);
void EDVRecBinContentsResizeColumnsOptimul(edv_recbin_struct *recbin);

/* Cell Setting */
static void EDVRecBinContentsSetCellIndex(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
);
static void EDVRecBinContentsSetCellName(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
);
static void EDVRecBinContentsSetCellSize(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size, const gboolean hide_link_size,
	const edv_size_format size_format,
	const gulong block_size
);
static void EDVRecBinContentsSetCellType(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
);
static void EDVRecBinContentsSetCellPermissions(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	gboolean hide_link_permissions
);
static void EDVRecBinContentsSetCellOwner(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
);
static void EDVRecBinContentsSetCellGroup(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
);
static void EDVRecBinContentsSetCellDateAccess(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
);
static void EDVRecBinContentsSetCellDateModified(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
);
static void EDVRecBinContentsSetCellDateChanged(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
);
static void EDVRecBinContentsSetCellDateDeleted(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	const edv_date_relativity date_relativity, const gchar *date_format
);
static void EDVRecBinContentsSetCellLinkedTo(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
);
static void EDVRecBinContentsSetCellOriginalLocation(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
);
static void EDVRecBinContentsSetCellCapacityUsed(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	const gulong recbin_size_max
);
static void EDVRecBinContentsSetRow(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row
);

static gint EDVRecBinContentsAppendObject(
	edv_recbin_struct *recbin,
	edv_recycled_object_struct *obj
);
static void EDVRecBinContentsAppendListing(
	edv_recbin_struct *recbin,
	const gboolean update_status_bar
);

gint EDVRecBinContentsFindRowByIndex(
	edv_recbin_struct *recbin,
	const guint index
);

/* Realize Listing */
void EDVRecBinContentsRealizeListing(edv_recbin_struct *recbin);

/* Get Listing */
void EDVRecBinContentsGetListing(
	edv_recbin_struct *recbin,
	const gboolean update_status_bar
);
void EDVRecBinContentsClear(edv_recbin_struct *recbin);

/* Renaming */
static void EDVRecBinContentsFPromptRenameApplyCB(
	gpointer data, const gchar *value
);
static void EDVRecBinContentsFPromptRenameCancelCB(gpointer data);
void EDVRecBinContentsPromptRename(
	edv_recbin_struct *recbin,
	const gint row,
	const gint column
);

/* Callbacks */
void EDVRecBinContentsObjectAddedNotify(
	edv_recbin_struct *recbin, const guint index
);
void EDVRecBinContentsObjectModifiedNotify(
	edv_recbin_struct *recbin, const guint index
);
void EDVRecBinContentsObjectRemovedNotify(
	edv_recbin_struct *recbin, const guint index
);


struct _EDVRecBinContentsListFPromptData {
	edv_recbin_struct	*recbin;
	gint		row,
			column;
};


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new error object.
 *
 *	The name specifies the name of the error object.
 */
static edv_recycled_object_struct *EDVRecBinNewErrorObject(
	const guint index,
	const gchar *name
)
{
	edv_recycled_object_struct *obj = EDVRecycledObjectNew();
	if(obj == NULL)
	    return(NULL);

	obj->name = STRDUP(name);
	obj->index = index;
	obj->type = EDV_OBJECT_TYPE_ERROR;

	return(obj);
}


/*
 *	Resets the Contents List's column headings to the names and
 *	ordering specified by the configuration.
 *
 *	All inputs assumed valid.
 */
static void EDVRecBinContentsResetColumns(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist
)
{
	gint i, width;
	const gchar *title = NULL;
	GList *glist;             
	GtkJustification justify = GTK_JUSTIFY_LEFT;
	GtkRcStyle *lists_rcstyle = core->lists_rcstyle;
	cfg_intlist_struct	*column_types_intlist,
				*column_width_intlist;
	const cfg_item_struct *cfg_list = core->cfg_list;
	edv_recbin_column_type column_type;

	/* Get the column types mapping */
	column_types_intlist = EDV_GET_INTLIST(
	    EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
	    return;

	/* Get the column widths */
	column_width_intlist = EDV_GET_INTLIST(
	    EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN_WIDTH
	);
	if(column_width_intlist == NULL)
	    return;

	gtk_clist_freeze(clist);

	/* Update the clist's column settings */
	gtk_clist_column_titles_active(clist);
	gtk_clist_column_titles_show(clist);
	gtk_clist_set_auto_sort(clist, FALSE);
	gtk_clist_set_sort_type(clist, GTK_SORT_DESCENDING);

#if 0
/* Already set */
	/* Change clist selection mode to GTK_SELECTION_EXTENDED
	 *
	 * The selection mode can change whenever the contents list is
	 * updated
	 */
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
#endif

	/* Iterate through each column */
	for(i = 0, glist = column_types_intlist->list;
	    glist != NULL;
	    i++, glist = g_list_next(glist)
	)
	{
	    column_type = (edv_recbin_column_type)glist->data;

	    /* Get the width for this column type */
	    width = (gint)g_list_nth_data(
		column_width_intlist->list,
		(guint)column_type
	    );

	    /* Get column title and justification  based on the
	     * column type
	     */
	    switch(column_type)
	    {
	      case EDV_RECBIN_COLUMN_TYPE_INDEX:
		title = "Index";
		justify = GTK_JUSTIFY_RIGHT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_NAME:
		title = "Name";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_SIZE:
		title = "Size";
		justify = GTK_JUSTIFY_RIGHT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_TYPE:
		title = "Type";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_PERMISSIONS:
		title = "Permissions";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_OWNER:
		title = "Owner";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_GROUP:
		title = "Group";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_DATE_ACCESS:
		title = "Date Access";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_DATE_MODIFIED:
		title = "Date Modified";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_DATE_CHANGED:
		title = "Date Changed";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_DATE_DELETED:
		title = "Date Deleted";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_LINKED_TO:
		title = "Linked To";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_ORIGINAL_LOCATION:
		title = "Original Location";
		justify = GTK_JUSTIFY_LEFT;
		break;
	      case EDV_RECBIN_COLUMN_TYPE_CAPACITY_USED:
		title = "Capacity Used";
		justify = GTK_JUSTIFY_LEFT;
		break;
	    }

	    gtk_clist_set_column_visibility(clist, i, TRUE);
	    gtk_clist_set_column_auto_resize(
		clist, i, FALSE
	    );
	    gtk_clist_set_column_title(clist, i, title);
	    gtk_clist_set_column_width(clist, i, width);
	    gtk_clist_set_column_justification(clist, i, justify);
	}
	/* Reset the rest of the columns */
	for(; i < clist->columns; i++)
	    gtk_clist_set_column_visibility(clist, i, FALSE);

	/* Set RC style after column headings have been mapped */
	if(lists_rcstyle != NULL)
	    gtk_widget_modify_style_recursive(
		GTK_WIDGET(clist), lists_rcstyle
	    );

	gtk_clist_thaw(clist);
}

/*
 *      Resizes the column to optimul size.
 */
void EDVRecBinContentsResizeColumnOptimul(
	edv_recbin_struct *recbin,
	const gint column_num
)
{
	gint ncolumns, width;
	GtkCList *clist;

	if(recbin == NULL)
	    return;

	clist = GTK_CLIST(recbin->contents_clist);
	ncolumns = clist->columns;
	if((column_num < 0) || (column_num >= ncolumns))
	    return;

	gtk_clist_freeze(clist);

	width = gtk_clist_optimal_column_width(clist, column_num);
	gtk_clist_set_column_width(clist, column_num, width);

	gtk_clist_thaw(clist);
}

/*
 *      Resizes all the columns to optimul sizes.
 */
void EDVRecBinContentsResizeColumnsOptimul(edv_recbin_struct *recbin)
{
	gint i, ncolumns;
	GtkCList *clist;

	if(recbin == NULL)
	    return;

	clist = GTK_CLIST(recbin->contents_clist);
	ncolumns = clist->columns;

	gtk_clist_freeze(clist);

	for(i = 0; i < ncolumns; i++)
	    EDVRecBinContentsResizeColumnOptimul(recbin, i);

	gtk_clist_thaw(clist);
}


/*
 *	Sets the cell's index value and attributes.
 */
static void EDVRecBinContentsSetCellIndex(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
)
{
	const gint border_minor = 2;
	const guint index = obj->index;
 	gchar num_str[40];

	/* Format the index string */
	if(index != 0)
	    g_snprintf(
		num_str, sizeof(num_str),
		"#%u",
		obj->index
	    );
	else
	    *num_str = '\0';

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column,
	    num_str
	);
	gtk_clist_set_shift(
	    clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's name value and attributes.
 */
static void EDVRecBinContentsSetCellName(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
)
{
	const gchar *name = obj->name;
	GdkPixmap *pixmap, *pixmap_hid;
	GdkBitmap *mask, *mask_hid;

	/* Get the icon */
	EDVMatchObjectIcon(
	    NULL, 0,
	    core->mimetype, core->total_mimetypes,
	    obj->type, name,
	    TRUE, obj->permissions,
	    0,			/* Small icons */
	    &pixmap, &mask,
	    NULL, NULL,
	    NULL, NULL,
	    &pixmap_hid, &mask_hid
	);

	/* Check if an alternate state icon should be used
	 *
	 * Hidden
	 */
	if(EDVIsObjectNameHidden(name))
	{
	    if(pixmap_hid != NULL)
	    {
		pixmap = pixmap_hid;
		mask = mask_hid;
	    }
	}

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(pixmap != NULL)
	    gtk_clist_set_pixtext(
		clist, row, column,
		(name != NULL) ? name : "(null)",
		EDV_LIST_PIXMAP_TEXT_SPACING,
		pixmap, mask
	    );
	else
	    gtk_clist_set_text(
		clist, row, column,
		(name != NULL) ? name : "(null)"
	    );
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's size value and attributes.
 */
static void EDVRecBinContentsSetCellSize(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	const gboolean hide_dir_size, const gboolean hide_link_size,
	const edv_size_format size_format,
	const gulong block_size
)
{
	const gint border_minor = 2;
	const edv_object_type type = obj->type;
	const gchar *s;

	/* Get size string */
	if(hide_dir_size && (type == EDV_OBJECT_TYPE_DIRECTORY))
	    s = "";
	else if(hide_link_size && (type == EDV_OBJECT_TYPE_LINK))
	    s = "";
	else if((type == EDV_OBJECT_TYPE_DEVICE_BLOCK) ||
		(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	)
	    s = "";
	else
	    s = EDVSizeStrFormat(
		obj->size, size_format, block_size, ',', TRUE
	    );

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, (s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, -border_minor
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's MIME Type value.
 */
static void EDVRecBinContentsSetCellType(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
)
{
	gchar *type_str;

	/* Get MIME Type type string for the given object */
	EDVMatchObjectTypeString(
	    core->mimetype, core->total_mimetypes,
	    obj->type, obj->permissions,
	    obj->name,
	    &type_str
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, (type_str != NULL) ? type_str : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's permissions value and attributes.
 */
static void EDVRecBinContentsSetCellPermissions(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column, gboolean hide_link_permissions
)
{
	const edv_permission_flags permissions = obj->permissions;
	gchar *s;

	/* Get permissions string */
	if(hide_link_permissions && (obj->type == EDV_OBJECT_TYPE_LINK))
	    s = NULL;
	else
	    s = EDVGetPermissionsString(permissions);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column,
	    (s != NULL) ? s : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);

	g_free(s);
}

/*
 *	Sets the cell's owner value and attributes.
 */
static void EDVRecBinContentsSetCellOwner(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
)
{
	/* Get owner name from object's user id */
	gchar *owner_name = EDVUIDGetNameFromUID(
	    core->uids_list,
	    obj->owner_id, NULL
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column,
	    (owner_name != NULL) ? owner_name : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);

	g_free(owner_name);
}

/*
 *	Sets the cell's group value and attributes.
 */
static void EDVRecBinContentsSetCellGroup(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
)
{
	/* Get group name from object's group id */
	gchar *group_name = EDVGIDGetNameFromGID(
	    core->gids_list,
	    obj->group_id, NULL
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column,
	    (group_name != NULL) ? group_name : ""
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);

	g_free(group_name);
}

/*
 *	Sets the cell's access date value and attributes.
 */
static void EDVRecBinContentsSetCellDateAccess(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->access_time;
	const gchar *date_str = (t > 0l) ?
	    EDVDateFormatString(t, date_format, date_relativity) : "";

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column,
	    date_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's modified date value and attributes.
 */
static void EDVRecBinContentsSetCellDateModified(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->modify_time;
	const gchar *date_str = (t > 0l) ?
	    EDVDateFormatString(t, date_format, date_relativity) : "";

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column,
	    date_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's changed date value and attributes.
 */
static void EDVRecBinContentsSetCellDateChanged(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	edv_date_relativity date_relativity, const gchar *date_format
)
{
	/* Get/format the date string */
	const gulong t = obj->change_time;
	const gchar *date_str = (t > 0l) ?
	    EDVDateFormatString(t, date_format, date_relativity) : "";

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column,
	    date_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's deleted date value and attributes.
 */
static void EDVRecBinContentsSetCellDateDeleted(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	const edv_date_relativity date_relativity, const gchar *date_format
)
{
	const gchar *date_str;
	const gulong t = obj->deleted_time;
	if(t > 0l)
	    date_str = EDVDateFormatString(
		t, date_format, date_relativity
	    );
	else
	    date_str = "";

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column,
	    date_str
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}


/*
 *	Sets the cell's linked to value and attributes.
 */
static void EDVRecBinContentsSetCellLinkedTo(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
)
{
	/* Get the link's target value */
	const gchar *target = ((obj->link_target != NULL) &&
	    (obj->type == EDV_OBJECT_TYPE_LINK)) ?
	    obj->link_target : "";

	/* Set the cell */
	gtk_clist_freeze(clist);
	gtk_clist_set_text(
	    clist, row, column, target
	);
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's original location value and attributes.
 */
static void EDVRecBinContentsSetCellOriginalLocation(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column
)
{
	GdkBitmap	*mask = NULL,
			*mask_hid = NULL,
			*mask_ext = NULL;
	GdkPixmap	*pixmap = NULL,
			*pixmap_hid = NULL,
			*pixmap_ext = NULL;
	const gchar *location = (obj->original_path != NULL) ?
	    obj->original_path : "";
	edv_object_struct *loc_obj = EDVObjectNew();
	if(loc_obj != NULL)
	{
	    struct stat lstat_buf;

	    EDVObjectSetPath(loc_obj, location);
	    if(!lstat((const char *)location, &lstat_buf))
	    {
		EDVObjectSetStat(loc_obj, &lstat_buf);
		EDVObjectUpdateLinkFlags(loc_obj);
		EDVMatchObjectIcon(
		    core->device, core->total_devices,
		    core->mimetype, core->total_mimetypes,
		    loc_obj->type,
		    loc_obj->full_path,
		    EDV_OBJECT_IS_LINK_VALID(loc_obj),
		    loc_obj->permissions,
		    0,			/* Small icons */
		    &pixmap, &mask,
		    NULL, NULL,
		    &pixmap_ext, &mask_ext,
		    &pixmap_hid, &mask_hid
		);
		/* Hidden */
		if(EDVIsObjectHidden(loc_obj))
		{
		    if(pixmap_hid != NULL)
		    {
			pixmap = pixmap_hid;
			mask = mask_hid;
		    }
		}
	    }
	    else
	    {
		edv_mime_type_struct *m = EDVMimeTypesListMatchType(
		    core->mimetype, core->total_mimetypes,
		    NULL,
		    EDV_MIME_TYPE_TYPE_INODE_DIRECTORY,
		    FALSE
		);
		if(m != NULL)
		{
		    pixmap = m->small_pixmap[EDV_MIME_TYPE_ICON_STATE_STANDARD];
		    mask = m->small_mask[EDV_MIME_TYPE_ICON_STATE_STANDARD];
		}
	    }
	    EDVObjectDelete(loc_obj);
	}

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(pixmap != NULL)
	    gtk_clist_set_pixtext(
		clist, row, column,
		location,
		EDV_LIST_PIXMAP_TEXT_SPACING,
		pixmap, mask
	    );
	else
	    gtk_clist_set_text(
		clist, row, column, location
	    );
	gtk_clist_set_shift(
	    clist, row, column, 0, 0
	);
	gtk_clist_thaw(clist);
}

/*
 *	Sets the cell's capacity used value and attributes.
 */
static void EDVRecBinContentsSetCellCapacityUsed(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row, const gint column,
	const gulong recbin_size_max
)
{
	gfloat usage_coeff;
	GdkBitmap *mask = NULL;
	GdkPixmap *pixmap;
	GtkCListColumn *column_ptr = ((column >= 0) && (column < clist->columns)) ?
	    &clist->column[column] : NULL;
	const gint	width = (column_ptr != NULL) ?
	    column_ptr->width : 60,
			height = GTK_CLIST_ROW_HEIGHT_SET(clist) ?
	    (clist->row_height - 4) : -1;

	if(recbin_size_max == 0l)
	{
	    gtk_clist_freeze(clist);
	    gtk_clist_set_text(
		clist, row, column, ""
	    );
	    gtk_clist_thaw(clist);
	    return;
	}

	/* Calculate this recycled object's usage coefficient */
	usage_coeff = CLIP(
	    ((gfloat)obj->size / (gfloat)recbin_size_max), 0.0f, 1.0f
	);

	/* Create the usage pixmap */
	pixmap = EDVNewProgressPixmap(
	    width, height,
	    usage_coeff,
	    TRUE,				/* Draw value */
	    GTK_ORIENTATION_HORIZONTAL,
	    FALSE,				/* Not reverse */
	    gtk_widget_get_style(GTK_WIDGET(clist)),
	    &mask
	);

	/* Set the cell */
	gtk_clist_freeze(clist);
	if(pixmap != NULL)
	    gtk_clist_set_pixmap(clist, row, column, pixmap, mask);
	else 
	    gtk_clist_set_text(clist, row, column, "");
	gtk_clist_set_cell_style(clist, row, column, NULL);
	gtk_clist_thaw(clist);

	GDK_PIXMAP_UNREF(pixmap);
	GDK_BITMAP_UNREF(mask);
}

/*
 *	Sets the Contents List's row to the values of the recycled
 *	object.
 *
 *	The clist specifies the Contents List.
 *
 *	The obj specifies the recycled object.
 *
 *	The row specifies the row.
 *
 *	All inputs assumed valid.
 */
static void EDVRecBinContentsSetRow(
	edv_core_struct *core, edv_recbin_struct *recbin,
	GtkCList *clist, edv_recycled_object_struct *obj,
	const gint row
)
{
	gboolean hide_dir_size, hide_link_size, hide_link_permissions;
	gulong recbin_size_max, block_size;
	gint i;
	const gchar *date_format;
	GList *glist;
	edv_size_format size_format;
	edv_date_relativity date_relativity;
	edv_recbin_column_type column_type;
	cfg_intlist_struct *column_types_intlist;
	const cfg_item_struct *cfg_list = core->cfg_list;

	/* Get column types mapping */
	column_types_intlist = EDV_GET_INTLIST(
	    EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	);
	if(column_types_intlist == NULL)
	    return;

	/* Get additional display options */
	size_format = (edv_size_format)EDV_GET_I(
	    EDV_CFG_PARM_SIZE_FORMAT
	);
	block_size = EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE);
	date_relativity = (edv_date_relativity)EDV_GET_I(
	    EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);
	hide_dir_size = EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_HIDE_DIR_SIZE);
	hide_link_size = EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_HIDE_LINK_SIZE);
	hide_link_permissions = EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_HIDE_LINK_PERMISSIONS);

	recbin_size_max = (gulong)EDV_GET_L(
	    EDV_CFG_PARM_RECBIN_SIZE_WARN
	);

	gtk_clist_freeze(clist);

	/* Iterate through each column */
	for(i = 0, glist = column_types_intlist->list;
	    glist != NULL;
	    i++, glist = g_list_next(glist)
	)
	{
	    column_type = (edv_recbin_column_type)glist->data;
	    switch(column_type)
	    {
	      case EDV_RECBIN_COLUMN_TYPE_INDEX:
		EDVRecBinContentsSetCellIndex(
		    core, recbin, clist, obj,
		    row, i
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_NAME:
		EDVRecBinContentsSetCellName(
		    core, recbin, clist, obj,
		    row, i
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_SIZE:
		EDVRecBinContentsSetCellSize(
		    core, recbin, clist, obj,
		    row, i, hide_dir_size, hide_link_size,
		    size_format, block_size
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_TYPE:
		EDVRecBinContentsSetCellType(
		    core, recbin, clist, obj,
		    row, i
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_PERMISSIONS:
		EDVRecBinContentsSetCellPermissions(
		    core, recbin, clist, obj,
		    row, i, hide_link_permissions
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_OWNER:
		EDVRecBinContentsSetCellOwner(
		    core, recbin, clist, obj,
		    row, i
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_GROUP:
		EDVRecBinContentsSetCellGroup(
		    core, recbin, clist, obj,
		    row, i
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_DATE_ACCESS:
		EDVRecBinContentsSetCellDateAccess(
		    core, recbin, clist, obj,
		    row, i, date_relativity, date_format
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_DATE_MODIFIED:
		EDVRecBinContentsSetCellDateModified(
		    core, recbin, clist, obj,
		    row, i, date_relativity, date_format
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_DATE_CHANGED:
		EDVRecBinContentsSetCellDateChanged(
		    core, recbin, clist, obj,
		    row, i, date_relativity, date_format
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_DATE_DELETED:
		EDVRecBinContentsSetCellDateDeleted(
		    core, recbin, clist, obj,
		    row, i, date_relativity, date_format

		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_LINKED_TO:
		EDVRecBinContentsSetCellLinkedTo(
		    core, recbin, clist, obj,
		    row, i
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_ORIGINAL_LOCATION:
		EDVRecBinContentsSetCellOriginalLocation(
		    core, recbin, clist, obj,
		    row, i
		);
		break;
	      case EDV_RECBIN_COLUMN_TYPE_CAPACITY_USED:
		EDVRecBinContentsSetCellCapacityUsed(
		    core, recbin, clist, obj,
		    row, i,
		    recbin_size_max
		);
		break;
	    }
	}

	gtk_clist_thaw(clist);
}

/*
 *	Appends the recycled object to the Contents List.
 *
 *	The clist specifies the Contents List.
 *
 *	The obj specifies the recycled object who's values will be
 *	used to append a new row on the Contents List and will be
 *	transfered to the Content List's row data. The obj should
 *	not be referenced again after this call.
 *
 *	Returns the new row index or -1 on error.
 *
 *	All inputs assumed valid.
 */
static gint EDVRecBinContentsAppendObject(
	edv_recbin_struct *recbin,
	edv_recycled_object_struct *obj
)
{
	GtkCList *clist = GTK_CLIST(recbin->contents_clist);
	const gint ncolumns = MAX(clist->columns, 1);
	gint i, new_row;
	gchar **strv;
	edv_core_struct *core = recbin->core;

	gtk_clist_freeze(clist);

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	/* Append the new row */
	new_row = gtk_clist_append(clist, strv);

	/* Delete the row cell values */
	g_free(strv);

	/* Failed to append row? */
	if(new_row < 0)
	{
	    gtk_clist_thaw(clist);
	    EDVRecycledObjectDelete(obj);
	    return(-1);
	}

	/* Set the new row's values */
	EDVRecBinContentsSetRow(
	    core, recbin, clist,
	    obj,
	    new_row
	);

	/* Set this recycled object as the new row's data */
	gtk_clist_set_row_data_full(
	    clist, new_row,
	    obj, EDVRecBinContentsItemDestroyCB
	);

	gtk_clist_thaw(clist);

	return(new_row);
}

/*
 *	Appends the recycled objects in the recycle bin directory to
 *	the Contents List.
 *
 *	The clist specifies the Contents List.  
 *
 *	If update_status_bar is TRUE then the status bar will be
 *	updated during this operation.
 *
 *	All inputs assumed valid.
 */
static void EDVRecBinContentsAppendListing(
	edv_recbin_struct *recbin,
	const gboolean update_status_bar
)
{
#ifdef HAVE_REGEX
	regex_t *regex_filter;
#endif
	gint nobjs, nobjs_loaded, last_progress_percent;
	const gchar	*recbin_index_file,
			*filter = recbin->contents_list_filter;
	GtkCList *clist = GTK_CLIST(recbin->contents_clist);
	edv_core_struct *core = recbin->core;
	const cfg_item_struct *cfg_list = core->cfg_list;
	edv_recycled_object_struct *obj;
	edv_recbin_index_struct *rbi_ptr;
	edv_status_bar_struct *sb = recbin->status_bar;

#if defined(HAVE_REGEX)
	/* Compile the regex search criteria */
	if(STRISEMPTY(filter) ?
	    FALSE : strcmp(filter, "*")
	)
	{
	    regex_filter = (regex_t *)g_malloc(sizeof(regex_t));
	    if(regcomp(
		regex_filter,
		filter,
#ifdef REG_EXTENDED
		REG_EXTENDED |		/* Use POSIX extended regex */
#endif
		REG_NOSUB		/* Do not report subpattern matches */
	    ))
	    {
		g_free(regex_filter);
		regex_filter = NULL;
	    }
	}
	else
	{
	    regex_filter = NULL;
	}
#else
	if(STRISEMPTY(filter) ?
	    TRUE : !strcmp(filter, "*")
	)
	    filter = NULL;
#endif

	/* Get the full path to the recycle bin index file */
	recbin_index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(STRISEMPTY(recbin_index_file))
	    return;

	/* Get the total number of recycled objects */
	nobjs = EDVRecBinIndexGetTotal(recbin_index_file);

	/* Open the recycle bin index file */
	rbi_ptr = EDVRecBinIndexOpen(recbin_index_file);

	last_progress_percent = -1;
	nobjs_loaded = 0;

#define UPDATE_PROGRESS	{				\
 if(update_status_bar && (nobjs > 0)) {			\
  const gint progress_percent = nobjs_loaded * 100 / nobjs; \
  if(progress_percent > last_progress_percent) {	\
   EDVStatusBarProgress(				\
    sb,							\
    (gfloat)progress_percent / 100.0f,			\
    TRUE						\
   );							\
   last_progress_percent = progress_percent;		\
  }							\
 }							\
}

	/* Begin reading the recycle bin index file */
	gtk_clist_freeze(clist);
	while(!EDVRecBinIndexNext(rbi_ptr))
	{
	    obj = rbi_ptr->obj;
	    if((obj != NULL) ? (obj->name != NULL) : FALSE)
	    {
		const gchar *name = obj->name;
		edv_recycled_object_struct *tar_obj;

		/* Filter check */
#if defined(HAVE_REGEX)
		if(regex_filter != NULL)
		{
		    if(regexec(
			regex_filter,
			name,
			0, NULL,
			0
		    ) == REG_NOMATCH)
		    {
			nobjs_loaded++;
			UPDATE_PROGRESS
			continue;
		    }
		}
#else
		if(filter != NULL)
		{
		    if(fnmatch(filter, name, 0) == FNM_NOMATCH)
		    {
			nobjs_loaded++;
			UPDATE_PROGRESS
			continue;
		    }
		}
#endif

		/* Make a copy of the recycled object obtained from
		 * the index file
		 */
		tar_obj = EDVRecycledObjectCopy(obj);
		if(tar_obj != NULL)
		{
		    /* Append/transfer the object to the listing */
		    EDVRecBinContentsAppendObject(recbin, tar_obj);
		    nobjs_loaded++;
		    UPDATE_PROGRESS
		}
	    }
	    else
	    {
		edv_recycled_object_struct *tar_obj = EDVRecBinNewErrorObject(
		    rbi_ptr->index,
		    NULL
		);
		if(tar_obj != NULL)
		{
		    /* Append/transfer the object to the listing */
		    EDVRecBinContentsAppendObject(recbin, tar_obj);
		    nobjs_loaded++;
		    UPDATE_PROGRESS
		}
	    }
	}
	gtk_clist_thaw(clist);

#undef UPDATE_PROGRESS

	/* Close the recycle bin index file */
	EDVRecBinIndexClose(rbi_ptr);

#ifdef HAVE_REGEX
	if(regex_filter != NULL)
	{
	    regfree(regex_filter);
	    g_free(regex_filter);
	}
#endif
}

/*
 *	Finds the row by index.
 *
 *	The index specifies the recycled object index to match
 *	with a row's recycled object index.
 *
 *	Returns the matched row or negative on error.
 */
gint EDVRecBinContentsFindRowByIndex(
	edv_recbin_struct *recbin,
	const guint index
)
{
	gint i, n;
	GtkCList *clist;
	edv_recycled_object_struct *obj;

	if(recbin == NULL)
	    return(-2);

	clist = GTK_CLIST(recbin->contents_clist);

	n = clist->rows;
	for(i = 0; i < n; i++)
	{
	    obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(clist, i));
	    if(obj == NULL)
		continue;

	    if(obj->index == index)
		return(i);
	}

	return(-1);
}

/*
 *	Updates all rows on the contents clist by getting the row data
 *	and updates the clist's rows.
 *
 *	This is designed to be called whenever the displayed values
 *	of each row need to be set again from internal data, usually
 *	when a MIME Type has been added/modified/removed. This function
 *	should not be used to `refresh' the list (get new values of
 *	disk object structures), use EDVRecBinContentsGetListing()
 *	instead.
 */
void EDVRecBinContentsRealizeListing(edv_recbin_struct *recbin)
{
	gint i, n;
	GtkCList *clist;
	cfg_item_struct *cfg_list;
	edv_core_struct *core;
	edv_recycled_object_struct *obj;

	if(recbin == NULL)
	    return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	gtk_clist_freeze(clist);

	/* Reset the columns */
	EDVRecBinContentsResetColumns(core, recbin, clist);

	/* Update the rows */
	n = clist->rows;
	for(i = 0; i < n; i++)
	{
	    obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(clist, i));
	    if(obj == NULL)
		continue;

	    EDVRecBinContentsSetRow(
		core, recbin, clist,
		obj,
		i				/* Row number */
	    );
	}

	/* Resize the columns */
	if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
	    EDVRecBinContentsResizeColumnsOptimul(recbin);

	gtk_clist_thaw(clist);
}


/*
 *	Clears the Contents List, resets the columns, and gets a new
 *	listing of recycled objects.
 *
 *	The path specifies the full path to the location.
 *
 *	If update_status_bar is TRUE then the status bar will be updated
 *	during this operation.
 */
void EDVRecBinContentsGetListing(
	edv_recbin_struct *recbin,
	const gboolean update_status_bar
)
{
	GtkCList *clist;
	cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Report the initial progress? */
	if(update_status_bar)
	{
	    EDVStatusBarMessage(
		recbin->status_bar,
		"Loading recycled objects...",
		FALSE
	    );
	    EDVStatusBarProgress(
		recbin->status_bar, 0.0f, TRUE
	    );
	}

	gtk_clist_freeze(clist);

	/* Clear the listing */
	EDVRecBinContentsClear(recbin);

	/* Reset the columns */
	EDVRecBinContentsResetColumns(core, recbin, clist);

	/* Append the list of recycled objects to the listing */
	EDVRecBinContentsAppendListing(recbin, update_status_bar);

	/* Resize the columns */
	if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
	    EDVRecBinContentsResizeColumnsOptimul(recbin);

	gtk_clist_thaw(clist);

	/* Report the final progress? */
	if(update_status_bar)
	{
	    /* Reset progress */
	    EDVStatusBarMessage(
		recbin->status_bar,
		"Recycled objects loaded",
		FALSE
	    );
	    EDVStatusBarProgress(
		recbin->status_bar, 0.0f, TRUE
	    );
	}
}

/*
 *	Deletes all items in the given clist.
 */
void EDVRecBinContentsClear(edv_recbin_struct *recbin)
{
	GtkCList *clist;

	if(recbin == NULL)
	    return;

	clist = GTK_CLIST(recbin->contents_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);
}


/*
 *	FPrompt rename apply callback.
 */
static void EDVRecBinContentsFPromptRenameApplyCB(
	gpointer data, const gchar *value
)
{
	gint row;
	edv_recbin_struct *recbin;
	EDVRecBinContentsListFPromptData *d = EDV_RECBIN_CONTENTS_LIST_FPROMPT_DATA(data);
	if(d == NULL)
	    return;

	recbin = d->recbin;
	row = d->row;

	/* Inputs valid? */
	if((recbin != NULL) && (row > -1) && (value != NULL))
	{
	    GtkWidget *toplevel = recbin->toplevel;
	    GtkCList *clist = GTK_CLIST(recbin->contents_clist);
	    edv_status_bar_struct *sb = recbin->status_bar;
	    edv_core_struct *core = recbin->core;

	    /* Get the selected object */
	    edv_recycled_object_struct *obj = EDV_RECYCLED_OBJECT(
		gtk_clist_get_row_data(clist, row)
	    );
	    if(obj != NULL)
	    {
		gboolean yes_to_all = FALSE;
		const guint index = obj->index;
		gchar *old_name = STRDUP(obj->name);
		const gchar *new_name = value;
		const gchar *error_msg;
		GList *modified_indicies_list;

		/* Rename */
		(void)EDVRecycledObjectOPRename(
		    core,
		    index,
		    new_name,
		    &modified_indicies_list,
		    toplevel,
		    FALSE,			/* Do not show progress */
		    TRUE,			/* Interactive */
		    &yes_to_all
		);

                /* Unmap the progress dialog */
                ProgressDialogBreakQuery(FALSE);
                ProgressDialogSetTransientFor(NULL);

                /* Check for errors */
                error_msg = EDVRecycledObjectOPGetError(core);
                if(!STRISEMPTY(error_msg))
                {
                    /* Report the error */
                    EDVPlaySoundError(core);
                    EDVMessageError(
                        "Rename Error",
                        error_msg,
                        NULL,
                        toplevel
                    );
                }

                if(modified_indicies_list != NULL)
                {
		    guint index;
		    gchar *msg;
		    GList *glist;

		    /* Notify about the modified recycled objects */
                    for(glist = modified_indicies_list;
                        glist != NULL;
                        glist = g_list_next(glist)
                    )
                    {
                        index = (guint)glist->data;
                        if(index == 0)
                            continue;

                        EDVRecycledObjectModifiedEmit(core, index);
                    }

		    /* Update the status bar message */
		    msg = g_strdup_printf(
                        "Object \"%s\" renamed to \"%s\"",
			old_name,
			new_name
		    );
                    EDVStatusBarMessage(sb, msg, FALSE);
                    g_free(msg);

		    /* Delete the modified indicies list */
		    g_list_free(modified_indicies_list);
                }
                else
                {
                    /* Did not get the modified indicies list so this
                     * implies failure
                     */
                    EDVStatusBarMessage(
                        sb,
                        "Rename object failed",
                        FALSE
                    );
 		}

		g_free(old_name);
	    }
	}

	g_free(d);
}

/*
 *	FPrompt rename cancel callback.
 */
static void EDVRecBinContentsFPromptRenameCancelCB(gpointer data)
{
	EDVRecBinContentsListFPromptData *d = EDV_RECBIN_CONTENTS_LIST_FPROMPT_DATA(data);
	if(d == NULL)
	    return;

	g_free(d);
}

/*
 *	Prompts to rename the object.
 *
 *	The row and column specifies the object on the list to map
 *	the rename prompt at. If column is -1 then the column that
 *	is set to EDV_RECBIN_COLUMN_TYPE_NAME will be used.
 */
void EDVRecBinContentsPromptRename(
	edv_recbin_struct *recbin,
	const gint row,
	const gint column
)
{
	gint cx, cy, px, py, pwidth, pheight, name_column;
	GtkWidget *toplevel;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	cfg_intlist_struct *column_type_intlist;
	edv_recycled_object_struct *obj;
	edv_core_struct *core;

	if((recbin == NULL) || FPromptIsQuery())
	    return;

	toplevel = recbin->toplevel;
	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Check and warn if write protect is enabled */
	if(EDVCheckWriteProtect(core, TRUE, toplevel))
	    return;

	EDVRecBinSyncData(recbin);

	/* Row does not exist? */
	if((row < 0) || (row >= clist->rows))
	    return;

	/* Find which column is displaying the name */
	name_column = -1;
	column_type_intlist = EDV_GET_INTLIST(
	    EDV_CFG_PARM_RECBIN_CONTENTS_COLUMN
	);
	if(column_type_intlist != NULL)
	{
	    gint i = 0;
	    GList *glist;

	    for(glist = column_type_intlist->list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		if((edv_recbin_column_type)glist->data ==
		    EDV_RECBIN_COLUMN_TYPE_NAME
		)
		{
		    name_column = i;
		    break;
		}
		i++;
	    }
	}
	/* No column displaying the name? */
	if(name_column < 0)
	    return;

	/* Get the cell's geometry */
	if(!gtk_clist_get_cell_geometry(
	    clist, name_column, row,
	    &cx, &cy, &pwidth, &pheight
	))
	    return;

	/* Get the root window relative coordinates */
	px = 0;
	py = 0;
	gdk_window_get_deskrelative_origin(
	    clist->clist_window, &px, &py
	);
	px += cx + 0;
	py += cy - 2;           /* Move up a little */

	/* Get the recycled object */
	obj = EDV_RECYCLED_OBJECT(gtk_clist_get_row_data(clist, row));
	if(obj == NULL)
	    return;

	/* Check if the object's name is a special notation that
	 * may not be renamed
	 */
	if(!STRISEMPTY(obj->name))
	{
	    const gchar *name = obj->name;
	    if(!strcmp((const char *)name, ".") ||
	       !strcmp((const char *)name, "..") ||
	       !strcmp((const char *)name, "/")
	    )
		return;
	}

	/* Is the specified column the name column or -1? */
	if((column == name_column) || (column < 0))
	{
	    gchar *value = STRDUP(obj->name);
	    EDVRecBinContentsListFPromptData *d = EDV_RECBIN_CONTENTS_LIST_FPROMPT_DATA(
		g_malloc(sizeof(EDVRecBinContentsListFPromptData))
	    );
	    if(d != NULL)
	    {
		d->recbin = recbin;
		d->row = row;
		d->column = column;
	    }

	    /* Map floating prompt to change values */
	    FPromptSetTransientFor(toplevel);
	    FPromptSetPosition(px, py);
	    FPromptMapQuery(
		NULL,			/* No label */
		value,
		NULL,			/* No tooltip message */
		FPROMPT_MAP_NO_MOVE,	/* Map code */
		pwidth, -1,		/* Width and height */
		0,			/* No buttons */
		d,			/* Callback data */
		NULL,			/* No browse callback */
		EDVRecBinContentsFPromptRenameApplyCB,
		EDVRecBinContentsFPromptRenameCancelCB
	    );

	    g_free(value);
	}
}


/*
 *	Recycled object added callback.
 */
void EDVRecBinContentsObjectAddedNotify(
	edv_recbin_struct *recbin, const guint index
)
{
	gint row;
	const gchar *recbin_index_file;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Get the full path to recycle bin index file */
	recbin_index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(STRISEMPTY(recbin_index_file))
	    return;

	/* Check if the recycled object specified by the index already
	 * exists in the list
	 */
	row = EDVRecBinContentsFindRowByIndex(recbin, index);
	if(row > -1)
	{
	    /* Recycled object already in list, do nothing (no update) */
	}
	else
	{
	    /* Recycled object not in list, so add it */
	    edv_recycled_object_struct *obj = EDVRecBinObjectStat(
		recbin_index_file, index
	    );
	    if(obj != NULL)
	    {
		/* Add the recycled object to the list */
		EDVRecBinContentsAppendObject(recbin, obj);

		/* Resize the columns */
		if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		    EDVRecBinContentsResizeColumnsOptimul(recbin);
	    }
	}
}

/*
 *	Recycled object modified callback.
 */
void EDVRecBinContentsObjectModifiedNotify(
	edv_recbin_struct *recbin, const guint index
)
{
	gint row;
	const gchar *recbin_index_file;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Get the full path to recycle bin index file */
	recbin_index_file = EDV_GET_S(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX);
	if(STRISEMPTY(recbin_index_file))
	    return;

	/* Update the modified recycled object in the list */
	row = EDVRecBinContentsFindRowByIndex(recbin, index);
	if(row > -1)
	{
	    /* Create a new recycled object describing the modified
	     * recycled object
	     */
	    edv_recycled_object_struct *obj = EDVRecBinObjectStat(
		recbin_index_file, index
	    );
	    if(obj != NULL)
	    {
		/* Update the recycled object */
		gtk_clist_freeze(clist);
		gtk_clist_set_row_data_full(
		    clist, row,
		    obj, EDVRecBinContentsItemDestroyCB
		);
		EDVRecBinContentsSetRow(
		    core, recbin, clist,
		    obj,
		    row
		);

		/* Resize the columns */
		if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		    EDVRecBinContentsResizeColumnsOptimul(recbin);

		gtk_clist_thaw(clist);
	    }
	}
	else
	{
	    /* The modified recycled object was not in the list, so
	     * add the modified recycled object to the list
	     */
	    edv_recycled_object_struct *obj = EDVRecBinObjectStat(
		recbin_index_file, index
	    );
	    if(obj != NULL)
	    {
		EDVRecBinContentsAppendObject(recbin, obj);

		/* Resize the columns */
		if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		    EDVRecBinContentsResizeColumnsOptimul(recbin);
	    }
	}
}

/*
 *	Recycled object removed callback.
 */
void EDVRecBinContentsObjectRemovedNotify(
	edv_recbin_struct *recbin, const guint index
)
{
	gint row;
	GtkCList *clist;
	const cfg_item_struct *cfg_list;
	edv_core_struct *core;

	if(recbin == NULL)
	    return;

	clist = GTK_CLIST(recbin->contents_clist);
	core = recbin->core;
	cfg_list = core->cfg_list;

	/* Remove all the rows who's object's index matches the
	 * removed index
	 */
	row = EDVRecBinContentsFindRowByIndex(recbin, index);
	if(row > -1)
	{
	    gtk_clist_freeze(clist);
	    do {
		gtk_clist_remove(clist, row);
		row = EDVRecBinContentsFindRowByIndex(recbin, index);
	    } while(row > -1);
	    gtk_clist_thaw(clist);

	    /* Resize the columns */
	    if(EDV_GET_B(EDV_CFG_PARM_RECBIN_CONTENTS_LIST_AUTO_RESIZE_COLUMNS))
		EDVRecBinContentsResizeColumnsOptimul(recbin);
	}
}
