/***************************************************************************
                          pointmatch.cpp  -  description
                             -------------------
    begin                : Tue Jan 6 2003
    copyright            : (C) 2003 by
    email                : mmdigitizer@earthlink.net
    $Log: pointmatch.cpp,v $
    Revision 1.6  2006/11/16 07:08:32  markmitch
    Added point match requirements

    Revision 1.5  2006/10/08 06:41:50  markmitch
    NO_UNDO branch moved to HEAD

    Revision 1.3.2.1  2006/03/14 07:10:34  markmitch
    Point Match on graphs with lines no longer crashes

    Revision 1.3  2005/03/20 01:47:06  markmitch
    After KDevelop 3 restructuring

    Revision 1.12  2004/09/27 04:52:26  markmitch
    KDevelop does not allow renaming source directory to src

    Revision 1.10  2004/07/21 15:14:24  markmitch
    Need stdlib for compile in suse

    Revision 1.9  2004/05/04 04:08:40  markmitch
    Replace abs by dabs

    Revision 1.8  2004/04/05 05:39:53  markmitch
    Remove Valgrind warning messages

    Revision 1.7  2004/03/07 16:36:52  markmitch
    Stop OSX compiler warning

    Revision 1.6  2004/01/15 08:24:48  markmitch
    Delete keypress. Documentation

    Revision 1.5  2004/01/14 06:54:38  markmitch
    Point match works well and has documentation

    Revision 1.4  2004/01/13 18:53:33  markmitch
    Point match works but needs bells and whistles

    Revision 1.3  2004/01/13 17:54:06  markmitch
    Testing point match

    Revision 1.2  2004/01/08 06:48:35  markmitch
    Reworked menu icons look great

    Revision 1.1  2004/01/07 07:48:07  markmitch
    Implementing point match


 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qmessagebox.h>
#include <qobject.h>

#include <math.h>
#include <stdlib.h>

#include "main.h"
#include "mmsubs.h"
#include "digitdebug.h"
#include "discretize.h"
#include "pointmatch.h"

PointMatch::PointMatch()
{
  DigitDebug::ctor(QString("pointmatch ") + QString::number((ulong) this, 16));
}

PointMatch::~PointMatch()
{
  DigitDebug::dtor(QString("pointmatch ") + QString::number((ulong) this, 16));
}

void PointMatch::convertImageToArray(const QImage &imageProcessed, int** imageArray,
  int* imageWidth, int* imageHeight)
{
  // compute bounds
  *imageWidth = imageProcessed.width();
  *imageHeight = imageProcessed.height();

  // allocate memory
  *imageArray = new int [(*imageWidth) * (*imageHeight)];
  CHECK_PTR_ENGAUGE(*imageArray);

  // initialize memory
  Discretize discretize;
  for (int x = 0; x < *imageWidth; x++)
    for (int y = 0; y < *imageHeight; y++)
    {
      (*imageArray) [FOLD2DINDEX(x, y, *imageHeight)] =
      (discretize.processedPixelIsOn(imageProcessed, x, y) ? PixelOnUnscanned : PixelOff);
    }
}

void PointMatch::convertSampleToArray(const QPointArray &samplePointPixels,
  bool** sampleMaskArray, int* sampleMaskWidth, int* sampleMaskHeight,
  int* xCenter, int* yCenter)
{
  // compute bounds
  bool first = true;
  unsigned int i;
  int xMin = *sampleMaskWidth, yMin = *sampleMaskHeight, xMax = 0, yMax = 0;
  for (i = 0; i < samplePointPixels.size(); i++)
  {
    int x = (samplePointPixels.at(i)).x();
    int y = (samplePointPixels.at(i)).y();
    if (first || (x < xMin))
      xMin = x;
    if (first || (x > xMax))
      xMax = x;
    if (first || (y < yMin))
      yMin = y;
    if (first || (y > yMax))
      yMax = y;

    first = false;
  }

  const int border = 1; // #pixels in border on each side
  
  *sampleMaskWidth = (xMax + border) - xMin + border + 1;
  *sampleMaskHeight = (yMax + border) - yMin + border + 1;

  // allocate memory
  *sampleMaskArray = new bool [(*sampleMaskWidth) * (*sampleMaskHeight)];
  CHECK_PTR_ENGAUGE(*sampleMaskArray);

  // initialize memory
  int x, y;
  for (x = 0; x < *sampleMaskWidth; x++)
    for (y = 0; y < *sampleMaskHeight; y++)
      (*sampleMaskArray) [FOLD2DINDEX(x, y, *sampleMaskHeight)] = false;

  int xSum = 0, ySum = 0;
  for (i = 0; i < samplePointPixels.size(); i++)
  {
    x = (samplePointPixels.at(i)).x() - xMin + border;
    y = (samplePointPixels.at(i)).y() - yMin + border;
    ASSERT_ENGAUGE((0 < x) && (x < *sampleMaskWidth));
    ASSERT_ENGAUGE((0 < y) && (y < *sampleMaskHeight));
    (*sampleMaskArray) [FOLD2DINDEX(x, y, *sampleMaskHeight)] = true;

    xSum += x;
    ySum += y;
  }

  // compute center of mass location  
  *xCenter = (int) ((double) xSum / (double) samplePointPixels.size() + 0.5);
  *yCenter = (int) ((double) ySum / (double) samplePointPixels.size() + 0.5);
}

bool PointMatch::correlation(bool* sampleMaskArray,
  int sampleMaskWidth, int sampleMaskHeight,
  int sampleXCenter, int sampleYCenter,
  int* imageArray, int imageWidth, int imageHeight,
  int x, int y, double* corr)
{
  *corr = 0.0;
  if ((x - sampleXCenter < 0) ||
    (y - sampleYCenter < 0) ||
    (imageWidth <= x - sampleXCenter + sampleMaskWidth) ||
    (imageHeight <= y - sampleXCenter + sampleMaskHeight))
  {
    // mask does not extend out of image
    return false;
  }

  for (int xS = 0; xS < sampleMaskWidth; xS++)
    for (int yS = 0; yS < sampleMaskHeight; yS++)
    {
      // are corresponding pixels in the sample and image on?
      bool sample = sampleMaskArray [FOLD2DINDEX(xS, yS, sampleMaskHeight)];
      bool image = (imageArray [FOLD2DINDEX(x - sampleXCenter + xS,
        y - sampleYCenter + yS, imageHeight)] != PixelOff);

      if (sample == image)
        *corr += 1.0;
      else
        *corr -= 1.0;
    }

    return true;
}

bool PointMatch::isolateSampleMatchPoint(QPointArray* samplePointPixels,
  const QImage &image, PointMatchSettings settings,
  int xStart, int yStart, int x, int y)
{
  if ((x < 0) || (y < 0) || (image.width() <= x) || (image.height() <= y))
    return false; // out of bounds

  Discretize discretize;
  if (!discretize.processedPixelIsOn(image, x, y))
    return false; // pixel is off

  if (dabs (x - xStart) > settings.pointSize / 2)
    return false; // point is too far from start
  if (dabs (y - yStart) > settings.pointSize / 2)
    return false; // point is too far from start

  bool found = (samplePointPixels->size() > 0);
  if (found)
    found = (samplePointPixels->find(QPoint(x, y)) >= 0); // n-squared search happening here
  if (found)
    return true; // already in list

  // add this point
  int newSize = samplePointPixels->size() + 1;
  bool resized = samplePointPixels->resize(newSize);
  ASSERT_ENGAUGE(resized);
  samplePointPixels->setPoint(newSize - 1, x, y);

  // recurse. diagonal points are included so single-pixel wide polygonal outlines will be traversed,
  // but for a 2x speed increase we only go diagonal if the adjacent nondiagonal pixels are off
  bool right =
    isolateSampleMatchPoint(samplePointPixels, image, settings, xStart, yStart, x + 1, y);
  bool up =
    isolateSampleMatchPoint(samplePointPixels, image, settings, xStart, yStart, x, y + 1);
  bool left =
    isolateSampleMatchPoint(samplePointPixels, image, settings, xStart, yStart, x - 1, y);
  bool down =
    isolateSampleMatchPoint(samplePointPixels, image, settings, xStart, yStart, x, y - 1);
  if (!right && !up)
    isolateSampleMatchPoint(samplePointPixels, image, settings, xStart, yStart, x + 1, y + 1);
  if (!up && !left)
    isolateSampleMatchPoint(samplePointPixels, image, settings, xStart, yStart, x - 1, y + 1);
  if (!left && !down)
    isolateSampleMatchPoint(samplePointPixels, image, settings, xStart, yStart, x - 1, y - 1);
  if (!down && !right)
    isolateSampleMatchPoint(samplePointPixels, image, settings, xStart, yStart, x + 1, y - 1);

  return true;
}

void PointMatch::matchSamplePoint(const QImage &imageProcessed,
  PointMatchSettings settings,
  const QPointArray &samplePointPixels, const QPointArray &pointsExisting,
  QValueList<PointMatchTriplet>* pointsCreated)
{
  ASSERT_ENGAUGE(pointsCreated != 0);
    
  // create sample point array
  bool* sampleMaskArray;
  int sampleMaskWidth, sampleMaskHeight;
  int sampleXCenter, sampleYCenter;
  convertSampleToArray(samplePointPixels, &sampleMaskArray,
    &sampleMaskWidth, &sampleMaskHeight,
    &sampleXCenter, &sampleYCenter);

  // create image array
  int* imageArray;
  int imageWidth, imageHeight;
  convertImageToArray(imageProcessed, &imageArray, &imageWidth, &imageHeight);

  removePixelsNearCurrentPoints(imageArray, imageWidth, imageHeight, pointsExisting,
    settings.pointSeparation);

  PointMatchList listCreated;
  scanImage(sampleMaskArray, sampleMaskWidth, sampleMaskHeight,
    sampleXCenter, sampleYCenter, settings, imageArray, imageWidth, imageHeight, &listCreated);  
  listCreated.sort();

  // copy sorted match points to output
  PointMatchTriplet* t;
  for (t = listCreated.first(); t != 0; t = listCreated.next())
    pointsCreated->append(*t);
  
  ASSERT_ENGAUGE(sampleMaskArray != 0);
  delete[] sampleMaskArray;

  ASSERT_ENGAUGE(imageArray);
  delete[] imageArray;
}
  
void PointMatch::recurseThroughOnRegion(bool* sampleMaskArray,
  int sampleMaskWidth, int sampleMaskHeight,
  int sampleXCenter, int sampleYCenter,
  int* imageArray, int imageWidth, int imageHeight, int x, int y,
  bool* firstMax, int* onCount, int* xSum, int* ySum,
  int* xMin, int* xMax, int* yMin, int* yMax,
  double* correlationMax)
{
  if ((x < 0) || (y < 0) ||
    (imageWidth <= x) || (imageHeight <= y))
  {
    // this point is off the screen
    return;
  }
  
  if (imageArray [FOLD2DINDEX(x, y, imageHeight)] != PixelOnUnscanned)
  {
    // we only consider points that are on, and have not already been scanned
    return;
  }

  // update statistics on this region
  *onCount += 1;
  *xSum += x;
  *ySum += y;
  if (x < *xMin)
    *xMin = x;
  if (*xMax < x)
    *xMax = x;
  if (y < *yMin)
    *yMin = y;
  if (*yMax < y)
    *yMax = y;

  // mark this point so it is never rescanned
  imageArray [FOLD2DINDEX(x, y, imageHeight)] = PixelOnScanned;
                    
  double corr;
  if (correlation(sampleMaskArray, sampleMaskWidth, sampleMaskHeight,
    sampleXCenter, sampleYCenter, imageArray, imageWidth, imageHeight, x, y,
    &corr))
  {
    // correlation for this point could be performed
    if (*firstMax || (corr > *correlationMax))
    {
      // this is the highest correlation so far, so save it
      *firstMax = false;
      *correlationMax = corr;
    }
  }

  // recurse through neighbors. originally only vertical neighbors were
  // included (so that separate points that just touched each other could
  // be separated), but diagonal neighbors were included so a diagonal line
  // (in a triangle for instance) could be handled
  recurseThroughOnRegion(sampleMaskArray, sampleMaskWidth, sampleMaskHeight,
    sampleXCenter, sampleYCenter, imageArray, imageWidth, imageHeight, x - 1, y - 1,
    firstMax, onCount, xSum, ySum, xMin, xMax, yMin, yMax, correlationMax);
  recurseThroughOnRegion(sampleMaskArray, sampleMaskWidth, sampleMaskHeight,
    sampleXCenter, sampleYCenter, imageArray, imageWidth, imageHeight, x - 1, y    ,
    firstMax, onCount, xSum, ySum, xMin, xMax, yMin, yMax, correlationMax);
  recurseThroughOnRegion(sampleMaskArray, sampleMaskWidth, sampleMaskHeight,
    sampleXCenter, sampleYCenter, imageArray, imageWidth, imageHeight, x - 1, y + 1,
    firstMax, onCount, xSum, ySum, xMin, xMax, yMin, yMax, correlationMax);
  recurseThroughOnRegion(sampleMaskArray, sampleMaskWidth, sampleMaskHeight,
    sampleXCenter, sampleYCenter, imageArray, imageWidth, imageHeight, x    , y - 1,
    firstMax, onCount, xSum, ySum, xMin, xMax, yMin, yMax, correlationMax);
  recurseThroughOnRegion(sampleMaskArray, sampleMaskWidth, sampleMaskHeight,
    sampleXCenter, sampleYCenter, imageArray, imageWidth, imageHeight, x    , y + 1,
    firstMax, onCount, xSum, ySum, xMin, xMax, yMin, yMax, correlationMax);
  recurseThroughOnRegion(sampleMaskArray, sampleMaskWidth, sampleMaskHeight,
    sampleXCenter, sampleYCenter, imageArray, imageWidth, imageHeight, x + 1, y - 1,
    firstMax, onCount, xSum, ySum, xMin, xMax, yMin, yMax, correlationMax);
  recurseThroughOnRegion(sampleMaskArray, sampleMaskWidth, sampleMaskHeight,
    sampleXCenter, sampleYCenter, imageArray, imageWidth, imageHeight, x + 1, y    ,
    firstMax, onCount, xSum, ySum, xMin, xMax, yMin, yMax, correlationMax);
  recurseThroughOnRegion(sampleMaskArray, sampleMaskWidth, sampleMaskHeight,
    sampleXCenter, sampleYCenter, imageArray, imageWidth, imageHeight, x + 1, y + 1,
    firstMax, onCount, xSum, ySum, xMin, xMax, yMin, yMax, correlationMax);
}

void PointMatch::removePixelsNearCurrentPoints(int* imageArray, int imageWidth, int imageHeight,
  const QPointArray &pointsExisting, int pointSeparation)
{
  int count = 0;
  
  for (unsigned int i = 0; i < pointsExisting.size(); i++)
  {
    int xPoint = (pointsExisting.at(i)).x();
    int yPoint = (pointsExisting.at(i)).y();

    // loop through rows of pixels
    int yMin = yPoint - pointSeparation;
    if (yMin < 0)
      yMin = 0;
    int yMax = yPoint + pointSeparation;
    if (imageHeight < yMax)
      yMax = imageHeight;

    for (int y = yMin; y < yMax; y++)
    {
      // pythagorean theorem gives range of x values
      int radical = pointSeparation * pointSeparation - (y - yPoint) * (y - yPoint);
      if (0 < radical)
      {
        int xMin = (int) (xPoint - sqrt(radical));
        if (xMin < 0)
          xMin = 0;
        int xMax = xPoint + (xPoint - xMin);
        if (imageWidth < xMax)
          xMax = imageWidth;

        // turn off pixels in this row of pixels
        for (int x = xMin; x < xMax; x++)
        {
          if (imageArray [FOLD2DINDEX(x, y, imageHeight)] != PixelOff)
          {
            imageArray [FOLD2DINDEX(x, y, imageHeight)] = PixelOff;
            ++count;
          }
        }
      }
    }
  }
}
  
void PointMatch::scanImage(bool* sampleMaskArray,
  int sampleMaskWidth, int sampleMaskHeight,
  int sampleXCenter, int sampleYCenter, PointMatchSettings settings,
  int* imageArray, int imageWidth, int imageHeight,
  PointMatchList* pointsCreated)
{
  bool success = true;

  try
  {
    // loop through image to first on-pixel
    for (int x = 0; x < imageWidth; x++)
      for (int y = 0; y < imageHeight; y++)
      {
        bool firstMax = true;
        int onCount = 0, xSum = 0, ySum = 0;
        int xMin = imageWidth, xMax = 0;
        int yMin = imageHeight, yMax = 0;
        double correlationMax;

        recurseThroughOnRegion(sampleMaskArray, sampleMaskWidth, sampleMaskHeight,
          sampleXCenter, sampleYCenter,
          imageArray, imageWidth, imageHeight,
          x, y, &firstMax, &onCount, &xSum, &ySum,
          &xMin, &xMax, &yMin, &yMax,
          &correlationMax);

        // save max if its correlation is positive and the point is not too big. a zero
        // correlation is would be expected from two uncorrelated sets of pixels, and
        // a negative correlation means the sets of pixels are inverses
        if (!firstMax && (correlationMax > 0) &&
          (xMax - xMin < settings.pointSize) &&
          (yMax - yMin < settings.pointSize))
        {
          PointMatchTriplet* p = new PointMatchTriplet;
          CHECK_PTR_ENGAUGE(p);

          if (onCount < 0)
            onCount = 1;
          p->x = (int) ((double) xSum / (double) onCount + 0.5);
          p->y = (int) ((double) ySum / (double) onCount + 0.5);
          p->correlation = correlationMax;
        
          pointsCreated->append(p);
        }
      }
  }
  catch (...)
  {
    // stack overflow occurred. Since stack is still full at this point,
    // simply set flag. The stack will be cleaned up automatically
    success = false;
  }          

  if (!success)
  {
    QMessageBox::warning(0, QObject::tr("Stack overflow!"),
      QObject::tr("Point matching failed for the current image, probably because the\n"
      "points are connected by lines. You may close it, erase the lines using \n"
      "another software tool, and try again, or you may manually digitize the \n"
      "points."));

#ifdef WIN32
    QMessageBox::critical(0, QObject::tr("Shutting down!"),
      QObject::tr("In Microsoft Windows, Engauge is unable to continue after a stack overflow."));
    exit(-1);
#endif
  }
}
