#include <glib.h>
#include <stdlib.h>
#include "entity.h"

/* This function does the actual setting of attributes.  The enode_attrib()
 * and enode_attrib_quiet () are just wrappers around this to deal with
 * events */
static EBufConst *
enode_attrib_real (ENode * node, gchar * attribute, EBuf * value)
{
    GSList *tmp;
    EBuf *attrib;
    static EBuf *empty = NULL;

    ECHECK_RETVAL (node != NULL, NULL);
    ECHECK_RETVAL (attribute != NULL, NULL);

    tmp = node->attribs;
    while (tmp) {
	EBuf *curattr = tmp->data;
	EBuf *curval;

	tmp = tmp->next;
	ECHECK_RETVAL (tmp != NULL, NULL);

	if (ebuf_equal_str (curattr, attribute)) {
	    curval = tmp->data;

	    if (value) {
		/* If the value is set, we exchange it with the current */
		ebuf_free (curval);
		tmp->data = value;
	    }

	    return (tmp->data);
	}

	tmp = tmp->next;
    }

    /* If we make it this far, we never found it above, and we need to append 
     * it to the list if it's a set */
    if (value) {
#if 0
	if (ebuf_empty (value)) {
	    /* Special case!  We have been asked to set an attribute
	     * to an empty value, but since empty is considered the
	     * same as not set, we just free the ebuf immediately and
	     * don't set the attribute */
	    ebuf_free (value);
	} else {
#endif
	    attrib = ebuf_new_with_str (attribute);
	    node->attribs = g_slist_append_tail (node->attribs, attrib,
						 &node->attribs_tail);
    	    node->attribs = g_slist_append_tail (node->attribs, value,
						 &node->attribs_tail);
    	    return (value);
#if 0
	}
#endif
    }

    /* If the value is not being set, we need to return empty */
    if (!empty)
	empty = ebuf_new_with_str ("");

#ifdef ECHECK_DEBUG
    if (!ebuf_equal_str (empty, "")) {
	g_warning
	    ("...oooOOO--> GACK!! Someone has screwed with my 'empty' EBuf!!!! <--OOOooo...");
    }
#endif				/* ECHECK_DEBUG */

    return (empty);
}

/* Get or set a node attribute.  If value is left out, returns current
 * value, and does not set.  This version will not notify the renderer
 * of a change in the attribute.  This is used to avoid infinite
 * loops inside the renderers, which can occur if the renderer
 * changes an attribute, then gets an event to change the attribute
 * ad infinitum. */
EBufConst *
enode_attrib_quiet (ENode * node, gchar * attribute, EBuf * value)
{
    EBuf *retval;
    EBuf *attrib;

    retval = enode_attrib_real (node, attribute, value);

    /* if they specified a value, send out an attrib set event, but note
     * that this will only go to watchers, not to the renderer itself */
    if (value) {
	attrib = ebuf_new_with_str (attribute);
	enode_event_set_attrib (node, attrib, value, FALSE);

	ebuf_free (attrib);
    }

    return (retval);
}

/* Get or set a node attribute.  If value is left out, returns current
 * value, and does not set */
EBufConst *
enode_attrib (ENode * node, gchar * attribute, EBuf * value)
{
    EBuf *retval;
    EBuf *attrib;

    if (!value) {
	/* It's a get, so we check to see if we're supposed to * notify
	 * someone of this.  If so, they can use the * enode_attrib_quiet to
	 * set the attribute to the desired * value.  This will overwrite the 
	 * entry we are on, and * return the correct thing. */
	enode_event_get_attrib (node, attribute);
    }

    retval = enode_attrib_real (node, attribute, value);

    /* if they specified a value, notify renderer of change */
    if (value) {
	/* TODO: Maybe implement and change to ebuf_adopt_str () */
	attrib = ebuf_new_with_str (attribute);
	enode_event_set_attrib (node, attrib, value, TRUE);

	ebuf_free (attrib);
    }

    return (retval);
}



/* Identical to first, except value can be specified as a Plain C string.. 
 * This is strictly for convenience in the C API.  NULL is returned on an
 * empty/unset attribute to make testing more convenient */
gchar *
enode_attrib_str (ENode * node, gchar * attribute, gchar * value)
{
    EBuf *retval;
    EBuf *val;

    ECHECK_RETVAL (node != NULL, NULL);
    ECHECK_RETVAL (attribute != NULL, NULL);

    if (value)
	val = ebuf_new_with_str (value);
    else
	val = NULL;

    retval = enode_attrib (node, attribute, val);

    if (ebuf_not_empty (retval))
	return (retval->str);
    else
	return (NULL);
}


/* Return true/false as to whether attribute is true */
gint enode_attrib_is_true (ENode * node, gchar * attrib)
{
    gint istrue = FALSE;
    EBuf *val;

    ECHECK_RETVAL (node != NULL, FALSE);
    ECHECK_RETVAL (attrib != NULL, FALSE);

    val = enode_attrib (node, attrib, NULL);

    if (ebuf_equal_strcase (val, "true") ||
	ebuf_equal_strcase (val, "yes") ||
	ebuf_equal_strcase (val, "on") || atoi (val->str)) {
	istrue = TRUE;
    }

    return (istrue);
}


/* Return a list of attributes that are set in this node */
GSList *
enode_list_set_attribs (ENode * node)
{
    GSList *tmp;
    GSList *attribs = NULL;

    ECHECK_RETVAL (node != NULL, NULL);

    tmp = node->attribs;
    while (tmp) {
	EBuf *attrib;
	EBuf *value;

	attrib = tmp->data;
	tmp = tmp->next;
	value = tmp->data;
	tmp = tmp->next;

	if (ebuf_not_empty (value))
	    attribs = g_slist_append (attribs, attrib);
    }

    return (attribs);
}

/* Return a list of all supported attributes for this node */
GSList *
enode_supported_attribs (ENode * node)
{
    GSList *list;

    ECHECK_RETVAL (node != NULL, NULL);

    list = element_supported_attribs_for_node (node);

    return (list);
}

/* Return a description of the attribute */
gchar *
enode_attrib_description (ENode * node, gchar * attribute)
{
    ElementAttr *e_attr;

    ECHECK_RETVAL (node != NULL, NULL);
    ECHECK_RETVAL (attribute != NULL, NULL);

    e_attr = element_attrib_info_for_node (node, attribute);

    if (e_attr)
	return (e_attr->description);
    else
	return (NULL);
}

/* Return the common value type for this attribute * (eg, string, integer
 * etc.) */
gchar *
enode_attrib_value_type (ENode * node, gchar * attribute)
{
    ElementAttr *e_attr;

    ECHECK_RETVAL (node != NULL, NULL);
    ECHECK_RETVAL (attribute != NULL, NULL);

    e_attr = element_attrib_info_for_node (node, attribute);
    if (e_attr)
	return (e_attr->value_desc);
    else
	return (NULL);
}

/* Returns more information about the value.  A comma * seperated list of
 * possible values, a range of integer * values etc. */
gchar *
enode_attrib_possible_values (ENode * node, gchar * attribute)
{
    ElementAttr *e_attr;
    e_attr = element_attrib_info_for_node (node, attribute);

    ECHECK_RETVAL (node != NULL, NULL);
    ECHECK_RETVAL (attribute != NULL, NULL);

    if (e_attr)
	return (e_attr->possible_values);
    else
	return (NULL);
}


/* Syncronize attributes to renderer by calling the * attribute set notify
 * method for each attribute set * in the node.  Should only be used by
 * renderers. */
void
enode_attribs_sync (ENode * node)
{
    ECHECK_RET (node != NULL);

    element_set_attrib_notify_all (node);
}


