#include <stdlib.h>
#include "entity.h"
#include "SDL.h"

#include "sdl-common.h"


/**sdl_sprite_render
 * Render a sdl_sprite to the given SDL surface.
 *
 */
void sdl_sprite_render(SDL_Rect * dest, SDL_Surface * screen,
	sdl_item * item)
{
  dest->x = item->x;
  dest->y = item->y;
  dest->w = item->width;
  dest->h = item->height;
  SDL_BlitSurface(item->surface, NULL, screen, dest);

  return;
}


/**sdl_sprite_refresh 
 * Set the refresh bit on our sprite to true.
 */
static gint
sdl_sprite_refresh (ENode * node, EBuf * attr, EBuf * value)
{

  return TRUE;
}


/**sdl_sprite_set_alpha
 * Sets the sprite alpha blending value.
 */
static gint
sdl_sprite_set_alpha (ENode * node, EBuf * attr, EBuf * value)
{
  sdl_item  * item = NULL;
  int temp_value = 255;

  item = enode_get_kv(node, "item");

  temp_value = atoi (value->str);

  if (temp_value < 1 || temp_value > 256) /* Bad alpha value. */
    {
      SDL_SetAlpha(item->surface, SDL_SRCALPHA, 255);
    }
  else /* Good alpha value. */
    {
      SDL_SetAlpha(item->surface, SDL_SRCALPHA, temp_value);
    }

  return TRUE;
}

/**sdl_sprite_new_pos
 * Set the sprite to a new position, and set the refresh bit.
 */
static gint 
sdl_sprite_new_pos (ENode * node, EBuf * attr, EBuf * value)
{
  char * xpos = NULL;
  char * ypos = NULL;
  sdl_item  * item = NULL;

  item = enode_get_kv(node, "item");

  if (NULL == item)
    {
      g_warning ("Sdl-sprite: No display item found.\n");
      return FALSE;
    }

  xpos = enode_attrib_str (node, "xpos", NULL);
  ypos = enode_attrib_str (node, "ypos", NULL);

  if (NULL == xpos || NULL == ypos)
    {
      g_warning ("Sdl-sprite: Bad x and y position\n");
      return FALSE;
    }

  item->x = atoi (xpos);
  item->y = atoi (ypos);

  enode_set_kv (node, "item", item);

  return TRUE;
}


/**sld_sprite_setup
 * Load the sprite, setup the struct.
 */
static void
sdl_sprite_setup (ENode * node)
{
  ENode * parent;
  GPtrArray * item_list = NULL;
  char * src_str = NULL;
  char * filename = NULL;
  char * xpos = NULL;
  char * ypos = NULL;
  SDL_Surface *image = NULL; 
  sdl_item * item;
  int alpha = 0;

  /* Pull out the garray. */
  parent = enode_parent (node, "sdl-window");

  if (NULL == parent)
    {
      g_warning("Sdl-sprite: Need parent of <sdl-window> type!\n");
      return;
    }

  /* Get item list. */
  item_list = enode_get_kv (parent, "item_list");

  if (NULL == item_list)
    {
      g_warning ("Sdl-sprite: Couldn't get sdl display list\n");
      return;
    }

  src_str = enode_attrib_str (node, "src", NULL);

  filename = eutils_file_search (node, src_str);
  if (filename) 
    {
      image = SDL_LoadBMP (filename);
      g_free (filename);
    } 
  else
    {
      g_warning ("Sdl-sprite: Couldn't find image '%s'\n", filename);
    }

  if (NULL == image)
    {
      g_warning ("Sdl-sprite: Couldn't load image '%s' \n", src_str);
      return;
    }

  xpos = enode_attrib_str (node, "ypos", NULL);
  ypos = enode_attrib_str (node, "xpos", NULL);

  if (NULL == xpos || NULL == ypos)
    {
      g_warning ("Sdl-sprite: Bad x and y position\n");
      return;
    }

  if (NULL != enode_attrib_str (node, "alpha", NULL))
    {
      alpha = atoi (enode_attrib_str (node, "alpha", NULL));

      if (alpha >= 0 || alpha <= 255)
        {
          SDL_SetAlpha(image, SDL_SRCALPHA, alpha);
        }
    }

  item = malloc (sizeof (sdl_item));

  /* Setup the item */
  item->type = SDL_SPRITE;
  item->name = "hi";
  item->width = image->w;
  item->height = image->h;
  item->x = atoi (xpos);
  item->y = atoi (ypos);
  item->needs_refresh = TRUE;
  item->surface = image;
 
  /* Push it on the display list */ 
  g_ptr_array_add (item_list, item);

  enode_set_kv(node, "item", item);
}


/**sdl_sprite_destroy
 * Free the sprite.
 */
static void
sdl_sprite_destroy (ENode * node)
{

}


/**sdl_sprite_register
 * Register the sdl sprite attributes. 
 */
void sdl_sprite_register (void)
{
  Element *element;
  ElementAttr *e_attr;

  /* Register sdl-sprite. */
  element = g_new0 (Element, 1);
  element->render_func = sdl_sprite_setup; 
  element->destroy_func = sdl_sprite_destroy;
  element->parent_func = NULL;
  element->tag = "sdl-sprite";
  element->description = "Create a new SDL window.";
  element_register (element);

  /* X position attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "xpos";
  e_attr->description = "X position of the sprite";
  e_attr->value_desc = "integer";
  e_attr->possible_values = "-1,*";
  e_attr->set_attr_func = sdl_sprite_new_pos;
  element_register_attrib (element, e_attr);

  /* Y position attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "ypos";
  e_attr->description = "y position of the sprite.";
  e_attr->value_desc = "integer";
  e_attr->possible_values = "-1,*";
  e_attr->set_attr_func = sdl_sprite_new_pos;
  element_register_attrib (element, e_attr);

/* Alpha blending attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "alpha";
  e_attr->description = "Alpha blending value of the sprite.";
  e_attr->value_desc = "integer";
  e_attr->possible_values = "-1,*";
  e_attr->set_attr_func = sdl_sprite_set_alpha;
  element_register_attrib (element, e_attr);


  /* force-able refresh attribute. */
  e_attr = g_new0 (ElementAttr, 1);
  e_attr->attribute = "_refresh";
  e_attr->description = "A force refresh";
  e_attr->value_desc = "boolean";
  e_attr->possible_values = "true,false";
  e_attr->set_attr_func = sdl_sprite_refresh;
  element_register_attrib (element, e_attr);
}
