/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-address-input.c,v 1.16 2004/09/05 22:45:56 hoa Exp $
 */

#include "etpan-address-input.h"
#include "etpan-subapp.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-subapp-thread.h"
#include "etpan-errors.h"
#include "etpan-input-common.h"
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <ncurses.h>
#include "etpan-abook-driver.h"
#include "etpan-cfg-abook.h"
#include "etpan-imf-helper.h"
#include "etpan-mime-tools.h"
#include "etpan-help-viewer.h"

static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd);
static void handle_fd(struct etpan_subapp * app, fd_set * fds);
static void handle_key(struct etpan_subapp * app, int key);
static void handle_resize(struct etpan_subapp * app);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_address_input_app_driver = {
  .name = "address-input",
  .always_handle_key = 0,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = set_fd,
  .handle_fd = handle_fd,
  .handle_key = handle_key,
  .handle_resize = handle_resize,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  struct etpan_input_common_app_state common_state;
  
  /* complete */
  int completed_once;

  int completed;
  int tab_count;
  size_t max_common;
  
  char * pattern;
  carray * complete_list;
  
  size_t pattern_size;
  size_t pattern_pos;
  
#if 0  
  char * alloc_complete_str;
  char * complete_str;
  size_t complete_str_len;
  size_t max_complete_list_len;
  carray * complete_list;
#endif
  
  int state;
};



enum {
  NORMAL,
  TRIED_NORMAL,
  RETRIEVING,
  CYCLE,
  FOUND,
};


static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd)
{
  etpan_subapp_thread_set_fd(app, fds, maxfd);
}

static void handle_fd(struct etpan_subapp * app, fd_set * fds)
{
  etpan_subapp_thread_handle_fd(app, fds);
}



static int etpan_address_input_display(struct etpan_subapp * app,
    WINDOW * w);

static void cancel_completion(struct etpan_subapp * app);

static void do_completion(struct etpan_subapp * app);

static void display(struct etpan_subapp * app, WINDOW * w)
{
  etpan_address_input_display(app, w);
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  state->completed_once = 0;
  state->completed = 0;
  state->tab_count = 0;
#if 0
  state->alloc_complete_str = NULL;
  state->complete_str = NULL;
  state->complete_str_len = 0;
  state->complete_list = carray_new(128);
  if (state->complete_list == NULL)
    goto free;
  state->max_complete_list_len = 0;
  state->max_common = 0;
#endif
  state->complete_list = NULL;
  state->max_common = 0;
  state->pattern_pos = 0;
  state->pattern_size = 0;
  state->pattern = NULL;
  state->state = NORMAL;
  
  r = etpan_input_common_init(&state->common_state);
  if (r != NO_ERROR)
    goto free_array;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free_array:
  carray_free(state->complete_list);
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}


struct etpan_subapp * etpan_address_input_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_address_input_app_driver);
}

char * etpan_address_input_get_value(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_input_common_get_value(app, &state->common_state);
}

int etpan_address_input_set(struct etpan_subapp * app,
    char * prompt, size_t size,
    char * default_str,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * data)
{
  struct app_state * state;
  
  state = app->data;
  
  cancel_completion(app);
  etpan_subapp_handle_resize(app);
  
  return etpan_input_common_set(app, &state->common_state,
      prompt, size, default_str, 0, upcall, data);
}


struct complete_abook_item {
  char * value;
  struct etpan_abook_entry * entry;
};


#if 0
static void replace_complete_string(struct etpan_subapp * app,
    char * str, size_t len)
{
  struct app_state * state;
  struct complete_abook_item * first;
  size_t offset;
  
  state = app->data;
  
  if (state->common_state.count - state->complete_str_len +
      len < state->common_state.size) {
    /* delete old string */
    offset = state->complete_str - state->alloc_complete_str;
    memmove(state->common_state.data + offset,
        state->common_state.data + offset +
        state->complete_str_len,
        state->common_state.count -
        (offset + state->complete_str_len));
        
    /* insert new string */
    memmove(state->common_state.data + offset + len,
        state->common_state.data + offset,
        state->common_state.count -
        (offset + state->complete_str_len));
      
    first = carray_get(state->complete_list, 0);
        
    /* insert completed string */
    memcpy(state->common_state.data + offset, str, len);
    
    /* update count */
    state->common_state.count -= state->complete_str_len;
    state->common_state.count += len;
    
    /* update pos */
    state->common_state.pos -= state->complete_str_len;
    state->common_state.pos += len;
  }
}
#endif

#if 0
static void replace_complete_string(struct etpan_subapp * app,
    char * str, size_t len)
{
  struct app_state * state;
  struct complete_abook_item * first;
  size_t offset;

#if 0  
  state = app->data;
  
  state->pattern_pos
#endif
  
#if 0  
  if (state->common_state.count - state->complete_str_len +
      len < state->common_state.size) {
    /* delete old string */
    offset = state->pattern_pos;
    memmove(state->common_state.data + offset,
        state->common_state.data + offset +
        state->complete_str_len,
        state->common_state.count -
        (offset + state->pattern_len));
        
    /* insert new string */
    memmove(state->common_state.data + offset + len,
        state->common_state.data + offset,
        state->common_state.count -
        (offset + state->pattern_len));
      
    first = carray_get(state->complete_list, 0);
        
    /* insert completed string */
    memcpy(state->common_state.data + offset, str, len);
    
    /* update count */
    state->common_state.count -= state->complete_str_len;
    state->common_state.count += len;
    
    /* update pos */
    state->common_state.pos -= state->complete_str_len;
    state->common_state.pos += len;
  }
#endif
}
#endif

#if 0
static void complete_address(struct etpan_subapp * app);
#endif

#if 0
static void complete_string(struct etpan_subapp * app)
{
  struct app_state * state;
  struct complete_abook_item * first;
  
  state = app->data;
        
  first = carray_get(state->complete_list, 0);
  
  if ((state->max_common == state->max_complete_list_len)
      && (state->max_common == state->complete_str_len))
    complete_address(app);
  else
    replace_complete_string(app, first->value, state->max_common);
}
#endif

static char * get_string_from_abook_entry(struct etpan_app * app,
    struct etpan_abook_entry * entry);

#if 0
static void complete_address(struct etpan_subapp * app)
{
  struct app_state * state;
  struct complete_abook_item * first;
#if 0
  char * name;
  char * addr;
#endif
  struct etpan_abook_entry * entry;
  char * str;
  char * mb_str;
#if 0
  struct mailimf_mailbox * mb;
#endif
  size_t len;
  
  state = app->data;
  
  first = carray_get(state->complete_list, 0);
  
  entry = first->entry;
  
  if (entry->addr == NULL) {
    complete_string(app);
    return;
  }

#if 0  
  if (entry->name != NULL) {
    name = strdup(entry->name);
    if (name == NULL)
      return;
  }
  else
    name = NULL;
  
  addr = strdup(entry->addr);
  if (addr == NULL) {
    if (name != NULL)
      free(name);
    return;
  }
  
  mb = mailimf_mailbox_new(name, addr);
  if (mb == NULL) {
    free(addr);
    if (name != NULL)
      free(name);
    return;
  }
  
  etpan_mailbox_decode(app->app, mb);
  mb_str = mailimf_mailbox_to_string(mb);
  mailimf_mailbox_free(mb);
  if (mb_str == NULL) {
    return;
  }
#endif
  mb_str = get_string_from_abook_entry(app->app, entry);
  if (mb_str == NULL)
    return;
  
  len = strlen(mb_str);
  
  str = malloc(len + 3);
  if (str == NULL) {
    free(mb_str);
    return;
  }
  snprintf(str, len + 3, "%s, ", mb_str);
  free(mb_str);
  
  replace_complete_string(app, str, strlen(str));
  free(str);
}
#endif

static int show_help(struct etpan_subapp * app);

#if 0
static int expand_address_list(struct etpan_subapp * app);
#endif





static void complete_address(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  int lines;
  int count;
  
  state = app->data;
  
  switch (key) {
  case KEY_F(1):
    show_help(app);
    break;
  }

#if 0  
  switch (key) {
  case '\n':
#if 0
    expand_address_list(app);
#endif
    etpan_input_common_handle_key(app, &state->common_state, key);
    break;
  }
#endif
  
#if 0
  ETPAN_APP_DEBUG((app->app, "retrieving ... %i", state->state));
  if (state->state == RETRIEVING) {
    switch (key) {
    case KEY_CTRL('G'):
      etpan_input_common_handle_key(app, &state->common_state, key);
      break;
    default:
      etpan_subapp_thread_cancel_all(app);
      state->state = TRIED_NORMAL;
      break;
    }
  }
#endif
  
  switch (state->state) {
  case NORMAL:
  case TRIED_NORMAL:
    switch (key) {
    case '\t':
      do_completion(app);
      etpan_subapp_handle_resize(app);
      break;
      
    default:
      etpan_input_common_handle_key(app, &state->common_state, key);
      break;
    }
    break;

  case RETRIEVING:
    switch (key) {
    case '\t':
      /* do nothing */
      break;
      
    default:
      etpan_input_common_handle_key(app, &state->common_state, key);
      state->state = TRIED_NORMAL;
      break;
    }
    break;
    
  case CYCLE:
    switch (key) {
    case '\t':
      lines = app->height - 1;
      count = (carray_count(state->complete_list) + lines - 1) / lines;
      
      if (count != 0) {
        state->tab_count ++;
        state->tab_count %= count;
      }
      break;
      
    default:
      etpan_input_common_handle_key(app, &state->common_state, key);
      state->state = TRIED_NORMAL;
      break;
    }
    break;
    
  case FOUND:
    switch (key) {
    case '\t':
      complete_address(app);
      break;
      
    default:
      etpan_input_common_handle_key(app, &state->common_state, key);
      state->state = TRIED_NORMAL;
      break;
    }
    break;
  }
}

static void handle_resize(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  if (state->completed_once) {
    app->top = 0;
    app->left = 0;
    app->height = app->app->height;
    app->width = app->app->width;
    etpan_subapp_set_relative_coord(app);
  }
  else {
    etpan_input_common_handle_resize(app, &state->common_state);
  }
}

static int display_init(struct etpan_subapp * app)
{
  app->show_cursor = 1;
  etpan_subapp_handle_resize(app);
  
  return NO_ERROR;
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_input_common_set_color(app, &state->common_state);
}

static void done(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  if (state->complete_list != NULL)
    carray_free(state->complete_list);
  
  etpan_input_common_done(&state->common_state);

  free(state);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;
  
  state = app->data;
  
  state->completed_once = 0;
  cancel_completion(app);
  
  etpan_input_common_leave(app, &state->common_state, new_app);
}


/* complete address */

static struct complete_abook_item *
complete_abook_item_new(char * value, struct etpan_abook_entry * entry)
{
  struct complete_abook_item * item;

  item = malloc(sizeof(struct complete_abook_item));
  if (item == NULL)
    return NULL;
  item->value = value;
  item->entry = entry;
  
  return item;
}

static void complete_abook_item_free(struct complete_abook_item * item)
{
  free(item);
}

static int add_match_list(carray * list,
    char * value, struct etpan_abook_entry * entry)
{
  struct complete_abook_item * item;
  int res;
  int r;

  item = complete_abook_item_new(value, entry);
  if (item == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  r = carray_add(list, item, NULL);
  if (r < 0) {
    complete_abook_item_free(item);
    res = ERROR_MEMORY;
    goto free;
  }

  return NO_ERROR;

 free:
  complete_abook_item_free(item);
 err:
  return res;
}

static int match_abook(char * pattern, int pattern_len, char * value)
{
  char * dup_value;
  int r;
  char * p;

  r = 0;

  if (value == NULL)
    goto err;

  dup_value = strdup(value);
  if (dup_value == NULL)
    goto err;

  for(p = dup_value ; * p != 0 ; p ++)
    * p = (char) toupper((unsigned char) * p);

  if (strncasecmp(pattern, dup_value, pattern_len) == 0)
    r = 1;
  
  free(dup_value);
  
 err:
  return r;
}

#if 0
static void do_completion(struct etpan_subapp * app)
{
  int r;
  struct app_state * state;
  char * value;
  char * p;
  size_t max_common;
  struct etpan_abook_config * abook_config;
  clistiter * cur;
  size_t value_size;

  cancel_completion(app);
  
  abook_config = app->app->config.abook_config;

  state = app->data;
  
  state->completed_once = 1;
  
  state->completed = 1;
  value_size = state->common_state.pos;
  state->alloc_complete_str = malloc(value_size + 1);
  if (state->alloc_complete_str == NULL)
    goto cancel;
  value = state->alloc_complete_str;
  
  strncpy(value, state->common_state.data, value_size);
  value[value_size] = '\0';

  /* go to last address */
  while ((p = strchr(value, ',')) != NULL)
    value = p + 1;
  while ((p = strchr(value, ':')) != NULL)
    value = p + 1;
  while ((p = strchr(value, ';')) != NULL)
    value = p + 1;
  
  /* skip blank */
  p = value;
  while ((* p == ' ') || (* p == '\t'))
    p ++;
  value = p;

  state->complete_str = value;
  if (state->complete_str == NULL)
    goto cancel;
  
  state->complete_str_len = strlen(state->complete_str);
  
  for(p = state->complete_str ; * p != 0 ; p ++)
    * p = (char) toupper((unsigned char) * p);
  
  state->max_complete_list_len = 0;

  for(cur = clist_begin(abook_config->entry_list) ;
      cur != NULL ; cur = cur->next) {
    struct etpan_abook_entry * entry;

    entry = cur->data;
    
    if (match_abook(state->complete_str,
            state->complete_str_len, entry->name)) {
      r = add_match_list(state->complete_list, entry->name, entry);
      if (r != NO_ERROR) {
        goto cancel;
      }
    }
    if (match_abook(state->complete_str,
            state->complete_str_len, entry->nick)) {
      r = add_match_list(state->complete_list, entry->nick, entry);
      if (r != NO_ERROR) {
        goto cancel;
      }
    }
    if (match_abook(state->complete_str,
            state->complete_str_len, entry->addr)) {
      r = add_match_list(state->complete_list, entry->addr, entry);
      if (r != NO_ERROR) {
        goto cancel;
      }
    }
  }

  max_common = 0;
  if (carray_count(state->complete_list) == 0) {
    /* no match */
  }
  else {
    struct complete_abook_item * first;
    
    /* find max common initial part of string */
    first = carray_get(state->complete_list, 0);
    
    max_common = strlen(first->value);
    if (max_common > state->max_complete_list_len)
      state->max_complete_list_len = max_common;

    if (carray_count(state->complete_list) == 1) {
      /* sole completion */
    }
    else {
      unsigned int i;

      /* find possible completion */
      for(i = 1 ; i < carray_count(state->complete_list) ; i ++) {
        struct complete_abook_item * matched;
        size_t j;
        size_t len;
        
        matched = carray_get(state->complete_list, i);
        
        len = strlen(matched->value);
        
        if (len < max_common)
          max_common = len;
        
        for(j = 0 ; j < max_common ; j ++)
          if (toupper((unsigned char) matched->value[j]) !=
              toupper((unsigned char) first->value[j])) {
            max_common = j;
            break;
          }
      
        if (max_common == 0)
          break;
      }
    }
  }
  
  state->max_common = max_common;
  
  return;
  
 cancel:
  cancel_completion(app);
  ETPAN_APP_LOG((app->app, "complete address - not enough memory"));
  state->completed = 0;
}
#endif

struct thread_lookup_info {
  int count;
  carray * addr_tab;
};

static int cmp_name(const void * a, const void *b)
{
  struct etpan_abook_entry * info_a;
  struct etpan_abook_entry * info_b;
  
  info_a = * (struct etpan_abook_entry **) a;
  info_b = * (struct etpan_abook_entry **) b;
  
  if (info_a->name == NULL) {
    if (info_b->name == NULL)
      return 0;
    else
      return -1;
  }
  else if (info_b->name == NULL) {
    return 1;
  }
  
  return strcmp(info_a->name, info_b->name);
}

static inline void add_complete_match(struct app_state * state,
    char * value, struct etpan_abook_entry * entry)
{
  struct complete_abook_item * item;
  int r;
  
  item = malloc(sizeof(* item));
  if (item == NULL)
    goto err;
  
  item->value = value;
  item->entry = entry;
  
  r = carray_add(state->complete_list, item, NULL);
  if (r < 0)
    goto free;
  
  return;
  
 free:
  free(item);
 err:
  return;
}

static void complete_address(struct etpan_subapp * app)
{
  char * mb_str;
  char * str;
  size_t len;
  struct complete_abook_item * first;
  struct app_state * state;
  size_t first_size;
  
  state = app->data;
  
  if (carray_count(state->complete_list) == 0)
    return;
  
  first = carray_get(state->complete_list, 0);
  first_size = strlen(first->value);
  
  mb_str = get_string_from_abook_entry(app->app, first->entry);
  if (mb_str == NULL)
    return;
    
  len = strlen(mb_str);
    
  str = malloc(len + 3);
  if (str == NULL) {
    free(mb_str);
    return;
  }
  snprintf(str, len + 3, "%s, ", mb_str);
  free(mb_str);
  
  len = strlen(str);
  
  /* do replacement */
  memmove(state->common_state.data +
      state->pattern_pos + len,
      state->common_state.data +
      state->pattern_pos + first_size,
      len - first_size);
  
  /* insert completed string */
  memcpy(state->common_state.data + state->pattern_pos,
      str, len);
  
  /* update count */
  state->common_state.count += len - first_size;
    
  /* update pos */
  state->common_state.pos += len - first_size;
  
  free(str);
  
  state->state = TRIED_NORMAL;
}  


static void complete_string(struct etpan_subapp * app)
{
  size_t max_common;
  struct complete_abook_item * first;
  struct app_state * state;
  size_t max_len;

  state = app->data;
  
  /* find completion */
  if (carray_count(state->complete_list) == 0) {
    /* no match */
    max_common = 0;
    max_len = 0;
  }
  else {
    unsigned int i;
    struct complete_abook_item * first;
    
    first = carray_get(state->complete_list, 0);
    max_common = strlen(first->value);
    max_len = max_common;
    
    for(i = 1 ; i < carray_count(state->complete_list) ; i ++) {
      struct complete_abook_item * item;
      size_t len;
      size_t j;
      
      item = carray_get(state->complete_list, i);
      max_len = strlen(item->value);
      
      for(j = 0 ; j < max_common ; j ++) {
        if (toupper((unsigned char) item->value[j]) !=
            toupper((unsigned char) first->value[j])) {
          max_common = j;
          break;
        }
      }
      
      if (max_common == 0)
        break;
    }
  }
  
  first = carray_get(state->complete_list, 0);

  if (max_common == max_len) {
    state->state = FOUND;
  }
  else {
    state->state = CYCLE;
  }
  
  if (max_common <= state->pattern_size)
    return;
  
  /* replace (pattern, pattern_size) with (first->value, max_common) */
  
  if (state->common_state.count - state->pattern_size + max_common <
      state->common_state.size) {
    memmove(state->common_state.data +
        state->pattern_pos + max_common,
        state->common_state.data +
        state->pattern_pos + state->pattern_size,
        max_common - state->pattern_size);

    /* insert completed string */
    memcpy(state->common_state.data + state->pattern_pos,
        first->value, max_common);
    
    /* update count */
    state->common_state.count += max_common - state->pattern_size;
    
    /* update pos */
    state->common_state.pos += max_common - state->pattern_size;
  }
}

static void thread_abook_lookup_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct thread_lookup_info * info;
  unsigned int i;
  carray * tab;
  int r;
  int max_common;
  struct app_state * state;
  unsigned int count;
  size_t pattern_len;
  
  state = app->data;
  
  info = data;
  
  /* merge response */

  free(op->arg);
  
  if (op->err == NO_ERROR) {
    tab = op->result;
    
    for(i = 0 ; i < carray_count(tab) ; i ++) {
      struct etpan_abook_entry * entry;
      
      entry = carray_get(tab, i);
    
      r = carray_add(info->addr_tab, entry, NULL);
      if (r < 0) {
        etpan_abook_entry_free(entry);
      }
    }
    carray_free(tab);
  }
  
  info->count --;
  if (info->count != 0)
    return;
  
  /* complete */
  ETPAN_APP_DEBUG((app->app, "request complete"));
  
  /* sort */
  
  qsort(carray_data(info->addr_tab), carray_count(info->addr_tab),
      sizeof(carray_data(info->addr_tab)[0]), cmp_name);
  
  for(i = 0 ; i < carray_count(info->addr_tab) ; i ++) {
    struct etpan_abook_entry * entry;
    
    entry = carray_get(info->addr_tab, i);
    ETPAN_APP_DEBUG((app->app, "complete : %s %s", entry->name, entry->addr));
  }
  
  pattern_len = state->pattern_size;
  max_common =  -1;
  
  for(i = 0 ; i < pattern_len ; i ++)
    state->pattern[i] = toupper((unsigned char) state->pattern[i]);
  
  state->complete_list = carray_new(carray_count(info->addr_tab));
  if (state->complete_list == NULL)
    return;
  
  count = 0;
  /* find possible completion */
  for(i = 0 ; i < carray_count(info->addr_tab) ; i ++) {
    struct etpan_abook_entry * entry;
    size_t len;
    char * value;
    int added;
    
    entry = carray_get(info->addr_tab, i);
    
    value = NULL;
    added = 0;
    if (match_abook(state->pattern, pattern_len, entry->name)) {
      value = entry->name;
      added = 1;
    }
    if (match_abook(state->pattern, pattern_len, entry->addr)) {
      value = entry->addr;
      added = 1;
    }
    if (match_abook(state->pattern, pattern_len, entry->nick)) {
      value = entry->nick;
      added = 1;
    }
    
    if (added) {
      add_complete_match(state, value, entry);
      count ++;
    }
    else {
      etpan_abook_entry_free(entry);
    }
  }
  
  complete_string(app);

#if 0  
  if (carray_count(state->complete_list) == 0) {
    /* no match */
    max_common = 0;
  }
  else {
    unsigned int i;
    char * first_value;
    
    first_value = carray_get(state->complete_list, 0);
    max_common = strlen(first_value);

    for(i = 1 ; i < carray_count(info->addr_tab) ; i ++) {
      char * entry;
      size_t len;
      size_t j;

      entry = carray_get(state->complete_list, i);
      
      for(j = 0 ; j < max_common ; j ++) {
        if (toupper((unsigned char) value[j]) !=
            toupper((unsigned char) first_value[j])) {
          max_common = j;
          break;
        }
      }
      
      if (max_common == 0)
        break;
    }
  }
#endif

#if 0  
  if (max_common != pattern_len) {
    /* do completion */
  }
#endif
}

static void do_completion(struct etpan_subapp * app)
{
  int r;
  struct app_state * state;
  char * value;
  char * p;
  size_t max_common;
  struct etpan_abook_config * abook_config;
  clistiter * cur;
  size_t value_size;
  size_t pos;
  struct thread_lookup_info * info;
  size_t size;
  unsigned int i;
  char * pattern;
  
  cancel_completion(app);
  
  abook_config = app->app->config.abook_config;

  state = app->data;
  
  state->state = RETRIEVING;
  
  state->completed_once = 1;
  
  state->completed = 1;
  
  value = state->common_state.data;
  /* go to last address */
  while ((p = strchr(value, ',')) != NULL)
    value = p + 1;
  while ((p = strchr(value, ':')) != NULL)
    value = p + 1;
  while ((p = strchr(value, ';')) != NULL)
    value = p + 1;
  
  /* skip blank */
  p = value;
  while ((* p == ' ') || (* p == '\t'))
    p ++;
  value = p;
  
  pos = value - state->common_state.data;
  size = state->common_state.pos - pos;
  
  pattern = malloc(size + 1);
  if (pattern == NULL)
    goto cancel;
  
  strncpy(pattern, value, size + 1);
  pattern[size] = 0;
  
  info = malloc(sizeof(* info));
  if (info == NULL) {
    free(pattern);
    goto cancel;
  }
  info->count = 0;
  info->addr_tab = carray_new(16);
  if (info->addr_tab == NULL) {
    free(info);
    free(pattern);
    goto cancel;;
  }

  state->pattern_pos = pos;
  state->pattern_size = size;
  state->pattern = pattern;
  
  for(i = 0 ; i < carray_count(abook_config->abook_tab) ; i ++) {
    struct etpan_abook * abook;
    char * dup_pattern;
    
    abook = carray_get(abook_config->abook_tab, i);
    dup_pattern = strdup(pattern);
    if (dup_pattern == NULL)
      continue;
    
    r = etpan_subapp_thread_abook_op_add(app,
        THREAD_ID_ADDR_INPUT_ABOOK_LOOKUP,
        ETPAN_THREAD_ABOOK_LOOKUP,
        abook, dup_pattern,
        thread_abook_lookup_callback, info,
        NULL);
    if (r != NO_ERROR) {
      free(dup_pattern);
      continue;
    }
    info->count ++;
  }

  if (info->count == 0)
    free(info);
  
  return;
  
 cancel:
  cancel_completion(app);
  ETPAN_APP_LOG((app->app, "complete address - not enough memory"));
  state->completed = 0;
}


static void cancel_completion(struct etpan_subapp * app)
{
  struct app_state * state;
  unsigned int i;

  state = app->data;

#if 0
  if (state->alloc_complete_str != NULL)
    free(state->alloc_complete_str);
  state->alloc_complete_str = NULL;
#endif
  if (state->pattern != NULL)
    free(state->pattern);
  
  state->state = NORMAL;
  state->pattern_pos = 0;
  state->pattern = NULL;
  
  state->max_common = 0;
  state->completed_once = 0;
  state->completed = 0;
  state->tab_count = 0;
#if 0
  state->complete_str = NULL;
  state->complete_str_len = 0;
  for(i = 0 ; i < carray_count(state->complete_list) ; i ++) {
    struct complete_abook_item * item;
    
    item = carray_get(state->complete_list, i);
    
    complete_abook_item_free(item);
  }
  carray_set_size(state->complete_list, 0);
  state->max_complete_list_len = 0;
#endif
  if (state->complete_list != NULL) {
    for(i = 0 ; i < carray_count(state->complete_list) ; i ++) {
      struct complete_abook_item * item;
      
      item = carray_get(state->complete_list, i);
      
      etpan_abook_entry_free(item->entry);
      complete_abook_item_free(item);
    }
    carray_free(state->complete_list);
  }
  state->complete_list = NULL;
}

static int etpan_address_input_display(struct etpan_subapp * app,
    WINDOW * w)
{
  struct app_state * state;
  int r;
  char * output;
  char * fill;
  unsigned int i;
  int y;
  int count;
  int lines;
  
  state = app->data;
  
  fill = app->app->fill;
  output = app->app->output;
  
  lines = app->display_height - 1;

  y = 0;
  i = state->tab_count * lines;
    
  count = 0;

  wattron(w, state->common_state.main_attr);

  if (state->complete_list != NULL) {
    if (carray_count(state->complete_list) == 0) {
      if (y < lines) {
        snprintf(output, app->display_width + 1, "no match%s", fill);
        mvwaddstr(w, y, 0, output);
        y ++;
      }
    }
    else {
      while (i < carray_count(state->complete_list)) {
        struct complete_abook_item * matched;
      
        if (y >= lines)
          break;
      
        matched = carray_get(state->complete_list, i);
        
        if (matched->entry->addr != NULL) {
          if (matched->entry->name != NULL)
            snprintf(output, app->display_width + 1, "%s (%s <%s>)%s",
                matched->value, matched->entry->name,
                matched->entry->addr, fill);
          else
            snprintf(output, app->display_width + 1, "%s (<%s>)%s",
                matched->value, matched->entry->addr,
                fill);
        }
        else {
            snprintf(output, app->display_width + 1, "%s%s", matched->value, fill);
        }
        mvwaddstr(w, y, 0, output);
        y ++;
        i ++;
      }
    
      if (carray_count(state->complete_list) == 1) {
        if (y < lines) {
          snprintf(output, app->display_width + 1,
              "sole completion, use TAB to complete%s", fill);
          mvwaddstr(w, y, app->left, output);
          y ++;
        }
      }
    }
  }
  
  while (y < lines) {
    mvwaddstr(w, y, 0, fill);
    y ++;
  }
  
  wattroff(w, state->common_state.main_attr);
  
  r = etpan_input_common_display(app, &state->common_state, w);
  if (r != NO_ERROR)
    return r;

  return NO_ERROR;
}


static int equal_abook(char * pattern, char * value)
{
  char * dup_value;
  int r;
  char * p;

  r = 0;

  if (value == NULL)
    goto err;

  dup_value = strdup(value);
  if (dup_value == NULL)
    goto err;

  for(p = dup_value ; * p != 0 ; p ++)
    * p = (char) toupper((unsigned char) * p);

  if (strcasecmp(pattern, dup_value) == 0)
    r = 1;
  
  free(dup_value);
  
 err:
  return r;
}

static char * get_string_from_abook_entry(struct etpan_app * app,
    struct etpan_abook_entry * entry)
{
  char * name;
  char * addr;
  struct mailimf_mailbox * mb;
  char * mb_str;
  
  if (entry->name != NULL) {
    name = strdup(entry->name);
    if (name == NULL)
      return NULL;
  }
  else
    name = NULL;
  
  addr = strdup(entry->addr);
  if (addr == NULL) {
    if (name != NULL)
      free(name);
    return NULL;
  }
  
  mb = mailimf_mailbox_new(name, addr);
  if (mb == NULL) {
    free(addr);
    if (name != NULL)
      free(name);
    return NULL;
  }
  
  etpan_mailbox_decode(app, mb);
  mb_str = mailimf_mailbox_to_string(mb);
  mailimf_mailbox_free(mb);
  
  return mb_str;
}

#if 0
static int expand_address_list(struct etpan_subapp * app)
{
  struct app_state * state;
  struct etpan_abook_config * abook_config;
  clistiter * cur;
  char * old_address_list;
  char * address_list;
  size_t remaining;
  char * address_p;
  char * dest_p;
  
  state = app->data;
  
  abook_config = app->app->config.abook_config;
  
  address_list = etpan_input_common_get_value(app, &state->common_state);
  old_address_list = strdup(address_list);
  if (old_address_list == NULL)
    goto err;
  
  remaining = state->common_state.size - 1;
  dest_p = address_list;
  
  address_p = old_address_list;
  while (* address_p != '\0') {
    char * p;
    char * end_word;
    int found;
    int try_to_match;
    size_t word_len;
    
    while ((* address_p == ' ') || (* address_p == '\t'))
      address_p ++;
    
    /* extract a word */
    p = address_p;
    while (* p != '\0') {
      if (* p == ',')
        break;
      if (* p == ':')
        break;
      if (* p == ';')
        break;
      p ++;
    }
    end_word = p;
    
    word_len = end_word - address_p;
    
    try_to_match = 1;
    /* if this is a group name, we skip */
    if (* end_word == ':')
      try_to_match = 0;
    
    found = 0;
    if (try_to_match) {
      char * word;
      
      /* get the word for match */
      word = malloc(word_len + 1);
      if (word == NULL)
        goto free;
      
      strncpy(word, address_p, word_len + 1);
      word[word_len] = '\0';
      
      /* strip spaces at the end */
      while (word_len > 0) {
        if ((word[word_len - 1] != ' ') && (word[word_len - 1] != '\t'))
          break;
        word[word_len - 1] = '\0';
        word_len --;
      }
      
      /* convert to upcase */
      for(p = word ; * p != '\0' ; p ++)
        * p = (char) toupper((unsigned char) * p);
      
      /* try to match the address book */
      for(cur = clist_begin(abook_config->entry_list) ;
          cur != NULL ; cur = cur->next) {
        struct etpan_abook_entry * entry;
        
        entry = cur->data;
        
        if (equal_abook(word, entry->name) || 
            equal_abook(word, entry->nick) ||
            equal_abook(word, entry->addr)) {
          char * mb_str;
          
          mb_str = get_string_from_abook_entry(app->app, entry);
          if (mb_str == NULL)
            goto free;
          
          word_len = strlen(mb_str);
          
          if (remaining < word_len + 2) {
            free(mb_str);
            goto restore;
          }
          
          * dest_p = ' ';
          strncpy(dest_p + 1, mb_str, word_len);
          free(mb_str);
          remaining -= word_len + 1;
          dest_p += word_len + 1;
          
          /* copy the separator */
          address_p = end_word;
          if (* end_word != '\0') {
            * dest_p = * address_p;
            dest_p ++;
            address_p ++;
          }
          
          found = 1;
          break;
        }
      }
      free(word);
    }
    
    /* if no match found */
    if (!found) {
      if (remaining < word_len + 2)
        goto restore;
      
      * dest_p = ' ';
      strncpy(dest_p + 1, address_p, word_len);
      remaining -= word_len + 1;
      dest_p += word_len + 1;
      
      /* copy the separator */
      address_p = end_word;
      if (* end_word != '\0') {
        * dest_p = * address_p;
        dest_p ++;
        address_p ++;
      }
    }
  }
  
  * dest_p = '\0';
  
  return NO_ERROR;
  
 free:
  free(old_address_list);
 err:
  return ERROR_MEMORY;

 restore:
  strcpy(address_list, old_address_list);
  return NO_ERROR;
}
#endif


#define HELP_TEXT \
"\
Help for address input\n\
----------------------\n\
\n\
This application will let you edit an address.\n\
Addresses should be separated with ','.\n\
\n\
- [TAB]      : Completion of address with the address book\n\
                 When you press [TAB] once, the name will be\n\
                 completed and when you press twice, the name\n\
                 will be replaced with the correct syntax.\n\
\n\
- [Enter]    : edition of address is finished\n\
- Ctrl-G     : cancel\n\
\n\
- F1         : help\n\
- Ctrl-L     : Console log\n\
\n\
(F1, ? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
