/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-header-input.c,v 1.10 2003/12/10 23:16:46 hoa Exp $
 */

#include "etpan-search-input.h"
#include "etpan-subapp.h"
#include "etpan-app.h"
#include "etpan-app-subapp.h"
#include "etpan-errors.h"
#include "etpan-input-common.h"
#include "etpan-help-viewer.h"
#include <stdlib.h>
#include <string.h>

static void handle_key(struct etpan_subapp * app, int key);
static void handle_resize(struct etpan_subapp * app);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_header_input_app_driver = {
  .name = "header-input",
  .always_handle_key = 0,
  .always_on_top = 0,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = NULL,
  .handle_fd = NULL,
  .handle_key = handle_key,
  .handle_resize = handle_resize,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

struct app_state {
  struct etpan_input_common_app_state common_state;
  
  /* complete */
  int completed_once;

  int completed;
  int tab_count;
  size_t max_common;
  
  char * alloc_complete_str;
  char * complete_str;
  size_t complete_str_len;
  size_t max_complete_list_len;
  carray * complete_list;
};

static int etpan_header_input_display(struct etpan_subapp * app,
    WINDOW * w);

static void cancel_completion(struct etpan_subapp * app);

static void do_completion(struct etpan_subapp * app);

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_header_input_display(app, w);
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  state->completed_once = 0;
  state->completed = 0;
  state->tab_count = 0;
  state->alloc_complete_str = NULL;
  state->complete_str = NULL;
  state->complete_str_len = 0;
  state->complete_list = carray_new(128);
  if (state->complete_list == NULL)
    goto free;
  state->max_complete_list_len = 0;
  state->max_common = 0;
  
  r = etpan_input_common_init(&state->common_state);
  if (r != NO_ERROR)
    goto free_array;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free_array:
  carray_free(state->complete_list);
 free:
  free(state);
 err:
  return ERROR_MEMORY;
}


struct etpan_subapp * etpan_header_input_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_header_input_app_driver);
}

char * etpan_header_input_get_value(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  return etpan_input_common_get_value(app, &state->common_state);
}

int etpan_header_input_set(struct etpan_subapp * app,
    char * prompt, size_t size,
    char * default_str,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * data)
{
  struct app_state * state;
  
  state = app->data;
  
  cancel_completion(app);
  etpan_subapp_handle_resize(app);
  
  return etpan_input_common_set(app, &state->common_state,
      prompt, size, default_str, 0, upcall, data);
}

static void replace_complete_string(struct etpan_subapp * app,
    char * str, size_t len)
{
  struct app_state * state;
  char * first;
  size_t offset;
  
  state = app->data;
  
  if (state->common_state.count - state->complete_str_len +
      len < state->common_state.size) {
    /* delete old string */
    offset = state->complete_str - state->alloc_complete_str;
    memmove(state->common_state.data + offset,
        state->common_state.data + offset +
        state->complete_str_len,
        state->common_state.count -
        (offset + state->complete_str_len));
        
    /* insert new string */
    memmove(state->common_state.data + offset + len,
        state->common_state.data + offset,
        state->common_state.count -
        (offset + state->complete_str_len));
      
    first = carray_get(state->complete_list, 0);
        
    /* insert completed string */
    memcpy(state->common_state.data + offset, str, len);
    
    /* update count */
    state->common_state.count -= state->complete_str_len;
    state->common_state.count += len;
    
    /* update pos */
    state->common_state.pos -= state->complete_str_len;
    state->common_state.pos += len;
  }
}

static void complete_string(struct etpan_subapp * app)
{
  struct app_state * state;
  char * first;
  
  state = app->data;
        
  first = carray_get(state->complete_list, 0);
  
  replace_complete_string(app, first, state->max_common);
}

static int show_help(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_input_common_handle_key(app, &state->common_state, key);
  
  switch (key) {
  case KEY_F(1):
    show_help(app);
    break;
    
  case '\t':

    if (!state->completed) {
      size_t old_pos;
      
      do_completion(app);
      
      if (state->max_common > 0) {
        old_pos = state->common_state.pos;
        complete_string(app);
        if (old_pos != state->common_state.pos)
          state->completed = 0;
      }
      
      etpan_subapp_handle_resize(app);
    }
    else {
      int lines;
      int count;
      
      lines = app->height - 1;
      count = (carray_count(state->complete_list) + lines - 1) / lines;
      
      if (count != 0) {
        state->tab_count ++;
        state->tab_count %= count;
      }
    }
    break;
    
  default:
    state->completed = 0;
    break;
  }
}

static void handle_resize(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  if (state->completed_once) {
    app->top = 0;
    app->left = 0;
    app->height = app->app->height;
    app->width = app->app->width;
    etpan_subapp_set_relative_coord(app);
  }
  else {
    etpan_input_common_handle_resize(app, &state->common_state);
  }
}

static int display_init(struct etpan_subapp * app)
{
  app->show_cursor = 1;
  etpan_subapp_handle_resize(app);
  
  return NO_ERROR;
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  etpan_input_common_set_color(app, &state->common_state);
}

static void done(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  carray_free(state->complete_list);

  etpan_input_common_done(&state->common_state);
  free(state);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;
  
  state = app->data;
  
  state->completed_once = 0;
  cancel_completion(app);
  
  etpan_input_common_leave(app, &state->common_state, new_app);
}

char * standard_headers[] = {
  "Newsgroups",
  "Organization",
  "Followup-To",
  "User-Agent",
};

static void do_completion(struct etpan_subapp * app)
{
  struct app_state * state;
  char * value;
  size_t max_common;
  size_t value_size;
  unsigned int i;

  cancel_completion(app);
  
  state = app->data;
  
  state->completed_once = 1;
  
  state->completed = 1;
  
  value_size = state->common_state.pos;
  state->alloc_complete_str = malloc(value_size + 1);
  if (state->alloc_complete_str == NULL)
    goto cancel;
  value = state->alloc_complete_str;
  
  strncpy(value, state->common_state.data, value_size);
  value[value_size] = '\0';
  
  state->complete_str = value;
  state->complete_str_len = strlen(state->complete_str);
  
  state->max_complete_list_len = 0;
  
  /* build list up */
  
  for(i = 0 ; i < sizeof(standard_headers) / sizeof(standard_headers[0]) ;
      i ++) {
    if (strncasecmp(state->complete_str, standard_headers[i],
            state->complete_str_len) == 0) {
      carray_add(state->complete_list, standard_headers[i], NULL);
    }
  }
  
  max_common = 0;
  if (carray_count(state->complete_list) == 0) {
    /* no match */
  }
  else {
    char * first;
    
    /* find max common initial part of string */
    first = carray_get(state->complete_list, 0);
    
    max_common = strlen(first);
    if (max_common > state->max_complete_list_len)
      state->max_complete_list_len = max_common;

    if (carray_count(state->complete_list) == 1) {
      /* sole completion */
    }
    else {
      unsigned int i;

      /* find possible completion */
      for(i = 1 ; i < carray_count(state->complete_list) ; i ++) {
        char * matched;
        size_t j;
        size_t len;
        
        matched = carray_get(state->complete_list, i);
        
        len = strlen(matched);
        
        if (len < max_common)
          max_common = len;
        
        for(j = 0 ; j < max_common ; j ++)
          if (matched[j] != first[j]) {
            max_common = j;
            break;
          }
      
        if (max_common == 0)
          break;
      }
    }
  }
  
  state->max_common = max_common;
  
  return;
  
 cancel:
  cancel_completion(app);
  ETPAN_APP_LOG((app->app, "complete header name - not enough memory"));
  state->completed = 0;
}

static void cancel_completion(struct etpan_subapp * app)
{
  struct app_state * state;

  state = app->data;

  if (state->alloc_complete_str != NULL)
    free(state->alloc_complete_str);
  state->alloc_complete_str = NULL;
  state->max_common = 0;
  state->completed_once = 0;
  state->completed = 0;
  state->tab_count = 0;
  state->complete_str = NULL;
  state->complete_str_len = 0;
  carray_set_size(state->complete_list, 0);
  state->max_complete_list_len = 0;
}

static int etpan_header_input_display(struct etpan_subapp * app,
    WINDOW * w)
{
  struct app_state * state;
  int r;
  char * output;
  char * fill;
  unsigned int i;
  unsigned int y;
  unsigned int count;
  unsigned int lines;
  
  state = app->data;
  
  fill = app->app->fill;
  output = app->app->output;
  
  lines = app->display_height - 1;

  y = 0;
  i = state->tab_count * lines;
    
  count = 0;

  wattron(w, state->common_state.main_attr);
  
  if (carray_count(state->complete_list) == 0) {
    if (y < lines) {
      snprintf(output, app->display_width + 1, "no match%s", fill);
      mvwaddstr(w, y, 0, output);
      y ++;
    }
  }
  else {
    while (i < carray_count(state->complete_list)) {
      char * matched;
      
      if (y >= lines)
        break;
      
      matched = carray_get(state->complete_list, i);
      
      snprintf(output, app->display_width + 1, "%s%s", matched, fill);
      mvwaddstr(w, y, 0, output);
      y ++;
      i ++;
    }
    if (carray_count(state->complete_list) == 1) {
      if (y < lines) {
        snprintf(output, app->display_width + 1,
            "sole completion%s", fill);
        mvwaddstr(w, y, app->left, output);
        y ++;
      }
    }
  }
  
  while (y < lines) {
    mvwaddstr(w, y, 0, fill);
    y ++;
  }
  
  wattroff(w, state->common_state.main_attr);
  
  r = etpan_input_common_display(app, &state->common_state, w);
  if (r != NO_ERROR)
    return r;

  return NO_ERROR;
}

#define HELP_TEXT \
"\
Help for header field name input\n\
--------------------------------\n\
\n\
This application will let you edit the name of the header field.\n\
\n\
- [TAB]      : Completion of field name with standard header field names.\n\
\n\
- [Enter]    : edition of header field name is finished\n\
- Ctrl-G     : cancel\n\
\n\
- F1         : help\n\
- Ctrl-L     : Console log\n\
\n\
(F1, ? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
