/***************************************************************************
 *   Copyright (C) 2003 by Gav Wood                                        *
 *   gav@cs.york.ac.uk                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 ***************************************************************************/

#define __GEDDEI_BUILD

#include "qfactoryexporter.h"

#include "signaltype.h"
#include "bufferdata.h"
#include "subprocessor.h"
#include "buffer.h"
using namespace Geddei;

class DownSample : public SubProcessor
{
	uint theCount, theScope, theStep;
	enum { Mean = 0, Max, Min };
	uint theConsolidate;

	virtual void processChunks(const BufferDatas &in, BufferDatas &out, const uint chunks) const;
	virtual const bool verifyAndSpecifyTypes(const SignalTypeRefs &inTypes, SignalTypeRefs &outTypes);
	virtual PropertiesInfo specifyProperties() const;
	virtual void initFromProperties(const Properties &properties);

public:
	DownSample() : SubProcessor("DownSample") {}
};

void DownSample::processChunks(const BufferDatas &ins, BufferDatas &outs, const uint chunks) const
{
	if(theCount <= 1)
		if(theScope > 1)
			for(uint i = 0; i < chunks; i++)
				outs[0].sample(i).copyFrom(ins[0].sample(i * theStep));
		else
			for(uint i = 0; i < chunks; i++)
				outs[0][i] = ins[0][i * theStep];
	else
	{	for(uint j = 0; j < chunks; j++)
			for(uint i = 0; i < theScope; i++)
				outs[0](j, i) = 0;
		for(uint j = 0; j < chunks; j++)
		{	for(uint i = 0; i < theCount; i++)
			{	BufferData d = ins[0].sample(i + j*theStep);
				const float *inSample = d.readPointer();
				if(theConsolidate == Mean)
					for(uint k = 0; k < theScope; k++)
						outs[0](j, k) += inSample[k];
				else if(theConsolidate == Max)
					for(uint k = 0; k < theScope; k++)
						if(outs[0](j, k) < inSample[k] || !k) outs[0](j, k) = inSample[k];
				else if(theConsolidate == Min)
					for(uint k = 0; k < theScope; k++)
						if(outs[0](j, k) > inSample[k] || !k) outs[0](j, k) = inSample[k];
			}
		}
		for(uint j = 0; j < chunks; j++)
			for(uint i = 0; i < theScope; i++)
				outs[0](j, i) /= theCount;
	}
}

const bool DownSample::verifyAndSpecifyTypes(const SignalTypeRefs &inTypes, SignalTypeRefs &outTypes)
{
	theScope = inTypes[0].scope();
	outTypes = inTypes[0];
	outTypes[0].asA<SignalType>().setFrequency(inTypes[0].frequency() / (float)theStep);
	return true;
}

void DownSample::initFromProperties(const Properties &properties)
{
	// TODO: setupDimensions according to properties if neccessary.
	theCount = properties.get("Spread").toInt();
	theStep = properties.get("Divisor").toInt();
	theConsolidate = properties["Consolidate"].toInt();
	setupIO(1, 1, max(theCount, theStep), theStep, 1);
}

PropertiesInfo DownSample::specifyProperties() const
{
	return PropertiesInfo("Spread", 1, "The number of samples from which to calculate the value. A larger number results in a smoother output, a lower number is faster. This should be >= 1 and <= Divisor.")
	                     ("Divisor", 2, "The sampling frequency divisor.")
	                     ("Consolidate", 0, "The method to determine the final sample when Spread > 1. { 0: Average; 1: Max; 2: Min }");
}

EXPORT_CLASS(DownSample, 0,3,0, SubProcessor);
