/*
    ext2resize.c -- ext2 resizer frontend
    Copyright (C) 1998, 1999, 2000 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char _ext2resize_c[] = "$Id: ext2resize.c,v 1.15 2001/04/18 22:31:21 adilger Exp $";

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include "ext2.h"

void usage(FILE *outfile, char *progname)
{
	fprintf(outfile,
		"usage: %s [-dfquvV] device [new_size[bkmgt]]\n\t"
		"\t-d, --debug    : turn debug info on\n"
		"\t-f, --force    : skip safety checks\n"
		"\t-q, --quiet    : be quiet (print only errors)\n"
		"\t-u, --unsafe   : skip buffer cache flushing\n"
		"\t-v, --verbose  : be verbose\n"
		"\t-V, --version  : print version and exit\n"
		"\tnew_size is in ext2 blocks (1k, 2k, or 4k) (default),\n"
		"\tdisk Blocks (512 byte), Kilo-, Mega-, Giga-, or Terabytes\n",
		progname);
	exit(1);
}

void parse_args(int *argc, char **argv[], int *flags)
{
	static struct option long_options[] = {
		{"debug", 0, 0, 'd'},
		{"force", 0, 0, 'f'},
		{"quiet", 0, 0, 'q'},
		{"unsafe", 0, 0, 'u'},
		{"verbose", 0, 0, 'v'},
		{"version", 0, 0, 'V'},
		{NULL, 0, 0, 0}};
	int c;
	int dummy;

	while ((c = getopt_long(*argc, *argv, "dfiquvV", long_options,
				&dummy)) != -1) {
		switch (c) {
		case 'd': *flags |= FL_DEBUG; *flags &= ~FL_QUIET; break;
		case 'f': *flags |= FL_FORCE; break;
		case 'q': *flags |= FL_QUIET;
			  *flags &= ~(FL_VERBOSE | FL_DEBUG); break;
		case 'u': *flags &= ~FL_SAFE; break;
		case 'v': *flags |= FL_VERBOSE; *flags &= ~FL_QUIET; break;
		case 'V': *flags |= FL_VERSION; *flags &= ~FL_QUIET; return;
		default: usage(stderr, *argv[0]);
		}
	}

	*argc -= optind;
	*argv += optind;
}

int main(int argc, char *argv[])
{
	struct ext2_fs		*fs;
	struct ext2_dev_handle	*handle;
	char			*dev;
	char			*progname;
	char			 mod = '\0';
	blk_t			 resize = 0;
	int			 flags = FL_SAFE;

	progname = strrchr(argv[0], '/') ? strrchr(argv[0], '/') + 1 : argv[0];
	parse_args(&argc, &argv, &flags);

	if (!(flags & FL_QUIET))
		ext2_print_version(stdout, progname);
	if (flags & FL_VERSION)
		return 0;

	if (argc < 1 || argc > 2)
		usage(stderr, progname);

	dev = argv[0];

	handle = ext2_make_dev_handle_from_file(dev, NULL, progname);
	if (handle == NULL)
		return 1;

	if (argc == 2) {
		sscanf(argv[1], "%i%c", &resize, &mod);

		switch (mod) {	/* Order of these options is important!! */
			case 't':
			case 'T': resize <<= 10; /* no break */
			case 'g':
			case 'G': resize <<= 10; /* no break */
			case 'm':
			case 'M': resize <<= 10; /* no break */
			case 'k': /* Need to open fs to find block size */
			case 'K':
			case  0 : break;
			case 'b':
			case 'B': resize >>= 1; break;
			default: usage(stderr, progname);
		}
	}

	if (mod)
		flags |= FL_KB_BLOCKS;
	if ((fs = ext2_open(handle, resize, flags)) == NULL) {
		fprintf(stderr, "%s: can't open %s\n", progname, dev);
		return 1;
	}

	if (!ext2_resize_fs(fs))
		return 2;
	ext2_close(fs);

	return 0;
}
