/*
 * Farsight Voice+Video library
 * Copyright (c) 2005 Philippe Khalaf <burger@speedy.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <farsight/farsight-plugin.h>

#include "msnavconf.h"

enum {
   ARG_0,
   ARG_REMOTE_SIP_HOST,
   ARG_REMOTE_SIP_PORT,
};

/* protocol specific params */
#define LOCAL_AUDIO_PORT  7078
#define LOCAL_VIDEO_PORT  9078
#define LOCAL_SIP_PORT    5060
//#define REMOTE_AUDIO_PORT 7078
//#define REMOTE_VIDEO_PORT 9078
//#define REMOTE_SIP_PORT   5060

static void     farsight_msnavconf_class_init    (FarsightMsnAVConfClass *klass);
static void     farsight_msnavconf_init          (FarsightMsnAVConf      *msn);

static void     farsight_msnavconf_connect       (FarsightProtocol *protocol);
static void     farsight_msnavconf_disconnect    (FarsightProtocol *protocol);
static GstElement *
                farsight_msnavconf_create_bin    (FarsightProtocol *protocol);

static void     farsight_msnavconf_set_property  (GObject          *object, 
                                            guint             prop_id,
                                            const GValue     *value, 
                                            GParamSpec       *pspec);
static void     farsight_msnavconf_get_property  (GObject          *object, 
                                            guint             prop_id, 
                                            GValue           *value,
                                            GParamSpec       *pspec);

static GObjectClass *parent_class = NULL;

GType 
farsight_msnavconf_get_type (void)
{
    static GType type = 0;

    if (type == 0) {
        static const GTypeInfo info = {
            sizeof (FarsightMsnAVConfClass),
            NULL,
            NULL,
            (GClassInitFunc) farsight_msnavconf_class_init,
            NULL,
            NULL,
            sizeof (FarsightMsnAVConf),
            0,
            (GInstanceInitFunc) farsight_msnavconf_init
        };

        type = g_type_register_static (FARSIGHT_PROTOCOL_TYPE,
                                       "FarsightMsnAVConfType",
                                       &info, 0);
    }

    return type;
}

static void 
farsight_msnavconf_class_init (FarsightMsnAVConfClass *klass)
{
    GObjectClass *gobject_class;
    FarsightProtocolClass *farsight_protocol_class;

    gobject_class = (GObjectClass *) klass;
    farsight_protocol_class = (FarsightProtocolClass*) klass;
    parent_class = g_type_class_peek_parent (klass);

    farsight_protocol_class->connect = farsight_msnavconf_connect;
    farsight_protocol_class->disconnect = farsight_msnavconf_disconnect;
    farsight_protocol_class->create_bin = farsight_msnavconf_create_bin;

    gobject_class->set_property = farsight_msnavconf_set_property;
    gobject_class->get_property = farsight_msnavconf_get_property;

    g_object_class_install_property (G_OBJECT_CLASS (klass), ARG_REMOTE_SIP_HOST,
            g_param_spec_string ("remote-sip-host", "Ip to the remote SIP",
                "The ip of the SIP server to connect to",
                "0.0.0.0", G_PARAM_READWRITE));

    g_object_class_install_property (gobject_klass, ARG_REMOTE_SIP_PORT,
            g_param_spec_uint ("remote-sip-port", "Port to the remote SIP", 
                "The port of the SIP server to connect to",
                0, G_MAXUINT16, 0, G_PARAM_READWRITE));

    gobject_class->finalize = farsight_msnavconf_finalize;
}

static void
farsight_msnavconf_finalize (GObject *object)
{
    FarsightMsnAVConfClass *msn = FARSIGHT_MSNAVCONF (object);

    g_free (msn->remote_sip_host);
    g_free (msn->remote_host);

    G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void 
farsight_msnavconf_init (FarsightMsnAVConf *msn)
{
    msn->remote_sip_host = NULL;

    msn->remote_host = NULL;
}

static void 
farsight_msnavconf_connect (FarsightProtocol *protocol)
{
    FarsightMsnAVConf *msn;

    g_return_if_fail (protocol != NULL);

    msn = FARSIGHT_MSNAVCONF (protocol);

    // Call SIP stack
    if (msn->remote_sip_host)
    {
    }
    else
    {
    }

    // this should get called from the SIP functions
    //FARSIGHT_PROTOCOL_CLASS (parent_class)->connected (protocol);
}

static void 
farsight_msnavconf_disconnect (FarsightProtocol *protocol)
{
    FarsightMsnAVConf *msn;

    g_return_if_fail (protocol != NULL);

    msn = FARSIGHT_MSNAVCONF (protocol);

    // Call SIP stack
}

static GstElement *
farsight_msnavconf_create_bin (FarsightProtocol *protocol)
{
    FarsightMsnAVConf *msn;
    GstElement *bin;
    gint caps = FARSIGHT_PROTOCOL (msn)->caps;

    g_return_val_if_fail (protocol != NULL, NULL);

    msn = FARSIGHT_MSNAVCONF (protocol);

    /* verify the caps and create the bin */
    g_return_val_if_fail (caps != FARSIGHT_CAPS_NONE);

    bin = gst_bin_new ("msnavconf_bin");

    // if video is on, create video pipeline
    if( caps & FARSIGHT_CAPS_CAN_SEND_VIDEO == FARSIGHT_CAPS_CAN_SEND_VIDEO
            || caps & FARSIGHT_CAPS_CAN_RECEIVE_VIDEO == FARSIGHT_CAPS_CAN_RECEIVE_VIDEO )
    {
        GstElement *rtpbin = NULL;
        GstElement *r263encoder = NULL;
        GstElement *r263depayloader = NULL;
        GstElement *h263decoder = NULL;

        // create our elements
        rtpbin = gst_element_factory_make ("rtpbin", "rtpbinvideo");
        gchar *ipport = g_strconcat (msn->remote_host, ":", REMOTE_VIDEO_PORT, NULL);
        g_object_set (G_OBJECT(rtpbin), "destaddrs", ipport, NULL);
        g_free (ipport);
        g_object_set (G_OBJECT(rtpbin), "localport", LOCAL_VIDEO_PORT, NULL);
        r263encoder = gst_element_factory_make ("r263enc", "r263encoderelement");
        g_object_set (G_OBJECT(rtpbin), "rtp-support", 1, NULL);
        r263depayloader = gst_element_factory_make ("r263depayloader", "r263depayloaderelement");
        h263decoder = gst_element_factory_make ("h263dec", "h263decoderelement");

        // add them to the bin
        gst_bin_add_many (GST_BIN(bin), rtpbin, r263encoder, 
                r263depayloader, h263decoder, NULL);

        // link them
        gst_element_link_pads (rtpbin, "sink", r263encoder, "src");
        gst_element_link_pads (rtpbin, "src", r263depayloader, "sink");
        gst_element_link_pads (r263depayloader, "src", h263decoder, "sink");

        // link ghost pads
        gst_element_add_pad (GST_ELEMENT (bin),
            gst_ghost_pad_new("videoin",
                gst_element_get_pad (r263encoder, "sink")));
        gst_element_add_pad (GST_ELEMENT (bin),
            gst_ghost_pad_new("videoout",
                gst_element_get_pad (h263decoder, "src")));
    }
    // if audio is on, create audio pipeline 
    if( caps & FARSIGHT_CAPS_CAN_SEND_AUDIO == FARSIGHT_CAPS_CAN_SEND_AUDIO
            || caps & FARSIGHT_CAPS_CAN_RECEIVE_AUDIO == FARSIGHT_CAPS_CAN_RECEIVE_AUDIO )
    {
        GstElement *rtpbin = NULL;
        GstElement *msgsmencoder = NULL;
        GstElement *msgsmdepayloader = NULL;
        GstElement *msgsmdecoder = NULL;

        // create our elements
        rtpbin = gst_element_factory_make ("rtpbin", "rtpbinaudio");
        gchar *ipport = g_strconcat (msn->remote_host, ":", REMOTE_AUDIO_PORT, NULL);
        g_object_set (G_OBJECT(rtpbin), "destaddrs", ipport, NULL);
        g_free (ipport);
        g_object_set (G_OBJECT(rtpbin), "localport", LOCAL_AUDIO_PORT, NULL);
        msgsmencoder = gst_element_factory_make ("msgsmenc", "msgsmencelement");
        msgsmdepayloader = gst_element_factory_make ("msgsmdepayloader", "msgsmdepayloaderelement");
        msgsmdecoder = gst_element_factory_make ("msgsmdecoder", "msgsmdecelement");

        // add them to the bin
        gst_bin_add_many (GST_BIN(bin), rtpbin, msgsmencoder, 
                msgsmdepayloader, msgsmdecoder, NULL);

        // link them
        gst_element_link_pads (rtpbin, "sink", msgsmencoder, "src");
        gst_element_link_pads (rtpbin, "src", msgsmdepayloader, "sink");
        gst_element_link_pads (msgsmdepayloader, "src", msgsmdecoder, "sink");

        // link ghost pads
        gst_element_add_pad (GST_ELEMENT (bin),
            gst_ghost_pad_new("audioin",
                gst_element_get_pad (msgsmencoder, "sink")));
        gst_element_add_pad (GST_ELEMENT (bin),
            gst_ghost_pad_new("audioout",
                gst_element_get_pad (msgsmdecoder, "src")));
    }

    return bin;
}

static void
farsight_msnavconf_set_property (GObject      *object, 
                           guint         prop_id,
                           const GValue *value, 
                           GParamSpec   *pspec)
{
    FarsightMsnAVConf *msn;

    g_return_if_fail (FARSIGHT_IS_MSNAVCONF (object));

    msn = FARSIGHT_MSNAVCONF (object);

    switch (prop_id) {
        case ARG_REMOTE_SIP_HOST:
            if (msn->remote_sip_host)
                g_free (msn->remote_sip_host);
            msn->remote_sip_host = g_value_dup_string (value);
            break;
        case ARG_REMOTE_SIP_PORT:
            msn->remote_sip_port = g_value_get_uint (value);
            break;
    }
}

static void
farsight_msnavconf_get_property (GObject    *object, 
                           guint       prop_id, 
                           GValue     *value,
                           GParamSpec *pspec)
{
    FarsightMsnAVConf *msn;

    g_return_if_fail (FARSIGHT_IS_MSNAVCONF (object));
    msn = FARSIGHT_MSNAVCONF (object);

    switch (prop_id) {
        case ARG_REMOTE_SIP_HOST:
            g_value_set_string (value, msn->remote_sip_host);
            break;
        case ARG_REMOTE_SIP_PORT:
            g_value_set_uint (value, msn->remote_sip_port);
            break;
    }
}

static gboolean 
init_plugin (FarsightPlugin *plugin)
{
    if (!farsight_protocol_register (plugin, FARSIGHT_TYPE_MSNAVCONF))
        return FALSE;
    return TRUE;
}

static FarsightPluginInfo plugin_info = {
    FARSIGHT_MAJOR_VERSION,
    FARSIGHT_MINOR_VERSION,

    "MSN Protocol Info",                                /* description */
    "0.1.0",                                            /* version */
    "Philippe Khalaf <burger@speedy.org>",              /* author */
    "http://farsight.sf.net/",                          /* homepage */
    NULL,                                               /* load */
    NULL                                                /* unload */
};

FARSIGHT_INIT_PLUGIN (init_plugin, plugin_info);
