#!/bin/sh
#
#	fhist - file history and comparison tools
#	Copyright (C) 2005 Peter Miller;
#	All rights reserved.
#
#	This program is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	This program is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with this program; if not, write to the Free Software
#	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
#
# MANIFEST: Test the fmerge functionality
#

work=/tmp/$$
PAGER=cat
export PAGER
LINES=24
export LINES
COLS=80
export COLS

activity="setup"

no_result()
{
	set +x
	echo "NO RESULT test of the fmerge functionality ($activity)" 1>&2
	cd $here
	rm -rf $work
	exit 2
}
fail()
{
	set +x
	echo "FAILED test of the fmerge functionality ($activity)" 1>&2
	cd $here
	rm -rf $work
	exit 1
}
pass()
{
	set +x
	echo PASSED 1>&2
	cd $here
	rm -rf $work
	exit 0
}
trap "fail" 1 2 3 15

#
# use the built-in error messages
#
FHIST_MESSAGE_LIBRARY=$work/no-such-dir
export FHIST_MESSAGE_LIBRARY
unset LANG
unset LANGUAGE

here=`pwd`
if test $? -ne 0 ; then exit 2; fi
mkdir $work
if test $? -ne 0 ; then exit 2; fi
cd $work
if test $? -ne 0 ; then no_result; fi

if [ "$1" != "" ]; then bin=$here/$1/bin; else bin=$here/bin; fi

# --------------------------------------------------------------------------
#
# Case: insert A
#
activity="insert A"
cat > base << 'fubar'
one
two
three
four
five
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
three
inserted line
inserted line
four
five
fubar
test $? -eq 0 || no_result

cp base fileB
test $? -eq 0 || no_result

cp fileA test.ok
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out
test $? -eq 0 || fail

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: delete A
#
activity="delete A"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
five
six
fubar
test $? -eq 0 || no_result

cp base fileB
test $? -eq 0 || no_result

cp fileA test.ok
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out
test $? -eq 0 || fail

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: insert A, delete A
#
activity="insert A, delete A"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
xxxxxx
yyyyyy
five
six
fubar
test $? -eq 0 || no_result

cp base fileB
test $? -eq 0 || no_result

cp fileA test.ok
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out
test $? -eq 0 || fail

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: insert B
#
activity="insert B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cp base fileA
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
three
xxxxxx
yyyyyy
four
five
six
fubar
test $? -eq 0 || no_result

cp fileB test.ok
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out
test $? -eq 0 || fail

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: delete B
#
activity="delete B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cp base fileA
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
five
six
fubar
test $? -eq 0 || no_result

cp fileB test.ok
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out
test $? -eq 0 || fail

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: insert B, delete B
#
activity="insert B, delete B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cp base fileA
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
xxxxxxx
yyyyyyyy
zzzzzzz
five
six
fubar
test $? -eq 0 || no_result

cp fileB test.ok
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out
test $? -eq 0 || fail

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: delete A, delete B
#
activity="delete A, delete B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
six
fubar
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
five
six
fubar
test $? -eq 0 || no_result

cat > test.ok << 'fubar'
one
two
/-/-/-/-/-/-/-/-/-/ BEGIN CONFLICT  [O3 A3 B3] /-/-/-/-/-/-/-/-/-/-/
/-/-/-/-/-/-/-/-/-/-/-/-/  END CONFLICT   /-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
six
fubar
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out 2> LOG
if test $? -ne 1
then
    cat LOG
    fail
fi

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: delete A, insert A, delete B
#
activity="delete A, insert A, delete B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
xxxxx
yyyyy
zzzzz
six
fubar
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
five
six
fubar
test $? -eq 0 || no_result

cat > test.ok << 'fubar'
one
two
/-/-/-/-/-/-/-/-/-/ BEGIN CONFLICT  [O3 A3 B3] /-/-/-/-/-/-/-/-/-/-/
xxxxx
yyyyy
zzzzz
/-/-/-/-/-/-/-/-/-/-/-/-/  END CONFLICT   /-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
six
fubar
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out 2> LOG
if test $? -ne 1
then
    cat LOG
    fail
fi

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: delete A, insert A, insert B
#
activity="delete A, insert A, insert B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
xxxxx
yyyyy
zzzzz
five
six
fubar
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
mmmmmmm
nnnnnnn
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > test.ok << 'fubar'
one
two
/-/-/-/-/-/-/-/-/-/ BEGIN CONFLICT  [O3 A3 B3] /-/-/-/-/-/-/-/-/-/-/
xxxxx
yyyyy
zzzzz
/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
mmmmmmm
nnnnnnn
/-/-/-/-/-/-/-/-/-/-/-/-/  END CONFLICT   /-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
five
six
fubar
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out 2> LOG
if test $? -ne 1
then
    cat LOG
    fail
fi

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: delete A, insert B
#
activity="delete A, insert B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
five
six
fubar
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
mmmmmmm
nnnnnnn
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > test.ok << 'fubar'
one
two
/-/-/-/-/-/-/-/-/-/ BEGIN CONFLICT  [O3 A3 B3] /-/-/-/-/-/-/-/-/-/-/
mmmmmmm
nnnnnnn
/-/-/-/-/-/-/-/-/-/-/-/-/  END CONFLICT   /-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
five
six
fubar
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out 2> LOG
if test $? -ne 1
then
    cat LOG
    fail
fi

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: delete A, delete B, insert B
#
activity="delete A, delete B, insert B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
five
six
fubar
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
mmmmmmm
nnnnnnn
ooooooo
four
five
six
fubar
test $? -eq 0 || no_result

cat > test.ok << 'fubar'
one
two
/-/-/-/-/-/-/-/-/-/ BEGIN CONFLICT  [O3 A3 B3] /-/-/-/-/-/-/-/-/-/-/
mmmmmmm
nnnnnnn
ooooooo
/-/-/-/-/-/-/-/-/-/-/-/-/  END CONFLICT   /-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
five
six
fubar
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out 2> LOG
if test $? -ne 1
then
    cat LOG
    fail
fi

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: insert A, delete B, insert B
#
activity="insert A, delete B, insert B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
xxxxxxx
yyyyyyy
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
mmmmmmm
nnnnnnn
ooooooo
four
five
six
fubar
test $? -eq 0 || no_result

cat > test.ok << 'fubar'
one
two
/-/-/-/-/-/-/-/-/-/ BEGIN CONFLICT  [O3 A3 B3] /-/-/-/-/-/-/-/-/-/-/
xxxxxxx
yyyyyyy
/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
mmmmmmm
nnnnnnn
ooooooo
/-/-/-/-/-/-/-/-/-/-/-/-/  END CONFLICT   /-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
four
five
six
fubar
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out 2> LOG
if test $? -ne 1
then
    cat LOG
    fail
fi

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: insert A, delete B
#
activity="insert A, delete B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
xxxxxxx
yyyyyyy
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
four
five
six
fubar
test $? -eq 0 || no_result

cat > test.ok << 'fubar'
one
two
/-/-/-/-/-/-/-/-/-/ BEGIN CONFLICT  [O3 A3 B3] /-/-/-/-/-/-/-/-/-/-/
xxxxxxx
yyyyyyy
/-/-/-/-/-/-/-/-/-/-/-/-/  END CONFLICT   /-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
four
five
six
fubar
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out 2> LOG
if test $? -ne 1
then
    cat LOG
    fail
fi

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: delete A, insert A, delete B, insert B
#
activity="delete A, insert A, delete B, insert B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
xxxxxxx
yyyyyyy
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
jjjjjjj
kkkkkkk
five
six
fubar
test $? -eq 0 || no_result

cat > test.ok << 'fubar'
one
two
/-/-/-/-/-/-/-/-/-/ BEGIN CONFLICT  [O3 A3 B3] /-/-/-/-/-/-/-/-/-/-/
xxxxxxx
yyyyyyy
/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
jjjjjjj
kkkkkkk
/-/-/-/-/-/-/-/-/-/-/-/-/  END CONFLICT   /-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
five
six
fubar
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out 2> LOG
if test $? -ne 1
then
    cat LOG
    fail
fi

diff test.ok test.out
test $? -eq 0 || fail

# --------------------------------------------------------------------------
#
# Case: insert A, insert B
#
activity="insert A, insert B"
cat > base << 'fubar'
one
two
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileA << 'fubar'
one
two
xxxxxxx
yyyyyyy
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > fileB << 'fubar'
one
two
jjjjjjj
kkkkkkk
three
four
five
six
fubar
test $? -eq 0 || no_result

cat > test.ok << 'fubar'
one
two
/-/-/-/-/-/-/-/-/-/ BEGIN CONFLICT  [O3 A3 B3] /-/-/-/-/-/-/-/-/-/-/
xxxxxxx
yyyyyyy
/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
jjjjjjj
kkkkkkk
/-/-/-/-/-/-/-/-/-/-/-/-/  END CONFLICT   /-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/-/
three
four
five
six
fubar
test $? -eq 0 || no_result

$bin/fmerge base fileA fileB > test.out 2> LOG
if test $? -ne 1
then
    cat LOG
    fail
fi

diff test.ok test.out
test $? -eq 0 || fail

#
# Only definite negatives are possible.
# The functionality exercised by this test appears to work,
# no other guarantees are made.
#
pass
