/******************************************************************** 
   Copyright (C) 2000 Bassoukos Tassos <abas@aix.meng.auth.gr>
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later
   version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*********************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gnome.h>

#include "guiprefs.h"
#include "main.h"
#include "hldat.h"
#include "pixmap.h"
#include "hldat.h"
#include "guiutils.h"

typedef struct {
  char *config_path;
  GtkWidget *pane;
} PanePrefs;

typedef struct {
  char *config_path;
  GtkCList *list;
  int num_columns;
  int *column_width;
} CListPrefs;

static void handle_window_resize(GtkWidget *widget,
				 GtkAllocation *allocation,
				 gpointer path){
  char *p=g_strconcat("/",APPLICATION_NAME,"/",(char *)path,NULL);
  char buf[512];
  
  sprintf(buf,"%d,%d",allocation->width,allocation->height);
  gnome_config_set_string(p,buf);
  g_free(p);
  gnome_config_sync();
}

void guiprefs_add_window(GtkWindow *w,const char *path){
  char *p=g_strconcat("/",APPLICATION_NAME,"/",path,NULL);
  char *s=gnome_config_get_string(p),**v=NULL;

  if(s!=NULL){
    v=g_strsplit(s,",",2);
    gtk_window_set_default_size(w,atoi(v[0]),atoi(v[1]));
    g_strfreev(v);
    g_free(s);
  }
  g_free(p);
  gtk_signal_connect(GTK_OBJECT(w),"size-allocate",
		     GTK_SIGNAL_FUNC(handle_window_resize),
		     (gpointer)path);
}

static void handle_paned_resize(GtkWidget *widget,
				GtkAllocation *allocation,
				gpointer data){
  PanePrefs *pp=(PanePrefs *)data;
  int pos;
  
  if(GTK_IS_VPANED(pp->pane)){
    pos=allocation->height;
  } else {
    pos=allocation->width;
  }
  gnome_config_set_int(pp->config_path,pos);
  gnome_config_sync();
}

void guiprefs_add_paned(GtkPaned *w,const char *path){
  PanePrefs *pp=g_malloc(sizeof(PanePrefs));
  char *p=g_strconcat("/",APPLICATION_NAME,"/",path,NULL);
  int pos=gnome_config_get_int(p);
  GList *l=gtk_container_children(GTK_CONTAINER(w));
  GtkWidget *h1=(GtkWidget *)l->data;;

  pp->config_path=p;
  if(pos!=0){
    gtk_paned_set_position(GTK_PANED(w),pos);
  }
  pp->pane=GTK_WIDGET(w);
  gtk_signal_connect(GTK_OBJECT(h1),"size-allocate",
		     GTK_SIGNAL_FUNC(handle_paned_resize),
		     (gpointer)pp);
  g_list_free(l);
}

static void handle_clist_resize_column(GtkCList *clist,
				       gint column,
				       gint width,
				       gpointer user_data){
  CListPrefs *cp=(CListPrefs*)user_data;
  char buf[512],num[64];
  int i;

  cp->column_width[column]=width;
  buf[0]=0;
  for(i=0;i<cp->num_columns;i++){
    sprintf(num,"%d",cp->column_width[i]);
    strcat(buf,num);
    if(i+1<cp->num_columns)
      strcat(buf,",");
  }
  gnome_config_set_string(cp->config_path,buf);
  gnome_config_sync();
}

void guiprefs_add_clist(GtkCList *w,const char *path,const char *def_widths){
  CListPrefs *cp=g_malloc(sizeof(CListPrefs));
  char *p=g_strconcat("/",APPLICATION_NAME,"/",path,NULL);
  char *s=gnome_config_get_string(p),**v=NULL;
  GtkArg arg;
  int i;

  cp->list=w;
  cp->config_path=p;
  arg.name="n_columns";
  gtk_object_getv(GTK_OBJECT(w),1,&arg);
  cp->num_columns=GTK_VALUE_INT(arg);
  cp->column_width=g_malloc(sizeof(*cp->column_width)*cp->num_columns);
  if(s==NULL){
    s=g_strdup(def_widths);
  }
  v=g_strsplit(s,",",cp->num_columns);
  for(i=0;i<cp->num_columns && v[i]!=NULL;i++){
    if((cp->column_width[i]=atoi(v[i]))>0)
      gtk_clist_set_column_width(w,i,cp->column_width[i]);
  }
  g_strfreev(v);
  g_free(s);
  gtk_signal_connect(GTK_OBJECT(w),"resize-column",
		     GTK_SIGNAL_FUNC(handle_clist_resize_column),
		     (gpointer)cp);
}

/*========================================*/
/*========================================*/
/*========================================*/
/*========================================*/

typedef struct {
  char *name;
  int icon;
  int level;
  void (*create)(GtkWidget *);
} PrefsPages;

struct _configelement;

typedef struct {
  void (*read)(struct _configelement *);
  void (*write)(struct _configelement *);
} ConfigClass;

typedef struct _configelement {
  ConfigClass *type;
  char *path;
  gpointer addr;
} ConfigElement;

gboolean prefs_dblclick_connect_query_bookmarks=TRUE;
gboolean prefs_dblclick_connect_query_trackers=TRUE;

gboolean tracker_sort_only_by_ip=TRUE;

int prefs_task_refresh_msec=100;

char *nickname=NULL;
gboolean nickname_append_max_speed=FALSE;
int user_icon_number=160;
int client_version=151;
int nickname_append_speed_seconds=60;
gboolean passwords_are_hidden=TRUE;
int notebook_tab_side=GTK_POS_LEFT;
gboolean notebook_has_icons=TRUE;
GdkColor notebook_tab_highlight_color;

gboolean dont_show_agreements=FALSE;
PrefsFont agreement_font=DEF_PREF_FONT;

gboolean auto_open_news=TRUE;
gboolean news_big_messagewindow=TRUE;
PrefsFont news_font=DEF_PREF_FONT;

gboolean auto_popup_pubchat=FALSE;

gboolean auto_open_files=TRUE;
gboolean mix_files_and_folders=FALSE;
gboolean hide_local_dotfiles=TRUE;
gboolean cache_remote_directories=TRUE;
char *default_local_dir=NULL;
gboolean strip_spaces_on_local_files=TRUE;
gboolean sanitize_local_filenames=FALSE;
gboolean download_convert_text=TRUE;
gboolean upload_convert_text=TRUE;
gboolean download_remove_null_files=TRUE;
gboolean download_queue=TRUE;
gboolean upload_queue=TRUE;
gboolean files_small_sizes=TRUE;
int retry_count,retry_delay;
int downloads_max_speed=0, uploads_max_speed=0;

gboolean prefs_messages_show_with_reply=FALSE;
gboolean prefs_messages_reply_quoted=FALSE;
PrefsFont chat_font=DEF_PREF_FONT;

gboolean privchat_auto_join=TRUE;
PrefsFont privchat_font=DEF_PREF_FONT;

gboolean auto_open_userlist=TRUE;
gboolean userlist_aa_canvas=TRUE;
gboolean userlist_sort_name=TRUE;
gboolean userlist_group_by_status=TRUE;
gboolean userlist_use_icons=TRUE;
gboolean userlist_no_connect=TRUE;
gboolean userlist_use_status_icons=TRUE;
GdkColor userlist_colors[4],userlist_highlight_color;
gushort userlist_highlight_color_alpha=0x8000;
PrefsFont userlist_font=DEF_PREF_FONT;
gboolean userlist_dock_right=FALSE;
gboolean enable_recursive_downloads=TRUE;
GList *hotline_data_files=NULL;
gboolean use_hotline_icons=TRUE;

static gboolean initialized_colors=FALSE;
static GtkWidget *property_window=NULL;
static GtkWidget *nickname_entry_widget=NULL;
static GtkWidget *default_local_dir_widget=NULL;
static GtkWidget *icon_files_widget=NULL;

#ifdef HAVE_ESD
static gboolean play_sound_types[HL_SOUND_MAX-HL_SOUND_CHAT];
static gboolean play_sounds=FALSE;
#endif

static void read_string(ConfigElement *e){
  if(*((char **)e->addr)!=NULL)
    free(*(char **)e->addr);
  *(char **)e->addr=gnome_config_get_string(e->path);
}
static void write_string(ConfigElement *e){
  if(*((char **)e->addr)==NULL)
    return;
  gnome_config_set_string(e->path,*((char **)e->addr));
}
static void read_number(ConfigElement *e){
  *(int *)e->addr=gnome_config_get_int(e->path);
}
static void write_number(ConfigElement *e){
  gnome_config_set_int(e->path,*(int *)e->addr);
}
static void read_bool(ConfigElement *e){
  *(gboolean *)e->addr=(gnome_config_get_int(e->path)==0)?FALSE:TRUE;
}
static void write_bool(ConfigElement *e){
  gnome_config_set_int(e->path,(*(gboolean *)e->addr)==FALSE?0:1);
}
static void read_color(ConfigElement *e){
  int red,green,blue;
  char *s;
  GdkColor *c=(GdkColor *)e->addr;
  s=gnome_config_get_string(e->path);
  sscanf(s,"%d %d %d",&red,&green,&blue);
  g_free(s);
  c->red=red<<8;
  c->green=green<<8;
  c->blue=blue<<8;
}
static void write_color(ConfigElement *e){
  char buf[256];
  sprintf(buf,"%d %d %d",((GdkColor *)e->addr)->red>>8,
	  ((GdkColor *)e->addr)->green>>8,((GdkColor *)e->addr)->blue>>8);
  gnome_config_set_string(e->path,buf);
}
static void read_stringlist(ConfigElement *e){
  GList **gl=e->addr;
  char *s,**v;
  int i;
  while(*gl!=NULL){
    free((*gl)->data);
    *gl=g_list_remove(*gl,(*gl)->data);
  }
  s=gnome_config_get_string(e->path);
  if(s==NULL) return;
  v=g_strsplit(s,":",0);
  for(i=0;v[i]!=NULL;i++)
    *gl=g_list_append(*gl,strdup(v[i]));
  g_strfreev(v);
}
static void write_stringlist(ConfigElement *e){
  GList **gl=e->addr,*l;
  char *s,**v;
  int i,count;

  for(l=*gl,count=0;l!=NULL;l=l->next,count++);
  v=calloc(sizeof(char *),count+1);
  v[count]=NULL;
  for(l=*gl,i=0;l!=NULL;l=l->next,i++)
    v[i]=l->data;
  s=g_strjoinv(":",v);
  gnome_config_set_string(e->path,s);
}
static void read_font(ConfigElement *e){
  PrefsFont *pf=(PrefsFont *)(e->addr);
  char buf[256];
  if(pf->font_name!=NULL)
    free(pf->font_name),pf->font_name=NULL;
  if(pf->font!=NULL)
    gdk_font_unref(pf->font),pf->font=NULL;
  sprintf(buf,"%s_use",e->path);
  pf->use_custom_font=gnome_config_get_int(buf)!=0?TRUE:FALSE;
  if(pf->use_custom_font==TRUE){
    sprintf(buf,"%s_font=fixed",e->path);
    pf->font_name=gnome_config_get_string(buf);
    pf->font=gdk_font_load(pf->font_name);
    if(pf->font!=NULL)
      gdk_font_ref(pf->font);
  }
}
static void write_font(ConfigElement *e){
  PrefsFont *pf=(PrefsFont *)(e->addr);
  char buf[256];
  sprintf(buf,"%s_use",e->path);
  gnome_config_set_int(buf,pf->use_custom_font);
  if(pf->font_name!=NULL){
    sprintf(buf,"%s_font",e->path);
    gnome_config_set_string(buf,pf->font_name);
  }
}


static ConfigClass conftypes[]={
  {read_string,write_string},
  {read_number,write_number},
  {read_bool,write_bool},
  {read_color,write_color},
  {read_font,write_font},
  {read_stringlist,write_stringlist}
};
#define STRING_TYPE (&conftypes[0])
#define NUMBER_TYPE (&conftypes[1])
#define BOOL_TYPE (&conftypes[2])
#define COLOR_TYPE (&conftypes[3])
#define FONT_TYPE (&conftypes[4])
#define STRLIST_TYPE (&conftypes[5])

static ConfigElement confitems[]={
  {STRING_TYPE,"nickname=Lazy User",&nickname},
  {BOOL_TYPE,"doubleclick_queries_bookmarks=0",
   &prefs_dblclick_connect_query_bookmarks},
  {BOOL_TYPE,"doubleclick_queries_trackers=1",
   &prefs_dblclick_connect_query_trackers},
  {BOOL_TYPE,"tracker_sort_only_by_ip=0",&tracker_sort_only_by_ip},
  {NUMBER_TYPE,"icon_number=160",&user_icon_number},
  {NUMBER_TYPE,"client_version=151",&client_version},
  {BOOL_TYPE,"nickname_append_max_speed=0",&nickname_append_max_speed},
  {NUMBER_TYPE,"nickname_append_speed_seconds=60",
   &nickname_append_speed_seconds},
  {BOOL_TYPE,"dont_show_agreements=0",&dont_show_agreements},
  {BOOL_TYPE,"passwords_are_hidden=1",&passwords_are_hidden},
  {NUMBER_TYPE,"task_refresh_delay=500",&prefs_task_refresh_msec},
  {BOOL_TYPE,"auto_open_news=1",&auto_open_news},
  {BOOL_TYPE,"news_big_messagewindow=1",&news_big_messagewindow},
  {BOOL_TYPE,"auto_open_userlist=1",&auto_open_userlist},
  {BOOL_TYPE,"userlist_aa_canvas=1",&userlist_aa_canvas},
  {BOOL_TYPE,"userlist_sort_name=1",&userlist_sort_name},
  {BOOL_TYPE,"userlist_group_by_status=1",&userlist_group_by_status},
  {BOOL_TYPE,"userlist_no_connect=1",&userlist_no_connect},
  {BOOL_TYPE,"userlist_use_icons=1",&userlist_use_icons},
  {BOOL_TYPE,"userlist_use_status_icons=1",&userlist_use_status_icons},
  {STRLIST_TYPE,"hotline_data_files=./hotline.dat",&hotline_data_files},
  {BOOL_TYPE,"use_hotline_icons=1",&use_hotline_icons},
  {COLOR_TYPE,"userlist_admin_ghost_color=128 0 0",
   &userlist_colors[ADMIN_GHOST_COLOR]},
  {COLOR_TYPE,"userlist_ghost_color=128 128 128",
   &userlist_colors[GHOST_COLOR]},
  {COLOR_TYPE,"userlist_admin_color=255 0 0",&userlist_colors[ADMIN_COLOR]},
  {COLOR_TYPE,"userlist_normal_color=0 0 0",&userlist_colors[NORMAL_COLOR]},
  {COLOR_TYPE,"userlist_highlight_color=32 32 127",&userlist_highlight_color},
  {NUMBER_TYPE,"userlist_highlight_color_alpha=32768",
   &userlist_highlight_color_alpha},
  {BOOL_TYPE,"auto_popup_pubchat=0",&auto_popup_pubchat},
  {BOOL_TYPE,"prefs_messages_show_with_reply=0",
   &prefs_messages_show_with_reply},
  {BOOL_TYPE,"prefs_messages_reply_quoted=0",&prefs_messages_reply_quoted},
  {BOOL_TYPE,"auto_open_files=1",&auto_open_files},
  {BOOL_TYPE,"mix_files_and_folders=0",&mix_files_and_folders},
  {BOOL_TYPE,"hide_local_dotfiles=1",&hide_local_dotfiles},
  {BOOL_TYPE,"cache_remote_directories=1",&cache_remote_directories},
  {STRING_TYPE,"default_local_dir=/tmp",&default_local_dir},
  {BOOL_TYPE,"strip_spaces_on_local_files=1",&strip_spaces_on_local_files},
  {BOOL_TYPE,"sanitize_local_filenames=0",&sanitize_local_filenames},
  {BOOL_TYPE,"download_convert_text=1",&download_convert_text},
  {BOOL_TYPE,"upload_convert_text=1",&upload_convert_text},
  {BOOL_TYPE,"enable_recursive_downloads=1",&enable_recursive_downloads},
  {BOOL_TYPE,"download_remove_null_files=1",&download_remove_null_files},
  {BOOL_TYPE,"download_queue=1",&download_queue},
  {BOOL_TYPE,"upload_queue=1",&upload_queue},
  {NUMBER_TYPE,"download_speed_limit=0",&downloads_max_speed},
  {NUMBER_TYPE,"upload_speed_limit=0",&uploads_max_speed},
  {BOOL_TYPE,"files_small_sizes=1",&files_small_sizes},
  DEF_FONT_CONFIG("userlist_font",userlist_font),
  DEF_FONT_CONFIG("agreement_font",agreement_font),
  DEF_FONT_CONFIG("news_font",news_font),
  DEF_FONT_CONFIG("chat_font",chat_font),
  {NUMBER_TYPE,"notebook_tab_side=0",&notebook_tab_side},
  {BOOL_TYPE,"notebook_has_icons=1",&notebook_has_icons},
  {BOOL_TYPE,"userlist_dock_right=0",&userlist_dock_right},
  {COLOR_TYPE,"notebook_tab_highlight_color=127 0 0",&notebook_tab_highlight_color},
  {NUMBER_TYPE,"retry_count=10",&retry_count},
  {NUMBER_TYPE,"retry_delay=10",&retry_delay},

#ifdef HAVE_ESD
  {BOOL_TYPE,"play_sounds=1",&play_sounds},
  {BOOL_TYPE,"play_sound_chat=1",&play_sound_types[0]},
  {BOOL_TYPE,"play_sound_logged_in=1",&play_sound_types[1]},
  {BOOL_TYPE,"play_sound_logout=1",&play_sound_types[2]},
  {BOOL_TYPE,"play_sound_broadcast=1",&play_sound_types[3]},
  {BOOL_TYPE,"play_sound_login=1",&play_sound_types[4]},
  {BOOL_TYPE,"play_sound_error=1",&play_sound_types[5]},
  {BOOL_TYPE,"play_sound_filedone=1",&play_sound_types[6]},
  {BOOL_TYPE,"play_sound_privchat=1",&play_sound_types[7]},
#endif
  {BOOL_TYPE,"privchat_auto_join=1",&privchat_auto_join},
  DEF_FONT_CONFIG("privchat_font",privchat_font),

  {NULL,NULL,NULL}
};

static void init_colors(){
  int i;

  if(initialized_colors==FALSE){
    initialized_colors=TRUE;
  } else {
    for(i=0;i<4;i++)
      gdk_colormap_free_colors(main_colormap,&userlist_colors[i],1);
  }
  for(i=0;i<4;i++)
    gdk_colormap_alloc_color(main_colormap,&userlist_colors[i],FALSE,TRUE);
}

gboolean load_prefs(){
  ConfigElement *ce;
  gnome_config_push_prefix("/" PACKAGE "/Main/Prefs/");
  for(ce=&confitems[0];ce->path!=NULL;ce++)
    ce->type->read(ce);
  gnome_config_pop_prefix();
  init_colors();
  return FALSE;
}

static void close_box(){
  property_window=NULL;
  gtk_idle_add((gboolean(*)(gpointer))load_prefs,NULL);
}

static void save_prefs(){
  ConfigElement *ce;
  gnome_config_push_prefix("/" PACKAGE "/Main/Prefs/");
  for(ce=&confitems[0];ce->path!=NULL;ce++)
    ce->type->write(ce);
  gnome_config_sync();
  gnome_config_pop_prefix();
}

static void get_prefs(){
  GList *r;
  if(nickname!=NULL)
    free(nickname);
  nickname=gtk_editable_get_chars(GTK_EDITABLE(nickname_entry_widget),0,-1);
  if(default_local_dir!=NULL)
    free(default_local_dir);
  default_local_dir=gtk_editable_get_chars(GTK_EDITABLE(default_local_dir_widget),0,-1);
  while(hotline_data_files!=NULL){
    free(hotline_data_files->data);
    hotline_data_files=g_list_remove(hotline_data_files,hotline_data_files->data);
  }
  for(r=GTK_CLIST(icon_files_widget)->row_list;r!=NULL;r=r->next){
    hotline_data_files=g_list_append(hotline_data_files,
				     strdup(GTK_CLIST_ROW(r)->cell[0].u.text));
  }
}
			       
static void apply_prefs(GtkWidget *button, gpointer dummy){
  if(property_window==NULL) return;
  get_prefs();
  save_prefs();
  gtk_widget_destroy(property_window);
}

static void prefs_changed(){
  if(property_window==NULL) return;
  gnome_dialog_set_sensitive(GNOME_DIALOG(property_window),1,TRUE);
}

static void boolean_state_changed(GtkWidget *w,gpointer data){
  if(property_window==NULL) return;
  *(gboolean *)data=
    gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
  prefs_changed();
}
static void integer_adjustment_changed(GtkAdjustment *w,gpointer data){
  *(gint *)data=(gint)(w->value);
  prefs_changed();
}
static void editable_changed(GtkEditable *w,gpointer data){
  prefs_changed();
}

static void widget_set_font(GtkWidget *w,PrefsFont *pf){
  GtkStyle *style;

  if(pf->font==NULL) return;
  style=gtk_style_copy(gtk_widget_get_style(main_app));
  gdk_font_unref(style->font);
  style->font=pf->font;
  gdk_font_ref(style->font);
  gtk_widget_set_style(w,style);
}

static void font_changed(GnomeFontPicker *fontpicker,
			      gpointer arg1,
			      gpointer user_data){
  PrefsFont *pf=(PrefsFont *)user_data;
  if(pf->font_name!=NULL)
    free(pf->font_name);
  pf->font_name=strdup(gnome_font_picker_get_font_name(fontpicker));
  if(pf->font!=NULL)
    gdk_font_unref(pf->font);
  pf->font=gnome_font_picker_get_font(fontpicker);
  gdk_font_ref(pf->font);
  gtk_label_set_text(GTK_LABEL(pf->label),pf->font_name);
  widget_set_font(pf->label,pf);
  prefs_changed();
}

static GtkWidget *create_boolean_checkbox(char *name,gboolean *value){
  GtkWidget *w=gtk_check_button_new_with_label(name);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w),*value);
  gtk_signal_connect(GTK_OBJECT(w),"toggled",
		     GTK_SIGNAL_FUNC(boolean_state_changed),
		     value);
  return w;
}

static void boolean_state_changed_activate(GtkWidget *w,gpointer data){
  gtk_widget_set_sensitive(GTK_WIDGET(data),gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)));
}

static void boolean_checkbox_activates(GtkToggleButton *t,GtkWidget *w){
  gtk_signal_connect(GTK_OBJECT(t),"toggled",
		     GTK_SIGNAL_FUNC(boolean_state_changed_activate),
		     w);
  boolean_state_changed_activate(GTK_WIDGET(t),w);
}

static void color_picker_changed(GnomeColorPicker *colorpicker,
				 guint arg1,
				 guint arg2,
				 guint arg3,
				 guint arg4,
				 gpointer user_data){
  GdkColor *c=(GdkColor *)user_data;
  gushort *alpha,aaa;
  alpha=(guint16 *)gtk_object_get_data(GTK_OBJECT(colorpicker),"alpha_ptr");
  if(alpha==NULL)
    alpha=&aaa;
  gnome_color_picker_get_i16(colorpicker,&c->red,&c->green,&c->blue,alpha);
  prefs_changed();
}

static GtkWidget *create_color_box(char *title,GdkColor *color,gushort *alpha){
  GtkWidget *c=gnome_color_picker_new();
  GtkWidget *f;

  gnome_color_picker_set_use_alpha(GNOME_COLOR_PICKER(c),alpha!=NULL?TRUE:FALSE);
  gnome_color_picker_set_dither(GNOME_COLOR_PICKER(c),TRUE);
  gnome_color_picker_set_i16(GNOME_COLOR_PICKER(c),
			     color->red,color->green,color->blue,(alpha==NULL)?0:*alpha);
  gtk_object_set_data(GTK_OBJECT(c),"alpha_ptr",alpha);
  gtk_signal_connect(GTK_OBJECT(c),"color-set",
		     GTK_SIGNAL_FUNC(color_picker_changed),
		     color);
  f=gtk_hbox_new(FALSE,0);
  gtk_box_pack_end(GTK_BOX(f),c,FALSE,FALSE,0);
  gtk_box_pack_end(GTK_BOX(f),gtk_label_new(title),FALSE,FALSE,0);

  return f;
}

static GtkWidget *create_string_entry(char *label,char **text,
				      GtkWidget **entry){
  GtkWidget *hbox=gtk_hbox_new(FALSE,0);

  *entry=gtk_entry_new();
  if(*text!=NULL)
    gtk_entry_set_text(GTK_ENTRY(*entry),*text);
  gtk_box_pack_start(GTK_BOX(hbox),gtk_label_new(label),FALSE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(hbox),*entry,TRUE,TRUE,0);
  gtk_signal_connect(GTK_OBJECT(*entry),"changed",
		     GTK_SIGNAL_FUNC(editable_changed),
		     (gpointer)text);
  return hbox;
}

static GtkWidget *create_font_picker(PrefsFont *pf,char *title,char *use){
  GtkWidget *t=gnome_font_picker_new(),*hbox=gtk_vbox_new(FALSE,1);
  GtkWidget *frame=gtk_frame_new(title);

  pf->picker=t;
  gtk_container_add(GTK_CONTAINER(frame),hbox);
  gtk_signal_connect(GTK_OBJECT(pf->picker),"font-set",
		     GTK_SIGNAL_FUNC(font_changed),
		     (gpointer)pf);
  pf->label=gtk_label_new("aa");
  gnome_font_picker_set_title(GNOME_FONT_PICKER(pf->picker),title); 
  if(pf->font_name!=NULL)
    gnome_font_picker_set_font_name(GNOME_FONT_PICKER(pf->picker),
				    pf->font_name);
  gtk_label_set_text(GTK_LABEL(pf->label),
		     gnome_font_picker_get_font_name(GNOME_FONT_PICKER(t))); 
  widget_set_font(pf->label,pf);
  gnome_font_picker_set_mode(GNOME_FONT_PICKER(pf->picker),
			     GNOME_FONT_PICKER_MODE_USER_WIDGET);
  gnome_font_picker_uw_set_widget(GNOME_FONT_PICKER(pf->picker),pf->label);
  t=create_boolean_checkbox(use,&pf->use_custom_font);
  gtk_box_pack_start(GTK_BOX(hbox),t,TRUE,TRUE,1);
  gtk_box_pack_start(GTK_BOX(hbox),pf->picker,TRUE,TRUE,1);
  boolean_checkbox_activates(GTK_TOGGLE_BUTTON(t),pf->picker);
  gtk_widget_show(pf->picker);
  return frame;
}

void guiprefs_widget_set_font(GtkWidget *w,PrefsFont *pf){
  if(pf->use_custom_font==FALSE){
    gtk_widget_set_style(w,gtk_widget_get_style(main_app));
    return;
  }
  widget_set_font(w,pf);
}

static GtkWidget *create_adjustment(GtkAdjustment *a,int *val){
  GtkWidget *s;
  gtk_signal_connect(GTK_OBJECT(a),"value-changed",
		     GTK_SIGNAL_FUNC(integer_adjustment_changed),
		     val);
  s=gtk_hscale_new(a);
  gtk_scale_set_value_pos(GTK_SCALE(s),GTK_POS_LEFT);
  return s;
}

typedef struct {
  char *name;
  int val;
} OMSettings;

static void optionmenu_select(GtkMenuItem *m,gpointer data){
  int *var=gtk_object_get_data(GTK_OBJECT(m),"VarPtr");
  OMSettings *s=(OMSettings *)data;
  
  if(*var==s->val)
    return;
  *var=s->val;
  prefs_changed();
}

static GtkWidget *create_optionmenu(char *name,int *var,OMSettings *s){
  GtkWidget *menu,*w,*mi;
  int i,sel=0;
  
  w=gtk_option_menu_new();
  menu=gtk_menu_new();
  for(i=0;s->name!=NULL;i++,s++){
    if(s->val==*var)
      sel=i;
    mi=gtk_menu_item_new_with_label(_(s->name));
    gtk_object_set_data(GTK_OBJECT(mi),"VarPtr",var);
    gtk_signal_connect(GTK_OBJECT(mi),"activate",
		       GTK_SIGNAL_FUNC(optionmenu_select),
		       (gpointer)s);
    gtk_menu_append(GTK_MENU(menu),mi);
  }
  gtk_option_menu_set_menu(GTK_OPTION_MENU(w),menu);
  gtk_option_menu_set_history(GTK_OPTION_MENU(w),sel);
  mi=gtk_hbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(mi),gtk_label_new(name),FALSE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(mi),w,TRUE,TRUE,0);
  return mi;
}

/* =========================== */

/* =========================== */

static void createUserPage(GtkWidget *vbox){
  GtkWidget *t,*t1;
  GtkAdjustment *a;

  t=create_string_entry(_("Nickname: "),&nickname,&nickname_entry_widget);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Append maximum speed to nickname"),&nickname_append_max_speed);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  a=(GtkAdjustment *)gtk_adjustment_new((float)nickname_append_speed_seconds,
					10.0,1000.0,10.0,1.0,60.0);
  t1=create_adjustment(a,&nickname_append_speed_seconds);
  boolean_checkbox_activates(GTK_TOGGLE_BUTTON(t),t1);
  gtk_box_pack_start(GTK_BOX(vbox),t1,FALSE,FALSE,0);
}

static void createMiscPage(GtkWidget *vbox){
  GtkWidget *f1,*t,*hbox,*tbl;
  GtkAdjustment *a;
  static OMSettings NBTS[]={
    {N_("Top"),GTK_POS_TOP},{N_("Left"),GTK_POS_LEFT},
    {N_("Bottom"),GTK_POS_BOTTOM},{N_("Right"),GTK_POS_RIGHT},
    {NULL,0}};
  static OMSettings CVID[]={
    {N_("Pre-1.5"),0},{N_("1.5.1"),151},
    {N_("1.7.1"),171},{N_("1.8"),180},
    {N_("1.8.3"),183},{NULL,0}};
    
  t=create_boolean_checkbox(_("Interactive connections in Bookmarks"),
			    &prefs_dblclick_connect_query_bookmarks);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Interactive connections in Trackers"),
			    &prefs_dblclick_connect_query_trackers);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Hide Passwords"),&passwords_are_hidden);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_optionmenu(_("Client Version:"),&client_version,CVID);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Ignore port number for duplicate servers in trackers"),&tracker_sort_only_by_ip);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);


  f1=gtk_frame_new(_("Notebook"));
  tbl=gtk_table_new(2,2,TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),f1,FALSE,FALSE,0);
  gtk_container_add(GTK_CONTAINER(f1),tbl);
  t=create_optionmenu(_("Notebook tab side: "),&notebook_tab_side,NBTS);
  gtk_table_attach_defaults(GTK_TABLE(tbl),t,0,2,0,1);
  t=create_boolean_checkbox(_("Notebook tabs have icons"),&notebook_has_icons);
  gtk_table_attach_defaults(GTK_TABLE(tbl),t,0,1,1,2);
  t=create_color_box(_("Tab activity color"),
		     &notebook_tab_highlight_color,NULL);
  gtk_table_attach_defaults(GTK_TABLE(tbl),t,1,2,1,2);
  
  f1=gtk_frame_new(_("Speeds"));
  hbox=gtk_hbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(vbox),f1,FALSE,FALSE,0);
  gtk_container_add(GTK_CONTAINER(f1),hbox);
  vbox=gtk_vbox_new(TRUE,0);
  gtk_box_pack_start(GTK_BOX(hbox),vbox,FALSE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(vbox),gtk_label_new(_("Task refresh speed:")),
		     TRUE,TRUE,0); 
  vbox=gtk_vbox_new(TRUE,0);
  gtk_box_pack_start(GTK_BOX(hbox),vbox,TRUE,TRUE,0);

  a=(GtkAdjustment *)gtk_adjustment_new((float)prefs_task_refresh_msec,
					10.0,1500.0,10.0,100.0,100.0);
  t=create_adjustment(a,&prefs_task_refresh_msec);
  gtk_box_pack_start(GTK_BOX(vbox),t,TRUE,TRUE,0);
}

static void createPrivMsgPage(GtkWidget *vbox){
  GtkWidget *frame,*t;
  frame=gtk_frame_new(_("Messages"));
  gtk_box_pack_start(GTK_BOX(vbox),frame,FALSE,FALSE,0);
  vbox=gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(frame),vbox);
  t=create_boolean_checkbox(_("Message window has reply textbox"),
			    &prefs_messages_show_with_reply);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("When replying, quote message"),
			    &prefs_messages_reply_quoted);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
}

static void createAgreementPage(GtkWidget *vbox){
  GtkWidget *t;

  t=create_boolean_checkbox(_("Auto-Agree Agreements"),&dont_show_agreements);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_font_picker(&agreement_font,
		      _("Custom Agreement Font"),_("Use custom font for the Agreement"));
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
}


static void createNewsPage(GtkWidget *vbox){
  GtkWidget *t;

  t=create_boolean_checkbox(_("Auto-Open News on connecting"),&auto_open_news);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Increase the size of the message window"),
			    &news_big_messagewindow);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_font_picker(&news_font,_("Custom News Font"),
		       _("Use custom font for News"));
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
}

static void createUserlistPage(GtkWidget *vbox){
  GtkWidget *frame,*t,*table,*box,*hbox;

  t=create_boolean_checkbox(_("Auto-Open userlist on connecting"),
			    &auto_open_userlist);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Userlist is on the right"),
			    &userlist_dock_right);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Use antialiased userlist (slower)"),
			    &userlist_aa_canvas);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Don't show connect/disconnect messages"),
			    &userlist_no_connect);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  
  hbox=gtk_hbox_new(TRUE,1);
  gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,FALSE,1);
  frame=gtk_frame_new(_("Sorting"));
  gtk_box_pack_start(GTK_BOX(hbox),frame,TRUE,TRUE,1);
  box=gtk_vbox_new(TRUE,0);
  gtk_container_add(GTK_CONTAINER(frame),box);
  t=create_boolean_checkbox(_("Sort userlist by nick"),&userlist_sort_name);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Group users by status"),&userlist_group_by_status);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Display status icons"),&userlist_use_status_icons);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);

  
  frame=gtk_frame_new(_("Colors"));
  gtk_box_pack_start(GTK_BOX(hbox),frame,TRUE,TRUE,1);
  table=gtk_table_new(2,2,TRUE);
  gtk_container_add(GTK_CONTAINER(frame),table);
  t=create_color_box(_("Normal"),&userlist_colors[NORMAL_COLOR],NULL);
  gtk_table_attach_defaults(GTK_TABLE(table),t,0,1,0,1);
  t=create_color_box(_("Administrator"),&userlist_colors[ADMIN_COLOR],NULL);
  gtk_table_attach_defaults(GTK_TABLE(table),t,1,2,0,1);
  t=create_color_box(_("Idle"),&userlist_colors[GHOST_COLOR],NULL);
  gtk_table_attach_defaults(GTK_TABLE(table),t,0,1,1,2);
  t=create_color_box(_("Idle Admin"),
		     &userlist_colors[ADMIN_GHOST_COLOR],NULL);
  gtk_table_attach_defaults(GTK_TABLE(table),t,1,2,1,2);
  t=create_color_box(_("Selection"),
		     &userlist_highlight_color,&userlist_highlight_color_alpha);
  gtk_table_attach_defaults(GTK_TABLE(table),t,0,1,2,3);

  t=create_font_picker(&userlist_font,_("Custom Userlist Font"),_("Use custom font"));
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
}

static void createPubChatPage(GtkWidget *vbox){
  GtkWidget *t;
  t=create_boolean_checkbox(_("Auto-Open userlist when receiving chat"),
			    &auto_popup_pubchat);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_font_picker(&chat_font,_("Custom Chat Font"),_("Use custom font in Chat"));
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
}

static void createFilesPage(GtkWidget *vbox){
  GtkWidget *t,*tbl,*frame;
  GtkAdjustment *a;

  t=create_boolean_checkbox(_("Auto-open Files on connecting"),
			    &auto_open_files);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Mix files and folders"),
			    &mix_files_and_folders);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Show small sizes"),
			    &files_small_sizes);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_string_entry(_("Default directory:"),&default_local_dir,
			&default_local_dir_widget);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);

 
  frame=gtk_frame_new(_("Retry options"));
  gtk_box_pack_start(GTK_BOX(vbox),frame,FALSE,FALSE,0);
  tbl=gtk_table_new(2,2,FALSE);
  gtk_container_add(GTK_CONTAINER(frame),tbl);
  gtk_table_attach(GTK_TABLE(tbl),
		   t=gtk_label_new(_("Retry count:")),
		   0,1,0,1,0,0,0,0);
  gtk_misc_set_alignment(GTK_MISC(t),1.0,0.5);
  a=(GtkAdjustment *)gtk_adjustment_new((float)retry_count,
					0.0,100.0,1.0,10.0,10.0);
  t=create_adjustment(a,&retry_count);
  gtk_table_attach_defaults(GTK_TABLE(tbl),t,
			    1,2,0,1);
  gtk_table_attach(GTK_TABLE(tbl),
		   t=gtk_label_new(_("Retry delay:")),
		   0,1,1,2,0,0,0,0);
  gtk_misc_set_alignment(GTK_MISC(t),1.0,0.5);
  a=(GtkAdjustment *)gtk_adjustment_new((float)retry_delay,
					0.0,100.0,1.0,10.0,10.0);
  t=create_adjustment(a,&retry_delay);
  gtk_table_attach_defaults(GTK_TABLE(tbl),t,1,2,1,2);
}

static void createDownloadsPage(GtkWidget *box){
  GtkWidget *t, *hbox;
  GtkAdjustment *a;
  t=create_boolean_checkbox(_("Convert Mac TEXT -> Unix text files"),
			    &download_convert_text);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Queue Downloads"),
			    &download_queue);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Hide local dotfiles"),
			    &hide_local_dotfiles);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Strip leading/trailing spaces of filenames"),
			    &strip_spaces_on_local_files);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Sanitize local filenames"),
			    &sanitize_local_filenames);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("After aborting, remove zero-length files"),
			    &download_remove_null_files);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Enable recursive download of folders"),
			    &enable_recursive_downloads);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box),hbox,FALSE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(hbox),gtk_label_new(_("Max bandwidth per download (kb) ")),
  				FALSE,FALSE,0);
  a=GTK_ADJUSTMENT(gtk_adjustment_new((float)downloads_max_speed,
				      0,256,1.0,10.0,0.0));
  t=create_adjustment(a,&downloads_max_speed);
  gtk_box_pack_start(GTK_BOX(hbox),t,TRUE,TRUE,0);
}

static void createUploadsPage(GtkWidget *box){
  GtkWidget *t, *hbox;
  GtkAdjustment *a;

  t=create_boolean_checkbox(_("Convert Unix text -> Mac TEXT files"),
			    &upload_convert_text);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Queue Uploads"),
			    &upload_queue);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Cache remote listings"),
			    &cache_remote_directories);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,0);
  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(box),hbox,FALSE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(hbox),gtk_label_new(_("Max bandwidth per upload (kb) ")),
  				FALSE,FALSE,0);
  a=GTK_ADJUSTMENT(gtk_adjustment_new((float)uploads_max_speed,
				      0,256,1.0,10.0,0.0));
  t=create_adjustment(a,&uploads_max_speed);
  gtk_box_pack_start(GTK_BOX(hbox),t,TRUE,TRUE,0);
}

static void add_data_file(GtkWidget *button,GnomeFileEntry *f){
  GtkWidget *e=gnome_file_entry_gtk_entry(f);
  char *a=gtk_editable_get_chars(GTK_EDITABLE(e),0,-1);
  if(a==NULL || strlen(a)==0) return;
  gtk_clist_append(GTK_CLIST(icon_files_widget),&a);
  g_free(a);
  gtk_entry_set_text(GTK_ENTRY(e),"");
  prefs_changed();
}
static void remove_data_file(GtkWidget *button,GnomeFileEntry *f){
  GList *l;
  char *m;
  GtkWidget *e=gnome_file_entry_gtk_entry(f);
  int r;

  if((l=GTK_CLIST(icon_files_widget)->selection)==NULL) return;
  r=GPOINTER_TO_INT(l->data);
  m=gtk_clist_get_row_data(GTK_CLIST(icon_files_widget),r);
  if(m!=NULL)
    gtk_entry_set_text(GTK_ENTRY(e),m);
  else {
    GtkCListRow *row=g_list_nth(GTK_CLIST(icon_files_widget)->row_list,r)->data;
    gtk_entry_set_text(GTK_ENTRY(e),row->cell[0].u.text);
  }
  gtk_clist_remove(GTK_CLIST(icon_files_widget),r);
  prefs_changed();
}

static void createIconPage(GtkWidget *vbox){
  GtkWidget *box,*use_icons,*hbox,*entry,*t;
  GList *l;
  int row;

  use_icons=t=create_boolean_checkbox(_("Use Icons"),&userlist_use_icons);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_boolean_checkbox(_("Use Hotline icons in buttons and menus"),&use_hotline_icons);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  boolean_checkbox_activates(GTK_TOGGLE_BUTTON(use_icons),t);

  box=gtk_hbox_new(FALSE,0);
  boolean_checkbox_activates(GTK_TOGGLE_BUTTON(use_icons),box);
  gtk_box_pack_start(GTK_BOX(vbox),box,FALSE,FALSE,0);
  gtk_box_pack_start(GTK_BOX(box),gtk_label_new(_("Choose icon:")),FALSE,FALSE,2);
  t=gutils_ibutton_new(&user_icon_number,GTK_SIGNAL_FUNC(prefs_changed),NULL);
  gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,2);

  hbox=gtk_hbox_new(0,FALSE);
  gtk_box_pack_start(GTK_BOX(vbox),hbox,TRUE,TRUE,0);
  vbox=gtk_vbox_new(0,FALSE);
  gtk_box_pack_start(GTK_BOX(hbox),vbox,TRUE,TRUE,0);
  entry=gnome_file_entry_new(NULL,_("Datafile"));
  gtk_box_pack_end(GTK_BOX(vbox),entry,FALSE,FALSE,0);
  icon_files_widget=gtk_clist_new(1);
  gtk_signal_connect(GTK_OBJECT(icon_files_widget),"row-move",
		     GTK_SIGNAL_FUNC(prefs_changed),NULL);
  gtk_clist_set_reorderable(GTK_CLIST(icon_files_widget),TRUE);
  t=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(t),
				 GTK_POLICY_NEVER,
				 GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(t),icon_files_widget);
  gtk_box_pack_start(GTK_BOX(vbox),t,TRUE,TRUE,0);
  vbox=gtk_vbox_new(0,FALSE);
  gtk_box_pack_start(GTK_BOX(hbox),vbox,FALSE,FALSE,0);
  gtk_box_pack_end(GTK_BOX(vbox),t=gtk_button_new_with_label(_("Add")),FALSE,FALSE,0);
  gtk_signal_connect(GTK_OBJECT(t),"clicked",GTK_SIGNAL_FUNC(add_data_file),entry);
  gtk_box_pack_end(GTK_BOX(vbox),t=gtk_button_new_with_label(_("Remove")),FALSE,FALSE,0);
  gtk_signal_connect(GTK_OBJECT(t),"clicked",GTK_SIGNAL_FUNC(remove_data_file),entry);
  for(l=hotline_data_files;l!=NULL;l=l->next){
    row=gtk_clist_append(GTK_CLIST(icon_files_widget),(char **)&l->data);
    gtk_clist_set_row_data(GTK_CLIST(icon_files_widget),row,l->data);
  }
}

static void createPrivChatPage(GtkWidget *vbox){
  GtkWidget *t;
  t=create_boolean_checkbox(_("Auto-Join private chats when invited"),
			    &privchat_auto_join);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
  t=create_font_picker(&privchat_font,_("Custom Private Chat Font"),_("Use custom font in Private Chat"));
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,0);
}

#ifdef HAVE_ESD
gboolean guiprefs_check_play_sound(int id){
  return (play_sounds && play_sound_types[id-HL_SOUND_CHAT])?TRUE:FALSE;
}

static void createSoundPage(GtkWidget *vbox){
  GtkWidget *frame,*t,*box;
  int i;
  static struct {int id;char *name;} sound_names[]={
    {HL_SOUND_LOGGED_IN,N_("Successful login")},
    {HL_SOUND_LOGIN,    N_("User login")},
    {HL_SOUND_LOGOUT,   N_("User logout")},
    {HL_SOUND_FILEDONE, N_("File transfer complete")},
    {HL_SOUND_CHAT,     N_("Public chat message")},
    {HL_SOUND_SERVMSG,  N_("Broadcast server message")},
    {HL_SOUND_DOORBELL, N_("Private chat requested")},
    {HL_SOUND_ERROR,    N_("Server error")},
    {0,NULL}
  };

  t=create_boolean_checkbox(_("Play Sounds"),&play_sounds);
  gtk_box_pack_start(GTK_BOX(vbox),t,FALSE,FALSE,1);
  frame=gtk_frame_new(_("Play sounds for these events:"));
  boolean_checkbox_activates(GTK_TOGGLE_BUTTON(t),frame);
  gtk_box_pack_start(GTK_BOX(vbox),frame,FALSE,FALSE,1);
  box=gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(frame),box);  
  for(i=0;sound_names[i].name!=NULL;i++){
    t=create_boolean_checkbox(_(sound_names[i].name),
			      &play_sound_types[sound_names[i].id-HL_SOUND_CHAT]);
    gtk_box_pack_start(GTK_BOX(box),t,FALSE,FALSE,1);
  }
}
#endif

static void prefs_select_page(GtkCTree *ctree,gint row,
			      gint column,GdkEventButton *ev,GtkNotebook *nb){
  GtkWidget *sl;
  int i;
  
  sl=gtk_ctree_node_get_row_data(ctree,gtk_ctree_node_nth(ctree,row));
  if(sl==NULL) return;
  i=gtk_notebook_page_num(GTK_NOTEBOOK(nb),sl);
  gtk_notebook_set_page(GTK_NOTEBOOK(nb),i);
}

void open_prefs(){
  static PrefsPages ppages[]={
    {N_("Fidelio"),410,0,createMiscPage},
#ifdef HAVE_ESD
    {N_("Sound"),189,1,createSoundPage},
#endif
    {N_("Icons"),0,1,createIconPage},
    {N_("User Info"),193,0,createUserPage},
    {N_("Agreement"),222,0,createAgreementPage},
    {N_("Files"),235,0,createFilesPage},
    {N_("Uploads"),211,1,createUploadsPage},
    {N_("Downloads"),210,1,createDownloadsPage},
    {N_("News"),413,0,createNewsPage},
    {N_("Users"),200,0,createUserlistPage},
    {N_("Private Messages"),220,1,createPrivMsgPage},
    {N_("Private Chat"),238,1,createPrivChatPage},
    {N_("Public Chat"),238,1,createPubChatPage},
    {NULL,0,0,NULL}
  };
  static int i,cur_level;
  GdkPixmap *icon;
  GdkBitmap *mask;
  GtkCTreeNode *parent=NULL,*last_node=NULL;
  GtkWidget *hbox,*tree,*nb,*sl,*vbox;
  GtkWidget *pw;
  gboolean has_children;
  char *name;

  if(property_window!=NULL) return;
  pw=gnome_dialog_new(_("Preferences"),GNOME_STOCK_BUTTON_CANCEL,GNOME_STOCK_BUTTON_OK,NULL);
  hbox=gtk_hpaned_new();
  tree=gtk_ctree_new(1,0);
  gtk_ctree_set_expander_style(GTK_CTREE(tree),GTK_CTREE_EXPANDER_NONE);
  gtk_ctree_set_line_style(GTK_CTREE(tree),GTK_CTREE_LINES_NONE);
  gtk_ctree_set_spacing(GTK_CTREE(tree),0);
  gtk_ctree_set_indent(GTK_CTREE(tree),16);
  gtk_clist_set_row_height(GTK_CLIST(tree),16);
  sl=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sl),
				 GTK_POLICY_NEVER,
				 GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(sl),tree);
  gtk_paned_pack1(GTK_PANED(hbox),sl,FALSE,FALSE);

  nb=gtk_notebook_new();
  gtk_paned_pack2(GTK_PANED(hbox),nb,TRUE,FALSE);
  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(nb),FALSE);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(pw)->vbox),hbox,TRUE,TRUE,0);
  guiprefs_add_paned(GTK_PANED(hbox),"Main/Prefs/PrefsPanePos");

  cur_level=ppages[0].level;
  for(i=0;ppages[i].name!=NULL;i++){
    sl=gtk_scrolled_window_new(NULL,NULL);
    vbox=gtk_vbox_new(FALSE,0);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sl),vbox);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sl),
				   GTK_POLICY_NEVER,
				   GTK_POLICY_AUTOMATIC);
    has_children=(ppages[i+1].level>ppages[i].level)?TRUE:FALSE;
    if(cur_level>ppages[i].level){
      parent=NULL;
    }
    cur_level=ppages[i].level;
    if(ppages[i].create!=NULL)
      ppages[i].create(vbox);
    gtk_notebook_append_page(GTK_NOTEBOOK(nb),sl,NULL);
    if(ppages[i].icon!=0)
      get_image_of_icon(ppages[i].icon,&icon,&mask);
    else {
      icon=NULL;
      mask=NULL;
    } 
    name=_(ppages[i].name);
    last_node=gtk_ctree_insert_node(GTK_CTREE(tree),parent,NULL,&name,
				    1,icon,mask,icon,mask,
				    has_children?FALSE:TRUE,has_children?TRUE:FALSE);
    gtk_ctree_node_set_row_data(GTK_CTREE(tree),last_node,sl);
    if(has_children)
      parent=last_node;
  }
  gtk_ctree_expand_recursive(GTK_CTREE(tree),NULL);
  gnome_dialog_set_sensitive(GNOME_DIALOG(pw),1,FALSE);
  gnome_dialog_button_connect_object(GNOME_DIALOG(pw),0,GTK_SIGNAL_FUNC(gtk_widget_destroy),
				     GTK_OBJECT(pw));
  gnome_dialog_button_connect(GNOME_DIALOG(pw),1,GTK_SIGNAL_FUNC(apply_prefs),NULL);
  gtk_signal_connect(GTK_OBJECT(tree),"select-row",
		     GTK_SIGNAL_FUNC(prefs_select_page),nb);
  gtk_signal_connect(GTK_OBJECT(pw),"destroy",
		     GTK_SIGNAL_FUNC(close_box),NULL);
  guiprefs_add_window(GTK_WINDOW(pw),"Main/Prefs/PrefsWindowSize");
  gtk_window_set_policy(GTK_WINDOW(pw), FALSE, TRUE, FALSE);
  gtk_widget_show_all(pw);
  property_window=pw;
  return;
}


/* ============================================== */

typedef struct {
  GtkWidget *iconbutton;
  GtkWidget *iconwidget;
  GtkWidget *dialog;
  GtkWidget *smallicons;
  GtkWidget *largeicons;
  GtkWidget *loadbar;

  int *iconvar;
  int iconsel;
  int origicon;
  GtkSignalFunc notify;
  gpointer data;
} IconButton;

static void update_icon(IconButton *ib,gpointer iconp){
  Icon *ic;
  int icon=GPOINTER_TO_INT(iconp);
  if(icon==ib->iconsel) return;
  ic=get_icon(icon);
  ib->iconsel=icon;
  if(ic!=NULL)
    gtk_pixmap_set(GTK_PIXMAP(ib->iconwidget),ic->pixmap,ic->mask);
  if(ib->notify!=NULL)
    ib->notify(ib->iconbutton,ib->data);
}
static void set_small_icon(GnomeIconList *iconlist,
			   gint arg1,
			   GdkEvent *event,
			   IconButton *ib){
  if(event==NULL) return;
  gtk_clist_unselect_all(GTK_CLIST(ib->largeicons));
  update_icon(ib,gnome_icon_list_get_icon_data(GNOME_ICON_LIST(ib->smallicons),arg1));
}

static void set_big_icon(GtkCList *clist,
			 gint row,
			 gint column,
			 GdkEventButton *event,
			 IconButton *ib){
  if(event==NULL) return;
  gnome_icon_list_unselect_all(GNOME_ICON_LIST(ib->smallicons),NULL,NULL);
  update_icon(ib,gtk_clist_get_row_data(GTK_CLIST(ib->largeicons),row));
}

static void add_icon(Icon *icon,gpointer data){
  IconButton *ib=data;
  int i;
  icon=get_icon(icon->number);
  if(icon==NULL) return;
  i=gnome_icon_list_append_imlib(GNOME_ICON_LIST(ib->smallicons),icon->image,
				 icon->name);
  gnome_icon_list_set_icon_data(GNOME_ICON_LIST(ib->smallicons),i,
				GINT_TO_POINTER(icon->number));
}

static void add_big_icon(Icon *icon,gpointer data){
  IconButton *ib=data;
  int i;
  char *ptr[5];
  char buf[20];
  icon=get_icon(icon->number);
  if(icon==NULL) return;
  ptr[2]="";ptr[1]=icon->name;ptr[0]=buf;
  sprintf(buf,"%5d",icon->number);
  i=gtk_clist_append(GTK_CLIST(ib->largeicons),ptr);
  gtk_clist_set_row_data(GTK_CLIST(ib->largeicons),i,GINT_TO_POINTER(icon->number));
  gtk_clist_set_pixmap(GTK_CLIST(ib->largeicons),i,2,icon->pixmap,icon->mask);
}

static void update_icons(IconButton *ib){
  int i;
  if(ib->loadbar)
    load_all_icons(ib->loadbar);
  gnome_icon_list_freeze(GNOME_ICON_LIST(ib->smallicons));
  gnome_icon_list_clear(GNOME_ICON_LIST(ib->smallicons));
  for_all_icons(TRUE,add_icon,ib);
  gnome_icon_list_thaw(GNOME_ICON_LIST(ib->smallicons));

  gtk_clist_freeze(GTK_CLIST(ib->largeicons));
  for_all_icons(FALSE,add_big_icon,ib);
  gtk_clist_thaw(GTK_CLIST(ib->largeicons));
  i=gnome_icon_list_find_icon_from_data(GNOME_ICON_LIST(ib->smallicons),
					GINT_TO_POINTER(ib->iconsel));
  if(i!=-1){
    gnome_icon_list_select_icon(GNOME_ICON_LIST(ib->smallicons),i);
    gnome_icon_list_moveto(GNOME_ICON_LIST(ib->smallicons),i,0.5);
  }
  i=gtk_clist_find_row_from_data(GTK_CLIST(ib->largeicons),
				 GINT_TO_POINTER(ib->iconsel));
  if(i!=-1){
    gtk_clist_select_row(GTK_CLIST(ib->largeicons),i,0);
    gtk_clist_moveto(GTK_CLIST(ib->largeicons),i,0,0.5,0.0);
  }
  if(ib->loadbar){
    gtk_widget_destroy(ib->loadbar);
    ib->loadbar=NULL;
  }
}

static void gutils_ibutton_dialog_destroyed(GtkWidget *w,IconButton *ib){
  ib->dialog=NULL;
}

static void gutils_ibutton_dialog_cancel(GtkWidget *w,IconButton *ib){
  *ib->iconvar=ib->origicon;
  ib->iconsel=-1;
  update_icon(ib,GINT_TO_POINTER(ib->origicon));
  gtk_widget_destroy(ib->dialog);
}
static void gutils_ibutton_dialog_ok(GtkWidget *w,IconButton *ib){
  *ib->iconvar=ib->iconsel;
  ib->origicon=ib->iconsel;
  gtk_widget_destroy(ib->dialog);
}

static void gutils_ibutton_clicked(GtkWidget *w,IconButton *ib){
  GtkWidget *nb,*box;
  char *titles[]={_("Icon Number"),_("Icon Name"),_("Icon"),"",NULL};

  if(ib->dialog!=NULL) return;
  ib->dialog=gnome_dialog_new("Pick an icon",GNOME_STOCK_BUTTON_CANCEL,
			      GNOME_STOCK_BUTTON_OK,NULL);
  gnome_dialog_button_connect(GNOME_DIALOG(ib->dialog),0,
			      GTK_SIGNAL_FUNC(gutils_ibutton_dialog_cancel),ib);
  gnome_dialog_button_connect(GNOME_DIALOG(ib->dialog),1,
			      GTK_SIGNAL_FUNC(gutils_ibutton_dialog_ok),ib);
  nb=gtk_notebook_new();
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(ib->dialog)->vbox),nb,TRUE,TRUE,0);
  ib->loadbar=gtk_progress_bar_new();
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(ib->dialog)->vbox),ib->loadbar,FALSE,FALSE,0);
  box=gtk_scrolled_window_new(NULL,NULL);
 
  ib->smallicons=gnome_icon_list_new_flags(100,NULL,GNOME_ICON_LIST_STATIC_TEXT);
  gtk_signal_connect(GTK_OBJECT(ib->smallicons),"select-icon",
		     GTK_SIGNAL_FUNC(set_small_icon),ib);  
  gtk_container_add(GTK_CONTAINER(box),ib->smallicons);
  gtk_notebook_append_page(GTK_NOTEBOOK(nb),box,gtk_label_new(_("Small icons")));

  box=gtk_scrolled_window_new(NULL,NULL);
  ib->largeicons=gtk_clist_new_with_titles(3,titles);
  gtk_container_add(GTK_CONTAINER(box),ib->largeicons);
  gtk_clist_set_column_visibility(GTK_CLIST(ib->largeicons),0,FALSE);
  gtk_signal_connect(GTK_OBJECT(ib->largeicons),"select-row",
		     GTK_SIGNAL_FUNC(set_big_icon),ib);
  gtk_clist_set_row_height(GTK_CLIST(ib->largeicons),18);
  gtk_notebook_append_page(GTK_NOTEBOOK(nb),box,gtk_label_new(_("Large icons")));
  gtk_signal_connect(GTK_OBJECT(ib->dialog),"destroy",
		     GTK_SIGNAL_FUNC(gutils_ibutton_dialog_destroyed),ib);
  gtk_window_set_policy(GTK_WINDOW(ib->dialog), FALSE, TRUE, FALSE);
  guiprefs_add_window(GTK_WINDOW(ib->dialog),"Main/Prefs/IconWindowSize");
  gtk_window_set_modal(GTK_WINDOW(ib->dialog),TRUE);
  gtk_widget_show_all(ib->dialog);
  
  update_icons(ib);
}

static void gutils_ibutton_destroyed(GtkWidget *w,IconButton *ib){
  if(ib->dialog!=NULL)
    gtk_widget_destroy(ib->dialog);
  free(ib);
}

GtkWidget *gutils_ibutton_new(int *iconvar,GtkSignalFunc modified,gpointer data){
  GtkWidget *b;
  GdkPixmap *i;GdkBitmap *m;
  IconButton *ib=calloc(1,sizeof(IconButton));

  b=gtk_button_new();
  ib->iconbutton=b;
  ib->iconvar=iconvar;
  ib->iconsel=-1;
  ib->origicon=*iconvar;
  ib->notify=NULL;

  pixmap_get("stipple",&i,&m);
  ib->iconwidget=gtk_pixmap_new(i,m);
  gtk_container_add(GTK_CONTAINER(ib->iconbutton),ib->iconwidget);

  gtk_signal_connect(GTK_OBJECT(b),"destroy",
		     GTK_SIGNAL_FUNC(gutils_ibutton_destroyed),ib);
  gtk_signal_connect(GTK_OBJECT(b),"clicked",
		     GTK_SIGNAL_FUNC(gutils_ibutton_clicked),ib);
  update_icon(ib,GINT_TO_POINTER(*iconvar));
  ib->notify=modified;
  ib->data=data;

  return b;
}
