/******************************************************************** 
   Copyright (C) 2000 Bassoukos Tassos <abas@aix.meng.auth.gr>
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later
   version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*********************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gnome.h>

#include "bookmarks.h"
#include "server.h"
#include "protocol.h"
#include "guiprefs.h"
#include "smalltrans.h"
#include "userlist.h"
#include "guiutils.h"
#include "hldat.h"
#include "userwidget.h"
#include "users.h"
#include "privs.h"
#include "chat.h"
#include "pixmap.h"

typedef struct {
  Connection *c;
  NotebookPage *notebook_page;
  UserWidget *user_list;
  UserlistOps *usersops;
  GtkWidget *user_text;
  GtkWidget *kick_dialog;
  GtkWidget *messagew;
  ChatWindowWidgets cww;
} UserNBPData;

#define USERS "UserPane"

static void users_add_userlist(UserNBPData *u);
static void pubchat_say(ChatWindowWidgets *cww, gpointer data, int emote);

static void users_destroy(GtkButton *b,gpointer data){
  UserNBPData *u=(UserNBPData *)data;
  hooks_set_data(u->c,USERS,NULL);
  if(u->user_list!=NULL)
    userwidget_destroy(u->user_list);
  if(u->notebook_page!=NULL)
    gutils_nbpage_destroy(u->notebook_page);
  if(u->usersops!=NULL)
    userlist_rm_ops(u->c,u->usersops);
  if(u->kick_dialog!=NULL)
    gtk_widget_destroy(u->kick_dialog);
  u->notebook_page=NULL;
  u->user_list=NULL;
  free(u);
}

static void users_destroy_hook(Connection *c,gpointer data){
  users_destroy(NULL,data);
}

void user_start_message(Connection *c,int socket){
  if(c!=NULL && socket!=-1)
    message_create_simple(c,NULL,socket,NULL);
}

static void users_refresh(GtkButton *b,gpointer data){
  UserNBPData *u=(UserNBPData *)data;
  userlist_refresh(u->c);
}

static GnomeUIInfo users_toolbar[] = {
  GNOMEUIINFO_ITEM_STOCK("Close",N_("Close userlist"),
			 users_destroy,
			 HL_STOCK_PIXMAP_CLOSE_PAGE),
  GNOMEUIINFO_ITEM_STOCK("Refresh",N_("Reload the userlist"),
			 users_refresh,
			 HL_STOCK_PIXMAP_REFRESH),
  GNOMEUIINFO_END
};

static void users_create_gui(UserNBPData *u){

  if(u->notebook_page!=NULL)
    return;
  u->notebook_page=gutils_nbpage_new("Main",users_toolbar,
				     u->c,_("Users"),"users",u);
  u->cww.user_data=u;
  u->cww.sendmsg=pubchat_say;
  chat_pane_create(u->notebook_page,&u->cww,FALSE);
  u->user_list=u->cww.userwidget;
  u->user_text=u->cww.textarea;
  u->messagew=u->cww.message;

  userwidget_lists_userlist(u->user_list);
  users_add_userlist(u);
}

static UserNBPData *users_init(Connection *c){
  UserNBPData *u=(UserNBPData *)calloc(sizeof(UserNBPData),1);

  u->c=c;
  u->notebook_page=NULL;
  u->user_list=NULL;
  u->kick_dialog=NULL;
  u->user_text=NULL;
  u->messagew=NULL;
  hooks_create(c,USERS,u,NULL,users_destroy_hook);
  return u;
}

static UserNBPData *get_uld(Connection *c){
  UserNBPData *u=(UserNBPData *)hooks_get_data(c,USERS);
  if(u==NULL)
    u=users_init(c);
  return u;
}

void check_users(Connection *c){
  UserNBPData *u=get_uld(c);
  if(u==NULL)
    u=users_init(c);
  if(auto_open_userlist==TRUE)
    users_create_gui(u);
}

void show_users(Connection *c){
  UserNBPData *u=(UserNBPData *)hooks_get_data(c,USERS);
  if(u==NULL){
    check_users(c);
    u=(UserNBPData *)hooks_get_data(c,USERS);
  }
  if(u->notebook_page==NULL)
    users_create_gui(u);
  gutils_nbpage_to_front(u->notebook_page);
}

/* =========================== */

static void users_add(Connection *c,int sock,gpointer data){
  UserNBPData *u=(UserNBPData *)data;
  char buf[2048];
  UserlistEntry *ule=get_user_by_socket(c,sock);
  if(userlist_no_connect==0){
    sprintf(buf,_("<< %s has connected\n"),ule->name);
    gtk_text_insert(GTK_TEXT(u->user_text),NULL,NULL,NULL,buf,-1);
  }
}

static void users_rm(Connection *c,int sock,gpointer data){
  UserNBPData *u=(UserNBPData *)data;
  char buf[2048];
  UserlistEntry *ule=get_user_by_socket(c,sock);
  if(userlist_no_connect==0){
    if(ule!=NULL && ule->name!=NULL)
      sprintf(buf,_(">> %s has disconnected\n"),ule->name);
    else
      sprintf(buf,_(">> Socket %d has disconnected\n"),sock);
    gtk_text_insert(GTK_TEXT(u->user_text),NULL,NULL,NULL,buf,-1);
  }
}

static UserlistOps uops={
  NULL,NULL,NULL,
  NULL,users_add,
  NULL,users_rm
};

static void users_add_userlist(UserNBPData *u){
  UserlistOps *up=malloc(sizeof(UserlistOps));
  *up=uops;
  u->usersops=up;
  userlist_add_ops(u->c,up,u);
}

/* =========================== */

static gboolean users_display_userinfo(gpointer data){
  HLTransaction *r=(HLTransaction *)data;
  HLObject *o=transaction_find_object(r,HLO_MESSAGE);
  
  if(o!=NULL){
    string_net_to_unix(o);
    message_display_text(r->data,o->data.string);
  }
  transaction_destroy(r);
  return FALSE;
}

static void users_user_query_reply(Connection *c,
				   HLTransaction *t,
				   HLTransaction *r,
				   gpointer data){
  if(r->w.error!=0){
    server_reply_default(c,t,r,strdup(_("Could not get user info:")));
    return;
  }
  r->data=c;
  transaction_destroy(t);
  transaction_read_objects(c,r);
  gtk_idle_add(users_display_userinfo,r);
}

static void users_send_user_query(Connection *c, gpointer data){
  HLTransaction *t;
  int socket=GPOINTER_TO_INT(data);

  if(socket==-1) 
    return;
  t=transaction_new(HLCT_GETUSERINFO,NULL,FALSE);
  t->data=data;
  transaction_add_object(t,create_number(HLO_SOCKET,socket));
  server_transaction_reply_set(c,t,users_user_query_reply,data);
  transaction_send(t,c);
}

void user_query_info(Connection *c,int socket){
  if(c==NULL || socket==-1) 
    return;
  server_transaction_start(c,users_send_user_query,GINT_TO_POINTER(socket));
}

/* ================================== */

static void users_send_update(Connection *c,gpointer p){
  HLTransaction *t=(HLTransaction *)p;
  server_set_default_reply(c,t);
  transaction_send(t,c);
}

static void users_update_common(Connection *c,HLTransaction *t,Task *s){
  transaction_add_object(t,create_number(HLO_ICON,user_icon_number));
  transaction_add_object(t,create_string(HLO_NICK,nickname));
  if(c->version>=170){
    task_update(s,TRUE,0.0);
    task_add(s);
    transaction_set_task(t,s,0.0,1.0);
  }
  server_transaction_start(c,users_send_update,t);
}

void users_update_self(Connection *c){
  HLTransaction *t=transaction_new(HLCT_CHANGENICKICON,c,(c->version>=170?TRUE:FALSE));
  Task *s=NULL;

  if(c->version>=170){
    s=task_new_sname(c);
    sprintf(s->text,_("Updating user info"));
  }
  users_update_common(c,t,s);
}

void users_register_self(Connection *c){
  HLTransaction *t=transaction_new(HLCT_AGREEAGREEMENT,c,TRUE);
  Task *s;

  s=task_new_sname(c);
  sprintf(s->text,_("Loging in ..."));
  users_update_common(c,t,s);
}

/* ================================== */

typedef struct {
  Connection *c;
  int socket;
  GtkWidget *kick_dialog;
} UserKick;

static void users_send_kickban(Connection *c,gpointer p){
  HLTransaction *t=(HLTransaction *)p;
  server_transaction_reply_set(c,t,server_reply_default,
			       strdup(_("Could not ban:")));
  free(t->data);
  transaction_send(t,c);
}

static void users_start_kickban(UserKick *uk,gboolean is_ban){
  HLTransaction *t;

  if(uk->socket==-1)
    return;
  if(uk->kick_dialog!=NULL)
    gtk_widget_destroy(uk->kick_dialog);
  uk->kick_dialog=NULL;
  t=transaction_new(HLCT_KICK,uk,FALSE);
  transaction_add_object(t,create_number(HLO_SOCKET,uk->socket));
  if(is_ban==TRUE)
    transaction_add_object(t,create_number(HLO_BAN,1));
  server_transaction_start(uk->c,users_send_kickban,t);
}

static void users_do_kick(GtkWidget *widget,gpointer p){
  users_start_kickban(p,FALSE);
}
static void users_do_ban(GtkWidget *widget,gpointer p){
  users_start_kickban(p,TRUE);
}

static void users_abort_kick(GtkWidget *widget,gpointer p){
  UserKick *uk=p;
  GtkWidget *w=uk->kick_dialog;
  if(w==NULL) {
    free(uk);
    return;
  }
  uk->kick_dialog=NULL;
  gtk_widget_destroy(w);
}
static gint users_close_kick(GtkWidget *widget,GdkEvent  *event,gpointer p){
  users_abort_kick(widget,p);
  return FALSE;
}

void user_start_kick(Connection *c, int socket){
  UserlistEntry *ule=NULL;
  UserKick *uk;
  char buf[2048];

  if(socket==-1 || c==NULL) 
    return;
  ule=get_user_by_socket(c,socket);
  if(ule==NULL)
    return;
  uk=malloc(sizeof(UserKick));
  uk->c=c;
  uk->socket=socket;
  uk->kick_dialog=gnome_dialog_new(_("Kick User?"),_("Ban"),_("Kick"),
				  GNOME_STOCK_BUTTON_CANCEL,NULL);
  gnome_dialog_set_parent(GNOME_DIALOG(uk->kick_dialog),
			  GTK_WINDOW(c->gui->main_window));
  gnome_dialog_button_connect(GNOME_DIALOG(uk->kick_dialog),2,users_abort_kick,uk);
  gnome_dialog_button_connect(GNOME_DIALOG(uk->kick_dialog),1,users_do_kick,uk);
  gnome_dialog_button_connect(GNOME_DIALOG(uk->kick_dialog),0,users_do_ban,uk);
  gtk_signal_connect(GTK_OBJECT(uk->kick_dialog),"delete_event",
		     GTK_SIGNAL_FUNC(users_close_kick),uk);
  sprintf(buf,_("Kick user %s ?"),ule->name);
  gtk_box_pack_end(GTK_BOX(GNOME_DIALOG(uk->kick_dialog)->vbox),
		   gtk_label_new(buf),FALSE,FALSE,0);
  gtk_widget_show_all(uk->kick_dialog);
}

/* ======================================= */

static void pubchat_say(ChatWindowWidgets *cww, gpointer data, int emote){
  UserNBPData *u=data;
  chat_pane_send_chat(u->c,cww,NULL,emote);
}

void pubchat_handle_received_message(HLTransaction *t){
  Connection *c=(Connection *)t->data;
  UserNBPData *u=(UserNBPData *)hooks_get_data(c,USERS);

  if((u==NULL || u->user_text==NULL) && auto_popup_pubchat==TRUE){
    show_users(c);
    u=(UserNBPData *)hooks_get_data(c,USERS);
  }
  if(u!=NULL && u->user_text!=NULL){
    play_sound(HL_SOUND_CHAT);
    chat_pane_handle_chat(&u->cww,t);
  } else {
    transaction_destroy(t);
  }
}
