//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Difference.hpp,v 1.4 2004/02/08 18:44:19 delpinux Exp $


// This class provides a toolkit to manipulate Differences. It inherits from
// the Shape class.

#ifndef DIFFERENCE_HPP
#define DIFFERENCE_HPP

#include <Object.hpp>
#include <list>

/*!  \class Difference

  This is the class which defines a Virtual Reality set operation: the
  \a difference.

  \author Stphane Del Pino.
*/
class Difference
  : public Shape
{
private:
  typedef std::list<ReferenceCounting<Object> > ObjectList;

public:
  typedef std::list<ReferenceCounting<Object> >::iterator iterator;
  typedef std::list<ReferenceCounting<Object> >::const_iterator const_iterator;

private:
  //! The list of shapes contained in the difference.
  ObjectList __objects;

  //! Returns \p true if the point \a p is inside the Difference.
  inline bool inShape (const TinyVector<3, real_t>& p) const;

  //! Prints the Difference informations using the stream \a s.
  std::ostream& put(std::ostream& s) const;

public:

  const const_iterator begin() const
  {
    return __objects.begin();
  }

  iterator begin()
  {
    return __objects.begin();
  }


  const const_iterator end() const
  {
    return __objects.end();
  }

  iterator end()
  {
    return __objects.end();
  }

  //! Adds the shape \a O inside the Difference.
  inline void push_back(const ReferenceCounting<Object>& o);

  //! Default constructor.
  Difference()
    : Shape(Shape::difference)
  {
    ;
  }

  //! Copy constructor.
  Difference(const Difference& D)
    : Shape(D),
      __objects(D.__objects)
  {
    ;
  }

  //! Destructor.
  ~Difference()
  {
    ;
  }

};

//! Adds the Object \a O inside the Difference.
void Difference::push_back(const ReferenceCounting<Object>& o)
{
  __objects.push_back(o);
}

//! Returns \p true if the point \a p is inside the Difference.
inline bool Difference::inShape (const TinyVector<3, real_t>& p) const
{
  assert(__objects.begin() != __objects.end());
  const_iterator i = __objects.begin();

  if (not((*(*i)).inside(p))) {
    return false;
  }
  ++i;

  for (; i != __objects.end(); i++) {
    if((*(*i)).inside(p)) return false;
  }

  return true;
}

#endif // DIFFERENCE_HPP

