/* GnuTLSAdapter.cc
 * GnuTLS adapter for TCP connection
 *
 * Copyright (C) 2004 Goedson Paixao
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "GnuTLSAdapter.hh"

#include <iostream>
#include <string>

using namespace Gabber;
using namespace std;

GnuTLSAdapter::GnuTLSAdapter(): 
     SSLAdapter()
{
     // initialize SSL library
     gnutls_global_init();
     gnutls_init(&_session, GNUTLS_CLIENT);
     gnutls_set_default_priority(_session);
     gnutls_certificate_allocate_credentials(&_cred);
     gnutls_credentials_set(_session, GNUTLS_CRD_CERTIFICATE, _cred);
}


void GnuTLSAdapter::freeResources() 
{
     gnutls_certificate_free_credentials(_cred);
     gnutls_deinit(_session);
     gnutls_global_deinit();
}


void GnuTLSAdapter::disconnect() 
{
     if(_connected) {
          _connected = false;
	  gnutls_bye(_session, GNUTLS_SHUT_RDWR);
     }
}


GIOStatus GnuTLSAdapter::send(const gchar* data, const gsize len, gsize* written) 
{
     g_assert(_connected);

     int bytes_written = gnutls_record_send(_session, data, len);

     if(bytes_written < 0) {
          _lastError = gnutls_strerror(bytes_written);
	  *(written) = 0;
	  return G_IO_STATUS_ERROR;

     }

     g_assert(bytes_written > 0);
     *(written) = bytes_written;
     return G_IO_STATUS_NORMAL;
}



GIOStatus GnuTLSAdapter::read(gchar* buffer, const gsize count, gsize* bytes_read) 
{
     *(bytes_read) = gnutls_record_recv(_session, buffer, count);

     g_assert(_connected);
     
     if(*(bytes_read) < 0) {
	  _lastError = gnutls_strerror(*(bytes_read));
	  return G_IO_STATUS_ERROR;
     }
     if(*(bytes_read) == count && gnutls_record_check_pending(_session) > 0) {
	  return G_IO_STATUS_AGAIN;
     }
     
     return G_IO_STATUS_NORMAL;
}



bool GnuTLSAdapter::registerSocket(const gint socket)
{
  int ret = 0;


     gnutls_transport_set_ptr(_session, (gnutls_transport_ptr) socket);
           
     ret = gnutls_handshake(_session);
     if (ret < 0) {
          _lastError = gnutls_strerror(ret);
	  return false;
     } else {
          _connected = true;
     }
     return true;
}

SSLAdapter *SSLAdapter::createAdapter() {
     return new GnuTLSAdapter();
}

