/***************************************************************************

  gbx.c

  interpreter startup

  (c) 2000-2004 Beno� Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gb_common.h"
#include "gb_alloc.h"
#include "gb_error.h"

#include <dlfcn.h>

#ifdef __GNU_LIBRARY__
#define _GNU_SOURCE
#include <getopt.h>
#endif

#include "gbx_class.h"
#include "gbx_exec.h"
#include "gbx_stack.h"
#include "gbx_trace.h"
#include "gb_file.h"
#include "gbx_library.h"
#include "gbx_project.h"
#include "gbx_local.h"
#include "gbx_watch.h"
#include "gbx_event.h"
#include "gbx_eval.h"
#include "gb_common_buffer.h"

#include "gbx_api.h"

#include "gbx_c_file.h"
#include "gbx_c_application.h"

extern void _exit(int) NORETURN;


PRIVATE void init(char *argv0, char *argvn)
{
  LIBRARY_init();
  FILE_init();
  LOCAL_init();
  EXEC_init();
  CLASS_init();
  CFILE_init();
  WATCH_init();

  PROJECT_init(argv0, argvn);

  STACK_init();

  LIBRARY_load_all(); /* loads gb.eval if needed */

  TRACE_init(); /* needs project and gb.eval */
}


PRIVATE void my_exit(int ret)
{
  LIBRARY_exit();
  LOCAL_exit();
  exit(ret);
}


PRIVATE void main_exit()
{
  OBJECT_exit();
  TRACE_exit();
  WATCH_exit();
  CLASS_exit();
  LIBRARY_exit();
  PROJECT_exit();
  LOCAL_exit();
  EVENT_exit();
  FILE_exit();
  STRING_exit();
  STACK_exit();
}


int main(int argc, char **argv)
{
  CLASS *class = NULL;
  CLASS_DESC_METHOD *startup = NULL;

  int ret = 0;
  int i, n;
  char *file = NULL;
  bool nopreload = FALSE;

  MEMORY_init();
  COMMON_init();

  if (argc >= 2 && strcmp(argv[1], "-x") == 0)
  {
    if (argc == 2)
    {
      fprintf(stderr, "gbx: no archive file.\n");
      my_exit(1);
    }

    EXEC_arch = TRUE;
    file = argv[2];

    LIBRARY_preload(file, argv);

    for (i = 1; i < (argc - 2); i++)
      argv[i] = argv[i + 2];

    argc -= 2;
  }
  else
  {
    for (i = 1; i < argc; i++)
    {
      if (strcmp(argv[i], "-h") == 0)
      {
        printf(
          "\n"
          "GAMBAS Interpreter version " VERSION " " __DATE__ " " __TIME__ "\n"
          COPYRIGHT
          "Usage: gbx [options] [<project file>] -- ...\n"
          "       gbx -x <archive file> ...\n\n"
          "Options:\n"
          "  -g   enter debugging mode\n"
          "  -V   display version\n"
          "  -h   display this help\n"
          "  -p   disable preloading\n"
          "  -x   execute an archive\n"
          "\n"
          );

        my_exit(0);
      }
      else if (strcmp(argv[i], "-V") == 0)
      {
        printf("gbx-" VERSION "\n");
        my_exit(0);
      }
      else if (strcmp(argv[i], "-g") == 0)
      {
        EXEC_debug = TRUE;
      }
      else if (strcmp(argv[i], "-f") == 0)
      {
        EXEC_fifo = TRUE;
      }
      else if (strcmp(argv[i], "-p") == 0)
      {
        nopreload = TRUE;
      }
      else
      {
        if (strcmp(argv[i], "--"))
        {
          file = argv[i];
          i++;
        }
        break;
      }
    }

    if (i < argc)
    {
      if (file && strcmp(argv[i], "--"))
      {
        fprintf(stderr, "gbx: too many project files.\n");
        my_exit(1);
      }

      i++;
    }

    n = i;

    if (!nopreload)
      LIBRARY_preload(file, argv);
    /*printf("argc = %d\n", argc);
    for (i = 0; i < argc; i++)
      printf("argv[%d] = '%s'\n", i, argv[i]);
    printf("\n");*/

    for (i = 1; i <= (argc - n); i++)
      argv[i] = argv[i + n - 1];

    argc -= n - 1;

    /*printf("argc = %d\n", argc);
    for (i = 0; i < argc; i++)
      printf("argv[%d] = '%s'\n", i, argv[i]);
    printf("\n");*/

  }


  TRY
  {
    init(argv[0], file);

    if (EXEC_debug)
    {
      TRACE_welcome();
      /*TRACE_start_function = &class->load->func[(long)startup->exec];*/
      TRACE_main(FALSE);
    }

    HOOK(main)(&argc, argv);

    PROJECT_argc = argc;
    PROJECT_argv = argv;

    /*EVAL_expression("cos(a) * CFloat(b + c) / a", 0);*/

    class = CLASS_get(PROJECT_startup);
    startup = (CLASS_DESC_METHOD *)CLASS_get_symbol_desc_kind(class, "main", CD_STATIC_METHOD, 0);
    if (startup == NULL)
      THROW(E_MAIN);

    CAPP_init(); /* needs startup class */
  }
  CATCH
  {
    ERROR_print();
    main_exit();
    _exit(1);
  }
  END_TRY

  TRY
  {
    EXEC.class = class;
    EXEC.object = NULL;
    EXEC.drop = TRUE;
    EXEC.nparam = 0;

    if (FUNCTION_is_native(startup))
    {
      EXEC.native = TRUE;
      EXEC.use_stack = FALSE;
      EXEC.desc = startup;

      EXEC_native();
    }
    else
    {
      EXEC.native = FALSE;
      EXEC.index = (int)startup->exec;
      //EXEC.func = &class->load->func[(long)startup->exec]

      EXEC_function();
    }

    ret = HOOK_DEFAULT(loop, WATCH_loop)();

    EVENT_check_post();
  }
  CATCH
  {
    if (ERROR_info.code)
    {
      ERROR_print();
      main_exit();
      _exit(1);
    }
    else
      ret = 0;
  }
  END_TRY

  main_exit();

  if (MEMORY_count)
    fprintf(stderr, "WARNING: %ld allocation(s) non freed.\n", MEMORY_count);

  MEMORY_exit();

  fflush(NULL);

  exit(ret);
}

