/***************************************************************************

  class.c

  Class loader

  (c) 2000-2004 Benot Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

#define __GBX_CLASS_C

#include "gb_common.h"
#include "gb_common_buffer.h"
#include "gb_alloc.h"
#include "gb_error.h"
#include "gb_limit.h"

#include <ctype.h>

#include "gb_buffer.h"
#include "gb_file.h"
#include "gbx_type.h"
#include "gbx_exec.h"
#include "gbx_debug.h"
#include "gb_magic.h"
#include "gbx_stream.h"
#include "gbx_trace.h"

#include "gbx_string.h"
#include "gbx_object.h"
#include "gbx_variant.h"
#include "gbx_number.h"

#include "gambas.h"

#include "gbx_class.h"


PRIVATE TABLE ClassTable;  /* Table des symboles contenant les classes */
PUBLIC CLASS *First = NULL;

#ifdef DEBUG
PRIVATE CLASS *Class;
#endif

PUBLIC void CLASS_init(void)
{
  if (sizeof(CLASS) > 128)
    printf("sizeof(CLASS) = %d !\n", sizeof(CLASS));

  TABLE_create_static(&ClassTable, sizeof(CLASS_SYMBOL), TF_IGNORE_CASE);

  CLASS_init_native();
}


PUBLIC TABLE *CLASS_get_table(void)
{
  return &ClassTable;
}


PRIVATE void CLASS_unload(CLASS *class)
{
  CLASS_DESC *desc;

  /*if (!class->ready && !CLASS_is_virtual(class))
    printf("WARNING: class %s was never loaded.\n", class->name);*/

  #if DEBUG_LOAD
    printf("Unloading class %s...\n", class->name);
  #endif

  if (CLASS_is_native(class))
  {
    desc = CLASS_get_symbol_desc_kind(class, "_exit", CD_STATIC_METHOD, 0);
    if (desc)
    {
      EXEC.desc = &desc->method;
      EXEC.native = TRUE;
      EXEC.class = class;
      EXEC.object = NULL;
      EXEC.nparam = 0;
      EXEC.drop = TRUE;
      EXEC.use_stack = FALSE;

      EXEC_native();
    }
  }

  if (class->free_name)
    FREE(&class->name, "CLASS_unload");

  if (!CLASS_is_native(class))
  {
    /*OBJECT_release(class, NULL);*/
    FREE(&class->load, "CLASS_unload");
    FREE(&class->data, "CLASS_unload");
  }
  else
  {
    if (class->signature)
      FREE(&class->signature, "CLASS_unload");
  }

  if (class->free_event)
    FREE(&class->event, "CLASS_unload");

  if (class->stat)
    FREE(&class->stat, "CLASS_unload");

  if (class->table)
    FREE(&class->table, "CLASS_unload");
}


PUBLIC void CLASS_exit(void)
{
  int i, n, nc, nb;
  CLASS_SYMBOL *csym;
  CLASS *class;

  #if TRACE_MEMORY
  printf("\n------------------- CLASS_exit (variables statiques) -------------------\n\n");
  #endif

  /* On compte le nombre de classes  librer */
  /* On libre les instances automatiques */

  nc = 0;

  for (i = 0; i < TABLE_count(&ClassTable); i++)
  {
    csym = (CLASS_SYMBOL *)TABLE_get_symbol(&ClassTable, i);
    class = csym->class;
    if (class->instance)
      OBJECT_UNREF(&class->instance, "CLASS_exit");
    if (!CLASS_is_native(class) && class->state)
    {
      /*printf("Must free: %s\n", class->name);*/
      nc++;
    }
  }

  /* On ne libre que les classes n'ayant aucun objet instanci */

  n = 0;
  while (n < nc)
  {
    nb = n;

    for (i = 0; i < TABLE_count(&ClassTable); i++)
    {
      csym = (CLASS_SYMBOL *)TABLE_get_symbol(&ClassTable, i);
      class = csym->class;

      /*if (!CLASS_is_native(class) && class->ready && !class->exit)
        printf("%s: %d ready = %d\n", class->name, class->count, class->ready);*/

      if (class->count == 0 && !CLASS_is_native(class) && class->state && !class->exit)
      {
        /*printf("Freeing %s\n", class->name);*/
        OBJECT_release(class, NULL);
        class->exit = TRUE;
        n++;
      }
    }

    if (n == nb) /* On n'a rien pu faire */
      break;
  }

  /* On force la libration de ce qui reste, tant pis */

  if (n < nc)
  {
    fprintf(stderr, "WARNING: circular references detected\n");
    for (i = 0; i < TABLE_count(&ClassTable); i++)
    {
      csym = (CLASS_SYMBOL *)TABLE_get_symbol(&ClassTable, i);
      class = csym->class;

      if (!CLASS_is_native(class) && class->state && !class->exit)
      {
        fprintf(stderr, "%s (%ld)\n", class->name, class->count);
        OBJECT_release(class, NULL);
        class->exit = TRUE;
        n++;
      }
    }
  }


  #if TRACE_MEMORY
  printf("\n------------------- CLASS_exit (dchargement) -------------------\n\n");
  #endif

  for (i = 0; i < TABLE_count(&ClassTable); i++)
  {
    csym = (CLASS_SYMBOL *)TABLE_get_symbol(&ClassTable, i);
    CLASS_unload(csym->class);
  }

  #if TRACE_MEMORY
  printf("\n------------------- CLASS_exit (libration) -------------------\n\n");
  #endif

  for (i = 0; i < TABLE_count(&ClassTable); i++)
  {
    csym = (CLASS_SYMBOL *)TABLE_get_symbol(&ClassTable, i);
    FREE(&csym->class, "CLASS_exit");
  }

  TABLE_delete_static(&ClassTable);
}


PUBLIC CLASS *CLASS_look(const char *name)
{
  CLASS_SYMBOL *csym;

  if (TABLE_find_symbol(&ClassTable, name, strlen(name), (SYMBOL **)&csym, NULL))
    return csym->class;
  else
    return NULL;
}


PUBLIC CLASS *CLASS_find(const char *name)
{
  CLASS_SYMBOL *csym;
  long index;

  if (name == NULL)
    name = COMMON_buffer;

#if DEBUG_LOAD
  printf("Looking for class %s...\n", name);
#endif

  if (TABLE_add_symbol(&ClassTable, name, strlen(name), (SYMBOL **)&csym, &index))
    return csym->class;

#if DEBUG_LOAD
  printf("Not found -> creating new one\n");
#endif

  ALLOC_ZERO(&csym->class, Max(128, sizeof(CLASS)), "CLASS_find");
  /*csym->class->id = index;*/
  csym->class->state = CS_NULL;
  csym->class->count = 0;
  csym->class->ref = 1;

  ALLOC(&csym->class->name, strlen(name) + 1, "CLASS_find");
  strcpy((char *)csym->class->name, name);

  csym->sym.name = csym->class->name;

  csym->class->free_name = TRUE;

  if (First == NULL)
    First = csym->class;

  csym->class->class = First;

  #if DEBUG_LOAD
  printf("New class %s !\n", name);
  #endif

  return csym->class;
}


PUBLIC long CLASS_count(void)
{
  return TABLE_count(&ClassTable);
}


PUBLIC CLASS *CLASS_get(const char *name)
{
  CLASS *class = CLASS_find(name);

  if (class->state == CS_NULL)
    CLASS_load(class);

  return class;
}


PUBLIC boolean CLASS_inherits(CLASS *class, CLASS *parent)
{
  for(;;)
  {
    class = class->parent;

    if (class == NULL)
      return FALSE;

    if (class == parent)
      return TRUE;
  }
}


PUBLIC long CLASS_find_symbol(CLASS *class, const char *name)
{
  long index;

  SYMBOL_find(class->table, class->n_desc, sizeof(CLASS_DESC_SYMBOL), TF_IGNORE_CASE, name, strlen(name), NULL, &index);
  
  //if (strcmp(name, "m1") == 0)
  //  printf("find: %s in %s -> %d\n", name, class->name, index);  
  
  return index;
}


PUBLIC long CLASS_find_symbol_with_prefix(CLASS *class, const char *name, const char *prefix)
{
  long index;

  SYMBOL_find(class->table, class->n_desc, sizeof(CLASS_DESC_SYMBOL), TF_IGNORE_CASE, name, strlen(name), prefix, &index);
  return index;
}


PUBLIC CLASS_DESC_SYMBOL *CLASS_get_symbol(CLASS *class, const char *name)
{
  long index;

  index = CLASS_find_symbol(class, name);
  if (index == NO_SYMBOL)
    return NULL;
  else
    return &class->table[index];
}


PUBLIC CLASS_DESC *CLASS_get_symbol_desc(CLASS *class, const char *name)
{
  CLASS_DESC_SYMBOL *cds = CLASS_get_symbol(class, name);

  if (cds == NULL)
    return NULL;
  else
    return cds->desc;
}


PUBLIC CLASS_DESC *CLASS_get_symbol_desc_kind(CLASS *class, const char *name, int kind, int kind2)
{
  CLASS_DESC *desc;

  desc = CLASS_get_symbol_desc(class, name);
  if (desc == NULL)
    return NULL;

  if ((CLASS_DESC_get_type(desc) != kind) && (CLASS_DESC_get_type(desc) != kind2))
    return NULL;

  return (CLASS_DESC *)desc;
}


PUBLIC CLASS_DESC_EVENT *CLASS_get_event_desc(CLASS *class, const char *name)
{
  long index;
  CLASS_DESC *cd;

  index = CLASS_find_symbol_with_prefix(class, name, ":");
  if (index == NO_SYMBOL)
    return NULL;

  cd = class->table[index].desc;

  if (CLASS_DESC_get_type(cd) != CD_EVENT)
    return NULL;

  return &cd->event;
}


PUBLIC const char *CLASS_DESC_get_signature(CLASS_DESC *cd)
{
  #if 0
  switch (CLASS_DESC_get_type(cd))
  {
    case CD_METHOD:
    case CD_STATIC_METHOD:

      return cd->method.help;

    case CD_EVENT:

      return cd->event.help;

    case CD_PROPERTY:
    case CD_STATIC_PROPERTY:

      return cd->property.help;

    default:

      return NULL;
  }
  #endif
  
  return NULL;
}


PUBLIC void CLASS_ref(void *object)
{
  ((OBJECT *)object)->ref++;

  printf("%s: ref(%s %p) -> %ld\n",
    TRACE_get_current_position(),
    OBJECT_class(object)->name, object, ((OBJECT *)object)->ref);
  fflush(stdout);
}


PUBLIC void CLASS_free(void *object)
{
  CLASS *class = OBJECT_class(object);

  /* Excution de la mthode _free */

  ((OBJECT *)object)->ref = 1; /* Empcher EXEC_leave de tenter de relibrer l'object ! */
  EXEC_special_inheritance(SPEC_FREE, class, object, 0, TRUE);
  ((OBJECT *)object)->ref = 0;

  (*class->free)(class, object);
}


PUBLIC void CLASS_unref(void **pobject, boolean can_free)
{
  OBJECT *object = *((OBJECT **)pobject);

  if (object->ref <= 0)
    fprintf(stderr, "*** %p REF = %ld !\n", object, object->ref);

  fprintf(stderr, "%s: unref(%s %p) -> %ld\n",
    TRACE_get_current_position(),
    OBJECT_class(object)->name, object, object->ref - 1);
  fflush(stdout);

  if ((--(object->ref) <= 0) && can_free)
  {
    fprintf(stderr, "FREE %p !\n", object);
    CLASS_free(object);
  }
}


PUBLIC void CLASS_do_nothing()
{
}

PUBLIC int CLASS_return_zero()
{
  return 0;
}


PRIVATE CLASS_DESC_SYMBOL *SortClassDesc;

PRIVATE int sort_desc(ushort *i1, ushort *i2)
{
  /*return strcmp(SortClassDesc[*i1].load.symbol.name, SortClassDesc[*i2].load.symbol.name);*/
  return compare_ignore_case(
    SortClassDesc[*i1].name,
    SortClassDesc[*i1].len,
    SortClassDesc[*i2].name,
    SortClassDesc[*i2].len
    );
}

PUBLIC void CLASS_sort(CLASS *class)
{
  ushort *sym;
  ushort i;

  if (class->n_desc == 0)
    return;

  ALLOC(&sym, sizeof(ushort) * class->n_desc, "CLASS_sort");

  for (i = 0; i < class->n_desc; i++)
    sym[i] = i;

  SortClassDesc = class->table;
  qsort(sym, class->n_desc, sizeof(ushort), (int (*)(const void *, const void *))sort_desc);

  for (i = 0; i < class->n_desc; i++)
    class->table[i].sort = sym[i];

  #if 0
  {
    SYMBOL *sym;    
  
    printf("\n%s\n", class->name);
    
    for (i = 0; i < class->n_desc; i++)
    {
      sym = (SYMBOL *)&class->table[i];
        
      printf("[%d] (%d) %.*s\n", i, (int)sym->sort, (int)sym->len, sym->name);
    }
  }
  #endif
  
  FREE(&sym, "CLASS_sort");
}

#define GET_IF_NULL(_callback) if (class->_callback == NULL) class->_callback = class->parent->_callback

PUBLIC void CLASS_inheritance(CLASS *class, CLASS *parent)
{
  if (class->parent != NULL)
    THROW(E_CLASS, class->name, "Multiple inheritance", "");

  class->parent = parent;

  CLASS_load(class->parent);

  GET_IF_NULL(check);

  if (parent->auto_create)
    class->auto_create = TRUE;
}


PUBLIC CLASS *CLASS_replace(const char *name)
{
  CLASS_SYMBOL *csym;
  char *new_name;
  CLASS *class;
  CLASS *new_class;
  int len;

  class = CLASS_find(name);
  if (class->state)
  {
    len = strlen(name);

    ALLOC(&new_name, len + 2, "CLASS_find_replace");
    sprintf(new_name, ".%s", name);

    new_class = CLASS_replace(new_name);
    FREE(&new_name, "CLASS_find_replace");

    new_name = (char *)new_class->name;

    TABLE_find_symbol(&ClassTable, name, len, (SYMBOL **)&csym, NULL);
    csym->class = new_class;

    TABLE_find_symbol(&ClassTable, new_name, len + 1, (SYMBOL **)&csym, NULL);
    csym->class = class;

    new_class->name = class->name;
    class->name = new_name;

    class = new_class;
  }

  return class;
}


PUBLIC void CLASS_make_description(CLASS *class, CLASS_DESC *desc, long n_desc, long *first)
{
  static const char *nonher[] = { "_new", "_free", "_init", "_exit", NULL };

  char *override = NULL;
  long n_override;
  long index;
  int i, j;
  const char *name;
  const char **pnonher;

  /* override */

  if (class->parent && class->parent->n_desc)
  {
    ALLOC_ZERO(&override, class->parent->n_desc, "CLASS_make_description");

    n_override = 0;

    for(i = 0; i < n_desc; i++)
    {
      if (CLASS_is_native(class))
        name = &(desc[i].gambas.name[1]);
      else
        name = desc[i].gambas.name;

      index = CLASS_find_symbol(class->parent, name);

      if (index != NO_SYMBOL)
      {
        n_override++;
        override[index] = TRUE;
      }
    }

    /* symboles non hritables */

    for (pnonher = nonher; *pnonher; pnonher++)
    {
      index = CLASS_find_symbol(class->parent, *pnonher);
      if ((index != NO_SYMBOL) && (!override[index]))
      {
        n_override++;
        override[index] = TRUE;
      }
    }

    class->n_desc = class->parent->n_desc + n_desc - n_override;
  }
  else
  {
    class->n_desc = n_desc;
  }

  /* Fabrication de la table */

  if (class->n_desc)
    ALLOC(&class->table, sizeof(CLASS_DESC_SYMBOL) * class->n_desc, "CLASS_make_description");

  i = 0;

  if (class->parent && class->parent->n_desc)
  {
    for (j = 0; j < class->parent->n_desc; j++)
    {
      if (!override[j])
      {
        class->table[i] = class->parent->table[j];
        class->table[i].sort = 0;
        i++;
      }
    }

    FREE(&override, "CLASS_make_description");
  }

  *first = i;

  for(j = 0; j < n_desc; j++, i++)
  {
    class->table[i].desc = &desc[j];

    /* On saute le caractre de type de symbole */
    if (CLASS_is_native(class))
      class->table[i].name = &(desc[j].gambas.name[1]);
    else
      class->table[i].name = desc[j].gambas.name;

    class->table[i].sort = 0;
    class->table[i].len = strlen(class->table[i].name);
  }
}


PUBLIC void CLASS_calc_info(CLASS *class, int n_event, int size_dynamic, boolean all, int size_static)
{
  if (class->parent)
  {
    if (all)
      class->off_event = size_dynamic;
    else
      class->off_event = class->parent->off_event + size_dynamic;

    class->n_event = class->parent->n_event + n_event;
  }
  else
  {
    if (all)
      class->off_event = size_dynamic;
    else
      class->off_event = sizeof(OBJECT) + size_dynamic;

    class->n_event = n_event;
  }

  class->size = class->off_event + sizeof(OBJECT_EVENT) + class->n_event * sizeof(ushort);

  class->size_stat = size_static;
  if (size_static)
    ALLOC_ZERO(&class->stat, class->size_stat, "CLASS_calc_info");
  else
    class->stat = NULL;
}


PUBLIC void CLASS_make_event(CLASS *class, int *first)
{
  int i, ev;

  *first = 0;

  if (class->n_event == 0)
    return;

  if (!CLASS_is_native(class))
  {
    if (class->parent == NULL)
    {
      class->event = class->load->event;
      class->free_event = FALSE;
      return;
    }
  }

  ALLOC_ZERO(&class->event, sizeof(CLASS_EVENT) * class->n_event, "CLASS_make_event");

  if (class->parent != NULL)
  {
    ev = class->parent->n_event;

    for (i = 0; i < ev; i++)
      class->event[i] = class->parent->event[i];
  }
  else
    ev = 0;

  class->free_event = TRUE;

  *first = ev;
}


PUBLIC int CLASS_get_inheritance(CLASS *class, CLASS **her)
{
  int nher = 0;

  while ((nher < MAX_INHERITANCE) && (class != NULL))
  {
    *her++ = class;
    nher++;
    class = class->parent;
  }

  return nher;
}


PUBLIC void *CLASS_auto_create(CLASS *class, int nparam)
{
  void *ob = class->instance;
  int i;

  if (ob)
  {
    if (!class->check(ob))
    {
      for (i = 0; i < nparam; i++)
        POP();
        
      return ob;
    }

    OBJECT_UNREF(&class->instance, "CLASS_auto_create");
    class->instance = NULL;
  }

  /*fprintf(stderr, "CLASS_auto_create: create %s\n", class->name);*/

  OBJECT_create(&class->instance, class, NULL, NULL, nparam);
  ob = class->instance;
  OBJECT_REF(ob, "CLASS_auto_create");

  return ob;
}


/*
PUBLIC CLASS *CLASS_copy(CLASS *class_src, char *prefix)
{
  CLASS *class;
  CLASS save;
  int len = strlen(prefix) + 1 + strlen(class_src->name) + 1;
  char *name;

  ALLOC(&name, len, "CLASS_copy");
  sprintf(name, "%s.%s", prefix, class_src->name);

  class = CLASS_find(name);

  if (!class->ready)
  {
    save = *class;
    *class = *class_src;

    class->name = name;
    class->ready = TRUE;
    class->copy = TRUE;
    class->template = class_src;
  }
  else
    FREE(&name, "CLASS_copy");

  return class;
}
*/

