/***************************************************************************

  code_t.h

  P-code assembler

  (c) 2000-2004 Beno� Minisini <gambas@users.sourceforge.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/

/*#define DEBUG*/

#define write_Zxxx(code, val)  write_short(code | ((short)val & 0x0FFF))
#define write_Z8xx(code, val)  write_short(code | ((short)val & 0x07FF))
#define write_ZZxx(code, val)  write_short(code | ((short)val & 0x00FF))

#define LAST_CODE start_code()


PUBLIC short CODE_stack_usage;
PUBLIC short CODE_stack;

#ifdef PROJECT_EXEC
#define cur_func EVAL
#else
PRIVATE FUNCTION *cur_func = NULL;
PRIVATE int last_line = 0;
#endif


PRIVATE void start_code(void)
{
  #ifndef PROJECT_EXEC
  if (JOB->debug && !JOB->nobreak)
    CODE_break();
  #endif
  cur_func->last_code = ARRAY_count(cur_func->code);
}

PRIVATE void write_short(short value)
{
  short *ptr;

  ptr = ARRAY_add(&cur_func->code);
  *ptr = value;
}


static void write_long(long value)
{
  ushort *ptr;

  ptr = ARRAY_add(&cur_func->code);
  *ptr = value & 0xFFFF;
  ptr = ARRAY_add(&cur_func->code);
  *ptr = (unsigned long)value >> 16;
}


PRIVATE void remove_last(void)
{
  ARRAY_remove_last(&cur_func->code);
  cur_func->last_code = ARRAY_count(cur_func->code);
}


PRIVATE void use_stack(int use)
{
  CODE_stack += use;
  CODE_stack_usage = Max(CODE_stack_usage, CODE_stack);
  #ifdef DEBUG
  printf("%04ld: %d\n", ARRAY_count(cur_func->code), CODE_stack);
  #endif
}

PUBLIC long CODE_get_current_pos()
{
  return ARRAY_count(cur_func->code);
}


#ifdef PROJECT_EXEC

PUBLIC void CODE_begin_function()
{
  CODE_stack = 0;
  CODE_stack_usage = 0;
}

PUBLIC void CODE_end_function()
{
  if (CODE_stack)
    ERROR_panic("Bad stack usage computed !");
}

#else

PUBLIC void CODE_begin_function(FUNCTION *func)
{
  cur_func = func;
  CODE_stack = 0;
  CODE_stack_usage = 0;
  if (func->start == NULL)
    last_line = (-1);
  else
    last_line = 0;
}

PUBLIC void CODE_end_function(FUNCTION *func)
{
  if (CODE_stack)
    ERROR_panic("Bad stack usage computed !");
}

#endif


PRIVATE ushort *get_last_code()
{
  if (cur_func->last_code < 0)
    return NULL;

  return &cur_func->code[cur_func->last_code];
}

#ifdef PROJECT_COMP

PUBLIC boolean CODE_popify_last(void)
{
  /*
  #ifdef DEBUG
  printf("CODE_is_last_popable ? ");
  if (!last_code) printf("FALSE, last_code = NULL");
  else printf("0x%04hX", *last_code);
  printf("\n");
  #endif
  */
  unsigned short *last_code, op;

  last_code = get_last_code();
  if (!last_code)
    return FALSE;

  op = *last_code & 0xFF00;

  if ((op >= C_PUSH_LOCAL) && (op <= C_PUSH_UNKNOWN))
  {
    *last_code += 0x0800;
    use_stack(-2);
    #ifdef DEBUG
    printf("Popify Last\n");
    #endif
    return TRUE;
  }

  if ((op & 0xF000) == C_PUSH_DYNAMIC)
  {
    *last_code += 0x1000;
    use_stack(-2);
    #ifdef DEBUG
    printf("Popify Last\n");
    #endif
    return TRUE;
  }

  /*
  if (op == C_CALL)
  {
    *last_code = C_CALL_POP | (*last_code & 0xFF);
    return TRUE;
  }
  */

  return FALSE;
}


PUBLIC boolean CODE_check_statement_last(void)
{
  unsigned short op;
  PCODE *last_code;

  last_code = get_last_code();
  if (!last_code)
    return FALSE;

  op = *last_code & 0xFF00;

  if (op == C_CALL)
    return TRUE;

  if (op == C_PUSH_RETURN)
    return TRUE;

  op >>= 8;

  if (op >= CODE_FIRST_SUBR && op <= CODE_LAST_SUBR)
    return TRUE;

  return FALSE;
}


PUBLIC boolean CODE_check_pop_local_last(short *local)
{
  PCODE *last_code;

  last_code = get_last_code();
  if (!last_code)
    return FALSE;

  if ((*last_code & 0xFF00) == C_POP_LOCAL)
  {
    *local = *last_code & 0xFF;
    return TRUE;
  }

  return FALSE;
}

#endif

PUBLIC void CODE_push_number(long value)
{
  LAST_CODE;

  use_stack(1);

  if (value >= -2048L && value < 2048L)
  {
    #ifdef DEBUG
    printf("PUSH QUICK %ld\n", value);
    #endif
    write_Zxxx(C_PUSH_QUICK, value);
  }
  else if (value >= -32768L && value < 32768L)
  {
    #ifdef DEBUG
    printf("PUSH INTEGER %ld\n", value);
    #endif
    write_short(C_PUSH_INTEGER);
    write_short((short)value);
  }
  else
  {
    #ifdef DEBUG
    printf("PUSH LONG %ld\n", value);
    #endif
    write_short(C_PUSH_LONG);
    write_long(value);
  }

}


PUBLIC void CODE_push_const(short value)
{
  LAST_CODE;

  use_stack(1);

  #ifdef DEBUG
  printf("PUSH CONST %d %s\n", value, TABLE_get_symbol_name(JOB->class->table, JOB->class->constant[value].index));
  #endif
  write_Zxxx(C_PUSH_CONST, value);
}


PUBLIC void CODE_push_local(short num)
{
  LAST_CODE;

  use_stack(1);

  #ifdef DEBUG
  if (num >= 0)
    printf("PUSH LOCAL %d\n", num);
  else
    printf("PUSH PARAM %d\n", (-1) - num);
  #endif
  write_ZZxx(C_PUSH_LOCAL, num);
}


#ifdef PROJECT_COMP

PUBLIC void CODE_pop_local(short num)
{
  LAST_CODE;

  use_stack(-1);

  #ifdef DEBUG
  if (num >= 0)
    printf("POP LOCAL #%d\n", num);
  else
    printf("POP PARAM #%d\n", (-1) - num);
  #endif
  write_ZZxx(C_POP_LOCAL, num);
}


PUBLIC void CODE_pop_ctrl(short num)
{
  LAST_CODE;

  use_stack(-1);

  #ifdef DEBUG
  printf("POP CTRL #%d\n", num);
  #endif

  write_ZZxx(C_POP_CTRL, num);
}


PUBLIC void CODE_pop_optional(short num)
{
  LAST_CODE;

  use_stack(-1);

  #ifdef DEBUG
  printf("POP OPTIONAL #%d\n", (-1) - num);
  #endif
  write_ZZxx(C_POP_OPTIONAL, num);
}

#endif /* PROJECT_COMP */


PUBLIC void CODE_push_array(short nparam)
{
  LAST_CODE;

  use_stack(1 - nparam);

  write_ZZxx(C_PUSH_ARRAY, nparam);
}


PUBLIC void CODE_push_global(short global, boolean is_static, boolean is_function)
{
  LAST_CODE;

  use_stack(1);

  #ifdef DEBUG
  printf("PUSH %s %d\n", is_static ? "STATIC" : "DYNAMIC", global);
  #endif

  if (is_function)
    write_Z8xx(C_PUSH_FUNCTION, global);
  else if (is_static)
    write_Z8xx(C_PUSH_STATIC, global);
  else
    write_Z8xx(C_PUSH_DYNAMIC, global);
}


#ifdef PROJECT_COMP

PUBLIC void CODE_pop_global(short global, boolean is_static)
{
  LAST_CODE;

  use_stack(-1);

  #ifdef DEBUG
  printf("POP %s %d\n", is_static ? "STATIC" : "DYNAMIC", global);
  #endif

  if (is_static)
    write_Z8xx(C_POP_STATIC, global);
  else
    write_Z8xx(C_POP_DYNAMIC, global);
}

#endif

/*
PUBLIC void CODE_push_symbol(short symbol)
{
  LAST_CODE;

  use_stack(0);

  #ifdef DEBUG
  printf("PUSH SYMBOL %s\n", TABLE_get_symbol_name(JOB->class->table, symbol));
  #endif

  write_short(C_PUSH_SYMBOL);
  write_short(symbol);
}


PUBLIC void CODE_pop_symbol(short symbol)
{
  LAST_CODE;

  use_stack(-2);

  #ifdef DEBUG
  printf("POP SYMBOL %s\n", TABLE_get_symbol_name(JOB->class->table, symbol));
  #endif

  write_short(C_POP_SYMBOL);
  write_short(symbol);
}
*/


PUBLIC void CODE_push_unknown(short symbol)
{
  LAST_CODE;

  use_stack(0);

  #ifdef DEBUG
  printf("PUSH UNKNOWN %s\n", TABLE_get_symbol_name(JOB->class->table, symbol));
  #endif

  write_short(C_PUSH_UNKNOWN);
  write_short(symbol);
}


#ifdef PROJECT_COMP

PUBLIC void CODE_pop_unknown(short symbol)
{
  LAST_CODE;

  use_stack(-2);

  #ifdef DEBUG
  printf("POP UNKNOWN %s\n", TABLE_get_symbol_name(JOB->class->table, symbol));
  #endif

  write_short(C_POP_UNKNOWN);
  write_short(symbol);
}

#endif


PUBLIC void CODE_push_class(short class)
{
  LAST_CODE;

  use_stack(1);

  #ifdef DEBUG
  printf("PUSH CLASS %d\n", class);
  #endif

  write_Z8xx(C_PUSH_CLASS, class);
}

#ifdef PROJECT_COMP

PUBLIC void CODE_jump()
{
  LAST_CODE;

  #ifdef DEBUG
  printf("JUMP\n");
  #endif
  write_short(C_JUMP);
  /**pos = CODE_get_current_pos();*/
  write_short(0);
}


PUBLIC void CODE_jump_if_true()
{
  /*
  ushort *last_code = get_last_code();
  ushort op;

  if (last_code && PCODE_is(*last_code, C_NOT))
  {
    remove_last();
    op = C_JUMP_IF_FALSE;
  }
  else
    op = C_JUMP_IF_TRUE;
  */

  use_stack(-1);

  #ifdef DEBUG
  printf("JUMP IF TRUE\n");
  #endif

  LAST_CODE;

  write_short(C_JUMP_IF_TRUE);
  /**pos = CODE_get_current_pos();*/
  write_short(0);
}


PUBLIC void CODE_jump_if_false()
{
  /*
  ushort *last_code = get_last_code();
  ushort op;

  if (last_code && PCODE_is(*last_code, C_NOT))
  {
    remove_last();
    op = C_JUMP_IF_TRUE;
  }
  else
    op = C_JUMP_IF_FALSE;
  */

  use_stack(-1);

  #ifdef DEBUG
  printf("JUMP IF FALSE\n");
  #endif

  LAST_CODE;

  write_short(C_JUMP_IF_FALSE);
  /**pos = CODE_get_current_pos();*/
  write_short(0);
}


PUBLIC void CODE_jump_first(short local)
{
  LAST_CODE;

  use_stack(-2);

  #ifdef DEBUG
  printf("JUMP FIRST LOCAL %d\n", local);
  #endif

  write_ZZxx(C_JUMP_FIRST, local);
}


PUBLIC void CODE_jump_next(void)
{
  LAST_CODE;

  use_stack(1);

  #ifdef DEBUG
  printf("JUMP NEXT\n");
  #endif
  write_short(C_JUMP_NEXT);
  /**pos = CODE_get_current_pos();*/
  write_short(0);
}


PUBLIC void CODE_jump_length(short src, short dst)
{
  if (src < 0 || src >= (ARRAY_count(cur_func->code) - 1))
    return;

  /*
  if (dst < 0 || dst > (ARRAY_length(cur_func->code)))
    return;
  */

  if (cur_func->code[src] == C_BREAK)
    cur_func->code[src + 2] = dst - (src + 2) - 1;
  else
    cur_func->code[src + 1] = dst - (src + 1) - 1;
}


PUBLIC void CODE_first(short local)
{
  LAST_CODE;

  use_stack(-1);

  #ifdef DEBUG
  printf("ENUM FIRST LOCAL %d\n", local);
  #endif

  write_ZZxx(C_FIRST, local);
}


PUBLIC void CODE_next(bool drop)
{
  LAST_CODE;

  use_stack(drop ? 0 : 1);

  #ifdef DEBUG
  printf("ENUM NEXT%s\n", drop ? " DROP" : "");
  #endif

  write_ZZxx(C_NEXT, drop ? 1 : 0);
  write_short(0);
}

#endif /* PROJECT_COMP */

PUBLIC void CODE_op(short op, short nparam, boolean fixed)
{
  LAST_CODE;

  use_stack(1 - nparam);

  #ifdef DEBUG
  printf("OP %d (%d)\n", op, nparam);
  #endif

  if (fixed)
    write_ZZxx(op, 0);
  else
    write_ZZxx(op, nparam);
}


PUBLIC void CODE_push_me(bool debug)
{
  LAST_CODE;

  use_stack(1);

  #ifdef DEBUG
  printf("PUSH ME\n");
  #endif

  write_ZZxx(C_PUSH_ME, debug ? 1 : 0);
}


PUBLIC void CODE_push_last()
{
  LAST_CODE;

  use_stack(1);

  #ifdef DEBUG
  printf("PUSH LAST\n");
  #endif

  write_short(C_PUSH_LAST);
}


PUBLIC void CODE_push_null()
{
  LAST_CODE;

  use_stack(1);

  #ifdef DEBUG
  printf("PUSH NULL\n");
  #endif

  write_short(C_PUSH_NULL);
}


/*
PRIVATE boolean change_last_call(ushort flag)
{
  ushort *last_code = get_last_code();

  if (!last_code)
    return FALSE;

  if ((*last_code & 0xFF00) == C_CALL)
  {
    *last_code = *last_code | flag;
    return TRUE;
  }
  else if ((ushort)((*last_code) & 0xFF00) >= (ushort)CODE_FIRST_SUBR)
  {
    *last_code = *last_code | flag;
    return TRUE;
  }
  else if (((*last_code & 0xFF00) == C_DROP) && flag == CODE_CALL_DROP)

  return FALSE;
}
*/

PUBLIC void CODE_dup(void)
{
  LAST_CODE;

  use_stack(1);

  #ifdef DEBUG
  printf("DUP\n");
  #endif

  write_short(C_DUP);
}


PUBLIC void CODE_return(int return_value)
{
  LAST_CODE;

  if (return_value)
  {
    use_stack(-1);
    write_ZZxx(C_RETURN, return_value);
  }
  else
    write_ZZxx(C_RETURN, 0);

  #ifdef DEBUG
  printf("RETURN (%d)\n", return_value ? 1 : 0);
  #endif
}


#ifdef PROJECT_COMP

static bool _allow_break = FALSE;

PUBLIC void CODE_allow_break(void)
{
	_allow_break = TRUE;
}

PUBLIC void CODE_break(void)
{
	if (!_allow_break)
		return;

  /*if (last_line < 0)
  {
    if (CODE_get_current_pos())
      return;
  }
  else
  {
    if (JOB->line == last_line)
      return;

    last_line = JOB->line;
  }*/

  #ifdef DEBUG
  printf("BREAK\n");
  #endif

  write_short(C_BREAK);
  _allow_break = FALSE;
}


PUBLIC void CODE_quit(void)
{
  LAST_CODE;

  #ifdef DEBUG
  printf("QUIT\n");
  #endif

  write_ZZxx(C_QUIT, 0);
}


PUBLIC void CODE_stop(void)
{
  LAST_CODE;

  #ifdef DEBUG
  printf("STOP\n");
  #endif

  write_ZZxx(C_QUIT, 1);
}

#endif /* PROJECT_COMP */


PUBLIC void CODE_push_char(char car)
{
  LAST_CODE;

  use_stack(1);
  write_ZZxx(C_PUSH_CHAR, car);

  #ifdef DEBUG
  printf("PUSH CHAR %d\n", car);
  #endif
}


/*
PUBLIC void CODE_push_zero(void)
{
  LAST_CODE;

  use_stack(1);
  write_short(C_PUSH_ZERO);

  #ifdef DEBUG
  printf("PUSH ZERO\n");
  #endif
}
*/


PUBLIC void CODE_push_void(void)
{
  LAST_CODE;

  use_stack(1);
  write_short(C_PUSH_VOID);

  #ifdef DEBUG
  printf("PUSH VOID\n");
  #endif
}


#ifdef PROJECT_COMP

PUBLIC void CODE_event(bool on)
{
  LAST_CODE;

  write_ZZxx(C_EVENT, on ? 1 : 0);

  #ifdef DEBUG
  printf("EVENT %s\n", on ? "ON" : "OFF");
  #endif
}

PUBLIC void CODE_stop_event(void)
{
  LAST_CODE;

  write_ZZxx(C_EVENT, 2);

  #ifdef DEBUG
  printf("STOP EVENT\n");
  #endif
}

#endif


PUBLIC void CODE_subr(short subr, short nparam, short optype, boolean output, boolean fixed)
{
  LAST_CODE;

  if (output)
    use_stack(0);
  else
    use_stack(1 - nparam);

  #ifdef DEBUG
  printf("SUBR %s %d (%d)\n", output ? "OUTPUT" : "", subr, nparam);
  #endif

  if (optype == 0)
  {
    if (fixed)
      nparam = 0;

    /*if (output)
      nparam |= CODE_CALL_OUTPUT;*/
  }
  else
  {
    nparam = optype;
  }

  subr += CODE_FIRST_SUBR;
  write_short(((subr & 0xFF) << 8) | (nparam & 0xFF));

}


PUBLIC void CODE_call(short nparam, boolean output)
{
  LAST_CODE;

  /* Une case de pile de moins, car la valeur de retour
     est stock� �la place de la fonction �appeler */

  if (output)
    use_stack(0);
  else
    use_stack(-nparam);

  #ifdef DEBUG
  printf("CALL %s ( %d )\n", output ? "OUTPUT" : "", nparam);
  #endif

  /*if (output)
    nparam |= CODE_CALL_OUTPUT;*/

  write_ZZxx(C_CALL, nparam);

}


PUBLIC void CODE_push_return(void)
{
  LAST_CODE;

  use_stack(1);
  write_short(C_PUSH_RETURN);

  #ifdef DEBUG
  printf("PUSH RETURN\n");
  #endif
}


#ifdef PROJECT_COMP

PUBLIC void CODE_try(void)
{
  LAST_CODE;

  write_short(C_TRY);
  write_short(0);

  #ifdef DEBUG
  printf("TRY\n");
  #endif
}


PUBLIC void CODE_end_try(void)
{
  LAST_CODE;

  write_short(C_END_TRY);

  #ifdef DEBUG
  printf("END TRY\n");
  #endif
}


PUBLIC void CODE_catch(void)
{
  LAST_CODE;

  write_short(C_CATCH);

  #ifdef DEBUG
  printf("CATCH\n");
  #endif
}

#endif


PUBLIC void CODE_drop(void)
{
  ushort *last_code = get_last_code();
  ushort subr;

  use_stack(-1);

  #ifdef DEBUG
  printf("DROP\n");
  #endif

  if (last_code)
  {
    switch(*last_code & 0xFF00)
    {
      case C_DROP:
        *last_code = (*last_code & 0xFF00) + (*last_code & 0xFF) + 1;
        return;

      case C_CALL:
        *last_code |= CODE_CALL_VOID;
        return;

      case C_PUSH_RETURN:
        ERROR_panic("C_PUSH_RETURN ?");
        remove_last();
        return;

      default:
        subr = (*last_code) >> 8;
        if (subr >= CODE_FIRST_SUBR && subr <= CODE_LAST_SUBR && (!(*last_code & CODE_CALL_VOID)))
        {
          *last_code |= CODE_CALL_VOID;
          return;
        }
    }
  }

  LAST_CODE;

  write_ZZxx(C_DROP, 1);
}


PUBLIC void CODE_push_special(short spec)
{
  LAST_CODE;

  use_stack(0);

  #ifdef DEBUG
  printf("PUSH SPECIAL %d\n", spec);
  #endif

  write_ZZxx(C_PUSH_SPECIAL, spec);
}


#ifdef PROJECT_COMP

PUBLIC void CODE_push_event(short event)
{
  LAST_CODE;

  use_stack(1);

  #ifdef DEBUG
  printf("PUSH EVENT %d\n", event);
  #endif

  write_ZZxx(C_PUSH_EVENT, event);
}

PUBLIC void CODE_new(ushort nparam, boolean array, boolean event)
{
  LAST_CODE;

  use_stack(1 - nparam);

  #ifdef DEBUG
  printf("NEW %s (%d)\n", (array ? "ARRAY" : (event ? "EVENT" : "")), nparam);
  #endif

  if (array)
    nparam |= CODE_NEW_ARRAY;

  if (event)
    nparam |= CODE_NEW_EVENT;

  write_ZZxx(C_NEW, nparam);
}

#endif



PUBLIC void CODE_push_boolean(boolean value)
{
  LAST_CODE;

  use_stack(1);
  write_ZZxx(C_PUSH_BOOLEAN, value);

  #ifdef DEBUG
  printf("PUSH %s\n", value ? "TRUE" : "FALSE");
  #endif
}


#ifdef CODE_DUMP

PUBLIC void CODE_dump(PCODE *code)
{
  int i;

  printf("\n");

  for (i = 0; i < ARRAY_count(code);)
    i += PCODE_dump(i, &code[i]);

  printf("\n");

}

#endif

/* PUBLIC void CODE_case(short local) */
/* { */
/*   LAST_CODE; */
/*  */
/*   use_stack(0); */
/*  */
/*   #ifdef DEBUG */
/*   printf("CASE (%d)\n", local); */
/*   #endif */
/*  */
/*   write_ZZxx(C_CASE, local); */
/* } */
