from logfile import logfile
from math import log
import string

class GAMESS(logfile):
    def __init__(self,filename):
        self.filename=filename

    def getGeoOpt(self):
        # Returns a tuple of three items:
        # The first is a string of lines to be written to the screen
        # The second is a list of numbers (to be plotted) (as string)
        # The third is a list of numbers (to be plotted) (as string)

        screenoutput=""; energy=[]; progress=[];

        inputfile=open(self.filename,"r")
        line=inputfile.readline()
        opttol = 1e-4
        endOfOpt = False
        while line:
            if line.find("OPTTOL")>=0:
                screenoutput+=line
                # Two possibilities:
                #           OPTTOL = 1.000E-04          RMIN   = 1.500E-03
                # INPUT CARD> $STATPT OPTTOL=0.0001 NSTEP=100 $END
                temp=line.strip().split()
                for i in range(len(temp)):
                    x=temp[i]
                    if x.find("OPTTOL")>=0:
                        if x=="OPTTOL":
                            opttol=float(temp[i+2])
                        else:
                            x=x.split('=')
                            opttol=float(x[1])
            elif line.find("EQUILIBRIUM GEOMETRY LOCATED")>=0:
# This is necessary if a frequency calculation follows a geometry optimisation
                endOfOpt = True
            elif not endOfOpt and line.find("FINAL")==1:
# Here is an example from Neil Berry:
#  FINAL R-B3LYP ENERGY IS     -382.0507446475 AFTER  10 ITERATIONS
# but in some cases the energy can be in position [3] not [4] so let's
# take the number after the "IS"
                temp = line.split()
                energy.append(temp[temp.index("IS")+1])
                screenoutput+=line
            elif not endOfOpt and line.find("MAXIMUM GRADIENT")>0:
                temp=line.strip().split()
                max=float(temp[3])
                deviation=0
                if max>opttol:
                    deviation+=log(max/opttol)
                rms=float(temp[7])
                if rms>opttol/3.:
                    deviation+=log(rms*3./opttol)
                progress.append(str(deviation))
                screenoutput+=line
                
            line=inputfile.readline()

        return screenoutput,energy,progress
    
    def getSCF(self):
        # Returns a tuple of two items:
        # The first is a string of lines to be written to the screen
        # The second is a list of numbers (to be plotted) (as string)

        inputfile = open(self.filename,"r")

        line = inputfile.readline()
        conv = 1e-5
        while line:
            if line.find("DENSITY CONV=")==5:
                conv = float(line.strip().split()[-1])
            if line.find("ITER EX DEM")==1:
# This is the section with the SCF information                
                screenoutput = line
                density = []
                line = inputfile.readline()
                while line!='\n':
# The SCF information is terminated by a blank line                    
                    screenoutput += line
                    try:
                        temp = int(line[0:4])
                    except ValueError:
# Occurs for:
#  * * *   INITIATING DIIS PROCEDURE   * * *
#  CONVERGED TO SWOFF, SO DFT CALCULATION IS NOW SWITCHED ON
#  DFT CODE IS SWITCHING BACK TO THE FINER GRID
                        pass
                    else:
                        den=float(line.split()[5])
                        progress=0
                        if den>conv:
                            progress=log(den/conv)
                        density.append(str(progress))
                    line=inputfile.readline()
                    
            line=inputfile.readline()

        inputfile.close()                

        return screenoutput,density
        
    def getIR_Raman(self):
        # Returns a tuple as follows:
        # Item 1: screenoutput (STRING)
        # Item 2: freqs (FLOATS
        # Item 3: symmetry (STRING)
        # Item 4: ir (FLOAT)
        # Item 5: raman (may be empty list) (FLOAT)

        screenoutput = ["Output from GAMESS follows..."]
        freqs = []
        ir = []
        symmetry = []
        raman = []
        
        inputfile = open(self.filename,"r")
        for line in inputfile:
            if line.find("NORMAL COORDINATE ANALYSIS IN THE HARMONIC APPROXIMATION")>=0:
                # Start of the frequency section

                # Need to get past the list of atomic weights
                hyphens = inputfile.next()
                blank = inputfile.next()
                line = inputfile.next()
                blank = inputfile.next()
                line = inputfile.next()
                numAtom = 0
                while line!="\n":
                    numAtom += 1
                    line = inputfile.next()

                # Print out the following lines which may contain some useful info:
                # e.g. WARNING, MODE 7 HAS BEEN CHOSEN AS A VIBRATION
                line = inputfile.next()
                while line.find("FREQUENCIES IN CM**-1")==-1:
                    screenoutput.append(line.strip())
                    line = inputfile.next()
                line = inputfile.next()

                blank = inputfile.next()
                freqNo = inputfile.next()
                while freqNo.find("SAYVETZ")==-1:
                    freq = inputfile.next().strip().split()
                    freqs.extend(map(float,freq[1:]))
                    reducedMass = inputfile.next()
                    irIntensity = inputfile.next().strip().split()
                    ir.extend(map(float,irIntensity[2:]))
                    blank = inputfile.next()
                    # Skip XYZ data for each atom plus
                    # the Sayvetz stuff at the end
                    for j in range(numAtom*3+10):
                        line = inputfile.next()
                    blank = inputfile.next()
                    freqNo = inputfile.next()

        inputfile.close()
        screenoutput.append("...End of output from GAMESS\n")
        if symmetry==[]:
            symmetry = ["?"]*len(freqs)
        return "\n".join(screenoutput),freqs,symmetry,ir,raman
                


    def getUV_Vis(self):
        # Returns a tuple of energy,wavelength,osc,sym,CIS
        
        pass

    def getenergylevels(self):
        # Returns an array of evalue, sym and the HOMO
        # Re: the number of the HOMO - the orbitals are numbered starting from zero
        # evalues are in eV

        # The details returned come from the *final* report of evalues and
        # the last list of symmetries in the log file
        # This is fine for GeoOpt and SP, but may be weird for TD and Freq(?)

        inputfile=open(self.filename,"r")
        line=inputfile.readline()
        while len(line)>0:
            if line.find("EIGENVECTORS")==10 or line.find("MOLECULAR OBRITALS")==10:
                # Take the last one of either in the file
                evalue=[]
                symmetry=[]
                line=inputfile.readline()
                blank=inputfile.readline() # blank line
                while line.find("END OF RHF")==-1:
                    line=inputfile.readline() # Eigenvector no
                    line=inputfile.readline()
                    evalue.extend(map(float,line.split()))
                    line=inputfile.readline()
                    symmetry.extend(line.split())
                    line=inputfile.readline()
                    while line!=blank and line.find("END OF RHF")==-1:
                        line=inputfile.readline()
            elif line.find("NUMBER OF OCCUPIED ORBITALS")>=0:
                temp=line.strip().split('=')
                HOMO=int(temp[-1])-1
            line=inputfile.readline()
                
        inputfile.close()

        return evalue,symmetry,HOMO,[]       

    def getbasisinfo(self,groups):
        # Returns an array of:
        # NAtoms, NBasis, NBsUse, [Optional components]
        # The items in [] are:
        # Overlap matrix, Molecular orbital coefficients, Atomic orbitals on each atom

        NAtoms=0; NBasis=0; NBsUse=0
        optional=[[],[],[]]

        inputfile=open(self.filename,"r")

        line=inputfile.readline()
        while len(line)>0:
            if line.find("TOTAL NUMBER OF ATOMS")==1:
                NAtoms=int(line.split()[-1])
            elif line.find("NUMBER OF CARTESIAN GAUSSIAN BASIS")==1 or line.find("TOTAL NUMBER OF BASIS FUNCTIONS")==1:
                # The first is from Julien's Example and the second is from Alexander's
                # I think it happens if you use a polar basis function instead of a cartesian one
                NBasis=int(line.split()[-1])
                if NBsUse==0:
                    NBsUse=NBasis
            elif line.find("TOTAL NUMBER OF MOS IN VARIATION SPACE")==1:
                # Note that this line is not always present, so by default
                # NBsUse is set equal to NBasis (see above).
                NBsUse=int(line.split()[-1])
            elif line.find("OVERLAP MATRIX")==0 or line.find("OVERLAP MATRIX")==1:
                # The first is for PC-GAMESS, the second for GAMESS
                # Read 1-electron overlap matrix
                overlap=[]
                for i in range(NBasis):
                    overlap.append([])
                    for j in range(NBasis):
                        overlap[i].append(None)
                base=0
                while base<NBasis:
                    blank=inputfile.readline()
                    line=inputfile.readline() # Basis fn number
                    blank=inputfile.readline()
                    for i in range(NBasis-base): # Fewer lines each time
                        line=inputfile.readline()
                        temp=line.split()
                        for j in range(4,len(temp)):
                            overlap[base+j-4][i+base]=float(temp[j])
                            overlap[i+base][base+j-4]=float(temp[j])
                    base=base+5

                optional[0]=overlap
            elif line.find("EIGENVECTORS")==10:
                line=inputfile.readline()
                blank=inputfile.readline() # blank line

                atomorb=[] # atomorb[0] will be a list of the atomic orbitals of atom#1
                for i in range(0,NAtoms):
                    atomorb.append([])
 
                MOCoeff=[]
                for i in range(0,NBsUse):
                    MOCoeff.append([])
                    for j in range(0,NBasis):
                        MOCoeff[i].append(None)
                        
                base=0
                while line.find("END OF RHF")==-1:
                    line=inputfile.readline() # Eigenvector no
                    line=inputfile.readline() # Evalues
                    line=inputfile.readline() # Symmetry
                    line=inputfile.readline()
                    i=0
                    while line!=blank and line.find("END OF RHF")==-1:
                        if base==0: # Just do this the first time 'round
                            atomno=int(line.split()[2])-1
                            atomorb[atomno].append(int(line.split()[0])-1)
                            
                        temp=line[15:] # Strip off the crud at the start
                        j=0
                        while j*11+4<len(temp):
                            MOCoeff[base+j][i]=float(temp[j*11:(j+1)*11])
                            j=j+1
                        line=inputfile.readline()
                        i=i+1
                    base=base+5
           
            line=inputfile.readline()

        optional[1]=MOCoeff
        optional[2]=atomorb

        inputfile.close()

        return NAtoms,NBasis,NBsUse,optional
            
            
