package GCCommandLine;

###################################################
#
#  Copyright 2005-2006 Tian
#
#  This file is part of GCstar.
#
#  GCstar is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  GCstar is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with GCstar; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
###################################################

use strict;

{
    package GCPseudoFrame;
    
    use File::Basename;
    use GCUtils;
    
    sub new
    {
        my ($proto, $parent, $options, $lang) = @_;
        my $class = ref($proto) || $proto;
        my $self  = {
                        options => $options,
                        imagePrefix => 'gcstar_',
                        lang => $lang,
                        parent => $parent,
                    };
        bless ($self, $class);
        
        return $self;
    }

    sub setCurrentModel
    {
        my ($self, $model) = @_;
        
        $self->{parent}->setModel($model);
    }

    sub refreshFilters
    {
    }
    
    sub transformTitle
    {
        my ($self, $title) = @_;
        return $title;
    }
    
    sub getUniqueImageFileName
    {
        my ($self, $suffix, $itemTitle, $imagesDir) = @_;
    
        $imagesDir ||= $self->{options}->images;
        $imagesDir =~ s+([^/])$+$1/+;
        my $imagePrefix;
        if ($self->{options}->useTitleForPics)
        {
            $imagePrefix = GCUtils::getSafeFileName($itemTitle);
            $imagePrefix .= '_';
        }
        else
        {
            $imagePrefix = $self->{imagePrefix};
        }

        if ( ! -e $imagesDir)
        {
            mkdir($imagesDir);
        }

        my $filePrefix = $imagesDir.$imagePrefix;
        my @tmp_filenames;
        @tmp_filenames = glob $filePrefix.'*.*';
        my $sysPrefix = $filePrefix;
        $sysPrefix =~ s/\\/\\\\/g if ($^O =~ /win32/i);
        my @numbers = sort {$b <=> $a} map {
            /$sysPrefix([0-9]*)\./ && $1;
        } @tmp_filenames;
        my $mostRecent = $numbers[0] || 0;

        my $picture = $filePrefix.$mostRecent.$suffix;

        while (-e $picture)
        {
            $mostRecent++;
            $picture = $filePrefix.$mostRecent.$suffix;
        }
        return $picture;
    }

    sub transformPicturePath
    {
        my ($self, $path, $file) = @_;

        return '' if !$path;
        $file ||= $self->{parent}->{file};
        $path = GCUtils::getDisplayedImage($path, '', $file);
        my $dir = undef;
        $dir = dirname($file) if $file;
        return GCUtils::pathToUnix(File::Spec->rel2abs($path,$dir))
            if !$self->{options}->useRelativePaths;
        return GCUtils::pathToUnix(File::Spec->abs2rel($path,$dir));
    }
    
    sub endProgress
    {
    }
    
    sub removeUpdatedMark
    {
    }
}

{

    package GCCommandExecution;

    use File::Temp qw/ :POSIX /;
    use File::Basename;
    use GCData;
    use GCDisplay;
    use GCExport;
    use GCImport;
    use GCPlugins;
    use GCModel;

    sub new
    {
        my ($proto, $options, $model, $plugin, $import, $export, $output) = @_;
        my $class = ref($proto) || $proto;
        my $self  = {};
        bless ($self, $class);

        GCPlugins::loadPlugins;
        $self->{lang} = $GCLang::langs{$options->lang};
        $self->{parent} = new GCPseudoFrame($self, $options, $self->{lang});

        $self->{modelFactory} = new GCModelsCache($self->{parent});
        $self->{model} = $self->{modelFactory}->getModel($model);

        $self->{plugin} = $pluginsMap{$model}->{$plugin};

        if ($import)
        {
            GCImport::loadImporters;
            foreach (@importersArray)
            {
                if ($_->getName =~ /$import/)
                {
                    $self->{importer} = $_;
                    last;
                }
            }
            $self->{importer}->setModel($self->{model});
            $self->{importOptions} = {};
            foreach (@{$self->{importer}->getOptions})
            {
                $self->{importOptions}->{$_->{name}} = $_->{default};
            }
        }
        if ($export)
        {
            GCExport::loadExporters;
            foreach (@exportersArray)
            {
                if ($_->getName eq $export)
                {
                    $self->{exporter} = $_;
                    last;
                }
            }
            $self->{exporter}->setModel($self->{model});
            $self->{exporter}->setLangName($options->lang);
            $self->{exportOptions} = {
                                         lang => $self->{lang}
                                     };
            foreach (@{$self->{exporter}->getOptions})
            {
                $self->{exportOptions}->{$_->{name}} = $_->{default};
            }
        }
        
        $self->{file} = $output;
        $self->{useStdOut} = 0;
        if (!$output)
        {
            (undef, $self->{file}) = tmpnam;
            $self->{useStdOut} = 1;
        }

        $self->{toBeRemoved} = [];

        $self->{data} = new GCItems($self->{parent});
        $self->{data}->{options} = $options;

        return $self;
    }
    
    sub DESTROY
    {
        my $self = shift;
        
        unlink $_ foreach (@{$self->{toBeRemoved}});
    }
    
    sub setModel
    {
        my ($self, $model) = @_;
        $self->{model} = $self->{modelFactory}->getModel($model);
        $self->{exporter}->setModel($self->{model})
            if $self->{exporter};
        $self->{importer}->setModel($self->{model})
            if $self->{importer};
    }
    
    sub setFields
    {
        my ($self, $fieldsFile) = @_;

        $self->{fields} = [];        
        open FIELDS, '<'.$fieldsFile;
        my $model = <FIELDS>;
        chop $model;
        while (<FIELDS>)
        {
            chop;
            push @{$self->{fields}}, $_;
        }
    }
    
    sub load
    {
        my ($self, $title) = @_;
        my @data;
        $self->{plugin}->{title} = $title;
        $self->{plugin}->{type} = 'load';
        $self->{plugin}->{urlField} = $self->{model}->{commonFields}->{url};
        $self->{plugin}->load;
        my $itemNumber = $self->{plugin}->getItemsNumber();
        $self->{plugin}->{type} = 'info';
        for (my $i = 0;
                $i < $itemNumber;
                $i++)
        {
            $self->{plugin}->{wantedIdx} = $i;
            my $info = $self->{plugin}->getItemInfo;
            foreach (@{$self->{model}->{managedImages}})
            {
                $info->{$_} = $self->downloadPicture($info->{$_});
            }
            push @data, $info;
        }
        $self->{data}->{itemArray} = \@data;
    }
    
    sub save
    {
        my $self = shift;
        my $previousFile = $self->{data}->{options}->file;
        $self->{data}->{options}->file($self->{file});
        $self->{data}->{model} = $self->{model};
        $self->{data}->save;
        $self->{data}->{options}->file($previousFile);
        open IN, $self->{file};
        if ($self->{useStdOut})
        {
            print $_ while (<IN>);
        }
        close IN;
        unlink $self->{file} if $self->{useStdOut};
    }
    
    sub open
    {
        my ($self, $file) = @_;
        $self->{data}->load($file, undef, undef, 1);
    }

    sub import
    {
        my ($self, $file, $prefs) = @_;

        $self->{importOptions}->{parent} = $self->{parent};
        $self->{importOptions}->{file} = $file;
        $self->{importOptions}->{fields} = $self->{fields};
        $self->parsePrefs($prefs, $self->{importOptions});
        $self->{importer}->{options} = $self->{importOptions};
        $self->{data}->{itemArray} = $self->{importer}->getItemsArray($file);
        $self->setModel($self->{importer}->getModelName);
        $self->{data}->{model} = $self->{model}
            if $self->{data};
    }
    
    sub export
    {
        my ($self, $prefs) = @_;

        $self->{exportOptions}->{parent} = $self->{parent};
        $self->{exportOptions}->{fields} = $self->{fields};
        $self->{exportOptions}->{originalList} = $self->{data};
        $self->{exportOptions}->{withPictures} = 1;
        $self->{exportOptions}->{file} = $self->{file};
        $self->{exportOptions}->{collection} = $self->{file};
        $self->{exportOptions}->{fieldsInfo} = $self->{model}->{fieldsInfo};
        $self->{exportOptions}->{items} = $self->{data}->{itemArray};
        $self->{exportOptions}->{defaultImage} = $ENV{GCS_SHARE_DIR}.'/logos/no.png';
        $self->parsePrefs($prefs, $self->{exportOptions});
        $self->{data}->{model} = $self->{model}
            if $self->{data};

        $self->{exporter}->process($self->{exportOptions});

        open IN, $self->{exportOptions}->{file};
        if ($self->{useStdOut})
        {
            print $_ while (<IN>);
        }
        close IN;
        unlink $self->{exportOptions}->{file} if $self->{useStdOut};
    }

    sub downloadPicture
    {
        my ($self, $pictureUrl) = @_;
    
        return '' if ! $pictureUrl;
        my ($name,$path,$suffix) = fileparse($pictureUrl, "\.gif", "\.jpg", "\.jpeg", "\.png");
        (undef, my $picture) = tmpnam;
        $picture .= $suffix;

        my $browser = LWP::UserAgent->new;
        $browser->proxy(['http'], $self->{parent}->{options}->proxy);
        $browser->agent($self->{agent});
        $browser->get($pictureUrl, ':content_file' => $picture);
        push @{$self->{toBeRemoved}}, $picture;
        return $picture;
    }
    
    sub parsePrefs
    {
        my ($self, $prefs, $cont) = @_;
        
        foreach (split /,/, $prefs)
        {
            my @option = split /=>/, $_;
            $option[0] =~ s/^\s*//g;
            $option[0] =~ s/\s*$//g;
            $option[1] =~ s/^\s*//g;
            $option[1] =~ s/\s*$//g;
            $cont->{$option[0]} = $option[1];
        }
    }
}

1;
