// This file is part of a XSLT engine working on Gdome documents. In fact,
// it just maps Gdome documents to libxml documents back and forth, and
// applies the transformation on libxml documents using libxlt.
// 
// Copyright (C) 2002: Luca Padovani <lpadovan@cs.unibo.it>
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
// For more information, please send an email to lpadovan@cs.unibo.it

#include "config.h"

#include <string>
#include <cassert>

#include <stdio.h>
#include <GdomeSmartDOM.hh>

#include "GdomeSmartDOMXSLTStylesheet.hh"

namespace GdomeSmartDOMExt {

  XSLTStylesheet::XSLTStylesheet(const Document& doc)
  {
    assert(doc);
    GdomeDocument* _doc = gdome_cast_doc(doc.gdome_object());
    assert(_doc);
    stylesheet = processStylesheet(_doc);
    GdomeException _exc = 0;
    gdome_doc_unref(_doc, &_exc);
    assert(_exc == 0);
  }

  XSLTStylesheet::~XSLTStylesheet()
  {
    if (stylesheet)
      {
	xsltFreeStylesheet(stylesheet);
	stylesheet = 0;
      }
  }

  Document
  XSLTStylesheet::apply(const Document& source) const
  {
    std::vector< std::pair<GdomeString,GdomeString> > noParams;
    return apply(source, noParams);
  }

  Document
  XSLTStylesheet::apply(const Document& source, const std::vector< std::pair<GdomeString,GdomeString> >& params) const
  {
    assert(source);

    char** _params = (char**) malloc(sizeof(char*) * (2 * params.size() + 1));
    for (unsigned i = 0; i < params.size(); i++)
      {
	std::string param = params[i].first;
	_params[2 * i] = strdup(param.c_str());
	std::string value = params[i].second;
	_params[2 * i + 1] = strdup(value.c_str());
      }
    _params[2 * params.size()] = 0;

    GdomeDocument* _source = gdome_cast_doc(source.gdome_object());
    assert(_source);
    GdomeDocument* _result = applyStylesheet(_source, stylesheet, const_cast<const char**>(_params));

    GdomeException _exc = 0;
    gdome_doc_unref(_source, &_exc);
    assert(_exc == 0);

    for (unsigned i = 0; i < 2 * params.size(); i++) free(_params[i]);
    free(_params);

    Document result(_result);
    gdome_doc_unref(_result, &_exc);
    assert(_exc == 0);

    return result;
  }

  void
  XSLTStylesheet::save(const Document& doc, const std::string& filename) const
  {
    assert(doc);
    GdomeException _exc = 0;
    GdomeDocument* _doc = gdome_cast_doc(doc.gdome_object());
    assert(_doc);
    if (saveResultToFilename(filename.c_str(), _doc, stylesheet, 0) < 0) throw SaveException();
    gdome_doc_unref(_doc, &_exc);
    assert(_exc == 0);
  }

  void
  XSLTStylesheet::save(const Document& doc, FILE* f) const
  {
    assert(doc);
    GdomeException _exc = 0;
    GdomeDocument* _doc = gdome_cast_doc(doc.gdome_object());
    assert(_doc);
    if (saveResultToFile(f, _doc, stylesheet) < 0) throw SaveException();
    gdome_doc_unref(_doc, &_exc);
    assert(_exc == 0);
  }

  void
  XSLTStylesheet::save(const Document& doc, int fd) const
  {
    assert(doc);
    GdomeException _exc = 0;
    GdomeDocument* _doc = gdome_cast_doc(doc.gdome_object());
    assert(_doc);
    if (saveResultToFd(fd, _doc, stylesheet) < 0) throw SaveException();
    gdome_doc_unref(_doc, &_exc);
    assert(_exc == 0);
  }

}
