/****************************************************************************
** clparam_list.cc
**
** Mike Borella <mike@borella.net>
**
** Methods for the command line parameter list class
**
** $Id: clparam_list.cc,v 1.4 2000/10/25 22:52:05 mborella Exp $
**
****************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <iostream>
#include <fstream>
#include <ctype.h>
#include "clparam_list.h"
#include "sysinfo.h"
#include "systime.h"
#include "userinfo.h"

//---------------------------------------------------------------------------
//
// Clparam_list::operator=()
//
// Copy constructor.
//
//---------------------------------------------------------------------------

Clparam_list& Clparam_list::operator=(const Clparam_list &rhs)
{
  // guard against self-assignment
  if (this != &rhs)
    {
      _params = rhs._params;
      _global_callback_list = rhs._global_callback_list;
      _include_list = rhs._include_list;
      _mandatory_list = rhs._mandatory_list;
    }
  
  return *this;
}

//---------------------------------------------------------------------------
//
// Clparam_list::add()
//
// Add a parameter to the list, checking for duplicates
//
//---------------------------------------------------------------------------

void Clparam_list::add(Clparam p) throw (EH)
{
  if (this->contains(p.internal_name()))
    {
      EH e("duplicate parameter");
      throw e;
    }

  if (!p.long_only() && this->contains(p.short_rep()))
    {
      EH e("duplicate parameter");
      throw e;
    }

  _params.push_back(p);
}

//---------------------------------------------------------------------------
//
// Clparam_list::have_callbacks()
//
// does the list have any global or parameter callbacks?
//
//---------------------------------------------------------------------------

bool Clparam_list::have_callbacks()
{
  bool ret = false;
  list<Clparam>::iterator iter;
  
  for (iter = _params.begin(); iter != _params.end(); iter++)
    ret |= !iter->callback().empty();
  if (ret || _global_callback_list.size() != 0)
    return true;
  else
    return false;
}

//---------------------------------------------------------------------------
//
// Clparam_list::contains()
//
// does the list contain this parameter? Based on internal name
//
//---------------------------------------------------------------------------

bool Clparam_list::contains(string s)
{
  list<Clparam>::iterator iter;
  
  for (iter = _params.begin(); iter != _params.end(); iter++)
    if (!strcmp(iter->internal_name().c_str(),s.c_str()))
      return true;
  return false;
}

//---------------------------------------------------------------------------
//
// Clparam_list::contains()
//
// does the list contain this parameter? Based on short representation
//
//---------------------------------------------------------------------------

bool Clparam_list::contains(char c)
{
  list<Clparam>::iterator iter;
  
  for (iter = _params.begin(); iter != _params.end(); iter++)
    if (iter->short_rep() == c)
      return true;
  return false;
}

//---------------------------------------------------------------------------
//
// Clparam_list::logdump()
//
// Dump contents to a log file
//
//---------------------------------------------------------------------------

void Clparam_list::logdump(Logfile& l)
{
  // preface
  l.write("dumping parameter list...");
  
  // do the global stuff here
  list<string>::iterator iter;
  for (iter = _include_list.begin(); iter != _include_list.end(); iter++)

    l.write("  include file: " + *iter);
  for (iter = _mandatory_list.begin(); iter != _mandatory_list.end(); iter++)
    l.write("  mandatory: " + *iter);
  for (iter = _global_callback_list.begin(); 
       iter != _global_callback_list.end(); iter++)
    l.write("  global_callback: " + *iter);
 
  // now do the parameters
  list<Clparam>::iterator iter2;
  for (iter2 = _params.begin(); iter2 != _params.end(); iter2++)
    iter2->logdump(l);

  // afterword
  l.write("finished dumping parameter list");
}

//***************************************************************************
// Generic (C and C++) private output functions 
//***************************************************************************

//---------------------------------------------------------------------------
//
// Clparam_list::type2str()
//
// Given a parameter type, return that type in a string
//
//---------------------------------------------------------------------------

string Clparam_list::type2str(param_t t, string lang)
{
  switch (t)
    {
    case flag_t:
      return "bool";
      break;
    case int_t:
      return "int";
      break;
    case float_t:
      return "float";
      break;
    case char_t:
      return "char";
      break;
    case string_t:
      if (lang == "c" || lang == "C")
	return "char *";
      else
	return "string";
      break;
    default:
      break;
    }
}

//---------------------------------------------------------------------------
//
// Clparam_list::str2comment()
//
// Given a string, return it in comment form in a new string
//
//---------------------------------------------------------------------------

string Clparam_list::str2comment(string s)
{
  string s2;
  s2 = "/* " + s + " */";
  return s2;
}

//---------------------------------------------------------------------------
//
// Clparam_list::str2comment()
//
// Given a string, return it in comment form in a new string
//
//---------------------------------------------------------------------------

string Clparam_list::str2upper(string s)
{
  string s2;
  string::iterator iter;
  
  for (iter = s.begin(); iter != s.end(); iter ++)
    s2 += toupper(*iter);

  return s2;
 
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_comments()
//
// Output the comments at the beginning of a file.  Generic for header, C and 
// C++ files
//
//---------------------------------------------------------------------------

void Clparam_list::output_comments(ofstream& o, string fn, string purpose)
{
  // opening comments with filename
  o << "/******************************************************************************" << endl;
  o << "**" << endl;
  o << "** " << fn << endl;
  o << "**" << endl;
  
  // timestamp the file
  Systime t;
  o << "** " << t.unix_ctime();
  
  // add the system information
  Sysinfo i;
  o << "** " << i.format() << endl;

  // add the user information
  Userinfo u;
  o << "** " << u.format() << endl;
  o << "**" << endl;
  
  // add the purpose - as if there is one :-)
  o << "** " << purpose << endl;
  o << "**" << endl;

  // do a little advertising for the program...
  o << "** Automatically created by " << PACKAGE << " v" << VERSION << endl;
  o << "**" << endl;
  o << "** See http://genparse.sourceforge.net/ for details and updates" 
    << endl;
  o << "**" << endl;

  // that's it - finish up
  o << "******************************************************************************/" << endl;
  o << endl;  
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_comments_function()
//
// Output a function comment banner
//
//---------------------------------------------------------------------------

void Clparam_list::output_comments_function(ofstream& o, string f, string c)
{
  o << "/*----------------------------------------------------------------------------" << endl;
  o << "**" << endl;
  o << "** " << f << "()" << endl;
  o << "**" << endl;
  o << "** " << c << endl;
  o << "**" << endl;
  o << "**--------------------------------------------------------------------------*/" << endl;  
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_includes()
//
// Output the general include files that were specified in the .gp file
//
//---------------------------------------------------------------------------

void Clparam_list::output_includes(ofstream& o)
{
  // loop through the list
  list<string>::iterator iter;
  for (iter = _include_list.begin(); iter != _include_list.end(); iter++)
    o << "#include <" << *iter << ">" << endl;
  o << endl;
}

//***************************************************************************
// C-only private output functions 
//***************************************************************************

//---------------------------------------------------------------------------
//
// Clparam_list::output_c_freeargs()
//
// Output a C function tp free to memory allocated for the parameters
//
//---------------------------------------------------------------------------

void Clparam_list::output_c_freeargs(ofstream& o)
{
  // comments
  output_comments_function(o, "free_args", "Call this to free the memory that was dynamically allocated by the parser.");

  // declaration
  o << endl;
  o << "void free_args(struct arg_t *my_args)" << endl;
  o << "{" << endl;

  // loop through args, free all the char * strings
  list<Clparam>::iterator iter;
  for (iter = _params.begin(); iter != _params.end(); iter++)
    if (iter->type() == string_t)
      {
	o << "  if (my_args->" << iter->external_name() 
	  << " != NULL) free(my_args->" << iter->external_name() << ");" 
	  << endl;
      }
  
  // finish up
  o << "  free(my_args);" << endl;
  o << "}" << endl << endl;
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_c_usage()
//
// Output a usage function in C
//
//---------------------------------------------------------------------------

void Clparam_list::output_c_usage(ofstream& o)
{
  // comments
  output_comments_function(o, "usage", 
			   "Print out usage information, then exit");

  // declaration
  o << endl;
  o << "void usage(char *executable)" << endl;
  o << "{" << endl;

  // front matter
  o << "  printf(\"usage: %s [ -";
  list<Clparam>::iterator iter;
  for (iter = _params.begin(); iter != _params.end(); iter++)
    {
      if (!iter->long_only()) 
	o << iter->short_rep();
    }
  o << " ] ";

  if (!_mandatory_list.empty())
    {
      list<string>::iterator iter2;
      for (iter2 = _mandatory_list.begin(); iter2 != _mandatory_list.end();
	   iter2++)
	o << *iter2 << " ";
    }
  o << "\\n\", executable);" << endl;

  // iterate through parameters
  for (iter = _params.begin(); iter != _params.end(); iter++)
    {
      if (!iter->long_only()) 
	o << "  printf(\"  [ -" << iter->short_rep() << " ] \");" << endl;

      // Allow for extra space for long-only params 
      if (!iter->long_rep().empty())
	{
	  if (iter->long_only())
	    o << "  printf(\"  [ --" << iter->long_rep() << "  ] \");" 
	      << endl;
	  else
	    o << "  printf(\"[ --" << iter->long_rep() << "  ] \");" << endl;
	}

      // Print the open (
      o << "  printf(\"(\");" << endl;
      
      // Print the type
      o << "  printf(\"type=\");" << endl;
      switch(iter->type())
	{
	case flag_t:
	  o << "  printf(\"FLAG";
	  break;
	case int_t:
	  o << "  printf(\"INTEGER";
	  break;
	case float_t:
	  o << "  printf(\"REAL";
	  break;
	case char_t:
	  o << "  printf(\"CHARACTER";
	  break;
	case string_t:
	  o << "  printf(\"STRING";
	  break;	  
	}
      if (!iter->high_range().empty() || !iter->low_range().empty()
	  || !iter->default_value().empty())
	o << ",";
      o << "\");" << endl;

      // Print the range
      if (!iter->low_range().empty())
	{
	  o << "  printf(\" range=" << iter->low_range() << "...";
	  if (!iter->high_range().empty())
	    o << iter->high_range();
	  o << ",\");" << endl;	    
	}
      else
	{
	  if (!iter->high_range().empty())
	    o << "  printf(\"range=..." << iter->high_range() << ",\");" 
	      << endl;	    
	}

      // Print default values, but not for flags
      if (!iter->default_value().empty() && iter->type() != flag_t)
	o << "  printf(\" default=" << iter->default_value() 
	  << "\");" << endl;
      o << "  printf(\")\\n\");" << endl;

      // Print the description, hopefully in a way that looks nice...
      int i = 0;
      while(i < iter->get_num_desc())
	{
	  o << "  printf(\"         " << iter->get_description(i) 
	    << "\\n\");" << endl;
	  i++;
	}
    }

  // back matter
  o << endl;
  o << "  exit(1);" << endl;
  o << "}" << endl << endl;
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_c_header()
//
// Output the header file
//
//---------------------------------------------------------------------------

void Clparam_list::output_c_header(string f) throw (EH)
{
  // make file .h extension
  string headerfile;
  headerfile = f + ".h";

  // tell the user what we're doing
  if (!_quiet)
    cout << "creating " << headerfile << "...";
  
  // open the header file
  ofstream o;
  o.open(headerfile.c_str());
  if (!o)
    {
      EH e("can't open header file: " + headerfile);
      throw e;
    }

  // print a nice friendly comment banner at the beginning of the file
  output_comments(o, headerfile, 
		  "Header file for command line parser");
  
  // print the includes 
  o << "#include <stdio.h>" << endl;
  output_includes(o);

  // add a bool type for C
  o << "#ifndef bool" << endl;
  o << "typedef enum bool_t" << endl;
  o << "{" << endl;
  o << "  false = 0, true" << endl;
  o << "} bool;" << endl;
  o << "#endif" << endl <<endl;

  // print the structure definition then flags
  o << "/* customized structure for command line parameters */" 
	     << endl;
  o << "struct arg_t" << endl;
  o << "{" << endl;
  list<Clparam>::iterator iter2;
  for (iter2 = _params.begin(); iter2 != _params.end(); iter2++)
      o << "  " << type2str(iter2->type(), "c") << " " 
		 << iter2->external_name() << ";" << endl;
  o << "  int optind;" << endl;
  o << "};" << endl << endl;

  // function prototypes
  o << "/* function prototypes */" << endl;
  o << "struct arg_t * " << _parsefunc << "(int, char **);" << endl;
  o << "void usage(char *);" << endl;
  o << "void free_args(struct arg_t *);" << endl << endl;

  // more prototypes, for the callbacks, global and local
  if (have_callbacks())
    {
      o << "/* global and local callbacks */" << endl;
      list<string>::iterator iter;
      for (iter = _global_callback_list.begin(); 
	   iter != _global_callback_list.end(); iter++)
	o << "int " << *iter << "(struct arg_t *);" << endl;
      for (iter2 = _params.begin(); iter2 != _params.end(); iter2++)
	{
	  if (!(iter2->callback()).empty())
	    {
	      o << "int param_" << iter2->external_name() << "_callback("
		<< type2str(iter2->type(), "c") << ");" << endl;
	    }
	}
    }

  // close the header file
  o.close();

  // tell the user what we're doing
  if (!_quiet)
    cout << "done" << endl;  
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_c_parser()
//
// Output the parsing (main) file
//
//---------------------------------------------------------------------------

void Clparam_list::output_c_parser(string f) throw (EH)
{
  // make file .c extension
  string cfile;
  cfile = f + ".c";

  // tell the user what we're doing
  if (!_quiet) 
    cout << "creating " << cfile << "...";

  // open the c file
  ofstream o;
  o.open(cfile.c_str());
  if (!o)
    {
      EH e("can't open C file: " + cfile);
      throw e;
    }

  // print a nice friendly comment banner at the beginning of the file
  output_comments(o, cfile, "C file for command line parser");
  
  // print the includes 
  o << "#include <string.h>" << endl;
  o << "#include <stdlib.h>" << endl;
  o << "#include <unistd.h>" << endl;
  o << "#include <stdlib.h>" << endl;
  o << "#include <getopt.h>" << endl;
  o << "#include \"" << f + ".h" << "\"" << endl << endl;
  
  // dump the usage function
  output_c_usage(o);

  // dump the free_args function
  output_c_freeargs(o);

  // comments
  output_comments_function(o, _parsefunc, "Parse the argv array into the command line structure");
  
  // declaration and local vars
  o << endl;
  o << "struct arg_t *" << _parsefunc << "(int argc, char *argv[])" << endl;
  o << "{" << endl;
  o << "  extern char *optarg;" << endl;
  o << "  extern int optind;" << endl;
  o << "  int option_index = 0;" << endl;
  o << "  int c;" << endl;
  o << "  struct arg_t *my_args;" << endl;
  o << "  int errflg = 0;" << endl << endl;

  // write the options structure
  o << "  static struct option long_options[] =" << endl;
  o << "  {" << endl;
  list<Clparam>::iterator iter2;
  for (iter2 = _params.begin(); iter2 != _params.end(); iter2++)
    if (iter2->type() == flag_t)
      {
	if (!iter2->long_only())
	  {
	    o << "    {\"" << iter2->long_rep() << "\", 0, 0, '" 
	      << iter2->external_name() << "'}," << endl;
	  }
	else
	  {
	    o << "    {\"" << iter2->long_rep() << "\", 0, 0, " 
	      << iter2->longopt_value() << "}," << endl;
	  }
      }
    else
      {
	if (!iter2->long_only())
	  {
	    // everything else does take an argument, so second value is 1
	    o << "    {\"" << iter2->long_rep() << "\", 1, 0, '" 
	      << iter2->short_rep() << "'}," << endl;
	  }
	else
	  {
	    o << "    {\"" << iter2->long_rep() << "\", 1, 0, " 
	      << iter2->longopt_value() << "}," << endl;
	  }
      }
  o << "    {0, 0, 0, 0}" << endl;
  o << "  };" << endl << endl;
  
  // malloc the struct arg_t and write default values
  o << "  my_args = (struct arg_t *) malloc (sizeof(struct arg_t));" << endl
    << endl;
  for (iter2 = _params.begin(); iter2 != _params.end(); iter2++)
    {
      switch(iter2->type())
	{
	case flag_t:  // all flags default to off
	  o << "  my_args->" << iter2->external_name() << " = false;" << endl;
	  break;
	case string_t: // copy default if there, NULL otherwise
	  if (!iter2->default_value().empty())
	    o << "  my_args->" << iter2->external_name() 
	      << " = strdup(\"" << iter2->default_value() << "\");" << endl;
	  else
	    o << "  my_args->" << iter2->external_name() << " = NULL;" 
	      << endl;
	  break;
	default: // all else - copy default if there
	  if (!iter2->default_value().empty())
	    o << "  my_args->" << iter2->external_name() 
	      << " = " << iter2->default_value() << ";" << endl;
	}
    }
  o << endl;

  // Make the getopt_long() call
  o << "  while ((c = getopt_long(argc, argv, \"";
  for (iter2 = _params.begin(); iter2 != _params.end(); iter2++)
    {
      if (iter2->long_only())
	continue;
      if (iter2->type() == flag_t)
	o << iter2->external_name();
      else
	o << iter2->external_name() << ":";
    }
  o << "\", long_options, &option_index)) != EOF)" << endl;

  // do the switch statement
  o << "    {" << endl;
  o << "      switch(c)" << endl;
  o << "        {" << endl;
  for (iter2 = _params.begin(); iter2 != _params.end(); iter2++)
    {
      if (iter2->long_only())
	o << "        case " << iter2->longopt_value() << ": " << endl;
      else
	o << "        case '" << iter2->short_rep() << "': " << endl;
      o << "          my_args->" << iter2->external_name();
      switch(iter2->type())
	{
	case string_t:
	  o << " = strdup(optarg);" << endl;
          break;
        case int_t:
	  o << " = atoi(optarg);" << endl;
          break;
        case float_t:
	  o << " = atof(optarg);" << endl;
	  break;
	case char_t:
	  o << " = *optarg;" << endl;
	  break;
	case flag_t:
	  o << " = true;" << endl;
	  if (iter2->short_rep() == 'h' && iter2->long_rep() == "help")
	    o << "          usage(argv[0]);" << endl;
	  break;
        default:
          break;
	}

      // do low range checking
      if (!iter2->low_range().empty())
	{
	  o << "          if (my_args->" << iter2->external_name()
	    << " < " << iter2->low_range() << ")" << endl;
	  o << "            {" << endl;
	  o << "              fprintf(stderr, \"parameter range error: " 
	    << iter2->external_name() << " must be >= " 
	    << iter2->low_range() << "\\n\");" << endl;
	  o << "              errflg++;" << endl;
	  o << "            }" << endl;
	}
      if (!iter2->high_range().empty())
	{
	  o << "          if (my_args->" << iter2->external_name()
	    << " > " << iter2->high_range() << ")" << endl;
	  o << "            {" << endl;
	  o << "              fprintf(stderr, \"parameter range error: " 
	    << iter2->external_name() << " must be <= " 
	    << iter2->high_range() << "\\n\");" << endl;
	  o << "              errflg++;" << endl;
	  o << "            }" << endl;
	}

      // do callbacks
      if (!iter2->callback().empty())
        {
          o << "          if (!param_" << iter2->external_name() 
	    << "_callback(my_args->" << iter2->external_name() 
	    << "))" << endl;
          o << "            usage(argv[0]);" << endl;
        }
      
      // do break statement
      o << "          break;" << endl << endl;            
    }

  // print default action for unknown parameters
  o << "        default:" << endl;
  o << "          usage(argv[0]);" << endl << endl;

  // print the end of the function, with global callback calls
  o << "        }" << endl;
  o << "    } /* while */" << endl << endl;
  o << "  if (errflg)" << endl;
  o << "    usage(argv[0]);" << endl << endl;
  list<string>::iterator iter;
  for (iter = _global_callback_list.begin(); 
       iter != _global_callback_list.end(); iter++)
    {
      o << "  if (!" << *iter << "(my_args))" << endl;
      o << "    usage(argv[0]);" << endl << endl;
    }
  o << "  if (optind >= argc)" << endl;
  o << "    my_args->optind = 0;" << endl;
  o << "  else" << endl;
  o << "    my_args->optind = optind;" << endl;
  o << "  return my_args;" << endl;
  o << "}" << endl;
  
  // close the C file
  o.close();  

  // tell the user what we're doing
  if (!_quiet)
    cout << "done" << endl;  
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_c_callbacks()
//
// Output the callbacks file in C. 
//
//---------------------------------------------------------------------------

void Clparam_list::output_c_callbacks(string f) throw (EH)
{
  // first figure out if we have to do anything at all - no callbacks = no
  // callback file
  if (have_callbacks())
    {
      // make the callback filename
      string cb_file;
      cb_file = f + "_cb.c";

      // tell the user what we're doing
      if (!_quiet)
	cout << "creating " << cb_file << "...";
      
      // open the callback file
      ofstream o;
      o.open(cb_file.c_str());
      if (!o)
	{
	  EH e("can't open callback file: " + cb_file);
	  throw e;
	}

      // do the main comments
      output_comments(o, cb_file, "Callback routines for command line parser");

      // use the header filename for the includes
      o << "#include <stdio.h>" << endl;
      o << "#include \"" << f + ".h" << "\"" << endl << endl;

      // do the global callbacks first
      list<string>::iterator iter;
      for (iter = _global_callback_list.begin(); 
	   iter != _global_callback_list.end(); iter++)
	{
	  output_comments_function(o, *iter, "User defined global callback.");
	  o << endl;
	  o << "int " << *iter << "(struct arg_t *a)" << endl;
	  o << "{" << endl;
	  o << "  return 1;" << endl;
	  o << "}" << endl << endl;
	}
      
      // do the individual parameter callbacks
      list<Clparam>::iterator iter2;
      for (iter2 = _params.begin(); iter2 != _params.end(); iter2++)
	{
	  if (!iter2->callback().empty())
	    {
	      string s;
	      s += "param_";
	      s += iter2->external_name();
	      s += "_callback";
	      output_comments_function(o, s, 
				       "User defined parameter callback.");
	      o << endl;
	      o << "int param_" << iter2->external_name() << "_callback("
		<< type2str(iter2->type(), "c") << " var)" << endl;
	      o << "{" << endl;
	      o << "  return 1;" << endl;
	      o << "}" << endl << endl;
	    }
	}
      
      // tell the user what we're doing
      if (!_quiet)
	cout << "done" << endl;
    }
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_c()
//
// Output the parsing engine in C.  
//
//---------------------------------------------------------------------------

void Clparam_list::output_c(string outfile) throw (EH)
{
  output_c_header(outfile);
  output_c_parser(outfile);
  output_c_callbacks(outfile);
}

//***************************************************************************
// C++-only output functions 
//***************************************************************************

//---------------------------------------------------------------------------
//
// Clparam_list::output_cpp_comments_class()
//
// Output a class comment banner
//
//---------------------------------------------------------------------------

void Clparam_list::output_cpp_comments_class(ofstream& o, string theclass, 
					     string comment)
{
  o << "/*----------------------------------------------------------------------------" << endl;
  o << "**" << endl;
  o << "** class " << theclass << endl;
  o << "**" << endl;
  o << "** " << comment << endl;
  o << "**" << endl;
  o << "**--------------------------------------------------------------------------*/" << endl;  
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_cpp_header()
//
// Output a C++ header file
//
//---------------------------------------------------------------------------

void Clparam_list::output_cpp_header(string f) throw (EH)
{
  // make file .h extension
  string filename;
  filename = f + ".h";
  
  // tell the user what we're doing
  if (!_quiet)
    cout << "creating " << filename << "...";
  
  // open the header file
  ofstream o;
  o.open(filename.c_str());
  if (!o)
    {
      EH e("can't open header file: " + filename);
      throw e;
    }
  
  // print a nice friendly comment banner at the beginning of the file
  output_comments(o, filename, "Header file for command line parser class");
  
  // do the ifdef
  o << "#ifndef " << str2upper(_parsefunc) << "_H" << endl;
  o << "#define " << str2upper(_parsefunc) << "_H" << endl;
  o << endl;

  // print the includes 
  o << "#include <iostream>" << endl;
  o << "#include <string>" << endl;
  output_includes(o);

  // dump the class
  output_cpp_comments_class(o, _parsefunc, "command line parser class");
  o << endl;
  o << "class " << _parsefunc << endl;
  o << "{" << endl;

  // private section
  o << "private:" << endl;
  o << "  " << str2comment("parameters") << endl;
  list<Clparam>::iterator iter;
  for (iter = _params.begin(); iter != _params.end(); iter++)
    o << "  " << type2str(iter->type(), "c++") << " _" 
      << iter->external_name() << ";" << endl;
  o << endl;
  o << "  " << str2comment("other stuff to keep track of") << endl;
  o << "  string _executable;" << endl;
  o << "  int _optind;" << endl;
  o << endl;

  // public section

  o << "public:" << endl;
  o << "  " << str2comment("constructor and destructor") << endl;
  o << "  " << _parsefunc << "(int, char **) throw(string);" << endl;
  o << "  ~" << _parsefunc << "(){}" << endl;
  o << endl;
  o << "  " << str2comment("usage function") << endl;
  o << "  void usage();" << endl;
  o << endl;
  o << "  " << str2comment("return next (non-option) parameter") << endl;
  o << "  int next_param() { return _optind; }" << endl;
  o << endl;
  o << "  " << str2comment("callback functions") << endl;
  list<string>::iterator iter2;
  for (iter2 = _global_callback_list.begin(); 
       iter2 != _global_callback_list.end(); iter2++)
    o << "  bool " << *iter2 << "();" << endl;
  for (iter = _params.begin(); iter != _params.end(); iter++)
    {
      if (!(iter->callback()).empty())
	  o << "  bool param_" << iter->external_name() << "_callback();" 
	    << endl;
    }
  o << endl;
  
  // functions to return short rep values
  for (iter = _params.begin(); iter != _params.end(); iter++)
    o << "  " << type2str(iter->type(), "c++") << " " << iter->external_name() 
      << "() { return _" << iter->external_name() << "; }" << endl;
  o << "};" << endl;
  o << endl;
  o << "#endif" << endl;

  // close the file
  o.close();

  // tell the user what we're doing
  if (!_quiet)
    cout << "done" << endl;
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_cpp_class()
//
// Output a C++ parser class
//
//---------------------------------------------------------------------------

void Clparam_list::output_cpp_class(string f) throw(EH)
{
  // make file extension
  string filename;
  filename = f + "." + _cppext;
  
  // tell the user what we're doing
  if (!_quiet)
    cout << "creating " << filename << "...";
  
  // open the header file
  ofstream o;
  o.open(filename.c_str());
  if (!o)
    {
      EH e("can't open header file: " + filename);
      throw e;
    }
  
  // print a nice friendly comment banner at the beginning of the file
  output_comments(o, filename, "Definition of command line parser class");
  
  // includes
  o << "#include <getopt.h>" << endl;
  o << "#include <stdlib.h>" << endl;
  o << "#include \"" << f << ".h\"" << endl;
 
  // include callback file if necessary
  if (have_callbacks())
    o << "#include \"" << f << "_cb." << _cppext << "\"" << endl;
  o << endl;

 
  // comments for constructor
  output_comments_function(o, _parsefunc + "::" + _parsefunc,
			   "Constructor method.");
  o << endl;

  // constructor
  o << _parsefunc << "::" << _parsefunc 
    << "(int argc, char *argv[]) throw (string)" << endl;
  o << "{" << endl;
  
  // local variables
  o << "  extern char *optarg;" << endl;
  o << "  extern int optind;" << endl;
  o << "  int option_index = 0;" << endl;
  o << "  int c;" << endl;
  o << endl;

  // options structure
  o << "  static struct option long_options[] =" << endl; 
  o << "  {" << endl;
  list<Clparam>::iterator iter;
  for (iter = _params.begin(); iter != _params.end(); iter++)
    {
      if (iter->type() == flag_t)
	{
	  if (iter->long_only())
	    o << "    {\"" << iter->long_rep() << "\", 0, 0, " 
	      << iter->longopt_value() << "}," << endl;
	  else
	    o << "    {\"" << iter->long_rep() << "\", 0, 0, '" 
	      << iter->short_rep() << "'}," << endl;
	}
      else
	{
	  if (iter->long_only())
	    o << "    {\"" << iter->long_rep() << "\", 1, 0, " 
	      << iter->longopt_value() << "}," << endl;
	  else
	    o << "    {\"" << iter->long_rep() << "\", 1, 0, '" 
	      << iter->short_rep() << "'}," << endl;
	}
    }
  o << "    {0, 0, 0, 0}" << endl;
  o << "  };" << endl;
  o << endl;

  // assign argv[0] to the executable name
  o << "  _executable += argv[0];" << endl;
  o << endl;

  // assign default values
  o << "  " << str2comment("default values") << endl;
  for (iter = _params.begin(); iter != _params.end(); iter++)
    {
      switch(iter->type())
        {
        case flag_t:  // all flags default to off
          o << "  _" << iter->external_name() << " = false;" << endl;
          break;
        case string_t: // copy default if there, NULL otherwise
          if (!iter->default_value().empty())
            o << "  _" << iter->external_name() 
              << " = \"" << iter->default_value() << "\";" << endl;
          break;
        default: // all else - copy default if there
          if (!iter->default_value().empty())
            o << "  _" << iter->external_name() 
              << " = " << iter->default_value() << ";" << endl;
        }
    }
  o << endl;

  // Make the getopt_long() call
  o << "  while ((c = getopt_long(argc, argv, \"";
  for (iter = _params.begin(); iter != _params.end(); iter++)
    {
      if (iter->long_only())
	continue;
      if (iter->type() == flag_t)
        o << iter->short_rep();
      else
        o << iter->short_rep() << ":";
    }
  o << "\", long_options, &option_index)) != EOF)" << endl;

 // do the switch statement
  o << "    {" << endl;
  o << "      switch(c)" << endl;
  o << "        {" << endl;
  for (iter = _params.begin(); iter != _params.end(); iter++)
    {
      if (iter->long_only())
	o << "        case " << iter->longopt_value() << ": " << endl;
      else
	o << "        case '" << iter->short_rep() << "': " << endl;
      o << "          _" << iter->external_name();
      switch(iter->type())
        {
        case string_t:
          o << " = optarg;" << endl;
          break;
        case int_t:
          o << " = atoi(optarg);" << endl;
          break;
        case float_t:
          o << " = atof(optarg);" << endl;
          break;
        case char_t:
          o << " = *optarg;" << endl;
          break;
        case flag_t:
          o << " = true;" << endl;
	  if (iter->short_rep() == 'h' && iter->long_rep() == "help")
	    o << "          this->usage();" << endl;
	  break;
        default:
          break;
        }

      // do low range checking
      if (!iter->low_range().empty())
        {
          o << "          if (_" << iter->external_name()
            << " < " << iter->low_range() << ")" << endl;
          o << "            {" << endl;
          o << "              string s;" << endl;
	  o << "              s += \"parameter range error: " 
	    << iter->external_name() << " must be >= " << iter->low_range() 
	    << "\";" << endl;
	  o << "              throw(s);" << endl;
          o << "            }" << endl;
        }
      if (!iter->high_range().empty())
        {
          o << "          if (_" << iter->external_name()
            << " > " << iter->high_range() << ")" << endl;
          o << "            {" << endl;
	  o << "              string s;" << endl;
	  o << "              s += \"parameter range error: " 
            << iter->external_name() << " must be <= " << iter->high_range() 
	    << "\";" << endl;
	  o << "              throw(s);" << endl;
          o << "            }" << endl;
        }

      // do callbacks
      if (!iter->callback().empty())
        {
          o << "          if (!param_" << iter->external_name() 
            << "_callback())" << endl;
          o << "            this->usage();" << endl;
        }
      
      // do break statement
      o << "          break;" << endl << endl;            
    }

  // print default action for unknown parameters
  o << "        default:" << endl;
  o << "          this->usage();" << endl << endl;
  o << "        }" << endl;
  o << "    } /* while */" << endl << endl;

  // save a pointer to the next argument
  o << "  _optind = optind;" << endl;

  // do global callbacks
  list<string>::iterator iter2;
  for (iter2 = _global_callback_list.begin(); 
       iter2 != _global_callback_list.end(); iter2++)
    {
      o << "  if (!" << *iter2 << "())" << endl;
      o << "    usage();" << endl << endl;
    }
  
  o << "}" << endl;
  o << endl;

  // comments for usage()
  output_comments_function(o, _parsefunc + "::usage", "Usage function.");
  o << endl;

  // usage function
  o << "void " << _parsefunc << "::usage()" << endl;
  o << "{" << endl;
  o << "  cout << \"usage: \" << _executable << \" [ -";
  for (iter = _params.begin(); iter != _params.end(); iter++)
    if (!iter->long_only())
      o << iter->short_rep();
  o << " ] "; 

  // mandatory parameters
  for (iter2 = _mandatory_list.begin(); iter2 != _mandatory_list.end(); 
       iter2++)
    o << " <" << *iter2 << ">"; 
  o << "\" << endl;" << endl;

  // info for each param
  for (iter = _params.begin(); iter != _params.end(); iter++)
    {
      if (!iter->long_only())
	{
	  o << "  cout << \"  [ -" << iter->short_rep() << " ] \";" << endl;
	  if (!iter->long_rep().empty())
	    o << "  cout << \"[ --" << iter->long_rep() << " ]  \";" << endl;
	}
      else
	{
	    o << "  cout << \"  [ --" << iter->long_rep() << " ]  \";" << endl;
	}
      
      // Print the open (
      o << "  cout << \"(\";" << endl;
      
      // Print the type
      o << "  cout << \"type=\";" << endl;
      switch(iter->type())
        {
        case flag_t:
          o << "  cout << \"FLAG";
          break;
        case int_t:
          o << "  cout << \"INTEGER";
          break;
        case float_t:
          o << "  cout << \"REAL";
          break;
        case char_t:
          o << "  cout << \"CHARACTER";
          break;
        case string_t:
          o << "  cout << \"STRING";
          break;          
        }
      if (!iter->high_range().empty() || !iter->low_range().empty()
          || !iter->default_value().empty())
        o << ",";
      o << "\";" << endl;

      // Print the range
      if (!iter->low_range().empty())
        {
          o << "  cout << \" range=" << iter->low_range() << "...";
          if (!iter->high_range().empty())
            o << iter->high_range();
          o << ",\";" << endl;     
        }
      else
        {
          if (!iter->high_range().empty())
            o << "  cout << \"range=..." << iter->high_range() << ",\";" 
              << endl;      
        }

      // Print default values, but not for flags
      if (!iter->default_value().empty() && iter->type() != flag_t)
        o << "  cout << \" default=" << iter->default_value() << "\";" << endl;
      o << "  cout << \")\\n\";" << endl;

      // Print the description, hopefully in a way that looks nice...
      int i = 0;
      while(i < iter->get_num_desc())
        {
          o << "  cout << \"         " << iter->get_description(i) 
            << "\\n\";" << endl;
          i++;
        }
    }

  o << "  exit(0);" << endl;
  o << "}" << endl;
  o << endl;

  // close the file
  o.close();

  // tell the user what we're doing
  if (!_quiet) 
    cout << "done" << endl;  
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_cpp_callbacks()
//
// Output a C++ parser callbacks file
//
//---------------------------------------------------------------------------

void Clparam_list::output_cpp_callbacks(string f) throw (EH)
{
  // first figure out if we have to do anything at all - no callbacks = no
  // callback file
  if (have_callbacks())
    {
      // make file extension
      string filename;
      filename = f + "_cb." + _cppext;
      
      // tell the user what we're doing
      if (!_quiet)
	cout << "creating " << filename << "...";
      
      // open the callbacks file
      ofstream o;
      o.open(filename.c_str());
      if (!o)
	{
	  EH e("can't open callbacks file: " + filename);
	  throw e;
	}
      
      // print a nice friendly comment banner at the beginning of the file
      output_comments(o, filename, "Callabcks for command line parser class");
      
      // includes
      o << "#include \"" << f << ".h\"" << endl;
      o << endl;
      
      // loop through global callbacks
      list<string>::iterator iter;
      for (iter = _global_callback_list.begin(); 
           iter != _global_callback_list.end(); iter++)
	{
	  // comments for 
	  output_comments_function(o, _parsefunc + "::" + *iter,
				   "Global callback.");
	  o << endl;
	  o << "bool " << _parsefunc << "::" << *iter << "()" << endl;
          o << "{" << endl;
          o << "  return true;" << endl;
          o << "}" << endl << endl;
	}

      // do the individual parameter callbacks
      list<Clparam>::iterator iter2;
      for (iter2 = _params.begin(); iter2 != _params.end(); iter2++)
        {
          if (!iter2->callback().empty())
            {
	      output_comments_function(o, _parsefunc + "::param_" 
				       + iter2->external_name(),
				       "Parameter callback.");
	      o << endl;
              o << "bool " << _parsefunc << "::" << "param_" << 
		iter2->external_name() << "_callback()" 
		<< endl;
              o << "{" << endl;
              o << "  return true;" << endl;
              o << "}" << endl << endl;
            }
        }

      // tell the user what we're doing
      if (!_quiet)
	cout << "done" << endl;
    } // if
}

//---------------------------------------------------------------------------
//
// Clparam_list::output_cpp()
//
// Output the parsing engine in C++
//
//---------------------------------------------------------------------------

void Clparam_list::output_cpp(string outfile)
{
  output_cpp_header(outfile);
  output_cpp_class(outfile);
  output_cpp_callbacks(outfile);
}


