
/*
  
   1998 Roberto Alameda 
  You may modify and distribute this file under the terms of the GNU
  General Public License, version 2, or any later version, at your
  convenience. See the file COPYING for details. 

*/


#include <config.h>

#if HAVE_UNISTD_H
# include <unistd.h>
# include <sys/types.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <limits.h>
#include <errno.h>
#include <sys/stat.h>


#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif


#include "gfont.h"

#include "mini-ofolder.xpm"
#include "font.xpm"
#include "emptypixmap.xpm"
#include "t1.xpm"
#include "tt.xpm"
#include "stipple.xbm"



/****************** Global variables ******************/
GdkVisual *visual;
GdkColormap *colormap;
GdkGC *bwGC, *defGC, *redGC, *blueGC;
GdkColorContext* mCC;
GtkStyle *stylebold;
GdkColor white, black;
GdkPixmap* emptypixmap;

GtkWidget* mainwindow;  // The main window widget
GtkWidget* fontclistw;  // Font list widget
GtkWidget* dirl;        // Directory name combo box
GtkWidget* stbar;       // Status bar below
GtkWidget* prbar;       // Progress bar below
GtkWidget* samplet;     // The sample text (a pixmap)
struct EntryDialog *fontsizeD, *characterD, *stringD;

bool antialiasing = FALSE;
bool kerning = FALSE;
bool enable_antialiasing = TRUE;

int xresolution, yresolution;  // Resolution of the display in dpi

GdkColor graypalette[5];  // Colors for antialiasing: white - light - medium - dark - black

TT_Engine ttengine;  // For the FreeType library
GSList *fontlist;    // All fonts in the present directory
#ifdef USE_GNOME
GnomeClient *client;
#endif


/* Variables used for drag&drop */
enum {
  TARGET_STRING,
  TARGET_ROOTWIN,
  TARGET_URL
};    

GtkTargetEntry dnd_target_table[] = {
  /*{ "STRING",     0, TARGET_STRING },*/
  /*{ "text/plain", 0, TARGET_STRING },*/
  { "text/uri-list", 0, TARGET_URL },
  { "application/x-rootwin-drop", 0, TARGET_ROOTWIN }       
};    

guint dnd_targets = sizeof(dnd_target_table) / sizeof(dnd_target_table[0]);  


/*************** Local static variables ****************/

/* Variables defined by the given arguments */
static char *firstdirectory = DEFAULT_FONTDIR; /* Directory to scan when app starts for the first time */
static char *geometry = NULL;



/************Function prototypes (internally used in this module) ***/
gint delete_event(GtkWidget *widget, GdkEvent *event, gpointer data);
Callback newfontdir, leave, showchar, showstring, fonttable, toggle_button;
static void gfont_init(int *argc, char*** argv);
static GtkWidget* makefontlist(void);
static void destroyrow(gpointer data);
static gint compareEntries(gconstpointer first, gconstpointer second);
static bool fillfontlist(GtkWidget* clist, char* directory);
static void okfontdir(gpointer diag);
static void load_directory(GtkWidget *entry);





/************** Menubar ******************/

#ifdef USE_GNOME
static GnomeUIInfo file_menu[] = {
  GNOMEUIINFO_ITEM_STOCK(N_("_Open..."), 
			 N_("Open a directory"), 
			 newfontdir, (void*)GNOME_STOCK_MENU_OPEN),
    GNOMEUIINFO_ITEM_STOCK(N_("Save _Names..."), 
			   N_("Save the names of all fonts"), 
			   save_names, (void*)GNOME_STOCK_MENU_SAVE_AS),
    GNOMEUIINFO_ITEM_STOCK(N_("_Print..."), 
			   N_("Print samples of the selected fonts"), 
			   printfonts, (void*)GNOME_STOCK_MENU_PRINT),
    GNOMEUIINFO_SEPARATOR,
    GNOMEUIINFO_MENU_EXIT_ITEM(leave,NULL),
    GNOMEUIINFO_END
};


static GnomeUIInfo view_menu[] = {
  GNOMEUIINFO_ITEM_STOCK(N_("_Properties..."), 
			 N_("See the properties of the selected fonts"), 
			 show_properties, (void*)GNOME_STOCK_MENU_PROP),
    GNOMEUIINFO_ITEM_STOCK(N_("_Errors..."), 
			   N_("See error messages"), 
			   show_message_window, 
			   (void*)GNOME_STOCK_MENU_BOOK_RED),
    GNOMEUIINFO_END
};

static GnomeUIInfo help_menu[] = {
  GNOMEUIINFO_MENU_ABOUT_ITEM(makeabout, NULL),
    GNOMEUIINFO_END
};

static GnomeUIInfo main_menu[] = {
  GNOMEUIINFO_MENU_FILE_TREE(file_menu),
  GNOMEUIINFO_MENU_VIEW_TREE(view_menu),
  GNOMEUIINFO_MENU_HELP_TREE(help_menu),
  GNOMEUIINFO_END
};


#else
static GtkItemFactoryEntry menu_items[] = {  
{ N_("/_File"),           NULL,          NULL,                               
  0, "<Branch>" },  
{ N_("/File/_Open"),      N_("<control>O"),  (GtkItemFactoryCallback)newfontdir,       
  0, NULL },  
{ N_("/File/_Print"),     N_("<control>P"),  (GtkItemFactoryCallback)printfonts,        
  0, NULL }, 
{ N_("/File/P_roperties"),N_("<control>R"),  (GtkItemFactoryCallback)show_properties,  
  0, NULL }, 
{ N_("/File/Save _Names"),N_("<control>N"),  (GtkItemFactoryCallback)save_names,  
  0, NULL }, 
{ N_("/File/sep1"),       NULL,          NULL,                            
  0, "<Separator>" },  
{ N_("/File/_Quit"),      N_("<control>Q"),  (GtkItemFactoryCallback)leave,            
  0, NULL },  
{ N_("/_View"),           NULL,          NULL,                               
  0, "<Branch>" },  
{ N_("/View/_Errors"),    NULL,          (GtkItemFactoryCallback)show_message_window, 
  0, NULL },
{ N_("/_Help"),           NULL,          NULL,                           
  0, "<LastBranch>" },  
{ N_("/Help/_About"),     NULL,          (GtkItemFactoryCallback)makeabout,        
  0, NULL }
};
#endif




/********************* Begin of program functions *****************/

#if 1
void info()
{
  Display *dp = GDK_DISPLAY();
  int screen = gdk_screen;

  printf("X%dR%d revision %d\n", ProtocolVersion(dp), XlibSpecificationRelease,
         ProtocolRevision(dp));
  printf("%s, vendor release %d\n", ServerVendor(dp), VendorRelease(dp));

  printf("Display: %s, screen %d, visual id %#lx, colormap id %#lx\n", DisplayString(dp), screen,
         XVisualIDFromVisual(GDK_VISUAL_XVISUAL(visual)), GDK_COLORMAP_XCOLORMAP(colormap));
  printf("Colormap cells: %d\n", colormap->size);
  printf("Depth of visual: %d planes\n", visual->depth);

  printf("Width:%d Height:%d\n", DisplayWidth(dp,screen), DisplayHeight(dp,screen));
  printf("WidthMM:%d HeightMM:%d\n", DisplayWidthMM(dp,screen), DisplayHeightMM(dp,screen));
  printf("Default root window id of display: %#lx\n", DefaultRootWindow(dp));

  printf("X %s support locale %s\n", (XSupportsLocale()==True)?"does":"does not",
         setlocale(LC_ALL, NULL));
  printf("Locale modifiers:%s\n", XSetLocaleModifiers(NULL));
}



void dumpGC(GdkGC* gc)
{
  XGCValues values;
  GC xgc = GDK_GC_XGC(gc);
  
  XGetGCValues(GDK_DISPLAY(), xgc, GCForeground | GCBackground, &values);
  printf("Foreground: %ld, Background:%ld\n", values.foreground, values.background);
}
#endif


#ifdef USE_GNOME
static struct poptOption options[] = {
  {"geometry", '\0', POPT_ARG_STRING, &geometry, 0,
    N_("Specify the geometry of the main window"),
    N_("GEOMETRY")
  },
  {0, '\0', 0, 0}              
}; 


/* Function called when session manager requests app to end */
static void
session_die(GnomeClient* client, gpointer client_data)
{
  leave(mainwindow, client_data);
  exit(0);
}


/* Called when session manager requests app to save state.
   We save the directory we were displaying
 */
static gint
save_session(GnomeClient *client, gint phase, GnomeSaveStyle save_style,
	     gint is_shutdown, GnomeInteractStyle interact_style,
	     gint is_fast, gpointer client_data)
{
  gchar** argv;
  guint argc;
  char *dir = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dirl)->entry));

  /* allocate 0-filled, so it will be NULL-terminated */
  argv = (gchar**)g_malloc0(sizeof(gchar*)*3);

  argv[0] = (gchar*)client_data;  /* save the directory we are in */
  argv[1] = dir;
  argc = 2;
  gnome_client_set_clone_command(client, argc, argv);
  gnome_client_set_restart_command(client, argc, argv);
  return TRUE;
}

#endif




/* Initialise all we need */
static void gfont_init(int argc, char** argv)
{
  struct stat buf;

  /* L10n */
  gtk_set_locale();
  bindtextdomain(PACKAGE, LOCALEDIR);   
  textdomain(PACKAGE);

  /* Init GTK and X11 stuff */
#ifdef USE_GNOME
  poptContext ctx;
  if (gnome_init_with_popt_table(PACKAGE, VERSION, argc, argv, 
				 options, 0, &ctx)) exit(1);
  poptSetOtherOptionHelp(ctx, "[OPTIONS] <directory>");

  mainwindow = gnome_app_new(PACKAGE, _("Font Viewer"));
  client = gnome_master_client();
  gtk_signal_connect(GTK_OBJECT(client), "save_yourself",
		     GTK_SIGNAL_FUNC(save_session), argv[0]);
  gtk_signal_connect(GTK_OBJECT(client), "die",
		     GTK_SIGNAL_FUNC(session_die), NULL);

  if (poptPeekArg(ctx) != NULL) firstdirectory = poptGetArg(ctx); 
  if (geometry != NULL) {
    gint x, y, w, h;
    if (gnome_parse_geometry(geometry, &x, &y, &w, &h)) {
      if (x != -1) 
	gtk_widget_set_uposition(mainwindow, x, y);
      if (w != -1) 
	gtk_window_set_default_size(GTK_WINDOW(mainwindow), w, h);
    }
    else {
      g_error(_("Could not parse geometry string `%s'"), geometry);
    }
  }
#else
  gtk_init(&argc, &argv);
  mainwindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(mainwindow), _("Font Viewer"));
  /* Check for filename or directory as first argument */
  if (argc > 1) firstdirectory = argv[1];   
#endif

  /* If a file was given as argument, display its directory instead */   
  if ((stat(firstdirectory, &buf) == 0) && S_ISREG(buf.st_mode))  
    firstdirectory = g_dirname(firstdirectory);

  gtk_rc_init();
  gtk_rc_parse("gfontviewrc");
  gtk_window_set_wmclass(GTK_WINDOW(mainwindow), PACKAGE, "FontViewer");
  gtk_widget_set_name(mainwindow, "main window");
  gtk_widget_ensure_style(mainwindow); 

  visual = gdk_visual_get_system();
  colormap = gdk_colormap_get_system();
  mCC = gdk_color_context_new(visual, colormap);
  if (visual->depth<8 || visual->depth==15) enable_antialiasing = FALSE;

  gdk_color_black(colormap, &black);
  gdk_color_white(colormap, &white);
 
  // Calculate screen resolution in dots per inch
  xresolution = (int)(0.5 + gdk_screen_width() / (gdk_screen_width_mm()/25.4));
  yresolution = (int)(0.5 + gdk_screen_height() / (gdk_screen_height_mm()/25.4));

  // Get selected basis font and make a bold version in stylebold
  // Have to access a private component for that...
  GdkFontPrivate *priv = (GdkFontPrivate *)mainwindow->style->font;
  const char *fnames = (const char*)priv->names->data;

  // It can be a fontset; get all components of it (separated by comma)
  gchar **fonts = g_strsplit(fnames, ",", -1);
  for (gint i=0; fonts[i]; i++) {
    // For all fonts in the fontset, find a bold version
    gint n=0;
    gchar **namecomps = g_strsplit(fonts[i], "-", 14);
    while (namecomps[n]) n++;  // Count elements in font definition
    if (n==15) {  // Only if all 15 are there, it is safe to change to bold
      g_free(namecomps[3]);
      namecomps[3] = g_strdup("bold");
      gchar *boldfont = g_strjoinv("-", namecomps);
      g_free(fonts[i]);
      fonts[i] = g_strdup(boldfont);
    }
    g_strfreev(namecomps);
  }
  // Make a complete fontset
  gchar *boldfonts = g_strjoinv(",", fonts);
  g_strfreev(fonts);

  // Attach the bold font or fontset to the stylebold style
  GdkFont *bold;
  stylebold = gtk_style_copy(mainwindow->style);
  switch (stylebold->font->type) {
  case GDK_FONT_FONT: 
    bold = gdk_font_load(boldfonts);
    break;
  case GDK_FONT_FONTSET:
    bold = gdk_fontset_load(boldfonts);
    break;
  }
  if (bold) {
    gdk_font_unref(stylebold->font);
    stylebold->font = bold;
  }

  /* Fill graypalette[] for antialiasing */
  char *list[] = {"white", "gray75", "gray50", "gray25", "black"};
  for (guint i=0; i<sizeof(list)/sizeof(char*); i++) {
    int failed = 1;
    if (gdk_color_parse(list[i], &graypalette[i]))
      graypalette[i].pixel = 
	gdk_color_context_get_pixel(mCC, graypalette[i].red, 
				    graypalette[i].green, 
				    graypalette[i].blue, &failed);
    if (failed) graypalette[i].pixel = black.pixel;
  }
  

  // Init T1lib
  T1_SetLogLevel(T1LOG_STATISTIC);
  T1_SetBitmapPad(16);
  T1_SetDeviceResolutions(xresolution, yresolution);
  if (T1_InitLib(NO_LOGFILE | IGNORE_FONTDATABASE | IGNORE_CONFIGFILE)==NULL){
    fprintf(stderr, _("Initialization of T1-library failed\n"));
    exit(1); 
  }
  if (enable_antialiasing) {
    if (T1_AASetBitsPerPixel(visual->depth) < 0) {
      fprintf(stderr, 
	      _("Cannot set T1-library antialiasing depth to %d: %s\n"), 
	      visual->depth, GetT1error(T1_errno));
      exit(1);
    }
    T1_AASetLevel(T1_AA_LOW);
    T1_AASetGrayValues(graypalette[0].pixel, graypalette[1].pixel, 
		       graypalette[2].pixel, graypalette[3].pixel, 
		       graypalette[4].pixel); 
  }
  
  // Init FreeType
  int error = TT_Init_FreeType(&ttengine);
  if (error) {
    fprintf(stderr, _("Could not create TT engine instance\n"));
    exit(1);
  }
  error = TT_Init_Kerning_Extension(ttengine);
  if (error) {
    fprintf(stderr, _("Could not init the kerning extension\n"));
    exit(1);
  }
  error = TT_Init_Post_Extension(ttengine);
  if (error) {
    fprintf(stderr, _("Could not init the PostScript extension\n"));
    exit(1);
  }
  if (enable_antialiasing) {
    // Gray palette: the values are indexes into graypalette[]
    guchar palette[5] = {0, 1, 2, 3, 4};  // white - light - medium - dark - black
    TT_Set_Raster_Gray_Palette(ttengine, palette);
  }
}





void delete_stringsw(GtkWidget*, gpointer data)
{
  FontData *fd = (FontData*)data;
  fd->refcount--;
  if (!fd->visible) destroyfont(fd);
}


void delete_image(GtkWidget*, gpointer data)
{
  GdkImage *image = (GdkImage*)data;
  //printf("Destroying image %p...\n", image);
  gdk_image_destroy(image);
}


void delete_pixmap(GtkWidget*, gpointer data)  // destroy callback
{
  GdkPixmap *pixmap = (GdkPixmap*)data;
  //printf("Destroying pixmap %p...\n", pixmap);
  gdk_pixmap_unref(pixmap);
}




/*
  Called when mouse key is released and the combo box was displayed
 */
static int dirl_release(GtkWidget *widget, GdkEvent *event, GtkCombo *combo) 
{
  /* If the popwin still has the grab, the click was not in the 
     list area, so ignore */
  if (!GTK_WIDGET_HAS_GRAB(combo->popwin)) {
    while (gtk_events_pending()) gtk_main_iteration(); 
    XSync(GDK_DISPLAY(), False);  // Get the combox box out of screen
    load_directory(combo->entry);
  }
  return FALSE;
}


/* 
   Called when user pressed ENTER in font directory entry combo
   or after user pressed OK in directory selection dialog
   It causes the list to be refreshed from the new directory
*/
static void load_directory(GtkWidget *entry)
{
  bool add_to_list = TRUE;  // Whether to add the dir to the combo list
  char *dir = gtk_entry_get_text(GTK_ENTRY(entry));
  if (*dir == '\0') {  // if empty string, do nothing
    return;
  }
  gdk_window_clear(samplet->window);  // Clear old font sample
  gtk_pixmap_set(GTK_PIXMAP(samplet), emptypixmap, NULL); 
  STATUSBAR_POP(stbar);
  if (fillfontlist(fontclistw, dir) == FALSE) return;
  // Only goes further if directory was correctly read

  // Go thru all list items of the list; if dir is already there, do not
  // add it again
  GList *list = GTK_LIST(GTK_COMBO(dirl)->list)->children; 
  while (list && list->data) {  // list->data is a list item
    gchar *ltext;
    GtkWidget *label = GTK_BIN(list->data)->child;
    if (!label || !GTK_IS_LABEL(label)) continue;
    gtk_label_get(GTK_LABEL(label), &ltext); 
    if (strcmp(dir, ltext) == 0) {
      add_to_list = FALSE;
      break;
    }
    list = list->next;
  }
  
  if (add_to_list) {
    /* Adding a new element to the list corrupts the string returned
       by gtk_entry_get_text; it cannot be accessed any more */
    GtkWidget *item = gtk_list_item_new_with_label(dir);
    gtk_widget_show(item); 
    /* Add new item to beginning of list */
    GList *iteml = g_list_alloc(); 
    iteml->data = item;
    gtk_list_prepend_items(GTK_LIST(GTK_COMBO(dirl)->list), iteml);
  }
}




void fonttable(GtkWidget *button, gpointer list)
{
  GList *selection = GTK_CLIST(list)->selection;
  if (selection == NULL) {
    errormsg(mainwindow, _("No row selected"));
    return;
  }
  
  double size = g_strtod(gtk_entry_get_text(GTK_ENTRY(fontsizeD->entry)), 
			 NULL);
  if (size<1.0 || size>1000.0) {
    errormsg(mainwindow, _("Invalid size"));
    return;
  }

  STATUSBAR_PUSH(stbar, _("Making font table..."));
  set_window_busy(mainwindow, TRUE);

  while (selection) {
    int row = GPOINTER_TO_INT(selection->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(list), row);
    
    switch (fd->fontType) {
    case T1FONT:
      t1_fonttable(fd, size);
      break;
    case TTFONT: 
      tt_fonttable(fd, size);
      break;
    }
    selection = selection->next;
  }
  STATUSBAR_POP(stbar);
  set_window_busy(mainwindow, FALSE);
}




/* 
   Gets the list of all directories with fonts under 'dirname'
   and appends it to 'list', giving the new list as result
 */
static GList* get_dirtree(GList *list, const char* dirname)
{  
  struct dirent *entry;
  DIR *dir;
  GString *file = g_string_new(NULL);
  bool dir_hasfonts = FALSE;
  
  dir = opendir(dirname);
  if (dir == NULL) return list;

  while ((entry=readdir(dir))) {
    struct stat sbuf;
    char *name = entry->d_name;
    size_t len = strlen(name);
    g_string_sprintf(file, "%s/%s", dirname, name);
    if (stat(file->str, &sbuf) != 0) continue;
  
    if (S_ISDIR(sbuf.st_mode) && 
	strcmp(name, ".")!=0 && 
	strcmp(name, "..")!=0) list = get_dirtree(list, file->str);
    if (dir_hasfonts==FALSE && S_ISREG(sbuf.st_mode)) {
      // Type 1
      if (len>4 && name[len-4] == '.' &&
	  (name[len-3] == 'p' || name[len-3] == 'P') &&
	  (name[len-2] == 'f' || name[len-2] == 'F') &&
	  (name[len-1] == 'a' || name[len-1] == 'A' ||
	   name[len-1] == 'b' || name[len-1] == 'B'))
	dir_hasfonts = TRUE;    
      // True Type
      if (len>4 && name[len-4] == '.' &&
	(name[len-3] == 't' || name[len-3] == 'T') &&
	(name[len-2] == 't' || name[len-2] == 'T') &&
	(name[len-1] == 'f' || name[len-1] == 'F' ||
	 name[len-1] == 'c' || name[len-1] == 'C'))
	dir_hasfonts = TRUE;
    }
  }    
  if (dir_hasfonts) list = g_list_append(list, g_strdup(dirname));
  closedir(dir);    
  g_string_free(file, TRUE);
  return list;
}




static GList* get_some_cool_dirs(void)
{
  GList *list = NULL;
  const char *dlist[] = {"/usr/share/ghostscript/fonts", "/usr/X11/lib/X11/fonts", "/usr/share/texmf/fonts", "/usr/share/fonts", "/usr/local/share/texmf/fonts", "/usr/local/fonts"};

  gchar *homefonts = g_strconcat(g_get_home_dir(), "/fonts", NULL);
  list = get_dirtree(list, homefonts);
  for (guint i=0; i<sizeof(dlist)/sizeof(char*); i++) 
    list = get_dirtree(list, dlist[i]);
  return list;
}



void toggle_button(GtkWidget*, gpointer data)
{
  bool *variable =(bool*)data;
  *variable = !*variable;  // Invert state of variable
}




static GtkWidget* makefontlist(void)
{
  char* titles[] = {"", N_("File name"), N_("Font name"),
		    N_("Glyphs"), N_("Kern pairs")};

  /* Translate titles */
  int n_titles = sizeof(titles)/sizeof(char*);
  for (int i=1; i<n_titles; i++)  
    titles[i] = _(titles[i]);

  /* Make window with clist */
  GtkWidget* scwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwindow), 
				 GTK_POLICY_ALWAYS, GTK_POLICY_AUTOMATIC);

  fontclistw = gtk_clist_new_with_titles(n_titles, titles);
  gtk_widget_ensure_style(fontclistw);
  gtk_clist_column_titles_passive(GTK_CLIST(fontclistw));
  gtk_clist_set_column_auto_resize(GTK_CLIST(fontclistw), 0, TRUE); 
  gtk_clist_set_column_width(GTK_CLIST(fontclistw), 1, 100);
  gtk_clist_set_column_width(GTK_CLIST(fontclistw), 2, 180);
  gtk_clist_set_selection_mode(GTK_CLIST(fontclistw), GTK_SELECTION_EXTENDED);
  gtk_clist_set_shadow_type(GTK_CLIST(fontclistw), GTK_SHADOW_ETCHED_OUT);
  gtk_container_border_width(GTK_CONTAINER(fontclistw), 0);

  gtk_signal_connect(GTK_OBJECT(fontclistw), "select_row",
		     GTK_SIGNAL_FUNC(select_fontlist), NULL);
  gtk_signal_connect(GTK_OBJECT(fontclistw), "unselect_row",
		     GTK_SIGNAL_FUNC(unselect_fontlist), NULL);

  gtk_signal_connect(GTK_OBJECT(fontclistw), "button_press_event",
		     GTK_SIGNAL_FUNC(fontlist_event), NULL);
  gtk_signal_connect(GTK_OBJECT(fontclistw), "button_release_event",
		     GTK_SIGNAL_FUNC(fontlist_event), NULL);

  /* Set height such that ca. 10 lines fit */
  gtk_widget_set_usize(fontclistw, -1, 10*(mainwindow->style->font->ascent+
					 mainwindow->style->font->descent));
  gtk_container_add(GTK_CONTAINER(scwindow), fontclistw);
  return scwindow;
}



// Destroy row in clist fontlist; called when displaying a new directory
// to free all structures related to fonts in the old directory
static void destroyrow(gpointer data)
{
  FontData *fd = (FontData*)data;
  fd->visible = FALSE; // No longer being displayed in list
  destroyfont(fd);
}


// Try to destroy a font if unused: free all memory related to its FontData
void destroyfont(FontData *fd)
{
  if (fd->refcount > 0) return;  // Still in use

  //printf("Destroying font %s...\n", fd->fontName);
  switch(fd->fontType) {
  case T1FONT: 
    T1_DeleteFont(fd->t1data.fontID);
    // If encoding is 'Custom x', free all components of encoding vector
    if (fd->t1data.encoding && 
	strcmp(fd->t1data.encoding[256], _("Custom")) == 0) {
      for (int i=0; i<=256; i++) g_free(fd->t1data.encoding[i]);
      g_free(fd->t1data.encoding);
    }
    break;
  case TTFONT:
    TT_Close_Face(fd->ttdata.face);  
    break;
  }
  g_free(fd->fontName);
  g_free(fd->fontFamily);
  g_free(fd->fontFile);
  g_free(fd);
}



/*
  Compare two font entries to determine sort order in the main list
 */
static gint compareEntries(gconstpointer first, gconstpointer second)
{
  FontData* fd1 = (FontData*)first;
  FontData* fd2 = (FontData*)second;
  
  // First Type 1 fonts, then TrueType
  if (fd1->fontType != fd2->fontType) 
    return (fd1->fontType==TTFONT);
  
  // Sort after family name
  int rc = strcmp(fd1->fontFamily, fd2->fontFamily);
  if (rc != 0) return rc;

  // If same family, sort after font name
  return strcmp(fd1->fontName, fd2->fontName);
}



/*
  As the name says, fills the list widget of fonts: it scans the directory,
  looks for fonts in it, creates the data structure FontData associated to
  each font, and then creates the rows of the widget. Each row has a FontData
  struct associated; when the list is destroyed, the function destroyrow will
  be called for each font.
  The FontData structs are stored in the linked list 'fontlist'

 */
static bool fillfontlist(GtkWidget* clist, char* directory)
{
  struct dirent *entry;
  struct stat buf;
  int nfiles=0, readfiles=0;
  static GtkWidget *t1pixmap = NULL;
  static GtkWidget *ttpixmap = NULL;

  g_slist_free(fontlist);
  fontlist = NULL;

  if (!t1pixmap) t1pixmap = makeicon(mainwindow, t1pixmap_data);
  if (!ttpixmap) ttpixmap = makeicon(mainwindow, ttpixmap_data);

  GString* fullname = g_string_new("");
  gtk_clist_clear(GTK_CLIST(clist));

  // Calculate number of entries in directory (variable nfiles)
  DIR* dir = opendir(directory);
  if (dir == NULL) {
    errormsg(mainwindow, _("Cannot open %s: %s"), 
	     directory, g_strerror(errno));
    return FALSE;
  }
  
  set_window_busy(mainwindow, TRUE);
  STATUSBAR_PUSH(stbar, _("Reading files..."));
  while (gtk_events_pending()) gtk_main_iteration();
  while((entry=readdir(dir))) nfiles++;
  rewinddir(dir);

  /* Read all files in directory and create 
     a sorted linked list (fontlist) with the valid ones */
  while((entry=readdir(dir))) {
    int fontID;
    char *fontname;

    gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 
			    double(++readfiles)/nfiles);
    while (gtk_events_pending()) gtk_main_iteration();

    g_string_sprintf(fullname, "%s/%s", directory, entry->d_name);
    if (stat(fullname->str, &buf)==0 && S_ISREG(buf.st_mode)) {
      FontData *fontdesc = NULL;
      if (fontdesc == NULL) {
	fontdesc = g_new(FontData, 1);
	fontdesc->refcount = 0;
	fontdesc->visible = TRUE;
	fontdesc->fontName = NULL;
      }
      fontdesc->fontFile = fullname->str;
      // File is either T1
      if (is_t1font(fullname->str, fontdesc)) {
	const char *afmpath[] = {"%s/%s", "%s/afm/%s"};
	fontlist = g_slist_insert_sorted(fontlist, fontdesc, compareEntries);
	fontdesc->fontFile = g_strdup(fullname->str);  
	
	// Look for an afm file
	int len = NAMLEN(entry);
	entry->d_name[len-3] = 'a'; 
	entry->d_name[len-2] = 'f';
	entry->d_name[len-1] = 'm';
	for (guint i=0; i<sizeof(afmpath)/sizeof(*afmpath); i++) {
	  g_string_sprintf(fullname, afmpath[i], directory, entry->d_name);
	  if (stat(fullname->str, &buf)==0 && S_ISREG(buf.st_mode)) {
	    if (!(buf.st_mode&S_IRUSR)) // If not readable, error
	      add_error(fontdesc, _("Cannot open afm file %s"), fullname->str);
	    else
	      T1_SetAfmFileName(fontdesc->t1data.fontID, fullname->str);
	    break;
	  }
	}	  
	fontdesc = NULL;
	continue;
      }
      // Or TT
      if (is_ttfont(fullname->str, fontdesc)) {
	fontlist = g_slist_insert_sorted(fontlist, fontdesc, compareEntries);
	fontdesc->fontFile = g_strdup(fullname->str);  
	fontdesc = NULL;
	continue;
      }
      // Or TT font collection
      if (is_ttfontcollection(fullname->str, fontdesc)) {
	int num = fontdesc->ttdata.properties.num_Faces;  // Number of fonts in collection
	TT_Close_Face(fontdesc->ttdata.face);
	for (int i=0; i<num; i++) {  // Make a list entry for each font in collection
	  FontData *fd = g_new(FontData, 1);
	  fd->fontFile = g_strdup(fullname->str);
	  int error = TT_Open_Collection(ttengine, fullname->str, i, 
					 &fd->ttdata.face);
	  if (error) {
	    add_error(fd, _("Cannot open font number %d in collection"), i);
	    g_free(fd->fontFile);
	    g_free(fd);
	    continue;
	  }
	  fill_ttfontdata(fd);
	  fontlist = g_slist_insert_sorted(fontlist, fd, compareEntries);
	}
	continue;
      }
      // Or unknown; so forget it
    }
  }
  closedir(dir);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 0.0);
  STATUSBAR_POP(stbar);
  while (gtk_events_pending()) gtk_main_iteration();

  // Put the linked list in the list widget
  gtk_clist_freeze(GTK_CLIST(clist));
  STATUSBAR_PUSH(stbar, _("Making entries..."));
  while (gtk_events_pending()) gtk_main_iteration();
  int row = 0;
  for (GSList* p=fontlist; p; p=p->next, row++) {
    gchar *rowentries[5];
    char buf[24], buf2[24];
    FontData* fd = (FontData*)p->data;
    rowentries[0] = NULL;
    rowentries[1] = g_basename(fd->fontFile);
    rowentries[2] = fd->fontName;
    if (fd->num_glyphs >= 0)
      sprintf(buf, "%u", fd->num_glyphs);
    else 
      strcpy(buf, "---");
    rowentries[3] = buf;
    if (fd->fontType == TTFONT) 
      sprintf(buf2, "%u", fd->ttdata.kernpairs);
    else
      strcpy(buf2, "---");
    rowentries[4] = buf2;

    gtk_clist_append(GTK_CLIST(clist), rowentries);
    if (fd->fontType == TTFONT)
      gtk_clist_set_pixmap(GTK_CLIST(clist), row, 0, 
			   GTK_PIXMAP(ttpixmap)->pixmap, 
			   GTK_PIXMAP(ttpixmap)->mask);
    else
      gtk_clist_set_pixmap(GTK_CLIST(clist), row, 0, 
			   GTK_PIXMAP(t1pixmap)->pixmap, 
			   GTK_PIXMAP(t1pixmap)->mask);
    
    // Each row has a FontData*
    gtk_clist_set_row_data_full(GTK_CLIST(clist), row, p->data, destroyrow);  
  }
  STATUSBAR_POP(stbar);
  gtk_clist_thaw(GTK_CLIST(clist));
  if (row > 0) {
    gtk_clist_select_row(GTK_CLIST(clist), 0, -1); // Select first row
    drawsample(clist, 0);
  }
  g_string_sprintf(fullname, _("%d fonts found"), row);
  STATUSBAR_PUSH(stbar, fullname->str);
  g_string_free(fullname, TRUE);

  set_window_busy(mainwindow, FALSE);
  return TRUE;
}



// The user selected to display a new directory
static void okfontdir(gpointer diag)
{
  GtkWidget* filesel = GTK_WIDGET(diag);
  char *newdir = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(dirl)->entry), newdir);
  gtk_widget_destroy(filesel);

  load_directory(GTK_COMBO(dirl)->entry);
}



// Selection window to choose a new directory
void newfontdir(GtkWidget*, gpointer)
{
  set_window_busy(mainwindow, TRUE);

  GtkWidget* filesel = 
    gtk_file_selection_new(_("Font Viewer directory selection"));
  gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(filesel));
  gtk_widget_set_sensitive(GTK_FILE_SELECTION(filesel)->file_list, FALSE);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), 
			    GTK_OBJECT(filesel));
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(okfontdir), 
			    GTK_OBJECT(filesel));
  char *actualdir = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dirl)->entry));
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel), actualdir);
  gtk_window_position(GTK_WINDOW(filesel), GTK_WIN_POS_CENTER);
  gtk_window_set_modal(GTK_WINDOW(filesel), TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(filesel), GTK_WINDOW(mainwindow));
  gtk_widget_show(filesel);
  set_window_busy(mainwindow, FALSE);
}



/* This function is called when the cursor is on the entry field 
   during a drag&drop operation */
void
combo_drag_data_received(GtkWidget *entry, GdkDragContext *context,
			 gint x, gint y, GtkSelectionData *data,
			 guint info, guint time)
{
  if ((data->length >= 0) && (data->format == 8)) { 
    char *src = (gchar*)data->data;
    char *p = src-1;
    int lines = 0;

    while ((p=strchr(p+1, '\n'))) lines++;
    if (src[strlen(src)-1] != '\n') lines++;
    if (lines > 1) {
      gtk_drag_finish(context, FALSE, FALSE, time);
      errormsg(mainwindow, _("Only one folder can be dropped"));
      return;
    }

    /* If we get a valid URL (starting with file:, to make sure the files 
       are local) load the given directory into the list */
    if (strncmp(src, "file:", 5) == 0) {
      g_strchomp(src);
      gtk_entry_set_text(GTK_ENTRY(entry), src+5);
      gtk_drag_finish(context, TRUE, FALSE, time);
      if (entry == GTK_COMBO(dirl)->entry) load_directory(entry);
      return;
    }
    else 
      errormsg(mainwindow, _("Only local folders can be dropped"));
  }
  gtk_drag_finish(context, FALSE, FALSE, time);
}                      



gint delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  STATUSBAR_PUSH(stbar, _("Exiting..."));
  while (gtk_events_pending()) gtk_main_iteration();
  return FALSE; // Send a destroy event
}


// Exit the program
void leave(GtkWidget*, gpointer data)
{
  T1_CloseLib();
  TT_Done_FreeType(ttengine);
  gtk_exit(0);
}





int main(int argc, char** argv)
{  
  gfont_init(argc, argv);


  //******** Main window
                                     
  gtk_container_border_width(GTK_CONTAINER(mainwindow), 0);
  gtk_window_set_policy(GTK_WINDOW(mainwindow), FALSE, TRUE, FALSE); /* shrink, grow, auto_shrink */
  gtk_widget_realize(mainwindow);	 
  gtk_signal_connect(GTK_OBJECT(mainwindow), "destroy", 
		     GTK_SIGNAL_FUNC(leave), NULL);
  gtk_signal_connect(GTK_OBJECT(mainwindow), "delete_event", 
		     GTK_SIGNAL_FUNC(delete_event), NULL);

  // Global vbox
  GtkWidget *vboxglobal = gtk_vbox_new(FALSE, 0);
#ifdef USE_GNOME
  gnome_app_set_contents(GNOME_APP(mainwindow), vboxglobal);
#else
  gtk_container_add(GTK_CONTAINER(mainwindow), vboxglobal);
#endif

  /* Menubar*/
#ifdef USE_GNOME
  gnome_app_create_menus(GNOME_APP(mainwindow), main_menu);
#else
  int i;
  gint nmenu_items = sizeof(menu_items) / sizeof (menu_items[0]);
  for (i=0; i<nmenu_items; i++) // Translate menu labels
    menu_items[i].path = _(menu_items[i].path);
  for (i=0; i<nmenu_items; i++) // Translate menu shortcuts
    if (menu_items[i].accelerator)  // Might be NULL...
      menu_items[i].accelerator = _(menu_items[i].accelerator);

  GtkAccelGroup *accel_group = gtk_accel_group_new();
  GtkItemFactory *item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, 
						      "<main>", accel_group);
  gtk_item_factory_create_items(item_factory, nmenu_items, menu_items, NULL);
  GtkWidget *menubar = gtk_item_factory_get_widget(item_factory, "<main>");
  gtk_accel_group_attach(accel_group, GTK_OBJECT(mainwindow));
  gtk_box_pack_start(GTK_BOX(vboxglobal), menubar, FALSE, FALSE, 0);
#endif

  // Main vertical container
  GtkWidget* vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_border_width(GTK_CONTAINER(vbox), 10);
  gtk_box_pack_start(GTK_BOX(vboxglobal), vbox, TRUE, TRUE, 0);


  //****** Current directory list combo
  GtkWidget* hboxl = gtk_hbox_new(FALSE, 10);
  gtk_box_pack_start(GTK_BOX(vbox), hboxl, FALSE, FALSE, 5);

  GtkWidget* dirb = gtk_button_new();
  GtkWidget* icondir = makeicon(mainwindow, miniofolder);
  gtk_container_add(GTK_CONTAINER(dirb), icondir);
  gtk_signal_connect(GTK_OBJECT(dirb), "clicked", 
		     GTK_SIGNAL_FUNC(newfontdir), NULL);
  gtk_box_pack_start(GTK_BOX(hboxl), dirb, FALSE, FALSE, 0);

  dirl = gtk_combo_new();
  GtkWidget *entry = GTK_COMBO(dirl)->entry;
  gtk_combo_disable_activate(GTK_COMBO(dirl)); // Do not popup list when ENTER
  gtk_combo_set_use_arrows_always(GTK_COMBO(dirl), TRUE);
  gtk_box_pack_start(GTK_BOX(hboxl), dirl, TRUE, TRUE, 0);
  // Reload file list when user presses ENTER
  gtk_signal_connect_object(GTK_OBJECT(entry), "activate", 
			    (GtkSignalFunc)load_directory, GTK_OBJECT(entry));
  // Also when user releases mouse button in combo box list
  gtk_signal_connect_after(GTK_OBJECT(GTK_COMBO(dirl)->list), 
			   "button_release_event", 
			   (GtkSignalFunc)dirl_release, dirl);

  // Entry field of the combo box is DnD target
  gtk_drag_dest_set(entry, GTK_DEST_DEFAULT_ALL,
		    dnd_target_table, dnd_targets-1,
		    (enum GdkDragAction)(GDK_ACTION_COPY | GDK_ACTION_MOVE)); 
  gtk_signal_connect(GTK_OBJECT(entry), "drag_data_received",
		     GTK_SIGNAL_FUNC(combo_drag_data_received), NULL);

  // Put some default dirs into the list
  GList *ilist = get_some_cool_dirs();
  if (ilist) {
    gtk_combo_set_popdown_strings(GTK_COMBO(dirl), ilist);
    g_list_free(ilist);
  }
  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(dirl)->entry), firstdirectory); 


  // Make the list of fonts
  GtkWidget* flist = makefontlist();
  gtk_box_pack_start(GTK_BOX(vbox), flist, TRUE, TRUE, 0);

  //******** Sample text
  // Use a viewport to clip pixmap
  GtkWidget* viewport = gtk_viewport_new(NULL, NULL);
  gtk_viewport_set_shadow_type(GTK_VIEWPORT(viewport), GTK_SHADOW_IN);
  gtk_box_pack_start(GTK_BOX(vbox), viewport, FALSE, FALSE, 0);

  GdkBitmap* mask;
  GtkStyle* style = gtk_widget_get_style(mainwindow);
  emptypixmap = gdk_pixmap_create_from_xpm_d(mainwindow->window, &mask,
					     &style->bg[GTK_STATE_NORMAL], 
					     emptypixmap_d);
  samplet = gtk_pixmap_new(emptypixmap, mask);
  gtk_container_add(GTK_CONTAINER(viewport), samplet);
  int ypixels = (int)(1.4*18*(yresolution/72.0));
  gtk_widget_set_usize(samplet->parent, 1, ypixels);


  //******** Input entries
  fontsizeD = make_entry_dialog(_("Font Size:"), "24");
  gtk_box_pack_start(GTK_BOX(vbox), fontsizeD->hbox, FALSE, FALSE, 0);

  GtkWidget* hboxe = gtk_hbox_new(TRUE, 10);
  gtk_container_border_width(GTK_CONTAINER(hboxe), 0);
  gtk_box_pack_start(GTK_BOX(vbox), hboxe, FALSE, FALSE, 0);

  characterD = make_entry_dialog(_("Character:"), "65");
  gtk_box_pack_start(GTK_BOX(hboxe), characterD->hbox, TRUE, TRUE, 0);

  stringD = make_entry_dialog(_("Test String:"), _("Test"));
  gtk_widget_set_usize(stringD->entry, 100, -1);
  gtk_box_pack_start(GTK_BOX(hboxe), stringD->hbox, TRUE, TRUE, 0);


  //********** Toggle buttons
  GtkWidget* hboxt = gtk_hbox_new(TRUE, 0);
  gtk_container_border_width(GTK_CONTAINER(hboxt), 3);
  gtk_box_pack_start(GTK_BOX(vbox), hboxt, FALSE, FALSE, 0);

  GtkWidget* aabutton = gtk_check_button_new_with_label(_("Antialias"));
  gtk_box_pack_start(GTK_BOX(hboxt), aabutton, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(aabutton), "toggled", 
		     GTK_SIGNAL_FUNC(toggle_button), &antialiasing);
  if (!enable_antialiasing) gtk_widget_set_sensitive(aabutton, FALSE);

  GtkWidget* kernbutton = gtk_check_button_new_with_label(_("Kerning"));
  gtk_box_pack_start(GTK_BOX(hboxt), kernbutton, TRUE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(kernbutton), "toggled", 
		     GTK_SIGNAL_FUNC(toggle_button), &kerning);


  //********** Separator
  GtkWidget* sep1 = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(vbox), sep1, FALSE, FALSE, 0);


  //******** Action buttons
  GtkWidget* hboxa = gtk_hbox_new(TRUE, 0);
  gtk_container_border_width(GTK_CONTAINER(hboxa), 0);
  gtk_box_pack_start(GTK_BOX(vbox), hboxa, FALSE, FALSE, 0);

  GtkWidget* charb = gtk_button_new_with_label(_("Make character"));
  gtk_object_set_data(GTK_OBJECT(charb), "makechar", GINT_TO_POINTER(1)); 
  gtk_signal_connect(GTK_OBJECT(charb), "clicked", 
		     GTK_SIGNAL_FUNC(show_char_or_string), fontclistw);
  gtk_box_pack_start(GTK_BOX(hboxa), charb, TRUE, TRUE, 0);

  GtkWidget* stringb = gtk_button_new_with_label(_("Make string"));
  gtk_object_set_data(GTK_OBJECT(stringb), "makechar", GINT_TO_POINTER(0)); 
  gtk_signal_connect(GTK_OBJECT(stringb), "clicked", 
		     GTK_SIGNAL_FUNC(show_char_or_string), fontclistw);
  gtk_box_pack_start(GTK_BOX(hboxa), stringb, TRUE, TRUE, 0);

  GtkWidget* tableb = gtk_button_new_with_label(_("Make font table"));
  gtk_signal_connect(GTK_OBJECT(tableb), "clicked", 
		     GTK_SIGNAL_FUNC(fonttable), fontclistw);
  gtk_box_pack_start(GTK_BOX(hboxa), tableb, TRUE, TRUE, 0);

  //********** Message and progress area  -- Last in window
#ifdef USE_GNOME
  GtkWidget *appbar = gnome_appbar_new(TRUE, TRUE, GNOME_PREFERENCES_USER);
  gnome_app_set_statusbar(GNOME_APP(mainwindow), appbar);
  gnome_app_install_menu_hints(GNOME_APP(mainwindow), main_menu);
  prbar = (GtkWidget*)gnome_appbar_get_progress(GNOME_APPBAR(appbar));
  stbar = appbar;
#else
  GtkWidget* hboxm = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_end(GTK_BOX(vbox), hboxm, FALSE, FALSE, 0);
  stbar = gtk_statusbar_new();
  gtk_box_pack_start(GTK_BOX(hboxm), stbar, TRUE, TRUE, 0);
  prbar = gtk_progress_bar_new();
  gtk_box_pack_end(GTK_BOX(hboxm), prbar, FALSE, FALSE, 0);
#endif
  gtk_widget_set_usize(prbar, 100, -1);

  
  //********** Final steps
  gtk_widget_show_all(mainwindow);
 
  // define GCs 
  GdkGCValues gcval;
  int gcmask;

  // defGC: fg is black, bg is the default window bg (possibly gray)
  defGC = gdk_gc_new(mainwindow->window); 
  gdk_gc_set_foreground(defGC, &black);
  gdk_gc_set_background(defGC, &style->bg[GTK_STATE_NORMAL]);
  // Set the bg colour of black_gc in the default style to white
  gdk_gc_set_background(style->black_gc, &white);

  // bwGC: black & white GC, with stippled fill pattern
  gcmask = GDK_GC_FUNCTION | GDK_GC_FOREGROUND | GDK_GC_BACKGROUND | 
    GDK_GC_FILL | GDK_GC_STIPPLE;

  gcval.function = GDK_COPY;
  gcval.foreground.pixel = black.pixel;
  gcval.background.pixel = white.pixel;
  gcval.fill = GDK_STIPPLED;
  gcval.stipple = gdk_bitmap_create_from_data(mainwindow->window, 
					      stipple_bits, 
					      stipple_width, stipple_height);
  bwGC = gdk_gc_new_with_values(mainwindow->window, &gcval,
				(GdkGCValuesMask)gcmask);
   
  // blueGC: blue & white GC, dashed lines
  GdkColor blue;
  gdk_color_parse("blue", &blue);
  gdk_colormap_alloc_color(colormap, &blue, FALSE, TRUE);
  gcmask = GDK_GC_FUNCTION | GDK_GC_FOREGROUND | GDK_GC_BACKGROUND | 
    GDK_GC_FILL;

  gcval.function = GDK_COPY;
  gcval.foreground.pixel = blue.pixel;
  gcval.background.pixel = white.pixel;
  gcval.fill = GDK_SOLID;
  blueGC = gdk_gc_new_with_values(mainwindow->window, &gcval,
				  (GdkGCValuesMask)gcmask);
  gdk_gc_set_line_attributes(blueGC, 0, GDK_LINE_ON_OFF_DASH, 
			     GDK_CAP_BUTT, GDK_JOIN_ROUND);
			   
  // redGC: red & white GC, with stippled fill pattern
  GdkColor red;
  gdk_color_parse("red", &red);
  gdk_colormap_alloc_color(colormap, &red, FALSE, TRUE);
  redGC = gdk_gc_new(mainwindow->window); 
  gdk_gc_copy(redGC, bwGC);
  gdk_gc_set_foreground(redGC, &red);
   
  GtkWidget *windowicon = makeicon(mainwindow, font_xpm);
  gdk_window_set_icon(mainwindow->window, NULL, 
		      GTK_PIXMAP(windowicon)->pixmap, NULL);

  make_message_window();  // The async messages
  while (gtk_events_pending()) gtk_main_iteration();
  load_directory(entry);

  // Geometry hints
  GdkGeometry geometry;
  GdkWindowHints geometry_mask; 

  geometry_mask = GDK_HINT_MIN_SIZE;
  geometry.min_width = 350;
  geometry.min_height = 300;

  gtk_window_set_geometry_hints(GTK_WINDOW(mainwindow), NULL, 
				&geometry, geometry_mask); 

  gtk_main();
  return 1;  // Should not reach
}
