
import glob
import os
import re
import urllib
import shutil
from xml.dom.minidom import parse
from gettext import gettext as _

import gobject
import gtk
import gnome
import gnomevfs
import wnck

from gimmie_globals import gimmie_is_panel_applet
from gimmie_base import Item, ItemSource, DisabledItemSource, Topic, gimmie_get_topic_for_uri
from gimmie_file import FileItem
from gimmie_recent import RecentlyUsed, RecentlyUsedOfMimeType
from gimmie_util import *
from gimmie_trash import trash_monitor
from gimmie_computer import DriveItem
from gimmie_tomboy import TomboySource


class RecentAggregate(ItemSource):
    '''
    This ItemSource subclass aggregates all the items from a list of
    ItemSources, by including the first Item encountered of a URI and
    filtering duplicates.
    '''
    def __init__(self, sources):
        ItemSource.__init__(self, name=_("Recently Used"), icon="stock_calendar")

        # Sources provide the real items we will display
        self.sources = [x for x in sources if x]
        for source in self.sources:
            self._listen_to_source(source)

    def _listen_to_source(self, source):
        source.connect("reload", lambda x: self.emit("reload"))

    def get_items(self):
        item_uris = {}

        # Find items matching recent uris
        for source in self.sources:
            for item in source.get_items():
                uri = item.get_uri()
                # Block special items the source might include like "Create
                # New Note" and avoid duplicate items for the same URI.
                if not item.is_special() and uri and uri not in item_uris:
                    item_uris[uri] = item

        items = item_uris.values()

        return items

    def do_reload(self):
        pass # Do nothing.


class DirectorySource(ItemSource):
    def __init__(self, path, icon = "stock_folder"):
        self.path = os.path.expanduser(path)
        dirname = os.path.basename(self.path)
        ItemSource.__init__(self, dirname, icon=icon)

        self.monitor = FileMonitor(self.path)
        self.monitor.connect("changed", lambda x, file: self.emit("reload"))
        self.monitor.open()

    def get_enabled(self):
        return os.path.exists(self.path)

    def add_name(self, paths, dirname, names):
        for name in names:
            paths.append(os.path.join(dirname, name))

    def do_reload(self):
        paths = []
        os.path.walk(self.path, self.add_name, paths)
        self.set_items([FileItem(x) for x in paths])


class DownloadsSource(DirectorySource):
    def __init__(self):
        if os.environ.has_key("DOWNLOAD_PATH"):
            path = os.environ["DOWNLOAD_PATH"]
        else:
            path = "~/Downloads"
        DirectorySource.__init__(self, path, icon="stock_internet")

        self.download_rdf_monitors = {}
        self.do_reload()

    def _get_download_rdf_paths(self):
        return glob.glob(os.path.expanduser("~/.mozilla/*/*.default/downloads.rdf"))

    def get_enabled(self):
        return DirectorySource.get_enabled(self) or self._get_download_rdf_paths() != []

    def do_reload(self):
        DirectorySource.do_reload(self)
        dir_items = DirectorySource.get_items(self)
        
        for download_rdf in self._get_download_rdf_paths():
            if not self.download_rdf_monitors.has_key(download_rdf):
                # Emit reload on download.rdf change
                mon = FileMonitor(download_rdf)
                mon.connect("changed", lambda x, file: self.emit("reload"))
                mon.open()
                self.download_rdf_monitors[download_rdf] = mon

            try:
                # Read previous download locations
                download_doc = parse(download_rdf)
                li_list = download_doc.getElementsByTagName("RDF:li")

                for li_elem in li_list:
                    uri = str(li_elem.getAttribute("RDF:resource"))
                    # Sometimes contains absolute paths instead of URLs
                    if os.path.isabs(uri):
                        uri = "file://" + uri
                    dir_items.append(FileItem(uri))
            except IOError, (errno, strerror):
                print " !!! Error parsing Mozilla %s file: %s" % (download_rdf, strerror)

        self.set_items(dir_items)


class AddSmartFolder(gtk.Dialog):
    '''
    Dialog to choose a stock Smart Folder to add to the sidebar,
    possibly prompting for some info (e.g. Tag names,
    modification date, etc).
    '''
    def __init__(self, parent):
        gtk.Dialog.__init__(self,
                            _("New Smart Folder"),
                            parent,
                            gtk.DIALOG_DESTROY_WITH_PARENT | gtk.DIALOG_NO_SEPARATOR,
                            (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                             gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))

    def get_sidebar_source(self):
        return None


class NewFromTemplateDialog(gtk.FileChooserDialog):
    '''
    Dialog to create a new document from a template
    '''
    __gsignals__ = {
        "response" : "override"
        }

    def __init__(self, parent, name, source_uri):
        # Extract the template's file extension
        try:
            self.file_extension = name[name.rindex('.'):]
            name = name[:name.rindex('.')]
        except ValueError:
            self.file_extension = None
        self.source_uri = source_uri

        gtk.FileChooserDialog.__init__(self,
                                       _("New Document"),
                                       parent,
                                       gtk.FILE_CHOOSER_ACTION_SAVE,
                                       (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                        gtk.STOCK_SAVE, gtk.RESPONSE_ACCEPT))
        self.set_current_name(name)
        self.set_current_folder(os.path.expanduser("~/Desktop"))
        self.set_do_overwrite_confirmation(True)
        self.set_default_response(gtk.RESPONSE_ACCEPT)

    def do_response(self, response):
        if response == gtk.RESPONSE_ACCEPT:
            file_uri = self.get_filename()

            # Add the file extension back unless the user did themselves
            if self.file_extension and not file_uri.endswith(self.file_extension):
                file_uri = "%s%s" % (self.source_uri, self.file_extension)

            # Create a new document from the template and display it
            try:
                if not self.source_uri:
                    # Create an empty file
                    f = open(file_uri, 'w')
                    f.close()
                else:
                    shutil.copyfile(self.source_uri, file_uri)
                launcher.launch_uri(file_uri)
            except IOError:
                pass

        self.destroy()


class NewDocumentMenuButton(ToolMenuButton):
    def __init__(self, tooltips):
        img = icon_factory.load_image("stock_new-template", gtk.ICON_SIZE_LARGE_TOOLBAR)

        ToolMenuButton.__init__(self, img, _("New Document"))
        self.set_tooltip(tooltips, _("Create a new document"))
        
        # Monitor the templates directory
        self.template_monitor = FileMonitor(os.path.expanduser("~/Templates"))
        self.template_monitor.connect("created", self._template_changed)
        self.template_monitor.connect("deleted", self._template_changed)
        self.template_monitor.open()
        self._template_changed(None, None)

    def _template_changed(self, monitor, uri):
        menu = gtk.Menu()
        self._add_templates(menu)
        self.set_menu(menu)

    def _add_template_item(self, menu, name, uri, icon_name):
        menu_item = gtk.ImageMenuItem(name)
        menu_item.set_image(icon_factory.load_image(icon_name, gtk.ICON_SIZE_LARGE_TOOLBAR))
        menu_item.show()
        menu_item.connect("activate", self._show_new_from_template_dialog, name, uri)
        return menu_item

    def _add_templates(self, menu):
        # Get all the templates
        template_dir = os.path.expanduser("~/Templates")
	try:
        	templates = os.listdir(template_dir)
	except:
		templates = []

        for template in templates:
            item = FileItem(os.path.join(template_dir, template))
            menu.append(self._add_template_item(menu,
                                                item.get_name(),
                                                item.get_uri(),
                                                item.get_icon(gtk.ICON_SIZE_LARGE_TOOLBAR)))
        if len(templates) == 0:
            empty_item = gtk.MenuItem(_("No templates available"))
            empty_item.set_sensitive(False)
            empty_item.show()
            menu.append(empty_item)

        sep = gtk.SeparatorMenuItem()
        sep.show()
        menu.append(sep)

        menu.append(self._add_template_item(menu, _("Empty File"), None, "stock_new"))

        sep = gtk.SeparatorMenuItem()
        sep.show()
        menu.append(sep)

        # Add a link to the templates directory
        mi = gtk.ImageMenuItem(_("Templates"))
        mi.set_image(icon_factory.load_image("gnome-fs-directory", gtk.ICON_SIZE_LARGE_TOOLBAR))
        mi.show()
        mi.connect("activate", lambda mi, uri: launcher.launch_uri(uri), template_dir)
        menu.append(mi)

    def _show_new_from_template_dialog(self, b, name, uri):        
        dlg = NewFromTemplateDialog(b.get_toplevel(), name, uri)
        dlg.show()


class DocumentsTopic(Topic):
    '''
    Lists recently opened/edited documents from ~/.recently-used, and some other
    categories of files such as movies and music.  Placeholders for recent email
    attachments, marked emails, and browser downloads.  In the future categories
    displayed should be pluggable.
    '''
    def __init__(self):
        Topic.__init__(self, _("Library"), uri="topic://Documents")

        from gimmie_running import RunningDocuments
        self.set_running_source_factory(lambda: RunningDocuments())

        # Tomboy loads notes in a worker thread, so begin it early.
        self.tomboy_source = TomboySource()

    def do_reload(self):
        source_list = [None,
                       RecentlyUsedOfMimeType(
                           _("Documents"),
                           "stock_new-presentation",
                           [  # Covers:
                              #   vnd.corel-draw
                              #   vnd.ms-powerpoint
                              #   vnd.ms-excel
                              #   vnd.oasis.opendocument.*
                              #   vnd.stardivision.*
                              #   vnd.sun.xml.*
                            re.compile("application/vnd.*"),
                              # Covers: x-applix-word, x-applix-spreadsheet, x-applix-presents
                            re.compile("application/x-applix-*"),
                              # Covers: x-kword, x-kspread, x-kpresenter, x-killustrator
                            re.compile("application/x-k(word|spread|presenter|illustrator)"),
                            re.compile("text/*"),
                            re.compile("image/*"),
                            "application/ms-powerpoint",
                            "application/msword",
                            "application/pdf",
                            "application/postscript",
                            "application/ps",
                            "application/rtf",
                            "application/x-abiword",
                            "application/x-asp",
                            "application/x-bittorrent",
                            "application/x-blender",
                            "application/x-cgi",
                            "application/x-dia-diagram",
                            "application/x-dvi",
                            "application/x-glade",
                            "application/x-gnucash",
                            "application/x-gnumeric",
                            "application/x-iso-image",
                            "application/x-jbuilder-project",
                            "application/x-magicpoint",
                            "application/x-mrproject",
                            "application/x-php",
                           ]),
                       self.tomboy_source, # Tomboy
                       RecentlyUsedOfMimeType(_("Music & Movies"),
                                              "gnome-mime-video",
                                              [re.compile("video/*"),
                                               re.compile("audio/*"),
                                               "application/ogg"]),
                       DisabledItemSource(_("F-Spot Photos"), "f-spot-logo"),
                       None,
                       DownloadsSource(),
                       DisabledItemSource(_("Attachments"), "stock_attach"),
                       ]
        
        source_list.insert(0, RecentAggregate(source_list))
        
        self.set_sidebar_source_list(source_list)

    def get_hint_color(self):
        return gtk.gdk.color_parse("lightgreen")

    def _add_place_item(self,
                        menu,
                        name,
                        uri,
                        icon_name = "gnome-fs-directory"):
        menu_item = gtk.ImageMenuItem(name)
        menu_item.set_image(icon_factory.load_image(icon_name, gtk.ICON_SIZE_LARGE_TOOLBAR))
        menu_item.show()
        menu_item.connect("activate", lambda mi, uri: launcher.launch_uri(uri), uri)
        return menu_item

    def _add_places(self, menu):
        # Include all the mounted drives
        computer = gimmie_get_topic_for_uri("topic://Computer")
        device_source = computer.get_source_for_uri("source://Devices")
        for item in device_source.get_items():
            if isinstance(item, DriveItem) and item.get_is_mounted():
                menu.append(self._add_place_item(menu,
                                                 item.get_name(),
                                                 item.get_uri(),
                                                 item.get_icon(gtk.ICON_SIZE_LARGE_TOOLBAR)))

        sep = gtk.SeparatorMenuItem()
        sep.show()
        menu.append(sep)

        # Include all the Gtk bookmarks
        for uri, name, mime, icon in places.get_places():
            menu.append(self._add_place_item(menu, name, uri, icon))

        sep = gtk.SeparatorMenuItem()
        sep.show()
        menu.append(sep)

        # Add the trash
        if trash_monitor.is_empty():
            menu.append(self._add_place_item(menu, _("Trash"), "trash://", "gnome-fs-trash-empty"))
        else:
            menu.append(self._add_place_item(menu, _("Trash"), "trash://", "gnome-fs-trash-full"))

    def _show_add_smart_folder_dialog(self, b):
        dlg = AddSmartFolder(b.get_toplevel())
        if dlg.run() == gtk.RESPONSE_ACCEPT:
            source = dlg.get_sidebar_source()
            ### FIXME: Add to sources sidebar
            print "Adding!"
        dlg.destroy()

    def get_toolbar_items(self, tooltips):
        tools = []

        btn = NewDocumentMenuButton(tooltips)
        btn.set_is_important(True)
        tools.append(btn)

        '''
        img = icon_factory.load_image("stock_new-dir", gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ToolButton(img, _("New Smart Folder"))
        btn.set_tooltip(tooltips, _("Add a new smart folder to the sidebar"))
        btn.set_is_important(True)
        btn.connect("clicked", self._show_add_smart_folder_dialog)
        tools.append(btn)
        '''

        img = icon_factory.load_image("gnome-fs-bookmark", gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = ToolMenuButton(img, _("Folders"))
        btn.set_tooltip(tooltips, _("Open commonly used document locations"))
        btn.set_is_important(True)
        menu = gtk.Menu()
        self._add_places(menu)
        btn.set_menu(menu)
        tools.append(btn)

        ### FIXME: Include this?
        '''
        img = icon_factory.load_image("gnome-fs-trash-full", gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ToolButton(img, _("Trash"))
        btn.set_tooltip(tooltips, _("Open the deleted files trashcan"))
        btn.set_is_important(True)
        btn.connect("clicked", lambda x: launcher.launch_uri("trash://"))
        tools.append(btn)
        '''

        if not gimmie_is_panel_applet():
            tools.append(None)

            img = icon_factory.load_image(gtk.STOCK_PRINT, gtk.ICON_SIZE_LARGE_TOOLBAR)
            btn = gtk.ToolButton(img, _("Printers"))
            btn.set_tooltip(tooltips, _("View printers available for document printing"))
            btn.set_is_important(True)
            btn.connect("clicked", lambda x: self._open_printers())
            tools.append(btn)

        return tools

    def _open_printers(self):
        topic = gimmie_get_topic_for_uri("topic://Computer")
        topicwin = topic.get_topic_window()
        topicwin.set_source_by_uri("source://Printers")
        topicwin.present()

    def get_context_menu_items(self):
        items = []

        img = icon_factory.load_image("stock_new-template", gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ImageMenuItem(_("New Document"))
        btn.set_image(img)
        menu = gtk.Menu()
        self._add_templates(menu)
        btn.set_submenu(menu)
        btn.show_all()
        items.append(btn)

        img = icon_factory.load_image("gnome-fs-bookmark", gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ImageMenuItem(_("Go"))
        btn.set_image(img)
        menu = gtk.Menu()
        self._add_places(menu)
        btn.set_submenu(menu)
        btn.show_all()
        items.append(btn)

        items.append(None)

        img = icon_factory.load_image(gtk.STOCK_PRINT, gtk.ICON_SIZE_LARGE_TOOLBAR)
        btn = gtk.ImageMenuItem(_("Printers"))
        btn.set_image(img)
        btn.connect("activate", lambda x: self._open_printers())
        btn.show_all()
        items.append(btn)
        
        return items
