/*
    Glurp - A GTK+ client for Music Player Daemon
    Copyright (C) 2004, 2005 Andrej Kacian

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    http://musicpd.org/glurp.shtml

*/

#include <gtk/gtk.h>

#include "structs.h"
#include "support.h"
#include "player.h"
#include "comm.h"

extern GlurpState *glurp;

void player_play_song(gint id) {
  mpd_Status *status;

  debug("PLAY requested");
  if(!glurp->conn) {
    debug("we're not connected, ignoring");
    return;
  }

  if( !(status = get_status(TRUE)) ) {
    debug("could not get mpd status");
    return;
  }

  if(!glurp->playlist) {
    debug("No songs loaded");
    statusbar_print("Playlist empty, cannot play");
    return;
  }

  if(status->state == MPD_STATUS_STATE_PLAY) {
    debug("mpd is already playing a song");
    statusbar_print("Already playing");
    return;
  }


  debug("Playing song with id %d", id);
  mpd_sendPlayIdCommand(glurp->conn, id);
  mpd_finishCommand(glurp->conn);

  if( check_mpd_error() ) {
    debug("Can't start playback");
    return;
  }

  glurp->play_state = MPD_STATUS_STATE_PLAY;

  statusbar_print("Starting playback");

  return;
}

void player_pause() {
  mpd_Status *status;

  debug("PAUSE requested");
  if(!glurp->conn) {
    debug("we're not connected, ignoring");
    return;
  }

  if( !(status = get_status(TRUE)) ) {
    debug("could not get mpd status");
    return;
  }

  if(status->state == MPD_STATUS_STATE_PLAY) {
    debug("pausing playback");
    statusbar_print("Pause");
    mpd_sendPauseCommand(glurp->conn, 1);
  } else if(status->state == MPD_STATUS_STATE_PAUSE) {
    debug("resuming playback");
    statusbar_print("Resuming playback");
    mpd_sendPauseCommand(glurp->conn, 0);
  }

  mpd_finishCommand(glurp->conn);

  if( check_mpd_error() ) {
    debug("Can't resume playback");
    return;
  }

  glurp->play_state = MPD_STATUS_STATE_PLAY;

  return;
}

void player_stop() {
  mpd_Status *status;

  debug("STOP requested");
  if(!glurp->conn) {
    debug("we're not connected, ignoring");
    return;
  }

  if( !(status = get_status(TRUE)) ) {
    debug("could not get mpd status");
    return;
  }

  if(status->state == MPD_STATUS_STATE_STOP) {
    debug("already stopped, ignoring");
    return;
  }

  debug("Stopping playback");
  statusbar_print("Stopping playback");
  mpd_sendStopCommand(glurp->conn);
  mpd_finishCommand(glurp->conn);

  if( check_mpd_error() ) {
    debug("Can't stop playback");
    return;
  }

  glurp->play_state = MPD_STATUS_STATE_STOP;

  return;
}

void player_prev() {
  mpd_Status *status;

  debug("PREV requested");
  if(!glurp->conn) {
    debug("we're not connected, ignoring");
    return;
  }

  if( !(status = get_status(TRUE)) ) {
    debug("could not get mpd status");
    return;
  }

  if( status->state != MPD_STATUS_STATE_PLAY && status->state != MPD_STATUS_STATE_PAUSE ) {
    debug("not playing, ignoring");
    return;
  }

  debug("switching to previous track");
  statusbar_print("Previous");
  mpd_sendPrevCommand(glurp->conn);
  mpd_finishCommand(glurp->conn);

  if( check_mpd_error() ) debug("Can't switch to prev track");

  return;
}

void player_next() {
  mpd_Status *status;

  debug("NEXT requested");
  if(!glurp->conn) {
    debug("we're not connected, ignoring");
    return;
  }

  if( !(status = get_status(TRUE)) ) {
    debug("could not get mpd status");
    return;
  }

  if( status->state != MPD_STATUS_STATE_PLAY && status->state != MPD_STATUS_STATE_PAUSE ) {
    debug("not playing, ignoring");
    return;
  }

  debug("switching to next track");
  statusbar_print("Next");
  mpd_sendNextCommand(glurp->conn);
  mpd_finishCommand(glurp->conn);

  if( check_mpd_error() ) debug("Can't switch to next track");

  return;
}
