/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include "packages.h"
#include "notebook.h"
#include "editor.h"
#include "userlist.h"

#include "gtkgmo.h"

typedef struct _mcp_registration mcp_registration;
struct _mcp_registration {
    char *name;
    mcp_version min_version;
    mcp_version max_version;
    package_init_f init;
};

typedef struct _mcp_overwrite mcp_overwrite;
struct _mcp_overwrite {
    char *org;
    char *replace;
};

typedef struct _mcp_alias mcp_alias;
struct _mcp_alias {
    char *name;
    char *alias;
};

GList *all_packages = NULL;
GList *overwrites = NULL;
GList *aliasses = NULL;

extern mcp_version *str_to_version(const char *version);
extern mcp_version *version_match(const mcp_version *a_min,
				  const mcp_version *a_max,
				  const mcp_version *b_min,
				  const mcp_version *b_max);
mcp_package *find_package(mcp_session *s, const char *name);

void mcp_negotiate_init(mcp_package *p);
void mcp_negotiate_handle(mcp_package *p, const char *command,
			  int argc, char **argv);
void mcp_negotiate_try_add_package(mcp_session *s, const char *name,
				   const mcp_version *from,
				   const mcp_version *to);

void mcp_dns_com_awns_status_init(mcp_package *p);
void mcp_dns_com_ben_tfstatus_init(mcp_package *p);

void mcp_dns_com_vmoo_client_init(mcp_package *p);

void dns_org_mud_moo_simpleedit_init(mcp_package *p);


int gm_package_name_matches(mcp_package *p, const char *name) {
    GList *l;
    mcp_alias *a;
    
    if(g_strcasecmp(p->name, name) == 0)
        return TRUE;
    for(l = aliasses; l; l = g_list_next(l)) {
        a = l->data;
        if(g_strcasecmp(p->name, a->name) == 0 && 
           g_strcasecmp(a->alias, name) == 0)
            return TRUE;
        if(g_strcasecmp(p->name, a->alias) == 0 && 
           g_strcasecmp(a->name, name) == 0)
            return TRUE;
    }
    return FALSE;
}

mcp_registration *find_registration(const char *name) {
    mcp_registration *r;
    GList *l;
    for(l = all_packages; l; l = g_list_next(l)) {
        r = l->data;
        if(g_strcasecmp(name, r->name) == 0)
            return r;
    }
    return NULL;
}

int is_overwritten(mcp_session *s, const char *name) {
    mcp_overwrite *o;
    GList *l;
    for(l = overwrites; l; l = g_list_next(l)) {
        o = l->data;
        if(g_strcasecmp(name, o->org) == 0 && find_package(s, o->replace))
            return TRUE;
    }
    return FALSE;
}

extern void mcp_package_free(mcp_package *p);
void remove_overwrites(mcp_session *s, const char *name) {
    mcp_overwrite *o;
    mcp_package *p;
    GList *l;
    for(l = overwrites; l; l = g_list_next(l)) {
        o = l->data;
        if(g_strcasecmp(name, o->replace) == 0 && (p = find_package(s, o->org))) {
            if(debug) printf("*** Package %s will be overwritten by package %s\n",
                             p->name, name);
            s->packages = g_list_remove(s->packages, p);
            mcp_package_free(p);
        }
    }
}   

void add_add_to_overwrites(const char *org, const char *replace) {
    mcp_overwrite *o = g_malloc(sizeof(mcp_overwrite));
    o->org = g_strdup(org);
    o->replace = g_strdup(replace);
    overwrites = g_list_prepend(overwrites, o);
}

void add_add_to_aliasses(const char *name, const char *alias) {
    mcp_alias *a = g_malloc(sizeof(mcp_alias));
    a->name = g_strdup(name);
    a->alias = g_strdup(alias);
    aliasses = g_list_prepend(aliasses, a);
}

void add_to_all_packages(const char *name,
			 int min_maj, int min_min,
			 int max_maj, int max_min,
			 package_init_f init) {
    mcp_registration *r = g_malloc(sizeof(mcp_registration));
    r->name = g_strdup(name);
    r->min_version.major = min_maj;
    r->min_version.minor = min_min;
    r->max_version.major = max_maj;
    r->max_version.minor = max_min;
    r->init = init;
    all_packages = g_list_prepend(all_packages, r);
}

void init_all_packages() {
    if(!all_packages) {
        add_to_all_packages("mcp-negotiate", 1, 0, 2, 0, mcp_negotiate_init);
        add_to_all_packages("dns-com-awns-status", 1, 0, 1, 0,
                            mcp_dns_com_awns_status_init);
        add_to_all_packages("dns-com-ben-tfstatus", 1, 0, 1, 0,
                            mcp_dns_com_ben_tfstatus_init);
        add_to_all_packages("dns-nl-vgmoo-client", 1, 0, 1, 0,
                            mcp_dns_com_vmoo_client_init);
        add_to_all_packages("dns-com-vmoo-client", 1, 0, 1, 0,
                            mcp_dns_com_vmoo_client_init);
        add_to_all_packages("dns-org-mud-moo-simpleedit", 1, 0, 1, 0,
                            dns_org_mud_moo_simpleedit_init);
        add_to_all_packages("dns-nl-vgmoo-userlist", 1, 0, 1, 0,
                            gm_userlist_mcp_init);
        add_to_all_packages("dns-com-vmoo-userlist", 1, 0, 1, 0,
                            gm_userlist_mcp_init);
    }
    if(!overwrites) {
        /*
          add_add_to_overwrites("dns-nl-vgmoo-userlist", "dns-com-vmoo-userlist");
        add_add_to_overwrites("dns-nl-vgmoo-pages",    "dns-com-vmoo-pages");
        add_add_to_overwrites("dns-nl-vgmoo-client",   "dns-com-vmoo-client");
        */
    }
    if(!aliasses) {
        add_add_to_aliasses("dns-com-vmoo-userlist", "dns-nl-vgmoo-userlist");
        add_add_to_aliasses("dns-com-vmoo-pages",    "dns-nl-vgmoo-pages");
        add_add_to_aliasses("dns-com-vmoo-client",   "dns-nl-vgmoo-client");
    }
}

void mcp_negotiate_startup(mcp_session *s) {
    mcp_version version = {1, 0};

    init_all_packages();
    mcp_negotiate_try_add_package(s, "mcp-negotiate", &version, &version);
}

void mcp_negotiate_try_add_package(mcp_session *s,
                                   const char *name,
                                   const mcp_version *from,
                                   const mcp_version *to) {
    mcp_registration *r;
    mcp_version *v;
    mcp_package *p;

    if(debug) printf("Trying to add package \"%s\"\n", name);
    if((r = find_registration(name))) {
        if(!is_overwritten(s, name)) {
            if(!find_package(s, name)) {
                if((v = version_match(from, to,
                                      &r->min_version, &r->max_version))) {
                    remove_overwrites(s, name);
                    p = g_malloc(sizeof(mcp_package));
                    p->name = g_strdup(r->name);
                    p->session = s;
                    p->version.major = v->major;
                    p->version.minor = v->minor;
                    p->handle = NULL;
                    p->free = NULL;
                    p->data = NULL;
                    gm_mcp_add_package(s, p);
                    g_free(v);
                    if(debug) printf("\x1b[35mAdded package %s with version "
                                     "%d.%d\x1b[0m\n",
                                     p->name, p->version.major, p->version.minor);
                    if(r->init)(r->init)(p);
                } else if(debug) {
                    printf("\tNo version match :(\n");
                }
            } else if (debug) {
                printf("Package already loaded !\n");
            }
        } else if(debug) {
            printf("\tPackage is overwritten...\n");
        }
    } else if(debug) {
        printf("\tNot supported package\n");
    }
}

void mcp_negotiatate_do_send_can(mcp_package *p, mcp_registration *r) {
    char *line = g_strdup_printf("package: %s min-version: %d.%d "
                                 "max-version: %d.%d",
                                 r->name,
                                 r->min_version.major,
                                 r->min_version.minor,
                                 r->max_version.major,
                                 r->max_version.minor);
    gm_mcp_send(p->session, p->name, "can", line);
}

void mcp_negotiatate_do_send_end(mcp_package *p) {
    gm_mcp_send(p->session, p->name, "end", "");
}

void mcp_negotiate_send_all_but_negotiate(mcp_package *p) {
    mcp_registration *r;
    GList *l;
    for(l = all_packages; l; l = g_list_next(l)) {
        r = l->data;
        if(g_strcasecmp(r->name, p->name) != 0) {
            mcp_negotiatate_do_send_can(p, r);
        }
    }
    if(p->version.major == 2 && p->version.minor >= 0) {
        mcp_negotiatate_do_send_end(p);
    }
}

void mcp_negotiate_send_can_negotiate(mcp_package *p) {
    mcp_registration *r;
    GList *l;
    for(l = all_packages; l; l = g_list_next(l)) {
        r = l->data;
        if(g_strcasecmp(r->name, p->name) == 0) {
            mcp_negotiatate_do_send_can(p, r);
        }
    }
}

void mcp_negotiate_init(mcp_package *p) {
    p->handle = mcp_negotiate_handle;
    mcp_negotiate_send_can_negotiate(p);
}

void mcp_negotiate_update_version(mcp_package *p, const mcp_version *from,
                                  const mcp_version *to) {
    mcp_registration *r;
    mcp_version *v;
    r = find_registration(p->name);

    if(debug) printf("Updatiting version on mcp-negotiate\n");
    if((v = version_match(from, to, &r->min_version, &r->max_version))) {
        p->version.major = v->major;
        p->version.minor = v->minor;
        g_free(v);
        mcp_negotiate_send_all_but_negotiate(p);
        if(debug) printf("\t\tNew version is %d.%d\n",
                         p->version.major, p->version.minor);
    } else if(debug) {
        printf("New version didn't match, it stays %d.%d\n",
               p->version.major, p->version.minor);
    }
}

void mcp_negotiate_handle_can(mcp_package *p, int argc, char **argv) {
    char *package, *min_version, *max_version;
    mcp_version *min, *max;
    int i;

    package = min_version = max_version = NULL;

    for(i = 0; i < argc; i+= 2) {
        if(g_strcasecmp(argv[i], "package") == 0) {
            package = argv[i + 1];
        } else if(g_strcasecmp(argv[i], "min-version") == 0) {
            min_version = argv[i + 1];
        } else if(g_strcasecmp(argv[i], "max-version") == 0) {
            max_version = argv[i + 1];
        }
    }

    if(package && min_version && max_version) {
        min = str_to_version(min_version);
        max = str_to_version(max_version);
        if(min && max) {
            if(g_strcasecmp(package, "mcp-negotiate") == 0)
                mcp_negotiate_update_version(p, min, max);
            else
                mcp_negotiate_try_add_package(p->session, package, min, max);
        }
        g_free(min);
        g_free(max);
    }
}

void mcp_negotiate_handle(mcp_package *p, const char *command,
                          int argc, char **argv) {
    if(g_strcasecmp("can", command) == 0) {
        mcp_negotiate_handle_can(p, argc, argv);
    } else if (debug) {
        if(g_strcasecmp("end", command) == 0) {
            printf("\tend command received - ignoring 4 now\n");
        } else {
            printf("\tHee, unknown function \"%s\" !\n", command);
        }
    }
}

/********************************************************************************
 *	Statusbar packages
 ********************************************************************************/
extern void gm_statusbar_set(const char *text);
void do_statusbar_set_text(mcp_package *p, const char *text) {
    world *w = p->session->belongs_to;
    g_free(w->status_msg);
    w->status_msg = g_strdup(text);
    if(gm_notebook_current_world() == w)
	gm_statusbar_set(w->status_msg);
}

void mcp_dns_com_awns_status_hanlde(mcp_package *p, const char *command,
                                    int argc, char **argv) {
    char *text;
    int i;
    if(strcmp(command, "") == 0) {
        text = NULL;
        for(i = 0; i < argc; i += 2) {
            if(g_strcasecmp("text", argv[i]) == 0) {
                text = argv[i + 1];
            }
        }
        if(text) {
            do_statusbar_set_text(p, text);
        }
    } else if(debug) {
        printf("\tHee, unknown function \"%s\" !\n", command);
    }
}

void mcp_dns_com_awns_status_init(mcp_package *p) {
    p->handle = mcp_dns_com_awns_status_hanlde;
}

void mcp_dns_com_ben_tfstatus_handle(mcp_package *p, const char *command,
                                     int argc, char **argv) {
    char *content;
    int i;
    if(g_strcasecmp(command, "update") == 0) {
        content = NULL;
        for(i = 0; i < argc; i += 2) {
            if(g_strcasecmp("content", argv[i]) == 0) {
                content = argv[i + 1];
            }
        }
        if(content) {
            do_statusbar_set_text(p, content);
        }
    } else if(debug) {
        printf("\tHee, unknown function \"%s\" !\n", command);
    }
}

void mcp_dns_com_ben_tfstatus_init(mcp_package *p) {
    p->handle = mcp_dns_com_ben_tfstatus_handle;
}

/********************************************************************************
 *	dns-[nl-vgmoo|com-vmoo]-client 1.0 - 1.0
 ********************************************************************************/

void do_mcp_update_screensize(mcp_package *p, world *w) {
    char *args = g_strdup_printf("cols: %d rows: %d",
                                 GTK_GMO(w->output)->width,
                                 GTK_GMO(w->output)->height);
    gm_mcp_send(p->session, p->name, "screensize", args);
    g_free(args);
}

void do_mcp_update_client_info(mcp_package *p) {
    char args[] = "name: \""PACKAGE"\" "
	"text-version: \""VERSION"\" "
	"internal-version: \""IVERSION"\" "
	"reg-id: \"\" "
	"flags: \"\"";
    gm_mcp_send(p->session, p->name, "info", args);
}

void mcp_dns_com_vmoo_client_init(mcp_package *p) {
    /* this function does nothing now, when the reconnect feature is implemented
     * in feature, de dns-nl-vgmoo-client-disconnect will be handled here
     */
    do_mcp_update_screensize(p, p->session->belongs_to);
    do_mcp_update_client_info(p);
}

void gm_mcp_update_screensize(world *w) {
    mcp_package *p;
    if(!w->mcp) return;
    if((p = find_package(w->mcp, "dns-com-vmoo-client")) || 
       (p = find_package(w->mcp, "dns-nl-vgmoo-client"))) {
        do_mcp_update_screensize(p, w);
    }
}

/********************************************************************************
 *	dns-org-mud-moo-simpleedit 1.0 - 1.0
 ********************************************************************************/

extern char *generate_key();

void do_upload_editor(mcp_package *p, const char *ref, const char *type,
                      GList *lines, int has_newlines) {
    GList *l;
    char *key = generate_key();
    char *args = g_strconcat("reference: ", ref,
			     " content*: \"\" type: ", type,
			     " _data-tag: ", key,
                             NULL);
    gm_mcp_send(p->session, p->name, "set", args);
    for(l = lines; l; l = g_list_next(l)) {
        gm_mcp_multiline_send(p->session, key, "content", l->data, has_newlines);
    }
    gm_mcp_multiline_end(p->session, key);
    g_free(args);
    g_free(key);
}

void gm_mcp_upload_editor(editor_t *e, const char *ref, const char *type,
                          GList *lines, int has_newlines) {
    mcp_package *p;
    if(!e->belongs_to->mcp) return;
    if((p = find_package(e->belongs_to->mcp, "dns-org-mud-moo-simpleedit"))) {
        if(debug) printf("Have to upload\n\n\n");
        do_upload_editor(p, ref, type, lines, has_newlines);
    }
}

void dns_org_mud_moo_simpleedit_line(mcp_multiline *m,
                                     const char *name, const char *line) {
    if(debug) printf("MCP_L: %s\n", line);
    gm_editor_add_line(EDITOR(m->data), line, strlen(line));
}


void dns_org_mud_moo_simpleedit_end(mcp_multiline *m) {
    if(debug) printf("MCP_E: end\n");
    gm_editor_all_data_received(EDITOR(m->data));
}


void dns_org_mud_moo_simpleedit_handle(mcp_package *p, const char *command,
                                       int argc, char **argv) {
    int i, content = 0;
    char *reference = NULL, *name = NULL, *type = NULL, *data_tag = NULL;
    editor_t *e;
    
    if(g_strcasecmp(command, "content") == 0) {
        for(i = 0; i < argc; i += 2) {
            if(g_strcasecmp(argv[i], "reference") == 0) {
                reference = argv[i + 1];
            } else if (g_strcasecmp(argv[i], "name") == 0) {
                name = argv[i + 1];
            } else if (g_strcasecmp(argv[i], "type") == 0) {
                type = argv[i + 1];
            } else if (g_strcasecmp(argv[i], "content*") == 0) {
                content = TRUE;
            } else if (g_strcasecmp(argv[i], "_data-tag") == 0) {
                data_tag = argv[i + 1];
            }
        }
        if(content && reference && name && type && data_tag) {
            e = gm_editor_new_from_mcp(p->session->belongs_to, name,
                                       reference, type);
            gm_mcp_multiline_new(p, data_tag, e,
                                 dns_org_mud_moo_simpleedit_line,
                                 dns_org_mud_moo_simpleedit_end);
        } else if(debug) {
            printf("simple-edit `content' command with invalid args :(\n");
        }
    } else if(debug) {
        printf("simpel-edit with unknown command (%s)\n", command);
    }
}

void dns_org_mud_moo_simpleedit_init(mcp_package *p) {
    p->handle = dns_org_mud_moo_simpleedit_handle;
}

void gm_mcp_update_friends(world *w, const char *s) {
    mcp_package *p;
    char *args;
    
    if(!w->mcp) return;
    if((p = find_package(w->mcp, "dns-com-vmoo-userlist")) ||
       (p = find_package(w->mcp, "dns-nl-vgmoo-userlist"))) {
        args = g_strconcat("friends: ", s, NULL);
        gm_mcp_send(w->mcp, p->name, "friends", args);
    }
}
