/* Low level interface for debugging DEC threads for GDB, the GNU debugger.
   Copyright 1996 Free Software Foundation, Inc.

This file is part of GDB.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include "defs.h"

#include <pthread_debug.h>

#include "gdbthread.h"

#include "target.h"
#include "inferior.h"
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/procfs.h>
#include <stdio.h>
#include <string.h>
#include <dlfcn.h>
#include "gdbcmd.h"

extern unsigned long current_thread;

static int dec_thread_active = 0;

pthreadDebugContext_t debug_context;

extern pthreadDebugCallbacks_t debug_callbacks;

extern void dec_thread_init ();

/* Two debug macros than can be used to check the memory state before
   and after calls to the debug library */
#define PROTECT_MEM
#define UNPROTECT_MEM

 /*
	Return the pthread_debug error string associated with errcode.
	If errcode is unknown, then return a message.
 */

struct string_map {
  int num;
  char *str;
};

static char *
ptd_err_string (errcode)
     int errcode;
{
  static struct string_map
    ptd_err_table[] = {
      {ESUCCESS,	"Successful completion"},
      {ENOENT,		"End of \"next thread\" sequence (no more)"},
      {ENOTSUP,		"Function not supported (usually callback error)"},
      {EINVAL,		"Invalid parameter"},
      {ESRCH,		"No such thread"},
      {ENOMEM,		"Insufficient memory"},
      {EPERM,		"Unable to perform operation (priv, state)"}
    };
  static int ptd_err_size = sizeof ptd_err_table / sizeof (struct string_map);
  int i;
  static char buf[50];

  for (i = 0; i < ptd_err_size; i++)
    if (ptd_err_table[i].num == errcode)
      return ptd_err_table[i].str;

  sprintf (buf, "Unknown pthread_debug error code: %d", errcode);

  return buf;
}

/*

LOCAL FUNCTION

	ptd_state_string - Convert a pthread_debug state code to a string

SYNOPSIS

	char * ptd_state_string (statecode)

DESCRIPTION

	Return the pthread_debug state string associated with statecode.  If
	statecode is unknown, then return a message.

 */

static char *
ptd_state_string (statecode)
     int statecode;
{
  static struct string_map
    ptd_state_table[] = {
      {PTHREAD_DEBUG_STATE_UNKNOWN,	"<bad value>"},
      {PTHREAD_DEBUG_STATE_RUNNING,	"running"},
      {PTHREAD_DEBUG_STATE_READY,	"ready"},
      {PTHREAD_DEBUG_STATE_BLOCKED,	"blocked"},
      {PTHREAD_DEBUG_STATE_NEW,		"new"},
      {PTHREAD_DEBUG_STATE_SORTING,	"sorting"},
      {PTHREAD_DEBUG_STATE_TERMINATED,	"terminated"},
      {PTHREAD_DEBUG_STATE_ZOMBIE,	"zombie"}
    };
  const int ptd_state_table_size = sizeof ptd_state_table / sizeof (struct string_map);
  int i;
  static char buf[50];

  for (i = 0; i < ptd_state_table_size; i++)
    if (ptd_state_table[i].num == statecode)
      return ptd_state_table[i].str;

  sprintf (buf, "Unknown pthread_debug state code: %d", statecode);

  return buf;
}

/* read the registers of the current thread (designated by the global variable
   current_thread and store the general registers in gregset and the floating
   point registers in fpgregset iff fpgregset is non null.
   Return 0 on success, -1 on error.
*/

int
dec_thread_get_registers (gregset, fpregset)
  gregset_t *gregset;
  fpregset_t *fpregset;
{
  pthreadDebugId_t thread;
  pthreadDebugThreadInfo_t info;
  int res;

  dec_thread_init ();

  /* Get the integer regs */

  PROTECT_MEM;

  res = pthreadDebugThdSeqInit (debug_context, &thread);

  while (res == ESUCCESS)
    {
      res = pthreadDebugGetThreadState (debug_context, thread, &info);

      if (info.teb == (pthread_t) current_thread)
	{
	  break;
	}
      res = pthreadDebugThdSeqNext (debug_context, &thread);
    }
  
  if (res != ESUCCESS) {
    pthreadDebugThdSeqDestroy (debug_context);
    UNPROTECT_MEM;
    return -1;
  }

  res = pthreadDebugGetReg (debug_context, thread, gregset);
  if (res != ESUCCESS) {
    error ("dec_thread_fetch_registers: pthreadDebugGetReg %s",
	   ptd_err_string (res));
    pthreadDebugThdSeqDestroy (debug_context);
    UNPROTECT_MEM;
    return -1;
  }

  /* And, now the fp regs */

  if (fpregset != NULL) {
    res = pthreadDebugGetFpreg (debug_context, thread, fpregset);

    if (res != ESUCCESS) {
      error ("dec_thread_fetch_registers: pthreadDebugGetReg %s",
	     ptd_err_string (res));
      pthreadDebugThdSeqDestroy (debug_context);
      UNPROTECT_MEM;
      return -1;
    }
  }

  pthreadDebugThdSeqDestroy (debug_context);
  UNPROTECT_MEM;
  return 0;
}

int
dec_thread_fetch_registers ()
{
  gregset_t gregset;
  fpregset_t fpregset;

  dec_thread_init ();

  if (dec_thread_get_registers (&gregset, &fpregset) == 0) {

    supply_gregset (&gregset);
    supply_fpregset (&fpregset);

    return 0;
  }

  return -1;
}

unsigned long
dec_thread_get_current_thread ()
{
  pthreadDebugId_t thread;
  pthreadDebugThreadInfo_t info;
  int res;

  dec_thread_init ();

  PROTECT_MEM;

  res = pthreadDebugThdSeqInit (debug_context, &thread);

  while (res == ESUCCESS)
    {
      res = pthreadDebugGetThreadState (debug_context, thread, &info);

      if (info.state == PTHREAD_DEBUG_STATE_RUNNING)
        {
          break;
        }
      res = pthreadDebugThdSeqNext (debug_context, &thread);
    }
 
  pthreadDebugThdSeqDestroy (debug_context);
  UNPROTECT_MEM;

  if (res != ESUCCESS) return 0;
  return (unsigned long) info.teb;
}

/* This routine must be called once before using the other functions of this
    file. */

void
dec_thread_init ()
{
  int val, i;
  void *symtable;
  struct minimal_symbol *msym;
  void* caller_context;
  char buf [64];
  struct value malloc_val;
  struct value *value;

  if (!dec_thread_active)
    {
#ifdef DEBUG
      printf_unfiltered ("in dec_thread_init\n");
#endif

      msym = lookup_minimal_symbol ("__pthread_dbg_symtable", NULL, NULL);

      if (msym != NULL)
        {
          symtable = (void *) SYMBOL_VALUE_ADDRESS (msym);
#ifdef DEBUG
	  printf_unfiltered ("dbg_symtable = %lx\n", symtable);
#endif
	}
      else
	{
          error ("dec_thread_init: unable to find __pthread_dbg_symtable\n");
	  return;
	}

      /* Initialize the thread debugging library.  This needs to be done after
         the process has started because it needs to call some functions
	 (e.g malloc) in the inferior */

      PROTECT_MEM;
      val = pthreadDebugContextInit (&caller_context, &debug_callbacks, 
				     symtable, &debug_context);
      UNPROTECT_MEM;

      if (val != ESUCCESS)
        error ("dec_thread_init: pthreadDebugContextInit: %s",
               ptd_err_string (val));

      dec_thread_active = 1;
    }
}

/* These routines implement the lower half of the pthread_debug interface.
   Ie: the *_clbk routines.  */

/* The next 4 routines are called by pthread_debug to tell us to stop and
   continue a particular process or thread.  Since GDB ensures that these are
   all stopped by the time we call anything in pthread_debug, these routines
   need to do nothing.  */

static int suspend_clbk (void *caller_context)
{
  return ESUCCESS;
}

static int resume_clbk (void *caller_context)
{
  return ESUCCESS;
}

static int hold_clbk (void *caller_context, pthreadDebugKId_t thread_id)
{
  return ESUCCESS;
}

static int unhold_clbk (void *caller_context, pthreadDebugKId_t thread_id)
{
  return ESUCCESS;
}

/* Common routine for reading and writing memory.  */

static int rw_common (
  int dowrite,
  void *caller_context,
  char *address,
  char *buffer,
  unsigned long size)
{
#ifdef DEBUG
  printf_unfiltered ("rw_common (%d, %p, %p, %p, %ld)\n",
    dowrite, caller_context, address, buffer, size);
#endif

  if (address == NULL)
    return EINVAL;

  while (size > 0)
    {
      int cc;

      cc = do_xfer_memory ((CORE_ADDR) address, buffer, size, dowrite, NULL);

      if (cc < 0)
	{
	  if (dowrite == 0)
	    print_sys_errmsg ("rw_common (): read", errno);
	  else
	    print_sys_errmsg ("rw_common (): write", errno);

	  return EINVAL;
	}
      size -= cc;
      buffer += cc;
      address += cc;
    }
  return ESUCCESS;
}

static int read_clbk (
    void *caller_context,
    void *address,
    void *buffer,
    unsigned long size)
{
  return rw_common (0, caller_context, address, buffer, size);
}

static int write_clbk (
    void *caller_context,
    void *address,
    void *buffer,
    unsigned long size)
{
  return rw_common (1, caller_context, address, buffer, size);
}

/* Get integer regs */

static int get_reg_clbk(void *caller_context,
  gregset_t *gregset,
  pthreadDebugKId_t thread_id)
{
#ifdef DEBUG
  printf_unfiltered ("in get_reg_clbk...\n");
#endif

  target_fetch_registers (-1);
  fill_gregset (gregset, -1);

  return ESUCCESS;
}

/* Set integer regs */

static int set_reg_clbk(void *caller_context,
  gregset_t *gregset,
  pthreadDebugKId_t thread_id)
{
#ifdef DEBUG
  printf_unfiltered ("in set_reg_clbk...\n");
#endif

  supply_gregset (gregset);
  target_store_registers (-1);

  return ESUCCESS;
}

/*
 * Routine to write a line of standard output
 */

static int output_clbk (void *caller_context, char *line)
{
  fprintf_filtered (gdb_stdout, "%s\n", line);
  return ESUCCESS;
}

/*
 * Routine to write a line of error
 */

static int error_clbk (void *caller_context, char *line)
{
  fprintf_filtered (gdb_stderr, "%s\n", line);
  return ESUCCESS;
}

/* Get floating-point regs.  */

static int get_fpreg_clbk (void *caller_context,
  fpregset_t *fpregset,
  pthreadDebugKId_t thread_id)
{
#ifdef DEBUG
  printf_unfiltered ("in get_fpreg_clbk...\n");
#endif

  target_fetch_registers (-1);
  fill_fpregset (fpregset, -1);

  return ESUCCESS;
}

/* Set floating-point regs.  */

static int set_fpreg_clbk (void *caller_context,
  fpregset_t *fpregset,
  pthreadDebugKId_t thread_id)
{
#ifdef DEBUG
  printf_unfiltered ("in set_fpreg_clbk...\n");
#endif

  supply_fpregset (fpregset);
  target_store_registers (-1);

  return ESUCCESS;
}

/*
 * Routine to allocate memory.
 */
static void *malloc_clbk (void *caller_context, size_t size)
{
  return xmalloc (size);
}

/*
 * Routine to deallocate memory
 */
static void free_clbk (void *caller_context, void *address)
{
  xfree (address);
}

static int kthdinfo_clbk (
  pthreadDebugClient_t        caller_context,
  pthreadDebugKId_t           thread_id,
  pthreadDebugKThreadInfo_p   thread_info)
{
  return ENOTSUP;
}

static int speckthd_clbk (
  pthreadDebugClient_t        caller_context,
  pthreadDebugSpecialType_t   type,
  pthreadDebugKId_t           *thread_id)
{
  return ENOTSUP;
}

pthreadDebugCallbacks_t debug_callbacks = {
  PTHREAD_DEBUG_VERSION,
  (pthreadDebugGetMemRtn_t) read_clbk,
  (pthreadDebugSetMemRtn_t) write_clbk,
  suspend_clbk,
  resume_clbk,
  kthdinfo_clbk,
  hold_clbk,
  unhold_clbk,
  (pthreadDebugGetFregRtn_t) get_fpreg_clbk,
  (pthreadDebugSetFregRtn_t) set_fpreg_clbk,
  (pthreadDebugGetRegRtn_t) get_reg_clbk,
  (pthreadDebugSetRegRtn_t) set_reg_clbk,
  (pthreadDebugOutputRtn_t) output_clbk,
  (pthreadDebugOutputRtn_t) error_clbk,
  malloc_clbk,
  free_clbk,
  speckthd_clbk
};

