/* Thin binding for the WTX protocol, for GDB.

   Copyright 2004 Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "defs.h"
#include "gdb_string.h"
#include "remote-wtxapi.h"

/* This layer is only implemented and tested on WTX 2.0 and 3.0 */

#if WTX_PROT_VERSION != 2 && WTX_PROT_VERSION != 3
#error
#endif

/* Conversion from an to a BOOL32.  */

#define TO_BOOL(b) (b ? TRUE : FALSE)

#ifndef WTX_PD_CURRENT
const TGT_ADDR_T WTX_PD_CURRENT;
#endif

#ifndef WTX_PD_ALL
const TGT_ADDR_T WTX_PD_ALL = -1;
#endif

#ifndef WTX_MOD_FIND_IN_ALL_PD
const TGT_ADDR_T WTX_MOD_FIND_IN_ALL_PD = -1;
#endif

#ifndef WTX_SYM_FIND_IN_ALL_PD
const int WTX_SYM_FIND_IN_ALL_PD = -1;
#endif

#if WTX_PROT_VERSION == 2
const WTX_CONTEXT_TYPE WTX_CONTEXT_PD = WTX_CONTEXT_ANY_TASK;
const WTX_ACTION_TYPE WTX_ACTION_PD_STOP = WTX_ACTION_STOP_ALL;
const int WTX_ERR_PD_INVALID_PD = WTX_ERROR;
#endif

const int wtxapi_symbol_copy_none        = 0;
const int wtxapi_symbol_copy_name        = 1 << 0;
const int wtxapi_symbol_copy_module_name = 1 << 1;

const pd_id_t invalid_pd_id                     = WTX_ERROR;
const evtpt_id_t invalid_module_id              = WTX_ERROR;
const evtpt_id_t invalid_evtpt_id               = WTX_ERROR;
const WTX_CONTEXT_ID_T invalid_context_id       = WTX_ERROR;
const WTX_AGENT_MODE_TYPE invalid_agent_mode    = WTX_ERROR;
const WTX_CONTEXT_STATUS invalid_context_status = WTX_ERROR;

/* Only one PD is allowed in WTX 2.0; NULL_PD is the value of this PD
   ID.  */

const pd_id_t NULL_PD = 0;

/* Symbol list: complete declaration. Based on WTX_SYMBOL and
   WTX_SYMBOL_LIST.  */

struct wtxapi_symbol_list
{
  /* First element of the list.  */
  WTX_SYMBOL *first;

  /* Current element (when going through the list with the iterator).  */
  WTX_SYMBOL *current;

  /* Pointer to the result of the WTX operation used to get the symbol
     list. It is used at deallocation time.  */
  void *wtx_result_to_cleanup;
};

/* Current WTX handle.  */

/* FIXME: Should really be static... For now, it is still used in remote-wtx.c,
   for the Tcl calls.  */

HWTX current_wtx_handle = 0;

/* Allocate a new wtxapi_symbol_list and initialize it with
   WTX_SYM. WTX_SYM should not be deallocated directly by the caller,
   it will be deallocated by free_wtxapi_symbol_list.  */

static struct wtxapi_symbol_list
  *new_wtxapi_symbol_list (WTX_SYM_LIST* wtx_sym);


static struct wtxapi_symbol_list
  *new_wtxapi_symbol_list_from_symbol (WTX_SYMBOL* wtx_sym);


/* Allocate a new wtxapi_module_info, and initialize it with
   WTX_MINFO. The caller can deallocate WTX_MINFO, as it is fully
   copied. It does not initialize wtx_minfo->undef_list; it should
   be handled separatly if needed.  */

static struct wtxapi_module_info
  *new_wtxapi_module_info (WTX_MODULE_INFO *wtx_minfo);

/* Build and initialize a WTX client handle. This handle can then be
   used to connect to a target server.  */

int
wtx_initialize ()
{
  return wtxInitialize (&current_wtx_handle) != WTX_ERROR;
}

/* Terminate use of WTX client handle.  */

int
wtx_terminate ()
{
  return wtxTerminate (current_wtx_handle) != WTX_ERROR;
}

/* Connect client to the target server.  */

int
wtx_tool_attach (const char *target_name, const char *tool_name)
{
  return wtxToolAttach (current_wtx_handle, target_name,
                        tool_name) != WTX_ERROR;
}

/* Check tool connection to the server.  */

int
wtx_tool_connected ()
{
  return wtxToolConnected (current_wtx_handle) == TRUE;
}

/* Detach from the target server.  */

int
wtx_tool_detach ()
{
  return wtxToolDetach (current_wtx_handle) != WTX_ERROR;
}

/* Clear any error for the tool.  */

int
wtx_err_clear ()
{
  return wtxErrClear (current_wtx_handle) != WTX_ERROR;
}

/* Return the last error for a handle.  */

WTX_ERROR_T
wtx_err_get ()
{
  return wtxErrGet (current_wtx_handle);
}

/* Add an error handler.  */

WTX_HANDLER_T
wtx_err_handler_add (WTX_HANDLER_FUNC p_func, void *p_client_data)
{
  return wtxErrHandlerAdd (current_wtx_handle, p_func, p_client_data);
}

/* Remove error handler for WTX handle.  */

int
wtx_err_handler_remove (WTX_HANDLER_T p_handler)
{
  return wtxErrHandlerRemove (current_wtx_handle, p_handler) != WTX_ERROR;
}

/* Fetch last WTX API error string.  */

const char *
wtx_err_msg_get ()
{
  if (current_wtx_handle == 0)
    return "Invalid WTX handle";
  else
    return wtxErrMsgGet (current_wtx_handle);
}

/* Get agent mode.  */

WTX_AGENT_MODE_TYPE
wtx_agent_mode_get ()
{
  return wtxAgentModeGet (current_wtx_handle);
}

/* Set the mode of the target agent.  */

int
wtx_agent_mode_set (WTX_AGENT_MODE_TYPE agent_mode)
{
  return wtxAgentModeSet (current_wtx_handle, agent_mode) != WTX_ERROR;
}

/* Create a new breakpoint.  */

evtpt_id_t
wtx_breakpoint_add (WTX_CONTEXT_TYPE context_type,
                    WTX_CONTEXT_ID_T context_id,
                    TGT_ADDR_T tgt_addr)
{
  return wtxBreakpointAdd (current_wtx_handle, context_type, context_id,
                           tgt_addr);
}

/* Create a new event point.  */

evtpt_id_t
wtx_eventpoint_add (struct wtxapi_evtpt *p_evtpt)
{
#if WTX_PROT_VERSION == 3
  WTX_EVTPT wtx_evtpt;
#else
  WTX_EVTPT_2 wtx_evtpt;
#endif
  memset (&wtx_evtpt, 0, sizeof (wtx_evtpt));
  wtx_evtpt.event.eventType = p_evtpt->event.event_type;
  wtx_evtpt.event.numArgs = p_evtpt->event.num_args;
  wtx_evtpt.event.args = p_evtpt->event.args;
  wtx_evtpt.context.contextType = p_evtpt->context.context_type;
  wtx_evtpt.context.contextId = p_evtpt->context.context_id;
  wtx_evtpt.action.actionType = p_evtpt->action.action_type;
  wtx_evtpt.action.actionArg = p_evtpt->action.action_arg;
  wtx_evtpt.action.callRtn = p_evtpt->action.call_rtn;
  wtx_evtpt.action.callArg = p_evtpt->action.call_arg;
  return wtxEventpointAdd (current_wtx_handle, &wtx_evtpt);
}

/* Delete eventpoint from the target.  */

int
wtx_eventpoint_delete (evtpt_id_t evtpt_id)
{
  return wtxEventpointDelete (current_wtx_handle, evtpt_id) != WTX_ERROR;
}

WTX_CONTEXT_STATUS
wtx_context_status_get (WTX_CONTEXT_TYPE context_type,
                        WTX_CONTEXT_ID_T context_id)
{
  return wtxContextStatusGet (current_wtx_handle, context_type, context_id);
}


/* Continue execution of target context.  */

int
wtx_context_cont (WTX_CONTEXT_TYPE context_type,
                  WTX_CONTEXT_ID_T context_id)
{
  return wtxContextCont (current_wtx_handle, context_type,
                         context_id) != WTX_ERROR;
}

/* Create a context on target.  */

WTX_CONTEXT_ID_T
wtx_context_create (struct wtxapi_context_desc *p_context_desc)
{
  int ix;
  WTX_CONTEXT_DESC context_desc;
  memset (&context_desc, 0, sizeof (context_desc));
#if WTX_PROT_VERSION == 3
  context_desc.wtxContextType = p_context_desc->context_type;

  if (p_context_desc->context_type == WTX_CONTEXT_PD)
    {
      context_desc.wtxContextDef.wtxPdContextDef.name =
        p_context_desc->wtx_context_def.pd_context.name;
      context_desc.wtxContextDef.wtxPdContextDef.options =
        p_context_desc->wtx_context_def.pd_context.options;
      context_desc.wtxContextDef.wtxPdContextDef.heapSize =
        p_context_desc->wtx_context_def.pd_context.heap_size;
      context_desc.wtxContextDef.wtxPdContextDef.lowPriority =
        p_context_desc->wtx_context_def.pd_context.low_priority;
      context_desc.wtxContextDef.wtxPdContextDef.highPriority =
        p_context_desc->wtx_context_def.pd_context.high_priority;
      context_desc.wtxContextDef.wtxPdContextDef.pagePoolList =
        p_context_desc->wtx_context_def.pd_context.page_pool_list;
      context_desc.wtxContextDef.wtxPdContextDef.destroyRtn =
        p_context_desc->wtx_context_def.pd_context.destroy_rtn;
      context_desc.wtxContextDef.wtxPdContextDef.linkPath =
        p_context_desc->wtx_context_def.pd_context.link_path;
      context_desc.wtxContextDef.wtxPdContextDef.redirIn =
        p_context_desc->wtx_context_def.pd_context.redir_in;
      context_desc.wtxContextDef.wtxPdContextDef.redirOut =
        p_context_desc->wtx_context_def.pd_context.redir_out;
      context_desc.wtxContextDef.wtxPdContextDef.redirErr =
        p_context_desc->wtx_context_def.pd_context.redir_err;
      context_desc.wtxContextDef.wtxPdContextDef.argc =
        p_context_desc->wtx_context_def.pd_context.argc;
      context_desc.wtxContextDef.wtxPdContextDef.argv =
        p_context_desc->wtx_context_def.pd_context.argv;
    }
  else
    {
      context_desc.wtxContextDef.wtxTaskContextDef.pdId =
        p_context_desc->wtx_context_def.task_context.pd_id;
      context_desc.wtxContextDef.wtxTaskContextDef.returnType =
        p_context_desc->wtx_context_def.task_context.return_type;
      context_desc.wtxContextDef.wtxTaskContextDef.name =
        p_context_desc->wtx_context_def.task_context.name;
      context_desc.wtxContextDef.wtxTaskContextDef.priority =
        p_context_desc->wtx_context_def.task_context.priority;
      context_desc.wtxContextDef.wtxTaskContextDef.options =
        p_context_desc->wtx_context_def.task_context.options;
      context_desc.wtxContextDef.wtxTaskContextDef.stackBase =
        p_context_desc->wtx_context_def.task_context.stack_base;
      context_desc.wtxContextDef.wtxTaskContextDef.stackSize =
        p_context_desc->wtx_context_def.task_context.stack_size;
      context_desc.wtxContextDef.wtxTaskContextDef.entry =
        p_context_desc->wtx_context_def.task_context.entry;
      context_desc.wtxContextDef.wtxTaskContextDef.redirIn =
        p_context_desc->wtx_context_def.task_context.redir_in;
      context_desc.wtxContextDef.wtxTaskContextDef.redirOut =
        p_context_desc->wtx_context_def.task_context.redir_out;
      context_desc.wtxContextDef.wtxTaskContextDef.redirErr =
        p_context_desc->wtx_context_def.task_context.redir_err;
      context_desc.wtxContextDef.wtxTaskContextDef.argc =
        p_context_desc->wtx_context_def.task_context.argc;
      context_desc.wtxContextDef.wtxTaskContextDef.argv =
        p_context_desc->wtx_context_def.task_context.argv;
    }

#else

  /* In WTX 2.0, we cannot create a new PD.  */

  if (p_context_desc->context_type == WTX_CONTEXT_PD)
    return invalid_context_id;

  context_desc.contextType = p_context_desc->context_type;
  context_desc.returnType =
    p_context_desc->wtx_context_def.task_context.return_type;
  context_desc.name = p_context_desc->wtx_context_def.task_context.name;
  context_desc.priority = p_context_desc->wtx_context_def.task_context.priority;
  context_desc.options = p_context_desc->wtx_context_def.task_context.options;
  context_desc.stackBase =
    p_context_desc->wtx_context_def.task_context.stack_base;
  context_desc.stackSize =
    p_context_desc->wtx_context_def.task_context.stack_size;
  context_desc.entry = p_context_desc->wtx_context_def.task_context.entry;
  context_desc.redirIn = p_context_desc->wtx_context_def.task_context.redir_in;
  context_desc.redirOut = p_context_desc->wtx_context_def.task_context.redir_out;

  context_desc.redirOut = p_context_desc->wtx_context_def.task_context.redir_out;
  for (ix = 0; ix < WTX_MAX_ARG_CNT; ix++)
    if (ix < p_context_desc->wtx_context_def.task_context.argc)
      context_desc.args [ix] =
        p_context_desc->wtx_context_def.task_context.argv [ix];
    else
      context_desc.args [ix] = NULL_PD;
#endif
  return wtxContextCreate (current_wtx_handle, &context_desc);
}

/* Resume execution of a target context.  */

int
wtx_context_resume (WTX_CONTEXT_TYPE context_type, WTX_CONTEXT_ID_T context_id)
{
  return wtxContextResume (current_wtx_handle, context_type,
                           context_id) != WTX_ERROR;
}

/* Add exit evpt notification.  */

evtpt_id_t
wtx_context_exit_notify_add (WTX_CONTEXT_TYPE context_type,
                             WTX_CONTEXT_ID_T context_id)
{
  return wtxContextExitNotifyAdd (current_wtx_handle, context_type,
                                  context_id);
}

/* Kill a target context.  */

int
wtx_context_kill (WTX_CONTEXT_TYPE context_type, WTX_CONTEXT_ID_T context_id)
{
#if WTX_PROT_VERSION == 3
  return wtxContextKill (current_wtx_handle, context_type, context_id,
                         WTX_PD_DELETE_OPTION_NONE) != WTX_ERROR;
#else
  return wtxContextKill (current_wtx_handle, context_type,
                         context_id) != WTX_ERROR;
#endif
}

/* Single step exec of target context.  */

int
wtx_context_step (WTX_CONTEXT_TYPE context_type, WTX_CONTEXT_ID_T context_id,
                  TGT_ADDR_T step_start, TGT_ADDR_T step_end)
{
  return wtxContextStep (current_wtx_handle, context_type, context_id,
                         step_start, step_end) != WTX_ERROR;
}

/* Suspend a target context.  */

int
wtx_context_suspend (WTX_CONTEXT_TYPE context_type,
                     WTX_CONTEXT_ID_T context_id)
{
  return wtxContextSuspend (current_wtx_handle, context_type,
                            context_id) != WTX_ERROR;
}

int
wtx_context_stop (WTX_CONTEXT_TYPE context_type,
                  WTX_CONTEXT_ID_T context_id)
{

#if WTX_PROT_VERSION == 3
  return wtxContextStop (current_wtx_handle, context_type,
                         context_id) != WTX_ERROR;
#else
  /* FIXME : wtxContextStop does not exist on WTX 2.0. For now, use
   wtxContextSuspend as a workaround.  */

  return wtxContextSuspend (current_wtx_handle, context_id,
                            context_id)  != WTX_ERROR;
#endif
}

/* List event points on TS.  */

struct wtxapi_evtpt_list *
wtx_eventpoint_list_get ()
{
  int ix;
  struct wtxapi_evtpt_list *evtpt_list;
#if WTX_PROT_VERSION == 3
  WTX_EVTPT_LIST *wtx_list;
#else
  WTX_EVTPT_LIST_2 *wtx_list;
#endif
  wtx_list = wtxEventpointListGet (current_wtx_handle);

  if (!wtx_list)
    return NULL;

  evtpt_list = (struct wtxapi_evtpt_list*)
    xmalloc (sizeof (struct wtxapi_evtpt_list));
  evtpt_list->n_evtpt = wtx_list->nEvtpt;
  evtpt_list->p_evtpt_info = (struct wtxapi_evtpt_info*)
    xmalloc (sizeof (struct wtxapi_evtpt_info)
                                         * wtx_list->nEvtpt);
  for (ix = 0; ix < wtx_list->nEvtpt; ix++)
    {
      evtpt_list->p_evtpt_info[ix].wtx_evtpt.event.event_type =
        wtx_list->pEvtptInfo [ix].wtxEvtpt.event.eventType;
      evtpt_list->p_evtpt_info[ix].wtx_evtpt.event.num_args =
        wtx_list->pEvtptInfo [ix].wtxEvtpt.event.numArgs;
      evtpt_list->p_evtpt_info[ix].wtx_evtpt.event.args = (TGT_ARG_T *)
        xmalloc (wtx_list->pEvtptInfo [ix].wtxEvtpt.event.numArgs *
                 sizeof (TGT_ARG_T));
      memcpy (evtpt_list->p_evtpt_info [ix].wtx_evtpt.event.args,
              wtx_list->pEvtptInfo [ix].wtxEvtpt.event.args,
              wtx_list->pEvtptInfo [ix].wtxEvtpt.event.numArgs *
              sizeof (TGT_ARG_T));
      evtpt_list->p_evtpt_info[ix].wtx_evtpt.context.context_type =
        wtx_list->pEvtptInfo [ix].wtxEvtpt.context.contextType;
      evtpt_list->p_evtpt_info[ix].wtx_evtpt.context.context_id =
        wtx_list->pEvtptInfo [ix].wtxEvtpt.context.contextId;
      evtpt_list->p_evtpt_info[ix].wtx_evtpt.action.action_type =
        wtx_list->pEvtptInfo [ix].wtxEvtpt.action.actionType;
      evtpt_list->p_evtpt_info[ix].wtx_evtpt.action.action_arg =
        wtx_list->pEvtptInfo [ix].wtxEvtpt.action.actionArg;
      evtpt_list->p_evtpt_info[ix].wtx_evtpt.action.call_rtn =
        wtx_list->pEvtptInfo [ix].wtxEvtpt.action.callRtn;
      evtpt_list->p_evtpt_info[ix].wtx_evtpt.action.call_arg =
        wtx_list->pEvtptInfo [ix].wtxEvtpt.action.callArg;
      evtpt_list->p_evtpt_info[ix].tool_id = wtx_list->pEvtptInfo [ix].toolId;
      evtpt_list->p_evtpt_info[ix].evtpt_num =
        wtx_list->pEvtptInfo [ix].evtptNum;
    }

  wtxResultFree (current_wtx_handle, wtx_list);
  return evtpt_list;
}

/* Free mem allocated by WTX API call.  */

int
wtx_result_free (void *p_result)
{
  return wtxResultFree (current_wtx_handle, p_result)  != WTX_ERROR;
}

/* Evaluate Gopher string on target.  */

WTX_GOPHER_TAPE *
wtx_gopher_eval (pd_id_t pd_id, const char *input_string)
{
#if WTX_PROT_VERSION == 3
  return wtxGopherEval (current_wtx_handle, pd_id, input_string);
#else
  return wtxGopherEval (current_wtx_handle, input_string);
#endif
}

/* Get info about memory pool.  */

WTX_MEM_INFO *
wtx_mem_info_get (pd_id_t pd_id)
{
#if WTX_PROT_VERSION == 3
  return wtxMemInfoGet (current_wtx_handle, pd_id);
#else
  return wtxMemInfoGet (current_wtx_handle);
#endif
}

/* Alloc blocks in memory pool.  */

TGT_ADDR_T
wtx_mem_alloc (pd_id_t pd_id, int num_bytes)
{
#if WTX_PROT_VERSION == 3
  return wtxMemAlloc (current_wtx_handle, pd_id, num_bytes);
#else
  return wtxMemAlloc (current_wtx_handle, num_bytes);
#endif
}

/* Perform checksum on target memory.  */

int
wtx_mem_checksum (pd_id_t pd_id, TGT_ADDR_T start_addr, int num_bytes)
{
#if WTX_PROT_VERSION == 3
  return wtxMemChecksum (current_wtx_handle, pd_id, start_addr, num_bytes);
#else
  return wtxMemChecksum (current_wtx_handle, start_addr, num_bytes);
#endif
}

/* Move a block of target memory.  */

int
wtx_mem_move (pd_id_t src_pd_id, TGT_ADDR_T src_addr,
              pd_id_t dst_pd_id, TGT_ADDR_T dest_addr, int num_bytes)
{
#if WTX_PROT_VERSION == 3
  return wtxMemMove (current_wtx_handle, src_pd_id, src_addr, dst_pd_id,
                     dest_addr, num_bytes) != WTX_ERROR;
#else
  return wtxMemMove (current_wtx_handle, src_addr, dest_addr,
                     num_bytes) != WTX_ERROR;
#endif
}

/* Free a block of target memory.  */

int
wtx_mem_free (pd_id_t pd_id, TGT_ADDR_T address)
{
#if WTX_PROT_VERSION == 3
  return wtxMemFree (current_wtx_handle, pd_id, address) != WTX_ERROR;
#else
  return wtxMemFree (current_wtx_handle, address) != WTX_ERROR;
#endif
}

/* Read memory from the target.  */

int
wtx_mem_read (pd_id_t pd_id, TGT_ADDR_T from_addr,
              void *to_addr, int num_bytes)
{
#if WTX_PROT_VERSION == 3
  return wtxMemRead (current_wtx_handle, pd_id, from_addr, to_addr,
                     num_bytes, 0);
#else
  return wtxMemRead (current_wtx_handle, from_addr, to_addr, num_bytes);
#endif
}

/* Read memory on WIDTH bytes.  */

int
wtx_mem_width_read (pd_id_t pd_id, TGT_ADDR_T from_addr,
                    void *to_addr, int num_bytes, int width)
{
#if WTX_PROT_VERSION == 3
  return wtxMemWidthRead (current_wtx_handle, pd_id, from_addr, to_addr,
                          num_bytes, width, 0);
#else
  return wtxMemWidthRead (current_wtx_handle, from_addr, to_addr, num_bytes,
                          width);
#endif
}

/* Write memory on the target.  */

int
wtx_mem_write (pd_id_t pd_id, void *from_addr,
               TGT_ADDR_T to_addr, int num_bytes)
{
#if WTX_PROT_VERSION == 3
  return wtxMemWrite (current_wtx_handle, pd_id, from_addr, to_addr, num_bytes, 0);
#else
  return wtxMemWrite (current_wtx_handle, from_addr, to_addr, num_bytes);
#endif
}

/* Write memory on the target on WIDTH bytes large.  */

int
wtx_mem_width_write (pd_id_t pd_id, void *from_addr,
                     TGT_ADDR_T to_addr, int num_bytes, int width)
{
#if WTX_PROT_VERSION == 3
  return wtxMemWidthWrite (current_wtx_handle, pd_id, from_addr, to_addr, num_bytes,
                           width, 0);
#else
  return wtxMemWidthWrite (current_wtx_handle, from_addr, to_addr, num_bytes, width);
#endif
}

/* Set target memory to given value.  */

int
wtx_mem_set (pd_id_t pd_id, TGT_ADDR_T addr, int num_bytes, int val)
{
#if WTX_PROT_VERSION == 3
  return wtxMemSet (current_wtx_handle, pd_id, addr, num_bytes, val);
#else
  return wtxMemSet (current_wtx_handle, addr, num_bytes, val);
#endif
}

/* Add memory to the agent pool.  */

int
wtx_mem_add_to_pool (pd_id_t pd_id, TGT_ADDR_T address, int size)
{
#if WTX_PROT_VERSION == 3
  return wtxMemAddToPool (current_wtx_handle, pd_id, address,
                          size) != WTX_ERROR;
#else
  return wtxMemAddToPool (current_wtx_handle, address, size) != WTX_ERROR;
#endif
}

/* Reallocate a block of target mem.  */

TGT_ADDR_T
wtx_mem_realloc (pd_id_t pd_id, TGT_ADDR_T address, int num_bytes)
{
#if WTX_PROT_VERSION == 3
  return wtxMemRealloc (current_wtx_handle, pd_id, address, num_bytes);
#else
  return wtxMemRealloc (current_wtx_handle, address, num_bytes);
#endif
}

/* Allocate aligned target memory.  */

TGT_ADDR_T
wtx_mem_align (pd_id_t pd_id, TGT_ADDR_T alignment, int num_bytes)
{
#if WTX_PROT_VERSION == 3
  return wtxMemAlign (current_wtx_handle, pd_id, alignment, num_bytes);
#else
  return wtxMemAlign (current_wtx_handle, alignment, num_bytes);
#endif
}

/* Scan target memory for pattern.  */

int
wtx_mem_scan (pd_id_t pd_id, int match, TGT_ADDR_T start_addr,
              TGT_ADDR_T end_addr, int num_bytes, void *pattern,
              TGT_ADDR_T *p_result)
{
#if WTX_PROT_VERSION == 3
  return wtxMemScan (current_wtx_handle, pd_id, TO_BOOL (match), start_addr,
                     end_addr, num_bytes, pattern, p_result) != WTX_ERROR;
#else
  return wtxMemScan (current_wtx_handle, TO_BOOL (match), start_addr, end_addr,
                     num_bytes, pattern, p_result) != WTX_ERROR;
#endif
}

/* Checks validity of target memory.  */

int
wtx_obj_module_checksum (pd_id_t pd_id, module_id_t module_id,
                         char *module_name)
{
#if WTX_PROT_VERSION == 3
  return wtxObjModuleChecksum (current_wtx_handle, pd_id, module_id,
                               module_name) != WTX_ERROR;
#else
  return wtxObjModuleChecksum (current_wtx_handle, module_id,
                               module_name) != WTX_ERROR;
#endif
}

/* Find obj module ID from name.  */

module_id_t
wtx_obj_module_find_id (pd_id_t pd_id, const char *module_name)
{
  module_id_t module_id;
#if WTX_PROT_VERSION == 3
  module_id = wtxObjModuleFindId (current_wtx_handle, pd_id, module_name);
#else
  module_id = wtxObjModuleFindId (current_wtx_handle, module_name);
#endif
  return module_id;
}

module_id_t
wtx_obj_module_in_system_find_id (const char *module_name)
{
#if WTX_PROT_VERSION == 3
  const pd_id_t current_pd = wtx_pd_current_get (current_wtx_handle);
  module_id_t module_id;

  module_id = wtxObjModuleFindId (current_wtx_handle, current_pd, module_name);
  if (module_id != WTX_ERROR)
    return module_id;

  return wtxObjModuleFindId (current_wtx_handle, WTX_PD_ALL, module_name);
#else
  return wtxObjModuleFindId (current_wtx_handle, module_name);
#endif
}

/* Find module name given its ID.  */

const char *
wtx_obj_module_find_name (pd_id_t pd_id, module_id_t module_id)
{
#if WTX_PROT_VERSION == 3
  return wtxObjModuleFindName (current_wtx_handle, pd_id, module_id);
#else
  return wtxObjModuleFindName (current_wtx_handle, module_id);
#endif
}

/* Allocate a new wtxapi_module_info.  */

static struct wtxapi_module_info *
new_wtxapi_module_info (WTX_MODULE_INFO *wtx_minfo)
{
  int ix;
  struct wtxapi_module_info *module_info;

  if (!wtx_minfo)
    return NULL;

  module_info = (struct wtxapi_module_info*)
    xmalloc (sizeof (struct wtxapi_module_info));
#if WTX_PROT_VERSION == 3
  module_info->pd_id = wtx_minfo->pdId;
  module_info->n_sections = wtx_minfo->nSections;
  module_info->section = (struct wtxapi_section_desc*)
    xmalloc (module_info->n_sections *
             sizeof (struct wtxapi_section_desc));
  for (ix = 0; ix < wtx_minfo->nSections; ix++)
    {
      module_info->section [ix].flags = wtx_minfo->section [ix].flags;
      module_info->section [ix].base_addr = wtx_minfo->section [ix].baseAddr;
      module_info->section [ix].length = wtx_minfo->section [ix].length;
    }
#else
  module_info->pd_id = NULL_PD;
  module_info->n_sections = wtx_minfo->nSegments;
  module_info->section = (struct wtxapi_section_desc*)
    xmalloc (module_info->n_sections *
             sizeof (struct wtxapi_section_desc));
  for (ix = 0; ix < wtx_minfo->nSegments; ix++)
    {
      module_info->section [ix].flags = wtx_minfo->segment [ix].flags;
      module_info->section [ix].base_addr = wtx_minfo->segment [ix].addr;
      module_info->section [ix].length = wtx_minfo->segment [ix].length;
    }
#endif
  module_info->module_id = wtx_minfo->moduleId;
  if (wtx_minfo->moduleName)
    module_info->module_name = xstrdup (wtx_minfo->moduleName);
  module_info->load_flag = wtx_minfo->loadFlag;
  module_info->undef_list = NULL;
  return module_info;
}

/* Give info on obj module.  */

struct wtxapi_module_info *
wtx_obj_module_info_get (pd_id_t pd_id, module_id_t module_id)
{
  struct wtxapi_module_info *module_info;
  WTX_MODULE_INFO *wtx_minfo;
#if WTX_PROT_VERSION == 3
  wtx_minfo = wtxObjModuleInfoGet (current_wtx_handle, pd_id, module_id);
#else
  wtx_minfo = wtxObjModuleInfoGet (current_wtx_handle, module_id);
#endif
  if (!wtx_minfo)
    return NULL;

  module_info = new_wtxapi_module_info (wtx_minfo);
  wtxResultFree (current_wtx_handle, wtx_minfo);
  return module_info;
}

/* Give info on obj mod.  */

struct wtxapi_module_info *
wtx_obj_module_info_and_path_get (pd_id_t pd_id, module_id_t module_id)
{
  int ix;
  WTX_MODULE_INFO *wtx_minfo;
  struct wtxapi_module_info *module_info;
#if WTX_PROT_VERSION == 3
  wtx_minfo = wtxObjModuleInfoAndPathGet (current_wtx_handle, pd_id, module_id);
#else
  wtx_minfo = wtxObjModuleInfoAndPathGet (current_wtx_handle, module_id);
#endif
  if (!wtx_minfo)
    return NULL;

  module_info = new_wtxapi_module_info (wtx_minfo);
  wtxResultFree (current_wtx_handle, wtx_minfo);
  return module_info;
}

/* List loaded obj modules.  */

struct wtxapi_module_list *
wtx_obj_module_list_get (pd_id_t pd_id)
{
  WTX_MODULE_LIST *wtx_mlist;
  struct wtxapi_module_list *list;
  int ix = 0;
#if WTX_PROT_VERSION == 3
  WTX_MOD_FIND_CRITERIA criteria;
  WTX_MODULE *current;

  memset (&criteria, 0, sizeof (criteria));
  criteria.options = WTX_MOD_FIND_ALL;
  criteria.pdId = pd_id;
  criteria.moduleId = 0;
  criteria.moduleName = NULL;
  criteria.ref = 0;

  wtx_mlist = wtxObjModuleListGet (current_wtx_handle, &criteria);

  if (!wtx_mlist)
    return NULL;

  list = (struct wtxapi_module_list*)
    xmalloc (sizeof (struct wtxapi_module_list));

  /* Note: as the module list is expected to have, say, less than 10
     elements, the full copy should not take much time.  */

  list->num_obj_mod = 0;
  for (current = wtx_mlist->pModule; current; current = current->next)
    list->num_obj_mod++;

  list->mod_id_array = (int *)
    xmalloc (list->num_obj_mod * sizeof (int));
  list->pd_id_array = (pd_id_t *)
    xmalloc (list->num_obj_mod * sizeof (pd_id_t));
  for (current = wtx_mlist->pModule; current; current = current->next)
    {
      list->mod_id_array [ix] = current->moduleId;
      list->pd_id_array [ix] = current->pdId;
      ix++;
    }

#else
  wtx_mlist = wtxObjModuleList (current_wtx_handle);

  if (!wtx_mlist)
    return NULL;

  list = (struct wtxapi_module_list*)
    xmalloc (sizeof (struct wtxapi_module_list));
  list->num_obj_mod = wtx_mlist->numObjMod;
  list->mod_id_array = (int *)
    xmalloc (list->num_obj_mod * sizeof (int));
  list->pd_id_array = (pd_id_t *)
    xmalloc (list->num_obj_mod * sizeof (pd_id_t));
  for (ix = 0; ix < list->num_obj_mod; ix++)
    list->mod_id_array [ix] = wtx_mlist->modIdList [ix];
  memset (list->pd_id_array, 0, list->num_obj_mod * sizeof (pd_id_t));
#endif
  wtxResultFree (current_wtx_handle, wtx_mlist);
  return list;
}

/* Load a new object module.  */

struct wtxapi_module_info *
wtx_obj_module_load (pd_id_t pd_id, char *filename, int load_flags)
{
  int ix;
  struct wtxapi_module_info *minfo;

#if WTX_PROT_VERSION == 3
  WTX_MODULE_FILE_DESC wtx_fdesc;
  WTX_MODULE_INFO *wtx_minfo;

  memset (&wtx_fdesc, 0, sizeof (wtx_fdesc));
  wtx_fdesc.filename = filename;
  wtx_fdesc.loadFlag = load_flags;

  wtx_minfo = wtxObjModuleLoad (current_wtx_handle, pd_id, &wtx_fdesc,
                                WTX_LOAD_FROM_TOOL);

  if (!wtx_minfo)
    return NULL;

  minfo = new_wtxapi_module_info (wtx_minfo);
  minfo->undef_list =
    new_wtxapi_symbol_list_from_symbol (wtx_minfo->undefSymList.pSymbol);
  if (minfo->undef_list)
    minfo->undef_list->wtx_result_to_cleanup = (void *)wtx_minfo;
#else
  WTX_LD_M_FILE_DESC wtx_fdesc;
  WTX_LD_M_FILE_DESC *wtx_minfo;

  memset (&wtx_fdesc, 0, sizeof (wtx_fdesc));
  wtx_fdesc.filename = filename;
  wtx_fdesc.loadFlag = load_flags;

  wtx_minfo = wtxObjModuleLoad (current_wtx_handle, &wtx_fdesc);

  if (!wtx_minfo)
    return NULL;

  minfo = (struct wtxapi_module_info *)
    xmalloc (sizeof (struct wtxapi_module_info));
  minfo->pd_id = WTX_PD_ALL;
  minfo->module_id = wtx_minfo->moduleId;
  if (wtx_minfo->filename)
    minfo->module_name = xstrdup (wtx_minfo->filename);
  minfo->load_flag = wtx_minfo->loadFlag;
  minfo->n_sections = wtx_minfo->nSections;
  minfo->section = (struct wtxapi_section_desc*)
    xmalloc (minfo->n_sections *
             sizeof (struct wtxapi_section_desc));
  for (ix = 0; ix < wtx_minfo->nSections; ix++)
    {
      minfo->section [ix].flags = wtx_minfo->section [ix].flags;
      minfo->section [ix].base_addr = wtx_minfo->section [ix].addr;
      minfo->section [ix].length = wtx_minfo->section [ix].length;
    }
  minfo->undef_list =
    new_wtxapi_symbol_list_from_symbol (wtx_minfo->undefSymList.pSymbol);
  if (minfo->undef_list)
    minfo->undef_list->wtx_result_to_cleanup = (void *)wtx_minfo;
#endif
  return minfo;
}

/* Unload an obj module from target.  */

int
wtx_obj_module_unload (pd_id_t pd_id, module_id_t mod_id)
{
#if WTX_PROT_VERSION == 3
  return wtxObjModuleUnload (current_wtx_handle, pd_id, 0,
                             mod_id) != WTX_ERROR;
#else
  return wtxObjModuleUnload (current_wtx_handle, mod_id) != WTX_ERROR;
#endif
}

/* Unload an obj. module from target.  */

int
wtx_obj_module_by_name_unload (pd_id_t pd_id, char *name)
{
#if WTX_PROT_VERSION == 3
  return wtxObjModuleByNameUnload (current_wtx_handle, pd_id, 0,
                                   name) != WTX_ERROR;
#else
  return wtxObjModuleByNameUnload (current_wtx_handle, name) != WTX_ERROR;
#endif
}

/* Send events matching expression.  */

int
wtx_register_for_event (const char *reg_exp)
{
  return wtxRegisterForEvent (current_wtx_handle, reg_exp) != WTX_ERROR;
}

/* Read register data from the target.  */

int
wtx_regs_get (WTX_CONTEXT_TYPE context_type,
              WTX_CONTEXT_ID_T context_id, WTX_REG_SET_TYPE reg_set,
              int first_byte, int num_bytes, void *reg_memory)
{
  return wtxRegsGet (current_wtx_handle, context_type, context_id, reg_set,
                     first_byte, num_bytes, reg_memory) != WTX_ERROR;
}

/* Write to registers on the target.  */

int
wtx_regs_set (WTX_CONTEXT_TYPE context_type, WTX_CONTEXT_ID_T context_id,
              WTX_REG_SET_TYPE reg_set, int first_byte, int num_bytes,
              void *reg_memory)
{
  return wtxRegsSet (current_wtx_handle, context_type, context_id, reg_set,
                     first_byte, num_bytes, reg_memory) != WTX_ERROR;
}

/* Convert str to a TGT_ADDR_T.  */

TGT_ADDR_T
wtx_str_to_tgt_addr (const char *str)
{
  return wtxStrToTgtAddr (current_wtx_handle, str);
}

/* Convert str to context ID.  */

WTX_CONTEXT_ID_T
wtx_str_to_context_id (const char *str)
{
  return wtxStrToContextId (current_wtx_handle, str);
}

/* Convert str ton context type.  */

WTX_CONTEXT_TYPE
wtx_str_to_context_type (const char *str)
{
  return wtxStrToContextType (current_wtx_handle, str);
}

/* Convert str to an int.  */

int
wtx_str_to_int32 (const char *str)
{
  return wtxStrToInt32 (current_wtx_handle, str);
}

/* Convert string to event type.  */

WTX_EVENT_TYPE
wtx_str_to_event_type (const char *str)
{
  return wtxStrToEventType (current_wtx_handle, str);
}

/* Convert event type to a string.  */

const char *
wtx_event_to_str_type (WTX_EVENT_TYPE event)
{
  return wtxEventToStrType (event);
}

/* Add symbol with given params.  */

int
wtx_sym_add (pd_id_t pd_id, const char *sym_name,
             TGT_ADDR_T sym_value, int sym_type)
{
#if WTX_PROT_VERSION == 3
  return wtxSymAdd (current_wtx_handle, pd_id, sym_name, sym_value,
                    sym_type) != WTX_ERROR;
#else
  return wtxSymAdd (current_wtx_handle, sym_name, sym_value,
                    (UINT8) sym_type) != WTX_ERROR;
#endif
}

static struct wtxapi_symbol_list *
new_wtxapi_symbol_list (WTX_SYM_LIST* wtx_sym)
{
  struct wtxapi_symbol_list *sym_list;

  if (!wtx_sym)
    return NULL;

  sym_list = (struct wtxapi_symbol_list*)
    xmalloc (sizeof (struct wtxapi_symbol_list));
  sym_list->wtx_result_to_cleanup = (void *) wtx_sym;
  sym_list->first = wtx_sym->pSymbol;
  sym_list->current = sym_list->first;
  return sym_list;
}

static struct wtxapi_symbol_list *
new_wtxapi_symbol_list_from_symbol (WTX_SYMBOL* wtx_sym)
{
  struct wtxapi_symbol_list *sym_list;

  if (!wtx_sym)
    return NULL;

  sym_list = (struct wtxapi_symbol_list*)
    xmalloc (sizeof (struct wtxapi_symbol_list));
  sym_list->wtx_result_to_cleanup = (void *) wtx_sym;
  sym_list->first = wtx_sym;
  sym_list->current = wtx_sym;
  return sym_list;
}

/* Find info on symbol. Based on wtxSymFind WTX 2.0, with two differences:
   _ one additional parameter: the protection domain ID (PD_ID);
   _ no filter on the type.  */

struct wtxapi_symbol_list *
wtx_sym_find (pd_id_t pd_id, char *sym_name,
              TGT_ADDR_T sym_value, int exact_name)
{
#if WTX_PROT_VERSION == 3
  WTX_SYM_FIND_CRITERIA criteria;
  memset (&criteria, 0, sizeof (criteria));
  criteria.pdId = pd_id;
  criteria.type = 0;
  criteria.nSymbols = 1;
  criteria.ref = 0;
  criteria.moduleId = 0;
  criteria.moduleName = NULL;

  if (sym_name)
    {
      criteria.findName = sym_name;
      criteria.findValue = 0;
      criteria.options |= WTX_SYM_FIND_BY_NAME;

      if (exact_name)
        criteria.options |= WTX_SYM_FIND_BY_EXACT_NAME;
    }
  else
    {
      criteria.findName = NULL;
      criteria.findValue = sym_value;
      criteria.options |= WTX_SYM_FIND_BY_VALUE;

      if (exact_name)
        criteria.options |= WTX_SYM_FIND_BY_EXACT_VALUE;
    }

  return new_wtxapi_symbol_list_from_symbol (wtxSymFind (current_wtx_handle,
                                                         &criteria));
#else
  return new_wtxapi_symbol_list_from_symbol (wtxSymFind (current_wtx_handle,
                                                         sym_name,
                                                         sym_value,
                                                         exact_name, 0, 0));
#endif
}

/* Get list of symbols. Based on wtxSymListGet WTX 2.0, with three differences:
   - one additional parameter: the protection domain ID (PD_ID);
   - no filter on the unknown symbols;
   - no filter on the module name/id.  */

struct wtxapi_symbol_list *
wtx_sym_list_get (pd_id_t pd_id,  char *substring, TGT_ADDR_T value)
{
  struct wtxapi_symbol_list *sym_list;
  WTX_SYM_LIST *wtx_sym_list;
#if WTX_PROT_VERSION == 3
  WTX_SYM_FIND_CRITERIA criteria;
  memset (&criteria, 0, sizeof (criteria));
  criteria.pdId = pd_id;
  criteria.type = 0;
  criteria.nSymbols = 0;
  criteria.ref = 0;
  criteria.moduleId = 0;
  criteria.moduleName = NULL;

  if (substring)
    {
      criteria.findName = substring;
      criteria.findValue = 0;
      criteria.options |= WTX_SYM_FIND_BY_NAME;
    }
  else
    {
      criteria.findName = NULL;
      criteria.findValue = value;
      criteria.options |= WTX_SYM_FIND_BY_VALUE;
    }

  wtx_sym_list = wtxSymListGet (current_wtx_handle, &criteria);
#else
  wtx_sym_list = wtxSymListGet (current_wtx_handle, substring, NULL, value, FALSE);
#endif
  sym_list = new_wtxapi_symbol_list (wtx_sym_list);
  return sym_list;
}

/* Get list of symbols.  */

struct wtxapi_symbol_list *
wtx_sym_list_by_module_id_get (pd_id_t pd_id, const char *string,
                               module_id_t module_id, TGT_ADDR_T value,
                               int list_unknown)
{
#if WTX_PROT_VERSION == 3
  return new_wtxapi_symbol_list
    (wtxSymListByModuleIdGet (current_wtx_handle, pd_id, string, module_id,
                              value, TO_BOOL (list_unknown)));
#else
  return new_wtxapi_symbol_list
    (wtxSymListByModuleIdGet (current_wtx_handle, string, module_id, value,
                              TO_BOOL (list_unknown)));
#endif
}

/* Get list of symbols.  */

struct wtxapi_symbol_list *
wtx_sym_list_by_module_name_get (pd_id_t pd_id, const char *string,
                                 const char *module_name, TGT_ADDR_T value,
                                 int list_unknown)
{
#if WTX_PROT_VERSION == 3
  return new_wtxapi_symbol_list
    (wtxSymListByModuleNameGet (current_wtx_handle, pd_id, string, module_name,
                                value, TO_BOOL (list_unknown)));
#else
  return new_wtxapi_symbol_list
    (wtxSymListByModuleNameGet (current_wtx_handle, string, module_name,
                                value, TO_BOOL (list_unknown)));
#endif
}

/* Remove a symbol from sym table.  */

int
wtx_sym_remove (pd_id_t pd_id, const char *sym_name, int sym_type)
{
#if WTX_PROT_VERSION == 3
  return wtxSymRemove (current_wtx_handle, pd_id, sym_name,
                       sym_type) != WTX_ERROR;
#else
  return wtxSymRemove (current_wtx_handle, sym_name, sym_type) != WTX_ERROR;
#endif
}

/* Return sym table info.  */

struct wtxapi_sym_tbl_info *
wtx_sym_tbl_info_get (pd_id_t pd_id)
{
  WTX_SYM_TBL_INFO *wtx_tbl_info;
  struct wtxapi_sym_tbl_info *tbl_info;
  pd_id_t current_pd_id;
#if WTX_PROT_VERSION == 3
  wtx_tbl_info = wtxSymTblInfoGet (current_wtx_handle, pd_id);
  current_pd_id = wtx_tbl_info->pdId;
#else
  wtx_tbl_info = wtxSymTblInfoGet (current_wtx_handle);
  current_pd_id = NULL_PD;
#endif

  if (!wtx_tbl_info)
    return NULL;

  tbl_info = (struct wtxapi_sym_tbl_info*)
    xmalloc (sizeof (struct wtxapi_sym_tbl_info));
  tbl_info->pd_id = current_pd_id;
  tbl_info->sym_num = wtx_tbl_info->symNum;
  tbl_info->same_name_ok = (wtx_tbl_info->sameNameOk == TRUE);
  wtxResultFree (current_wtx_handle, wtx_tbl_info);
  return tbl_info;
}

/* Reset the target.  */

int
wtx_target_reset ()
{
  return wtxTargetReset (current_wtx_handle) != WTX_ERROR;
}

/* Get target server name.  */

const char *
wtx_ts_name_get ()
{
  return wtxTsNameGet (current_wtx_handle);
}

/* Get the target CPU type code.  */

int
wtx_target_cpu_type_get ()
{
  return wtxTargetCpuTypeGet (current_wtx_handle);
}

/* Check for floating point processor.  */

int
wtx_target_has_fpp_get ()
{
  return wtxTargetHasFppGet (current_wtx_handle);
}

/* Get edianness of target.  */

WTX_ENDIAN_T
wtx_target_endian_get ()
{
  return wtxTargetEndianGet (current_wtx_handle);
}

/* Get target boot line info.  */

const char *
wtx_target_bootline_get ()
{
  return wtxTargetBootlineGet (current_wtx_handle);
}

/* Return name of current tool.  */

const char *
wtx_tool_name_get ()
{
  return wtxToolNameGet (current_wtx_handle);
}

/* Return the Tornado version.  */

const char *
wtx_ts_version_get ()
{
  return wtxTsVersionGet (current_wtx_handle);
}

/* Unregister for some events.  */

int
wtx_unregister_for_event (char *reg_exp)
{
  return wtxUnregisterForEvent (current_wtx_handle, reg_exp) != WTX_ERROR;
}

/* Call func on target within agent.  */

int
wtx_direct_call (TGT_ADDR_T entry, void *p_ret_val,
                 TGT_ARG_T arg0, TGT_ARG_T arg1, TGT_ARG_T arg2,
                 TGT_ARG_T arg3, TGT_ARG_T arg4, TGT_ARG_T arg5,
                 TGT_ARG_T arg6, TGT_ARG_T arg7, TGT_ARG_T arg8,
                 TGT_ARG_T arg9)
{
#if WTX_PROT_VERSION == 3
  return wtxDirectCall (current_wtx_handle, entry, p_ret_val, 10, arg0, arg1,
                        arg2, arg3, arg4, arg5, arg6, arg7, arg8,
                        arg9) != WTX_ERROR;
#else
  return wtxDirectCall (current_wtx_handle, entry, p_ret_val, arg0, arg1,
                        arg2, arg3, arg4, arg5, arg6, arg7, arg8,
                        arg9) != WTX_ERROR;
#endif
}

/* Get info about target and server.  */

struct wtxapi_ts_info *
wtx_ts_info_get ()
{
  WTX_TS_INFO *wtx_tsinfo = wtxTsInfoGet (current_wtx_handle);
  struct wtxapi_ts_info *ts_info;
  if (!wtx_tsinfo)
    return NULL;

  ts_info = (struct wtxapi_ts_info*)
    xmalloc (sizeof (struct wtxapi_ts_info));
  ts_info->tgt_link_desc = wtx_tsinfo->tgtLinkDesc;
  ts_info->tgt_info = wtx_tsinfo->tgtInfo;
  if (wtx_tsinfo->version)
    ts_info->version = xstrdup (wtx_tsinfo->version);
  if (wtx_tsinfo->userName)
    ts_info->user_name = xstrdup (wtx_tsinfo->userName);
  if (wtx_tsinfo->lockMsg)
    ts_info->lock_msg = xstrdup (wtx_tsinfo->lockMsg);
#if WTX_PROT_VERSION == 3
  ts_info->pd_initialized = (wtx_tsinfo->pdInitialized == TRUE);
#else
  ts_info->pd_initialized = 1;
#endif
  return ts_info;
}

/* Reattach to the target.  */

int
wtx_target_attach ()
{
  return wtxTargetAttach (current_wtx_handle) != WTX_ERROR;
}

/* Probe to see if registry is running.  */

int
wtx_probe ()
{
  return wtxProbe (current_wtx_handle) != WTX_ERROR;
}

/* Set WTX timeout.  */

int
wtx_timeout_set (int msec)
{
  return wtxTimeoutSet (current_wtx_handle, msec) != WTX_ERROR;
}

/* Get the current timeout.  */

int
wtx_timeout_get (int *p_msec)
{
  UINT32 wtx_timeout;
  int result = wtxTimeoutGet (current_wtx_handle, &wtx_timeout) != WTX_ERROR;
  *p_msec = wtx_timeout;
  return result;
}

/* Create a new protection domain.  */

pd_id_t
wtx_pd_create (const char *name, int options, int heap_size,
               int low_priority, int high_priority,
               TGT_ADDR_T page_pool_list, const char *link_path)
{
#if WTX_PROT_VERSION == 3
  return wtxPdCreate (current_wtx_handle, name, options, heap_size, low_priority,
                      high_priority, page_pool_list, link_path);
#else
  return NULL_PD;
#endif
}

/* Get kernel Protection Domain ID.  */

pd_id_t
wtx_pd_kernel_get ()
{
#if WTX_PROT_VERSION == 3
  return wtxPdKernelGet (current_wtx_handle);
#else
  return NULL_PD;
#endif
}

/* Get the current Protection Domain.  */

pd_id_t
wtx_pd_current_get ()
{
#if WTX_PROT_VERSION == 3
  return wtxPdCurrentGet (current_wtx_handle);
#else
  return NULL_PD;
#endif
}

/* Set the current Protection Domain.  */

int
wtx_pd_current_set (pd_id_t pd_id)
{
#if WTX_PROT_VERSION == 3
  return wtxPdCurrentSet (current_wtx_handle, pd_id) != WTX_ERROR;
#else
  return pd_id == NULL_PD;
#endif
}

/* Get the list of allocated PDs.  */

/* NOTE: it performs a full copy of the PD description list. It should
   not be a problem, we should not expect to many PDs. If there are
   some speed issues, the solution would be to have the same kind of
   model we use for wtxapi_symbol_list. To be investigated.  */

struct wtxapi_pd_desc_q *
wtx_pd_info_q_get ()
{
  struct wtxapi_pd_desc_q *pd_desc_list;
#if WTX_PROT_VERSION == 3
  WTX_PD_DESC_Q *wtx_pd_desc_q = wtxPdInfoQGet (current_wtx_handle);
  WTX_PD_DESC_Q *current_wtx_desc = wtx_pd_desc_q;
  struct wtxapi_pd_desc_q *current_desc;
  int ix;

  if (wtx_pd_desc_q == NULL)
    return NULL;

  pd_desc_list = (struct wtxapi_pd_desc_q*)
    xmalloc (sizeof (struct wtxapi_pd_desc_q));
  current_desc = pd_desc_list;

  for (current_wtx_desc = wtx_pd_desc_q;
       current_wtx_desc != NULL;
       current_wtx_desc = current_wtx_desc->pNext)
    {
      current_desc->pd_desc.pd_id = current_wtx_desc->wtxPdDesc.pdId;
      if (current_wtx_desc->wtxPdDesc.pdName)
        current_desc->pd_desc.pd_name =
          xstrdup (current_wtx_desc->wtxPdDesc.pdName);
      current_desc->pd_desc.pd_flags = current_wtx_desc->wtxPdDesc.pdFlags;
      if (current_wtx_desc->wtxPdDesc.pdLinkPathStr)
        current_desc->pd_desc.pd_link_path_str =
          xstrdup (current_wtx_desc->wtxPdDesc.pdLinkPathStr);
      current_desc->pd_desc.pd_link_path_count =
        current_wtx_desc->wtxPdDesc.pdLinkPathCount;
      if (current_wtx_desc->wtxPdDesc.pdLinkPathCount)
        {
          current_desc->pd_desc.pd_link_path = (pd_id_t *)
            xmalloc (current_wtx_desc->wtxPdDesc.pdLinkPathCount
                     * sizeof (pd_id_t));
          for (ix = 0; ix < current_wtx_desc->wtxPdDesc.pdLinkPathCount; ix++)
            current_desc->pd_desc.pd_link_path [ix]
              = current_wtx_desc->wtxPdDesc.pdLinkPath [ix];

        }
      else
        {
          current_desc->pd_desc.pd_link_path = NULL;
        }
      if (current_wtx_desc->wtxPdDesc.pdAttachToCount)
        {
          current_desc->pd_desc.pd_attach_to_count =
            current_wtx_desc->wtxPdDesc.pdAttachToCount;
          current_desc->pd_desc.pd_attach_to = (pd_id_t *)
            xmalloc (current_wtx_desc->wtxPdDesc.pdAttachToCount
                     * sizeof (pd_id_t));
          for (ix = 0; ix < current_wtx_desc->wtxPdDesc.pdAttachToCount; ix++)
            current_desc->pd_desc.pd_attach_to [ix]
              = current_wtx_desc->wtxPdDesc.pdAttachTo [ix];
        }
      else
        {
          current_desc->pd_desc.pd_attach_to = NULL;
        }

      if (current_wtx_desc->pNext != NULL)
        {
          current_desc->next = (struct wtxapi_pd_desc_q*)
            xmalloc (sizeof (struct wtxapi_pd_desc_q));
          current_desc = current_desc->next;
        }
      else
        current_desc->next = NULL;
    }
  wtxResultFree (current_wtx_handle, wtx_pd_desc_q);
#else
  pd_desc_list = (struct wtxapi_pd_desc_q*)
    xmalloc (sizeof (struct wtxapi_pd_desc_q));
  memset (&pd_desc_list, 0, sizeof (pd_desc_list));
  pd_desc_list->pd_desc.pd_name = xstrdup ("kernel");
  pd_desc_list->pd_desc.pd_link_path_str = xstrdup ("<none>");
#endif
  return pd_desc_list;
}

/* Test target server availability.  */

int
wtxapi_target_server_available_p ()
{
  WTX_TS_INFO *ts_info = wtxTsInfoGet (current_wtx_handle);
  return ts_info != NULL;
}


/* Symbol list handling.  */

struct wtxapi_symbol *
get_current_wtxapi_symbol (struct wtxapi_symbol_list *list)
{
  struct wtxapi_symbol *sym;
  if (!list || !list->current)
    return NULL;

  sym = (struct wtxapi_symbol *) xmalloc (sizeof (struct wtxapi_symbol));
  copy_current_wtxapi_symbol (list, sym, wtxapi_symbol_copy_full);
  return sym;
}

void
copy_current_wtxapi_symbol (struct wtxapi_symbol_list *list,
                            struct wtxapi_symbol *to,
                            int options)
{
  if (!to || !list || !list->current)
    return;

  to->status = list->current->status;
#if WTX_PROT_VERSION == 3
  to->pd_id = list->current->pdId;
#else
  to->pd_id = NULL_PD;
#endif
  if (options != wtxapi_symbol_copy_dont_copy_name
      && options != wtxapi_symbol_copy_dont_copy_strings
      && list->current->name)
    to->name = xstrdup (list->current->name);
  else
    to->name = NULL;

  if (options != wtxapi_symbol_copy_dont_copy_module_name
      && options != wtxapi_symbol_copy_dont_copy_strings
      && list->current->moduleName)
    to->module_name = xstrdup (list->current->moduleName);
  else
    to->module_name = NULL;

  to->exact_name = (list->current->exactName == TRUE);
  to->value = list->current->value;
  to->type = list->current->type;
  to->type_mask = list->current->typeMask;
}

int
go_to_first_element_in_wtxapi_sym_list (struct wtxapi_symbol_list *list)
{
  if (!list || !list->first)
    return 0;

  list->current = list->first;
  return 1;
}

int
go_to_next_element_in_wtxapi_sym_list (struct wtxapi_symbol_list *list)
{
  if (!list->current || !list->current->next)
    return 0;

  list->current = list->current->next;
  return 1;
}

/* Deallocation functions.  */

void
free_wtxapi_evtpt_list (struct wtxapi_evtpt_list *to_free)
{
  int ix;
  for (ix = 0; ix < to_free->n_evtpt; ix++)
    {
      xfree (to_free->p_evtpt_info[ix].wtx_evtpt.event.args);
    }
  xfree (to_free->p_evtpt_info);
  xfree (to_free);
}

void
free_wtxapi_module_list (struct wtxapi_module_list *to_free)
{
  xfree (to_free->mod_id_array);
  xfree (to_free->pd_id_array);
  xfree (to_free);
}

void
free_wtxapi_pd_desc_q (struct wtxapi_pd_desc_q *to_free)
{
  struct wtxapi_pd_desc_q* current;
  struct wtxapi_pd_desc_q* element_to_deallocate = NULL;

  for (current = to_free; current != NULL; current = current->next)
    {
      if (element_to_deallocate)
        xfree (element_to_deallocate);
      if (current->pd_desc.pd_name)
        xfree (current->pd_desc.pd_name);
      if (current->pd_desc.pd_link_path_str)
        xfree (current->pd_desc.pd_link_path_str);
      if (current->pd_desc.pd_link_path)
        xfree (current->pd_desc.pd_link_path);
      if (current->pd_desc.pd_attach_to)
        xfree (current->pd_desc.pd_attach_to);
      element_to_deallocate = current;
    }
}

void
free_wtxapi_symbol (struct wtxapi_symbol* to_free)
{
  if (to_free->name)
    xfree (to_free->name);

  if (to_free->module_name)
    xfree (to_free->module_name);

  xfree (to_free);
}

void
free_wtxapi_symbol_list (struct wtxapi_symbol_list* to_free)
{
  if (to_free->wtx_result_to_cleanup != NULL)
    wtxResultFree (current_wtx_handle, to_free->wtx_result_to_cleanup);

  xfree (to_free);
}

void
free_wtxapi_module_info (struct wtxapi_module_info *to_free)
{
  if (to_free->undef_list)
    free_wtxapi_symbol_list (to_free->undef_list);
  xfree (to_free->module_name);
  xfree (to_free->section);
  xfree (to_free);
}

void
free_wtxapi_ts_info (struct wtxapi_ts_info *to_free)
{
  if (to_free->version)
    xstrdup (to_free->version);

  if (to_free->user_name)
    xstrdup (to_free->user_name);

  if (to_free->lock_msg)
    xstrdup (to_free->lock_msg);

  xfree (to_free);
}


/* cleanup functions.  */

void
cleanup_wtxapi_evtpt_list (void *to_free)
{
  free_wtxapi_evtpt_list ((struct wtxapi_evtpt_list *) to_free);
}

void
cleanup_wtxapi_module_list (void *to_free)
{
  free_wtxapi_module_list ((struct wtxapi_module_list *) to_free);
}

void
cleanup_wtxapi_pd_desc_q (void *to_free)
{
  free_wtxapi_pd_desc_q ((struct wtxapi_pd_desc_q *) to_free);
}

void
cleanup_wtxapi_symbol (void *to_free)
{
  free_wtxapi_symbol ((struct wtxapi_symbol *) to_free);
}

void
cleanup_wtxapi_symbol_list (void *to_free)
{
  free_wtxapi_symbol_list ((struct wtxapi_symbol_list*) to_free);
}

void
cleanup_wtxapi_module_info (void *to_free)
{
  free_wtxapi_module_info ((struct wtxapi_module_info *) to_free);
}

void
cleanup_wtxapi_ts_info (void *to_free)
{
  free_wtxapi_ts_info ((struct wtxapi_ts_info *) to_free);
}

