/*
 * GNoise
 *
 * Copyright (C) 1999-2001 Dwight Engen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: ladspa.c,v 1.2 2001/01/20 19:40:11 dengen Exp $
 *
 */

#include "config.h"

#ifdef ENABLE_LADSPA
#include <dirent.h>
#include <dlfcn.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "edit.h"
#include "gnoise.h"
#include "ladspa.h"


typedef struct plug_info
{
    guint             plugs;		/* number of plugins in file */
    char              file[PATH_MAX+1];	/* path to plugin.so */
    void              *dll;		/* handle to dll or NULL if not open */
    LADSPA_Descriptor *desc;		/* descriptor */
    LADSPA_Handle     *hand;		/* handle to instantiated instance */
    LADSPA_Data       *ctl_value;	/* array of control values */
} plug_info_t;

typedef struct
{
    guint              indx;
    plug_info_t        *pi;
    GtkWidget          *widget;
} plug_menu_t;


static void      ladspa_ctl_value_changed (GtkObject *obj, gpointer ctl_val);
static void      ladspa_dlg_apply (GtkButton *button, gpointer user_data);
static void      ladspa_dlg_cancel (GtkButton *button, gpointer user_data);
static void      ladspa_dlg_create (GtkButton *button, gpointer user_data);
static gboolean  ladspa_dlg_destroy(GtkWidget *widget, GdkEvent *event, gpointer user_data);
static void      ladspa_dlg_free (plug_info_t *pi);
       void      ladspa_plug_run (float *, chnl_indx channels, smpl_indx samples);
       chnl_indx ladspa_plug_channels(void);

static GList       *pm_list = NULL;
static GList       *pi_list = NULL;
static GtkWidget   *win_ladspa;
static plug_menu_t *plug_run;
static char        default_lib_path[] =
		   "/usr/local/lib/ladspa:/usr/lib/ladspa:./plugins:.";

static void
ladspa_dlg_create (GtkButton *button, gpointer user_data)
{
    LADSPA_Descriptor *(*sym)(int);
    plug_menu_t       *pmi = (plug_menu_t *)user_data;
    plug_info_t       *pi = pmi->pi;
    guint              port;

    GtkWidget *MainVBox;
    GtkWidget *PluginFrame;
    GtkWidget *PortsVBox;
    GtkWidget *PortVBox;
    GtkWidget *PortLabel;
    GtkWidget *PortValue;
    GtkWidget *PluginButtonBox;
    GtkWidget *Apply;
    GtkWidget *Cancel;
    GtkAdjustment *Adj;
    
    if (win_ladspa != NULL)
	return;

    pi->dll = dlopen(pi->file, RTLD_LAZY);
    if (dlerror() != NULL || pi->dll == NULL)
	return;

    sym = dlsym(pi->dll, "ladspa_descriptor");
    if (dlerror() != NULL || !sym)
    {
	log("LADSPA", "Unable to find ladspa_descriptor\n");
	goto dll_close;
    }

    pi->desc = (*sym)(pmi->indx);
    if (pi->desc->PortCount == 0)
    {
	log("LADSPA", "error, plugin has no ports!\n");
	goto dll_close;
    }

    pi->hand = pi->desc->instantiate(pi->desc, 44100);
    if (pi->hand == NULL)
    {
	log("LADSPA", "unable to instantiate plugin\n");
	goto dll_close;
    }


    win_ladspa = gtk_window_new(GTK_WINDOW_DIALOG);
    gtk_window_set_title(GTK_WINDOW(win_ladspa), ("LADSPA Plugin"));


    MainVBox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show (MainVBox);
    gtk_container_add(GTK_CONTAINER(win_ladspa), MainVBox);

    PluginFrame = gtk_frame_new((pi->desc->Name));
    gtk_widget_show(PluginFrame);
    gtk_box_pack_start(GTK_BOX (MainVBox), PluginFrame, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER (PluginFrame), 2);
    gtk_frame_set_label_align(GTK_FRAME (PluginFrame), 0.5, 0.5);

    PortsVBox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(PortsVBox);
    gtk_container_add(GTK_CONTAINER (PluginFrame), PortsVBox);
    gtk_container_set_border_width(GTK_CONTAINER (PortsVBox), 4);

    pi->ctl_value = calloc(pi->desc->PortCount, sizeof(LADSPA_Data));

    for (port = 0; port < pi->desc->PortCount; port++)
    {
	LADSPA_PortDescriptor		p_desc = pi->desc->PortDescriptors[port];
	LADSPA_PortRangeHintDescriptor	p_hint = pi->desc->PortRangeHints[port].HintDescriptor;
	LADSPA_Data			lower;
	LADSPA_Data			upper;
	
	if (!LADSPA_IS_PORT_CONTROL(p_desc))
	    continue;

	lower = 0;
	upper = 1;

	if (LADSPA_IS_HINT_BOUNDED_BELOW(p_hint) ||
	    LADSPA_IS_HINT_BOUNDED_ABOVE(p_hint))
	{
	    if (LADSPA_IS_HINT_BOUNDED_BELOW(p_hint))
	    {
		lower = pi->desc->PortRangeHints[port].LowerBound;
		/* FIXME: 
	         *  if (LADSPA_IS_HINT_SAMPLE_RATE(hint)) 
	         *	indicate somehow value will be * samplerate...
	         */
	    }

	    if (LADSPA_IS_HINT_BOUNDED_ABOVE(p_hint))
	    {
		upper = pi->desc->PortRangeHints[port].UpperBound;
		/* FIXME: 
	         *  if (LADSPA_IS_HINT_SAMPLE_RATE(hint)) 
	         *	indicate somehow value will be * samplerate...
	         */
	    }

	    PortVBox = gtk_vbox_new(FALSE, 0);
	    gtk_widget_show(PortVBox);
	    gtk_box_pack_start(GTK_BOX(PortsVBox), PortVBox, FALSE, TRUE, 3);

	    PortLabel = gtk_label_new(pi->desc->PortNames[port]);
	    gtk_widget_set_name (PortLabel, "PortLabel");
	    gtk_widget_show(PortLabel);
	    gtk_box_pack_start(GTK_BOX(PortVBox), PortLabel, FALSE, FALSE, 1);
	    gtk_misc_set_alignment(GTK_MISC (PortLabel), 0, 0.5);

	    Adj = GTK_ADJUSTMENT(gtk_adjustment_new (0, lower, upper, 0, 0.5, 0));
	    PortValue = gtk_hscale_new(Adj);
	    gtk_widget_show(PortValue);
	    gtk_box_pack_start(GTK_BOX(PortVBox), PortValue, FALSE, TRUE, 0);
	    gtk_scale_set_value_pos(GTK_SCALE(PortValue), GTK_POS_LEFT);
	    gtk_scale_set_digits(GTK_SCALE(PortValue), 2);
	    gtk_signal_connect(GTK_OBJECT(Adj), "value_changed",
			       GTK_SIGNAL_FUNC(ladspa_ctl_value_changed),
			       &pi->ctl_value[port]);

	    /* connect control location to ladspa control port */
	    if (LADSPA_IS_PORT_INPUT(p_desc))
	    {
		//log("LADSPA", "connecting dll:%p port %d to %p\n", pi->dll, port, &pi->ctl_value[port]);
		pi->desc->connect_port(pi->hand, port, &pi->ctl_value[port]);
	    }

	    log("LADSPA", "%s %g-%g\n", pi->desc->PortNames[port], lower, upper);
	} else {
	    log("LADSPA", "unknown hint 0x%x\n", p_hint);
	}
    }


    PluginButtonBox = gtk_hbutton_box_new();
    gtk_widget_show(PluginButtonBox);
    gtk_box_pack_start(GTK_BOX(MainVBox), PluginButtonBox, FALSE, TRUE, 0);

    Apply = gtk_button_new_with_label("Apply");
    gtk_widget_show (Apply);
    gtk_container_add (GTK_CONTAINER (PluginButtonBox), Apply);
    GTK_WIDGET_SET_FLAGS (Apply, GTK_CAN_DEFAULT);

    Cancel = gtk_button_new_with_label("Cancel");
    gtk_widget_show(Cancel);
    gtk_container_add(GTK_CONTAINER (PluginButtonBox), Cancel);
    GTK_WIDGET_SET_FLAGS (Cancel, GTK_CAN_DEFAULT);

    gtk_signal_connect(GTK_OBJECT(Apply), "clicked",
		       GTK_SIGNAL_FUNC (ladspa_dlg_apply), pmi);
    gtk_signal_connect(GTK_OBJECT(Cancel), "clicked",
		      GTK_SIGNAL_FUNC(ladspa_dlg_cancel), pi);
    gtk_signal_connect(GTK_OBJECT(win_ladspa), "delete_event",
		       GTK_SIGNAL_FUNC(ladspa_dlg_destroy), pi);
    gtk_signal_connect(GTK_OBJECT(win_ladspa), "destroy_event",
		       GTK_SIGNAL_FUNC(ladspa_dlg_destroy), pi);

    gtk_widget_show(win_ladspa);
    goto ok;

dll_close:
    if (dlclose(pi->dll) < 0)
	log("LADSPA", "dlclose failed...\n");
ok:
    log("LADSPA", "created instance:%p dll:%p\n", pi, pi->dll);
    return;
}


static void
ladspa_dlg_free(plug_info_t *pi)
{
    dlclose(pi->dll);
    pi->dll = NULL;
    free(pi->ctl_value);
    pi->desc = NULL;
    win_ladspa = NULL;
}


static gboolean
ladspa_dlg_destroy(GtkWidget *widget, GdkEvent *event, gpointer user_data)
{
    ladspa_dlg_free((plug_info_t *)user_data);
    return FALSE;
}


static void
ladspa_dlg_cancel(GtkButton *button, gpointer user_data)
{
    gtk_widget_destroy(win_ladspa);
    ladspa_dlg_free((plug_info_t *)user_data);
}


static void
ladspa_ctl_value_changed(GtkObject *obj, gpointer ctl_val)
{
    GtkAdjustment *adj;

    adj = GTK_ADJUSTMENT(obj);
    *(LADSPA_Data *)ctl_val = adj->value;
    //log("LADSPA", "Value at %p changed to %f\n", ctl_val, adj->value);
}


/* FIXME: reference to global ws */
static void
ladspa_dlg_apply(GtkButton *button, gpointer user_data)
{
    plug_menu_t *pmi = (plug_menu_t *)user_data;

    log("LADSPA", "Apply plugin %s\n", pmi->pi->file);
    plug_run = pmi;
    edit_cmd(ws, EDIT_PLUGIN, NULL, FALSE, NULL);
    gtk_widget_destroy(win_ladspa);
    ladspa_dlg_free(pmi->pi);
}


chnl_indx
ladspa_plug_channels(void)
{
    plug_menu_t       *pmi = plug_run;
    LADSPA_Descriptor *desc = pmi->pi->desc;
    guint              port;
    guint              channels = 0;


    for (port = 0; port < desc->PortCount; port++)
    {
	LADSPA_PortDescriptor p_desc = desc->PortDescriptors[port];
	
	if (LADSPA_IS_PORT_AUDIO(p_desc) && LADSPA_IS_PORT_INPUT(p_desc))
	    channels++;
    }
    log("LADSPA", "plugin supports %d channels\n", channels);
    return channels;
}


void
ladspa_plug_run(float *io_buf, chnl_indx channels, smpl_indx samples)
{
    plug_menu_t       *pmi = plug_run;
    LADSPA_Descriptor *desc = pmi->pi->desc;
    LADSPA_Handle     *hand = pmi->pi->hand;
    guint              port;

    log("LADSPA", "running plugin on buf:%p\n", io_buf);

    /* attach input and output control ports
     * to the io buffer
     */
    channels *= 2;
    for (port = 0; port < desc->PortCount; port++)
    {
	LADSPA_PortDescriptor p_desc = desc->PortDescriptors[port];
	
	if (!LADSPA_IS_PORT_AUDIO(p_desc) || channels == 0)
	    continue;

	log("LADSPA", "connecting port:%d\n", port);
	desc->connect_port(hand, port, io_buf);
	channels--;
    }

    log("LADSPA", "activating plugin\n");
    if (desc->activate)
	desc->activate(hand);

    log("LADSPA", "running plug for samples:%d\n", samples);
    desc->run(hand, samples);

    log("LADSPA", "deactivating plug\n");
    if (desc->deactivate)
	desc->deactivate(hand);

    log("LADSPA", "cleanup plug\n");
    desc->cleanup(hand);
}






static void
ladspa_enumerate_dll(const char *lib_name, GtkWidget *menu)
{
    LADSPA_Descriptor *(*sym)(int);
    LADSPA_Descriptor *desc;
    plug_info_t       *pi;
    guint              plugs;
    void              *dll;
    char	      *er;

    dll = dlopen(lib_name, RTLD_LAZY);
    if (dlerror() != NULL || dll == NULL)
	return;

    if ((pi = malloc(sizeof(plug_info_t))) == NULL)
	goto dll_close;
    pi->dll = dll;
    strcpy(pi->file, lib_name);
    pi_list = g_list_prepend(pi_list, pi);

    /* if the dll has a symbol called ladspa_descriptor
     * it is (hopefully anyway) a ladspa plugin
     */
    sym = dlsym(pi->dll, "ladspa_descriptor");
    if ((er = dlerror()) != NULL || !sym)
    {
	log("LADSPA", "can't find desc %s\n", er);
	goto pi_free;
    }

    for (plugs=0; (desc = (*sym)(plugs)); plugs++)
    {
	plug_menu_t  *pm;

	if (LADSPA_IS_INPLACE_BROKEN(desc->Properties))
	{
	    log("LADSPA", "Unable to use plugin %s, plugin cannot modify buffers inplace\n", desc->Name);
	    continue;
	}

	log("LADSPA", "  found plugin %s\n", desc->Name);
	if ((pm = malloc(sizeof(plug_menu_t))) == NULL)
	    break;

	pm->indx = plugs;
	pm->pi = pi;
	pm->widget = gtk_menu_item_new_with_label(desc->Name);
	gtk_widget_show(pm->widget);
	gtk_container_add(GTK_CONTAINER(menu), pm->widget);
	gtk_signal_connect(GTK_OBJECT(pm->widget), "activate",
		    GTK_SIGNAL_FUNC(ladspa_dlg_create), pm);
	pm_list = g_list_prepend(pm_list, pm);
    }
    if ((pi->desc = malloc(sizeof(LADSPA_Descriptor *) * plugs)) == NULL)
	goto pi_free;

    pi->plugs = plugs;
    goto dll_close;

pi_free:
    free(pi);

dll_close:
    if (dlclose(dll) <0)
	log("LADSPA", "dlclose failed!\n");
    return;
}



void
ladspa_enumerate_menu(GtkWidget *menu)
{
    char       lib_path[PATH_MAX+1];
    const char *ch;
    const char *end;
    GList      *list;

    ch = getenv("LADSPA_PATH");
    if (!ch)
    {
	log("LADSPA", "LADSPA_PATH not set in environment, searching default path\n");
	ch = default_lib_path;
    }

    /* free any previous plugin entries */
    for(list = pm_list; list; list = pm_list)
    {
	plug_menu_t *pm = list->data;

	gtk_widget_destroy(pm->widget);
	pm_list = g_list_remove(pm_list, pm);
	free(pm);
    }

    for(list = pi_list; list; list = pi_list)
    {
	plug_info_t *pi = list->data;

	pi_list = g_list_remove(pi_list, pi);
	free(pi);
    }


    /* search the : seperated directory list */
    do
    {
	end = strchr(ch, ':');
	if (end == NULL)
	    end = strchr(ch, '\0');

	if (end)
	{
	    DIR           *dir;
	    struct dirent *dirent;

	    strncpy(lib_path, ch, end-ch);
	    lib_path[end-ch] = '\0';

	    if (lib_path[strlen(lib_path)-1] == '/')
		lib_path[strlen(lib_path)-1] = '\0';

	    log("LADSPA", "scaning:%s\n", lib_path);
	
	    if ((dir = opendir(lib_path)))
	    {
		while ((dirent = readdir(dir)))
		{
		    char lib_name[PATH_MAX+1];

		    strcpy(lib_name, lib_path);
		    strcat(lib_name, "/");
		    strcat(lib_name, dirent->d_name);
		    ladspa_enumerate_dll(lib_name, menu);
		}
		closedir(dir);
	    }
	}

	ch = end+1;
    } while (end && *end != '\0');
}

#endif /* ENABLE_LADSPA */
