/*
 * GNoise
 *
 * Copyright (C) 1999-2001 Dwight Engen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: snd_buf.c,v 1.10 2002/01/13 02:51:16 dengen Exp $
 *
 */


#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>
#include "snd_buf.h"


static void	snd_file_markers_save(snd_buf_t *sb);
static gboolean	snd_file_chunk_find(snd_buf_t *sb, char *chunk_name,
				    void *chunk, size_t chunk_size);

static void snd_file_markers_save(snd_buf_t *sb)
{
    int		i;
    off_t	pos;
    char	zerobuf[4];
    cue_chunk_t	chunk;
    cue_point_t point;

    if (sb->markers <= 0)
	return;

    /* ensure chunk will start on mod 4 byte boundary */
    memset(zerobuf, 0, sizeof(zerobuf));
    pos = lseek(sb->fd, 0, SEEK_END);
    if (pos % 4)
	write(sb->fd, zerobuf, 4 - (pos % 4));

    strcpy(chunk.chunk_id, "cue ");
    chunk.chunk_size = 4 + sb->markers * sizeof(point);
    chunk.cue_points = sb->markers;
    write(sb->fd, &chunk, sizeof(chunk));

    for(i = 0; i < sb->markers; i++)
    {
	point.ident = i;
	point.position = sb->marker[i];
	strcpy(point.chunk, "data");
	point.chunk_start = 0;
	point.block_start = 0;
	point.sample = sb->marker[i];
	write(sb->fd, &point, sizeof(point));
    }
}



static gboolean snd_file_chunk_find(snd_buf_t *snd_buf, char *chunk_name,
				    void *chunk, size_t chunk_size)
{
    off_t	  pos;
    void_chunk_t *void_chunk;

    void_chunk = chunk;

    /* chunks start on a 4 byte aligned offset */
    pos = lseek(snd_buf->fd, 0, SEEK_CUR);
    pos = ((pos+3)/4)*4;
    lseek(snd_buf->fd, pos, SEEK_SET);

    for(;;)
    {
	if (read(snd_buf->fd, void_chunk, chunk_size) < chunk_size)
	{
	    log("SND_BUF", "%s chunk not found\n", chunk_name);
	    return FALSE;
	}

	if (strncmp(void_chunk->chunk_id, chunk_name, 4))
	{
	    log("SND_BUF", "chunk find %s skipping unknown chunk %4.4s\n",
		chunk_name, &void_chunk->chunk_id);
	    if (lseek(snd_buf->fd, void_chunk->chunk_size, SEEK_CUR) < 0)
	    {
		log("SND_BUF", "unable to seek to next chunk %s\n", strerror(errno));
		return FALSE;
	    }
	} else {
	    return TRUE;
	}
    }
}



gboolean snd_file_save(snd_buf_t *sb)
{
    /* truncate file to just header and data */
    if (ftruncate(sb->fd, sizeof(wave_header_t) + sizeof(data_chunk_t) +
	      sb->data_size) < 0)
    {
	log("SND_BUF", "ftruncate failed %s\n", strerror(errno));
	return FALSE;
    }

    /* write header */
    snd_file_header_save(sb->fd, sb->info);

    /* msync mmaped area */
    snd_file_sync(sb);

    snd_file_markers_save(sb);
    return TRUE;
}



gboolean snd_file_header_save(int fd, snd_info_t si)
{
    wave_header_t	header;
    data_chunk_t	data;
    guint32		bytes;

    bytes = si.samples * si.channels * (si.sample_bits / 8);
    memset(&header, 0, sizeof(header));
    memcpy(header.group_id,  "RIFF", 4);
    memcpy(header.riff_type, "WAVE", 4);
    header.size = bytes+sizeof(header);

    memcpy(header.fmt.chunk_id, "fmt ", 4);
    header.fmt.chunk_size = sizeof(fmt_chunk_t) - 8;
    header.fmt.format = WAVE_FORMAT_PCM;
    header.fmt.channels = si.channels;
    header.fmt.sample_rate = si.sample_rate;
    header.fmt.block_align = si.channels * (si.sample_bits / 8);
    header.fmt.avg_bytes_per_second = si.sample_rate * header.fmt.block_align;
    header.fmt.bits_per_sample = si.sample_bits;

    memcpy(data.chunk_id, "data", 4);
    data.chunk_size = bytes;

    lseek(fd, 0, SEEK_SET);
    write(fd, &header, sizeof(header));
    write(fd, &data, sizeof(data));
    return TRUE;
}



gboolean snd_file_open(snd_buf_t *snd_buf)
{
    wave_header_t	header;
    data_chunk_t	data_chunk;
    cue_chunk_t		cue_chunk;
    struct stat		stat;
    int			i;
    
    snd_buf->fd = open(snd_buf->file, O_RDWR);
    if (snd_buf->fd < 0)
    {
	log("SND_BUF", "unable to open file %s %s\n", snd_buf->file, strerror(errno));
	return FALSE;
    }

    if (fstat(snd_buf->fd, &stat) < 0)
    {
	log("SND_BUF", "unable to stat file %s\n", snd_buf->file);
	goto err1;
    }

    snd_buf->st_dev = stat.st_dev;
    snd_buf->st_ino = stat.st_ino;

    if (read(snd_buf->fd, &header, sizeof(wave_header_t)) < sizeof(wave_header_t))
    {
	log("SND_BUF", "unable to read header of file %s\n", snd_buf->file);
	goto err1;
    }

    /* check header to make sure it looks like a wav file we support */
    if ( (strncmp(header.group_id,      "RIFF", 4)) ||
	 (strncmp(header.riff_type,     "WAVE", 4)) ||
	 (strncmp(header.fmt.chunk_id,  "fmt ", 4)) )
    {
	log("SND_BUF", "unsupported file format\n");
	goto err1;
    }

    /* we only support WAVE_FORMAT_PCM type wave files */
    if (header.fmt.format != WAVE_FORMAT_PCM)
    {
	log("SND_BUF", "only PCM wave file format is supported\n");
	goto err1;
    }

    /* skip to the next chunk after the fmt chunk */
    if (header.fmt.chunk_size > 16)
	lseek(snd_buf->fd, header.fmt.chunk_size - 16, SEEK_CUR);

    /* find data chunk */
    if (!snd_file_chunk_find(snd_buf, "data", &data_chunk, sizeof(data_chunk)))
    {
	log("SND_BUF", "no data chunk found!\n");
	goto err1;
    }

    snd_buf->data_offset = lseek(snd_buf->fd, 0, SEEK_CUR);
    snd_buf->data_size = data_chunk.chunk_size;

    /* find cue chunk */
    lseek(snd_buf->fd, data_chunk.chunk_size, SEEK_CUR);
    if (snd_file_chunk_find(snd_buf, "cue ", &cue_chunk, sizeof(cue_chunk)))
    {
	for(i = 0; i < cue_chunk.cue_points; i++)
	{
	    cue_point_t cue_point;

	    if (read(snd_buf->fd, &cue_point, sizeof(cue_point)) < sizeof(cue_point))
	    {
		log("SND_BUF", "Unable to read cue point %d\n", i);
		goto err1;
	    }
	    snd_file_marker_add(snd_buf, cue_point.sample);
	}
    }

    snd_buf->info.channels = header.fmt.channels;
    snd_buf->info.sample_rate = header.fmt.sample_rate;
    snd_buf->info.sample_bits = header.fmt.bits_per_sample;
    snd_buf->info.samples = snd_buf->data_size / (snd_buf->info.sample_bits/8)
			    / snd_buf->info.channels;
    snd_buf->mmap_file = NULL;
    snd_buf->mmap_size = snd_buf->data_size + snd_buf->data_offset;

    /* verify file is at least this large */
    if (stat.st_size < snd_buf->mmap_size)
    {
	log("SND_BUF", "Unable to open, size mismatch, file may be corrupt.\n");
	log("SND_BUF", "  (WAV header reports:%d file is:%d)\n",
		snd_buf->mmap_size, stat.st_size);
	goto err1;
    }

    log("SND_BUF", "file         %s\n", g_basename(snd_buf->file));

    log("SND_BUF", "channels     %d\n", snd_buf->info.channels);
    log("SND_BUF", "sample size  %d bit\n", snd_buf->info.sample_bits);
    log("SND_BUF", "sample rate  %d Hz\n", snd_buf->info.sample_rate);
    log("SND_BUF", "sample data  %ld\n", snd_buf->data_size);
    log("SND_BUF", "samples      %ld\n", snd_buf->info.samples);
    log("SND_BUF", "avg b/sec    %d\n", header.fmt.avg_bytes_per_second);
    log("SND_BUF", "blk align    %d\n", header.fmt.block_align);
    log("SND_BUF", "data offset  %d\n", snd_buf->data_offset);
    
    /* mmap the entire file */
    snd_buf->mmap_file = mmap(0, snd_buf->mmap_size, PROT_READ | PROT_WRITE,
			      MAP_SHARED, snd_buf->fd, 0);
    if (snd_buf->mmap_file == MAP_FAILED)
    {
	log("SND_BUF", "unable to mmap %s\n", strerror(errno));
	close(snd_buf->fd);
	return FALSE;
    }

    /* set pointers to mmaped areas */
    snd_buf->header = (wave_header_t *)snd_buf->mmap_file;
    snd_buf->data_chunk = (data_chunk_t *)(char *)snd_buf->mmap_file + snd_buf->data_offset - sizeof(data_chunk_t);
    snd_buf->data = (char *)snd_buf->mmap_file + snd_buf->data_offset;
    

    /* fix corrupt header older gnoise versions may've written */
    {
    fmt_chunk_t *fmt;

    fmt = &snd_buf->header->fmt;
    if ((fmt->block_align != fmt->channels * (fmt->bits_per_sample / 8)) ||
	 fmt->avg_bytes_per_second != fmt->sample_rate * fmt->block_align)
    {
	fmt->block_align = fmt->channels * (fmt->bits_per_sample / 8);
	fmt->avg_bytes_per_second = fmt->sample_rate * fmt->block_align;
	log("SND_BUF", "fixed wav header blk align and avg b/sec\n");
    }
    }

#if 0
    log("SND_BUF", "mmaped %ld bytes\n", snd_buf->mmap_size);
    log("SND_BUF", "addr %p sample_data is %p\n", snd_buf->mmap_file, snd_buf->data);
#endif
    return TRUE;

err1:
    close(snd_buf->fd);
    snd_buf->fd = -1;
    return FALSE;
}



void snd_file_close(snd_buf_t *snd_buf)
{
    /* write markers to cue chunk */
    if (snd_buf->marker != NULL)
    {
	free(snd_buf->marker);
	snd_buf->markers = 0;
    }

    if (snd_buf->mmap_file)
    {
	munmap(snd_buf->mmap_file, snd_buf->mmap_size);
	snd_buf->mmap_file = NULL;
    }
    
    if (snd_buf->fd >= 0)
    {
	close(snd_buf->fd);
	snd_buf->fd = -1;
    }
}



void snd_file_sync(snd_buf_t *snd_buf)
{
    msync(snd_buf->mmap_file, snd_buf->mmap_size, MS_SYNC);
}



void snd_file_resize(snd_buf_t *snd_buf, smpl_indx samples)
{
    size_t new_mmap_size;
    char   *new_vma;

    snd_buf->data_size = samples * snd_buf->info.channels
			 * (snd_buf->info.sample_bits/8);
    snd_buf->info.samples = samples;
    new_mmap_size = snd_buf->data_size + snd_buf->data_offset;

    // set file to new length (shrink or grow)
    if (ftruncate(snd_buf->fd, new_mmap_size) < 0)
    {
	log("SND_BUF", "fatal: ftruncate failed %s\n", strerror(errno));
	exit(-1);	// !!FIXME, would be nice to save changes first...
    }

    if ((new_vma = mremap(snd_buf->mmap_file, snd_buf->mmap_size,
			  new_mmap_size, 0 /* MREMAP_MAYMOVE */)) == MAP_FAILED)
    {
	log("SND_BUF", "fatal: mremap failed %s\n", strerror(errno));
	exit(-1);	// !!FIXME, would be nice to save changes first...
    }

    snd_buf->mmap_file	= new_vma;
    snd_buf->mmap_size	= new_mmap_size;
    snd_buf->header	= (wave_header_t *)snd_buf->mmap_file;
    snd_buf->data_chunk = (data_chunk_t *)((char *)snd_buf->mmap_file + snd_buf->data_offset - sizeof(data_chunk_t));
    snd_buf->data	= (char *)snd_buf->mmap_file + snd_buf->data_offset;
    snd_buf->data_chunk->chunk_size = snd_buf->data_size;
    log("SND_BUF", "resized to samples:%d bytes:%d\n", samples, snd_buf->data_size);
}



void snd_file_marker_add(snd_buf_t *sb, smpl_indx sample)
{
    int i;

    sb->marker = realloc(sb->marker, (sb->markers+1) * sizeof(smpl_indx));
    if (sb->marker == NULL)
    {
	log("MARKER", "Unable to realloc memory\n");
	sb->markers = 0;
	return;
    }

    for(i = 0; i < sb->markers; i++)
	if (sb->marker[i] >= sample)
	    break;

    memmove(&sb->marker[i+1], &sb->marker[i],
	    (sb->markers - i) * sizeof(sb->marker[0]));
    sb->marker[i] = sample;
    sb->marker_sel = i;
    sb->markers++;
}



void snd_file_marker_del(snd_buf_t *sb, int indx)
{
    if (indx == sb->markers - 1)
	sb->marker_sel--;

    memmove(&sb->marker[indx], &sb->marker[indx+1],
	    (sb->markers - indx) * sizeof(sb->marker[0]));
    sb->markers--;
}
