/** -*- Mode: C++; tab-width: 4 -*-
 * vim: sw=4 ts=4:
 *
 * Gnome Apt menu code
 *
 * 	(C) 1998 Havoc Pennington <hp@pobox.com>
 * 	    2002-2005 Filip Van Raemdonck <mechanix@debian.org>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * 	$Id$
 *
 **/

#include <apt-pkg/error.h>

#include "cachecontrol.h"
#include "drawtree.h"
#include "gaptpixbuf.h"
#include "menus.h"
#include "pkgtree.h"
#include "pkgutil.h"
#include "preferences.h"
#include "sources.h"

#define GNOME_APT_LOGO "/usr/share/pixmaps/gnome-apt.png"

static void set_order_cb (GtkRadioAction*, GtkRadioAction*, gpointer);
static void set_grouping_cb (GtkRadioAction*, GtkRadioAction*, gpointer);
static void set_color_marker_cb (GtkToggleAction*, gpointer);
static void set_column_shown_cb (GtkToggleAction*, gpointer);

// File
static void apt_prefs_cb   (GtkWidget* w, gpointer data);
static void sources_cb (GtkWidget*, gpointer);
static void exit_cb        (GtkWidget* w, gpointer data);

// Actions
static void update_cb      (GtkWidget* w, gpointer data);
static void complete_run_cb (GtkWidget* w, gpointer data);
static void upgrade_cb      (GtkWidget* w, gpointer data);
static void smart_upgrade_cb(GtkWidget* w, gpointer data);
static void fix_broken_cb   (GtkWidget* w, gpointer data);

/* Package */
static void install_pkg_cb     (GtkWidget* w, gpointer data);
static void configure_pkg_cb   (GtkWidget* w, gpointer data);
static void upgrade_pkg_cb     (GtkWidget* w, gpointer data);
static void purge_pkg_cb       (GtkWidget* w, gpointer data);
static void remove_pkg_cb      (GtkWidget* w, gpointer data);
static void keep_pkg_cb        (GtkWidget* w, gpointer data);

/* View */
static void details_cb      (GtkWidget* w, gpointer data);
static void pending_cb (GtkWidget*, gpointer);

// Help
static void about_cb       (GtkWidget* w, gpointer data);
static void icons_cb       (GtkWidget* w, gpointer data);

static GtkActionEntry entries[] = {
	{ "menufile", NULL, "_File" },
	{ "menuactions", NULL, "_Actions" },
	{ "menupackage", NULL, "_Package" },
	{ "menuview", NULL, "_View" },
	{ "menuhelp", NULL, "_Help" },
	{ "viewcolumns", NULL, "Columns" },
	{ "viewsort", NULL, "Sort order" },
	{ "viewcategories", NULL, "Categories" },
	{ "viewcolors", NULL, "Color markers" },
	{ "fileprefs", GTK_STOCK_PREFERENCES, N_("Prefere_nces"), NULL, N_("Configure the application"), G_CALLBACK (apt_prefs_cb) },
	{ "filerepos", GTK_STOCK_INDEX, N_("_Repository lists"), NULL, N_("Tell GNOME Apt where to look for packages"),
	  G_CALLBACK (sources_cb) },
	{ "filequit", GTK_STOCK_QUIT, N_("_Quit"), "<control>Q", N_("Quit the application"), G_CALLBACK (exit_cb) },
	{ "actionupdate", GTK_STOCK_REFRESH, N_("_Update package list"), "<control>U",
	  N_("Download the latest package info from the configured sources"), G_CALLBACK (update_cb) },
	{ "actioncommit", GTK_STOCK_EXECUTE, N_("_Commit changes"), "<control>R", N_("Install and remove packages as requested"),
	  G_CALLBACK (complete_run_cb) },
	{ "actionupgrade", NULL, N_("Mark up_grades"), NULL, N_("Mark all out-of-date packages for upgrade, without removals or installs"),
	  G_CALLBACK (upgrade_cb) },
	{ "actiondistup", NULL, N_("_Smart mark upgrades"), NULL, N_("Mark upgrades, removing or installing packages if needed"),
	  G_CALLBACK (smart_upgrade_cb) },
	{ "actionfix", NULL, N_("_Fix broken"), NULL, N_("Fix broken packages, removing or installing packages if needed"),
	  G_CALLBACK (fix_broken_cb) },
	{ "pkgremove", GTK_STOCK_DELETE, N_("Uninstal_l"), NULL, N_("Uninstall package, keeping configuration"), G_CALLBACK (remove_pkg_cb) },
	{ "pkgkeep", NULL, N_("_Keep"), NULL, N_("Don't change this package"), G_CALLBACK (keep_pkg_cb) },
	{ "pkginstall", NULL, N_("_Install/Upgrade"), NULL, N_("Install this package"), G_CALLBACK (install_pkg_cb) },
	{ "viewdetails", GTK_STOCK_EDIT, N_("Package details"), "<control>D", N_("Display package information in a separate window"),
	  G_CALLBACK (details_cb) },
	{ "viewchanges", NULL, N_("Pending changes"), NULL, N_("Show what actions would be performed when completing the install run"),
	  G_CALLBACK (pending_cb) },
	/*{ "helpcontents", GTK_STOCK_HELP, N_("_Contents"), "F1", N_("View help for this application"), NULL },*/
	{ "helplegend", NULL, N_("_Icon help"), NULL, N_("Explain what all the icons in the package list mean"), G_CALLBACK (icons_cb) },
	{ "helpabout", GTK_STOCK_ABOUT, N_("_About GNOME Apt"), NULL, N_("About this application"), G_CALLBACK (about_cb) }
};
static GtkToggleActionEntry col_toggle_entries[] = {
	{ "colname", NULL, N_("Package name"), NULL, N_("Whether to show the package name column"), G_CALLBACK (set_column_shown_cb), TRUE },
	{ "coliver", NULL, N_("Installed version"), NULL, N_("Whether to show the installed version column"), G_CALLBACK (set_column_shown_cb),
	  TRUE },
	{ "colaver", NULL, N_("Available version"), NULL, N_("Whether to show the available version column"), G_CALLBACK (set_column_shown_cb),
	  TRUE },
	{ "colsect", NULL, N_("Package section"), NULL, N_("Whether to show the package section column"), G_CALLBACK (set_column_shown_cb), TRUE },
	{ "colpri", NULL, N_("Package priority"), NULL, N_("Whether to show the package priority column"), G_CALLBACK (set_column_shown_cb),
	  TRUE },
	{ "colstatus", NULL, N_("Package status"), NULL, N_("Whether to show the package status column"), G_CALLBACK (set_column_shown_cb), TRUE },
	{ "coldesc", NULL, N_("Description"), NULL, N_("Whether to show the package description column"), G_CALLBACK (set_column_shown_cb),
	  TRUE },
};
static GtkRadioActionEntry sort_radio_entries[] = {
	{ "sortnone", NULL, N_("Don't sort"), NULL, N_("Show packages in random order"), 0 },
	{ "sortalpha", NULL, N_("Sort alphabetically"), NULL, N_("Order packages alphabetically"), 1 },
	{ "sortsect", NULL, N_("Sort by section"), NULL, N_("Order packages by the section they are in"), 2 },
	{ "sortstatus", NULL, N_("Sort by status"), NULL, N_("Order packages by the status they have"), 3 },
	{ "sortpri", NULL, N_("Sort by priority"), NULL, N_("Order packages by the priority they have"), 4 },
};
static GtkRadioActionEntry group_radio_entries[] = {
	{ "groupnone", NULL, N_("Don't group"), NULL, N_("Group all packages together"), 0 },
	{ "groupalpha", NULL, N_("Group alphabetically"), NULL, N_("Group packages by their first letter"), 1 },
	{ "groupsect", NULL, N_("Group by section"), NULL, N_("Group packages from the same section"), 2 },
	{ "groupstatus", NULL, N_("Group by status"), NULL, N_("Group packages with the same status"), 3 },
	{ "grouppri", NULL, N_("Group by priority"), NULL, N_("Group packages with the same priority"), 4 },
};
static GtkToggleActionEntry mark_toggle_entries[] = {
	{ "markbroken", NULL, N_("Mark broken packages (red)"), NULL, N_("Whether to mark packages which are broken in red"),
	  G_CALLBACK (set_color_marker_cb), TRUE },
	{ "markorphans", NULL, N_("Mark orphaned packages (green)"), NULL, N_("Whether to mark orphaned packages in red"),
	  G_CALLBACK (set_color_marker_cb), TRUE }
};

static const gchar* ui_desc =
"<ui>"
"  <menubar name='GAptMenu'>"
"    <menu action='menufile'>"
"      <menuitem action='fileprefs'/>"
"      <menuitem action='filerepos'/>"
"      <separator/>"
"      <menuitem action='filequit'/>"
"    </menu>"
"    <menu action='menuactions'>"
"      <menuitem action='actionupdate'/>"
"      <menuitem action='actioncommit'/>"
"      <menuitem action='actionupgrade'/>"
"      <menuitem action='actiondistup'/>"
"      <menuitem action='actionfix'/>"
"    </menu>"
"    <menu action='menupackage'>"
"      <menuitem action='pkgremove'/>"
"      <menuitem action='pkgkeep'/>"
"      <menuitem action='pkginstall'/>"
"    </menu>"
"    <menu action='menuview'>"
"      <menuitem action='viewdetails'/>"
"      <menuitem action='viewchanges'/>"
"      <separator/>"
"      <menu action='viewcolumns'>"
"        <menuitem action='colname'/>"
"        <menuitem action='coliver'/>"
"        <menuitem action='colaver'/>"
"        <menuitem action='colsect'/>"
"        <menuitem action='colpri'/>"
"        <menuitem action='colstatus'/>"
"        <menuitem action='coldesc'/>"
"      </menu>"
"      <menu action='viewsort'>"
"        <menuitem action='sortnone'/>"
"        <menuitem action='sortalpha'/>"
"        <menuitem action='sortsect'/>"
"        <menuitem action='sortstatus'/>"
"        <menuitem action='sortpri'/>"
"      </menu>"
"      <menu action='viewcategories'>"
"        <menuitem action='groupnone'/>"
"        <menuitem action='groupalpha'/>"
"        <menuitem action='groupsect'/>"
"        <menuitem action='groupstatus'/>"
"        <menuitem action='grouppri'/>"
"      </menu>"
"      <menu action='viewcolors'>"
"        <menuitem action='markbroken'/>"
"        <menuitem action='markorphans'/>"
"      </menu>"
"    </menu>"
"    <menu action='menuhelp'>"
"      <menuitem action='helplegend'/>"
"      <menuitem action='helpabout'/>"
"    </menu>"
"  </menubar>"
"</ui>";

/* Menu-handling code */

static const int COLTYPES = 7;
struct MenuWidgets {
	GtkWidget* main_install;
	GtkWidget* main_remove;
	GtkWidget* main_keep;

	map<GAptPkgTree::CategoryType,gint> main_group;
	map<GAptPkgTree::SortType,gint> main_order;

	DrawTree::ColumnType main_column[COLTYPES];
};

static MenuWidgets widgets;

struct SBar {
	GtkStatusbar* bar;
	guint cid;
};

static void
selcb (GtkItem* item, gpointer data) {
	SBar* sb = (SBar*) data;
	const gchar* tt = (gchar*) g_object_get_data (G_OBJECT (item), "tooltip");
	gtk_statusbar_push (GTK_STATUSBAR (sb->bar), sb->cid, tt);
}
static void
deselcb (GtkItem* item, gpointer data) {
	SBar* sb = (SBar*) data;
	gtk_statusbar_pop (GTK_STATUSBAR (sb->bar), sb->cid);
}

static void
proxycb (GtkUIManager* uim, GtkAction* ac, GtkWidget* proxy, gpointer data) {
	gchar* tt;
	SBar* sb = (SBar*) data;

	if (!GTK_IS_MENU_ITEM (proxy)) {
		return;
	}

	g_object_get (ac, "tooltip", &tt, NULL);
	if (tt) {
		g_object_set_data (G_OBJECT (proxy), "tooltip", tt);
		g_signal_connect (GTK_MENU_ITEM (proxy), "select", G_CALLBACK (selcb), sb);
		g_signal_connect (GTK_MENU_ITEM (proxy), "deselect", G_CALLBACK (deselcb), sb);
	}
}

static GtkUIManager*
gapt_create_menus_with_data (const gchar* name, gint rb_group, gint rb_sort, GtkStatusbar* bar, gpointer data) {
	GtkActionGroup* ag;
	GtkUIManager* uim;
	GError* err = NULL;
	SBar* sb = new SBar;

	sb->bar = bar;
	sb->cid = gtk_statusbar_get_context_id (bar, "GNOME Apt Menutips");

	ag = gtk_action_group_new (name);
	gtk_action_group_add_actions (ag, entries, G_N_ELEMENTS (entries), data);
	gtk_action_group_add_toggle_actions (ag, col_toggle_entries, G_N_ELEMENTS (col_toggle_entries), data);
	gtk_action_group_add_toggle_actions (ag, mark_toggle_entries, G_N_ELEMENTS (mark_toggle_entries), data);
	gtk_action_group_add_radio_actions (ag, sort_radio_entries, G_N_ELEMENTS (sort_radio_entries), rb_sort, G_CALLBACK (set_order_cb), data);
	gtk_action_group_add_radio_actions (ag, group_radio_entries, G_N_ELEMENTS (group_radio_entries), rb_group, G_CALLBACK (set_grouping_cb), data);

	uim = gtk_ui_manager_new();
	g_signal_connect (uim, "connect-proxy", G_CALLBACK (proxycb), sb);

	gtk_ui_manager_insert_action_group (uim, ag, 0);

	if (!gtk_ui_manager_add_ui_from_string (uim, ui_desc, -1, &err)) {
		g_message ("Building menus failed: %s", err->message);
		g_error_free (err);
		exit (EXIT_FAILURE);
	}

	return uim;
}

GtkWidget*
gnome_apt_create_main_menu (GtkWindow* app, GtkStatusbar* bar, GAptPkgList* pkglist) {
	GtkUIManager* uim;
	GtkAccelGroup* ag;
	DrawTree* view = pkglist->getView();
	GAptPkgTree* tree = view->getModel();
	gint group_initial = -1, sort_initial = -1;

	g_return_if_fail (widgets.main_group.empty());
  g_return_if_fail(tree != 0);

	/* Column visibility */
	widgets.main_column[0] = DrawTree::ColumnName;
	widgets.main_column[1] = DrawTree::ColumnCurrent;
	widgets.main_column[2] = DrawTree::ColumnAvailable;
	widgets.main_column[3] = DrawTree::ColumnSection;
	widgets.main_column[4] = DrawTree::ColumnPriority;
	widgets.main_column[5] = DrawTree::ColumnStatus;
	widgets.main_column[6] = DrawTree::ColumnDescription;
	for (int i = 0; i < COLTYPES; ++i) {
		col_toggle_entries[i].is_active = view->is_visible (widgets.main_column[i]);
  }

	/* Grouping */
	widgets.main_group[GAptPkgTree::CategoryNone] = 0;
	widgets.main_group[GAptPkgTree::CategoryAlpha] = 1;
	widgets.main_group[GAptPkgTree::CategoryStatus] = 3;
	widgets.main_group[GAptPkgTree::CategorySection] = 2;
	widgets.main_group[GAptPkgTree::CategoryPriority] = 4;

	map<GAptPkgTree::CategoryType, gint>::iterator j = widgets.main_group.find (tree->get_category());
	if (j == widgets.main_group.end()) {
		g_warning ("Category not found in main menu");
	} else {
		group_initial = j->second;
	}

	/* Sorting */
	widgets.main_order[GAptPkgTree::SortNone] = 0;
	widgets.main_order[GAptPkgTree::SortAlpha] = 1;
	widgets.main_order[GAptPkgTree::SortStatus] = 3;
	widgets.main_order[GAptPkgTree::SortSection] = 2;
	widgets.main_order[GAptPkgTree::SortPriority] = 4;

	map<GAptPkgTree::SortType, gint>::iterator k = widgets.main_order.find (tree->get_sort());
	if (k == widgets.main_order.end()) {
		g_warning ("Sort order not found in main menu");
	} else {
		sort_initial = k->second;
	}

	for (guint j = 0; j < G_N_ELEMENTS (mark_toggle_entries); j++) {
		mark_toggle_entries[j].is_active = view->MarkRows[j];
	}

	uim = gapt_create_menus_with_data ("GNOME Apt menu", group_initial, sort_initial, bar, pkglist);
	ag = gtk_ui_manager_get_accel_group (uim);
	gtk_window_add_accel_group (app, ag);

	widgets.main_install = gtk_ui_manager_get_widget (uim, "/GAptMenu/menupackage/pkginstall");
	widgets.main_keep    = gtk_ui_manager_get_widget (uim, "/GAptMenu/menupackage/pkgkeep");
	widgets.main_remove  = gtk_ui_manager_get_widget (uim, "/GAptMenu/menupackage/pkgremove");

	return gtk_ui_manager_get_widget (uim, "/GAptMenu");
}

void
gnome_apt_menus_selection_changed (DrawTree* view) {
	GAptPkgTree* tree = view->getModel();

	pkgCache::Package* p = 0;
	if (view->selected_node()) {
		p = view->selected_node()->package();
	}

  bool install = false;
  bool keep    = false;
  bool remove  = false;

  if (p != 0) {
		pkgCache::PkgIterator i(*(tree->cache()), p);
		install = Util::can_change_install (i, tree);
		keep = Util::can_change_keep (i, tree);
		remove = Util::can_change_remove (i, tree);
  }

  gtk_widget_set_sensitive(widgets.main_install,  install);
  gtk_widget_set_sensitive(widgets.main_keep,  keep);
  gtk_widget_set_sensitive(widgets.main_remove,  remove);
}

/* Menu callbacks */
static void
set_color_marker_cb (GtkToggleAction* a, gpointer data) {
	guint n = G_N_ELEMENTS (mark_toggle_entries);
	GAptPkgList* pkglist = (GAptPkgList*) (data);
	g_return_if_fail (pkglist);

	for (guint j = 0; j < n; ++j) {
		if (!strcmp (gtk_action_get_name (GTK_ACTION (a)), mark_toggle_entries[j].name)) {
			DrawTree* view = pkglist->getView();
			view->MarkRows[j] = gtk_toggle_action_get_active (a);
			ga_debug ("setting color for %d to %d", j, view->MarkRows[j]);
			view->queue_display_update();
			break;
		}
	}
}

static void
set_column_shown_cb (GtkToggleAction* a, gpointer data) {
	GAptPkgList* pkglist = (GAptPkgList*) (data);
	g_return_if_fail (pkglist);

	for (int j = 0; j < COLTYPES; ++j) {
		if (!strcmp (gtk_action_get_name (GTK_ACTION (a)), col_toggle_entries[j].name)) {
			pkglist->getView()->set_visible (widgets.main_column[j], gtk_toggle_action_get_active (a));
			break;
}
}
}

static void
set_order_cb (GtkRadioAction* a, GtkRadioAction* c, gpointer data) {
	gint o = gtk_radio_action_get_current_value (a);
	map<GAptPkgTree::SortType, gint>::iterator i = widgets.main_order.begin();
	GAptPkgList* pkglist = (GAptPkgList*) data;
	g_return_if_fail (pkglist);

	while (i != widgets.main_order.end()) {
		if (o == i->second) {
			break;
		}
		++i;
	}
	if (i == widgets.main_order.end()) {
		g_warning ("Failed to find sort order method!");
		return;
	}

	pkglist->getView()->getModel()->set_sort ((GAptPkgTree::SortType) i->first);
}

static void
set_grouping_cb (GtkRadioAction* a, GtkRadioAction* c, gpointer data) {
	gint o = gtk_radio_action_get_current_value (a);
	map<GAptPkgTree::CategoryType, gint>::iterator i = widgets.main_group.begin();
	GAptPkgList* pkglist = (GAptPkgList*) data;
	g_return_if_fail (pkglist);

	while (i != widgets.main_group.end()) {
		if (o == i->second) {
			break;
		}
		++i;
	}
	if (i == widgets.main_group.end()) {
		g_warning ("Failed to find sort order method!");
		return;
	}

	pkglist->getView()->getModel()->set_category ((GAptPkgTree::CategoryType) i->first);
}

// Other stuff

static void
details_cb (GtkWidget* w, gpointer data) {
  GAptPkgList* pkglist = static_cast<GAptPkgList*>(data);

  g_return_if_fail(pkglist != 0);

	pkglist->details (true);
}

static void
pending_cb (GtkWidget* wdg, gpointer data) {
	GAptPkgList* pkglist = static_cast<GAptPkgList*> (data);

	GtkWidget* dialog = gtk_dialog_new_with_buttons (_("Pending Changes"), NULL,
	      GtkDialogFlags (GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR),
	      GTK_STOCK_CLOSE, GTK_RESPONSE_OK, NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
	gnome_apt_setup_dialog (dialog);
	gtk_window_set_default_size (GTK_WINDOW (dialog), 400, 360);

	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
	      gapt_cache_control_pending (GAPT_CACHE_CONTROL (gapt_cache_control_new (gnome_apt_cache_file()))),
	      TRUE, TRUE, GAPT_PAD);

	gtk_widget_show_all (dialog);
	gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
}

///////////////////////////////// was in app.cc

static void
update_cb (GtkWidget* w, gpointer data) {
	GAptPkgList* pkglist = static_cast<GAptPkgList*> (data);

  g_return_if_fail(pkglist != 0);

  GAptCacheFile* cf = gnome_apt_cache_file();
	if (cf) {
		GObject* cc = gapt_cache_control_new (cf);
		if (!gapt_cache_control_update (GAPT_CACHE_CONTROL (cc))) {
			if (!_error->empty()) {
				gnome_apt_error_dialog (_("Update failed"));
			}
		} else if (!_error->empty()) {
			/* Should be harmless (e.g. Release file misses) */
			_error->DumpErrors();
		}
  }
  else {
    g_warning(__FUNCTION__);
    g_warning("No cache file");
  }
}

static void
apt_prefs_cb (GtkWidget* w, gpointer data) {
	gnome_apt_preferences()->edit();
}

static void
sources_cb (GtkWidget* w, gpointer data) {
	gnome_apt_sources()->edit();
}

static void
exit_cb (GtkWidget* w, gpointer data) {
	GAptPkgList* pkglist = static_cast<GAptPkgList*> (data);

  g_return_if_fail(pkglist != 0);

  gnome_apt_quit();
}

static void
complete_run_cb (GtkWidget* w, gpointer data) {
	GAptPkgList* pkglist = static_cast<GAptPkgList*> (data);

  g_return_if_fail(pkglist != 0);

  GAptCacheFile* cf = gnome_apt_cache_file();
	if (cf) {
		GObject* cc = gapt_cache_control_new (cf);
		if (!gapt_cache_control_complete (GAPT_CACHE_CONTROL (cc))) {
			if (!_error->empty()) {
				gnome_apt_error_dialog (_("Run failed"));
			}
      }
  }
  else {
    g_warning(__FUNCTION__);
    g_warning("No cache file");
  }
}

static void
upgrade_cb (GtkWidget* w, gpointer data) {
	GAptPkgList* pkglist = static_cast<GAptPkgList*> (data);
	g_return_if_fail (pkglist);

  GAptCacheFile* cf = gnome_apt_cache_file();
  if (cf) {
		cf->MarkUpgrades();
		pkglist->getView()->queue_recalc_rows (true);
		pkglist->getView()->getModel()->update_status();
	} else {
    g_warning(__FUNCTION__);
    g_warning("No cache file");
  }
}

static void
smart_upgrade_cb (GtkWidget* w, gpointer data) {
	GAptPkgList* pkglist = static_cast<GAptPkgList*> (data);
	g_return_if_fail (pkglist);

	GAptCacheControl* cc = GAPT_CACHE_CONTROL (pkglist->getCacheControl());
	gapt_cache_control_smart_mark_upgrades (GAPT_CACHE_CONTROL (cc));
	pkglist->getView()->queue_recalc_rows (true);
	pkglist->getView()->getModel()->update_status();
}

static void
fix_broken_cb (GtkWidget* w, gpointer data) {
	GAptPkgList* pkglist = static_cast<GAptPkgList*> (data);
	g_return_if_fail (pkglist);

  GAptCacheFile* cf = gnome_apt_cache_file();
  if (cf) {
		cf->Fix();
		if (_error->PendingError()) {
			string errmsg;
			_error->PopMessage (errmsg);
			gnome_apt_error_dialog (errmsg.c_str());
		}

		pkglist->getView()->queue_recalc_rows (true);
		pkglist->getView()->getModel()->update_status();
	} else {
    g_warning(__FUNCTION__);
    g_warning("No cache file");
  }
}

static void
install_pkg_cb (GtkWidget* w, gpointer data) {
	GAptPkgList* pkglist = static_cast<GAptPkgList*> (data);

  g_return_if_fail(pkglist != 0);

  pkglist->install();
}

static void
remove_pkg_cb (GtkWidget* w, gpointer data) {
	GAptPkgList* pkglist = static_cast<GAptPkgList*> (data);

  g_return_if_fail(pkglist != 0);

  pkglist->remove();
}

static void
keep_pkg_cb (GtkWidget* w, gpointer data) {
	GAptPkgList* pkglist = static_cast<GAptPkgList*> (data);

  g_return_if_fail(pkglist != 0);
  
  pkglist->keep();
}

static void
about_cb (GtkWidget* w, gpointer data) {
	GAptPkgList* pkglist = static_cast<GAptPkgList*> (data);

  g_return_if_fail(pkglist != 0);

  static GtkWidget* about = NULL;

  if (about != NULL)
    {
      gdk_window_show(about->window);
      gdk_window_raise(about->window);
      return;
    }
 
	const gchar* authors[] = { 
		"Jason Gunthorpe <jgg@debian.org> - Project leader",
		"Manoj Srivastava <srivasta@datasync.com> - Dependency Expert",
		"Ben Gertzfield <che@debian.org> - Packaging and Releases",
		"Adam Heath <doogie@debian.org> - FTP method author",
		"Filip Van Raemdonck <mechanix@debian.org> - GNOME frontend",
		"",
		"Past Contributors:",
		"Brian White <bcwhite@verisim.com> - Project originator",
		"Tom Lees <tom@lpsg.demon.co.uk> - DPKG documentation and ideas",
		"Behan Webster <behanw@verisim.com> - Original GUI design",
		"Scott Ellis <storm@gate.net> - Original packaging and beta releases",
		"Branden Robinson <branden@purdue.edu> - Man Page Documentation",
		"Havoc Pennington <hp@pobox.com> - GNOME frontend",
		"Diego Lages <dlages@dcc.ufrj.br> - GNOME frontend",
		"Olivier Gravel Aubin <ogaubin@ucalgary.ca> - GNOME frontend",
		"Mitch Blevins <mblevin@debian.org> - GNOME package maintainer",
		NULL 
	};

	GdkPixbuf* logo = gdk_pixbuf_new_from_file (GNOME_APT_LOGO, NULL);
	gchar* text = g_strconcat (
	      _("Gnome Apt is a tool for managing the software packages installed"
	      " on your Debian GNU/Linux system.\n"),
	      /* This legal text should not be translated */
	      " Gnome Apt is distributed in the hope that it will be useful, "
	      "but WITHOUT ANY WARRANTY; without even the implied "
	      "warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR "
	      "PURPOSE. See the full text of the GPL for details.",
	      NULL);

	about = GTK_WIDGET (g_object_new (GTK_TYPE_ABOUT_DIALOG,
	      "name", "Apt: A Package Tool (GNOME Frontend)",
	      "version", VERSION,
	      "copyright", "(c) 1998-2003 Individual contributors, GNU General Public License.",
	      "comments", text,
	      "authors", authors,
	      "logo", logo,
	      NULL));
	g_object_unref (G_OBJECT (logo));
	g_free (text);

	gnome_apt_setup_dialog (about);

	g_signal_connect (G_OBJECT (about), "destroy", G_CALLBACK (gtk_widget_destroyed), &about);
	gtk_widget_show_all (about);
}

static void
make_icon_help (GAptPixbufType pbt, GtkWidget* vbox, const gchar* text) {
	GdkPixbuf* pixbuf;
  GtkWidget* hbox;
  GtkWidget* pix;
  GtkWidget* label;

	hbox = gtk_hbox_new (FALSE, 0);

	pixbuf = gapt_pixbuf_new (pbt);
	pix = gtk_image_new_from_pixbuf (pixbuf);
	g_object_unref (G_OBJECT (pixbuf));
	gtk_misc_set_alignment (GTK_MISC (pix), 0.5, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), pix, FALSE, FALSE, 0);

  label = gtk_label_new(text);
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, GAPT_PAD);

	gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, GAPT_PAD_SMALL);
}

static void 
icons_cb       (GtkWidget* w, gpointer data)
{
  static GtkWidget* icons = NULL;

  if (icons != NULL)
    {
      gdk_window_show(icons->window);
      gdk_window_raise(icons->window);
      return;
    }

	icons = gtk_dialog_new_with_buttons (_("GNOME Apt Icon Legend"), NULL,
	      GtkDialogFlags (GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR),
	      GTK_STOCK_CLOSE, GTK_RESPONSE_OK, NULL);
  gnome_apt_setup_dialog(icons);
	g_signal_connect (G_OBJECT (icons), "destroy",
		G_CALLBACK (gtk_widget_destroyed), &icons);
	g_signal_connect_swapped (G_OBJECT (icons), "response",
		G_CALLBACK (gtk_widget_destroy), G_OBJECT (icons));

	GtkWidget* vbox = gtk_vbox_new (TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), GAPT_PAD);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (icons)->vbox), vbox, TRUE, TRUE, 0);

	make_icon_help (GAPT_PIXBUF_SECTION, vbox, _("Package section"));
	make_icon_help (GAPT_PIXBUF_PACKAGE, vbox, _("A package"));
	make_icon_help (GAPT_PIXBUF_DEPENDS, vbox, _("Dependency"));
	make_icon_help (GAPT_PIXBUF_RECOMMENDS, vbox, _("Recommends relationship"));
	make_icon_help (GAPT_PIXBUF_SUGGESTS, vbox, _("Suggests relationship"));
	make_icon_help (GAPT_PIXBUF_REPLACES, vbox, _("Replaces relationship"));
	make_icon_help (GAPT_PIXBUF_CONFLICTS, vbox, _("Conflicts relationship"));
 
	gtk_widget_show_all (icons);
}
