/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * gcm-config-xorg.c
 * Copyright (C) Nicolas Bruguier 2006 <gandalfn@club-internet.fr>
 * 
 * gcm-config-xorg is free software.
 * 
 * You may redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option)
 * any later version.
 * 
 * gcm-config-xorg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with gnome-compiz-preferences.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

 
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

/*
 * Standard gettext macros.
 */
#ifdef ENABLE_NLS
#  include <libintl.h>
#  undef _
#  define _(String) dgettext (PACKAGE, String)
#  ifdef gettext_noop
#    define N_(String) gettext_noop (String)
#  else
#    define N_(String) (String)
#  endif
#else
#  define textdomain(String) (String)
#  define gettext(String) (String)
#  define dgettext(Domain,Message) (Message)
#  define dcgettext(Domain,Message,Type) (Message)
#  define bindtextdomain(Domain,Directory) (Domain)
#  define _(String) (String)
#  define N_(String) (String)
#endif

#include "gcm-xorg.h"
#include "gcm-xorg-template.h"

static gboolean aiglx = FALSE;
static gboolean xgl = FALSE;
static const char * template_file = NULL;
static const char * xorg_conf_output = NULL;
static const char * xorg_conf_backup = NULL;

static const struct poptOption options[] = {
	{ "aiglx", 'a', POPT_ARG_NONE | POPT_ARGFLAG_ONEDASH, 
	  &aiglx, 'a', N_("AiGLX configuration"), NULL },
	{ "xgl", 'x', POPT_ARG_NONE | POPT_ARGFLAG_ONEDASH, 
	  &xgl, 'x', N_("Xgl configuration"), NULL },
	{ "template", 't', POPT_ARG_STRING | POPT_ARGFLAG_ONEDASH, 
	  (char **) &template_file, 't', N_("Template file"), NULL },
	{ "output", 'o', POPT_ARG_STRING | POPT_ARGFLAG_ONEDASH, 
	  (char **) &xorg_conf_output, 'o', N_("Output file"), NULL },
	{ NULL, '\0', 0, NULL, '\0', NULL, NULL }
};

static int 
on_depth_choice(GCMXorgTemplate * template, gchar ** vals, gint def)
{
	GtkWidget * dialog;
	gint cpt;
	gint result = 0;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *image;
	GtkWidget *label;
	GtkWidget *alignment;
	GSList *radiobutton_group = NULL, * item;
	
	dialog = gtk_dialog_new_with_buttons(_("Default Depth"),
										 NULL,
										 GTK_DIALOG_DESTROY_WITH_PARENT,
										 GTK_STOCK_OK,
                                         GTK_RESPONSE_ACCEPT,
										 NULL);
	hbox = gtk_hbox_new(FALSE, 5);
	gtk_widget_show (hbox);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, FALSE, FALSE, 5);

	image = gtk_image_new_from_stock("gtk-dialog-question", GTK_ICON_SIZE_DIALOG);
	gtk_widget_show(image);
	gtk_box_pack_start(GTK_BOX(hbox), image, FALSE, FALSE, 0);

	label = gtk_label_new(_("<b>Please select your desired\ndefault color depth.</b>"));
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_label_set_use_markup(GTK_LABEL(label), TRUE);

	alignment = gtk_alignment_new(0.5, 0.5, 1, 1);
	gtk_widget_show(alignment);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), alignment, TRUE, TRUE, 0);
	gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 0, 0, 45, 0);

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_widget_show(vbox);
	gtk_container_add(GTK_CONTAINER(alignment), vbox);
 
	for (cpt = 0; vals[cpt]; cpt++)
	{
		GtkWidget *radiobutton;
		gchar * name = g_strdup_printf(_("%i bits"), atoi(vals[cpt]));
		
		radiobutton = gtk_radio_button_new_with_mnemonic(NULL, name);
		gtk_widget_show(radiobutton);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobutton), cpt == def);
		gtk_box_pack_start(GTK_BOX(vbox), radiobutton, FALSE, FALSE, 0);
		gtk_radio_button_set_group(GTK_RADIO_BUTTON (radiobutton), radiobutton_group);
	    radiobutton_group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(radiobutton));
	}

	gtk_dialog_run(GTK_DIALOG(dialog));	
	
	for (item = radiobutton_group; item; item = item->next)
	{
		if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(item->data)))
			break;
		result++;
	}
	return atoi(vals[cpt - 1 - result]);
}

static void
restart_xorg(void)
{
	GtkWidget * dialog;
	gchar * buffer;
	
	buffer = g_strdup_printf(_("In order to complete the configuration, xorg needs to be restarted.\n\n"
							   "Please make sure that you save all your work before restarting.\n\n"
							   "If you have some trouble after restart, you can restore initial configuration like this:\n"
							   "sudo cp %s %s"), xorg_conf_backup, xorg_conf_output);
	dialog = gtk_message_dialog_new (NULL,
								     GTK_DIALOG_DESTROY_WITH_PARENT,
									 GTK_MESSAGE_INFO,
									 GTK_BUTTONS_OK_CANCEL,
									 buffer);
	gtk_window_set_title(GTK_WINDOW (dialog), _("Restart Xorg"));
	gtk_window_set_modal(GTK_WINDOW (dialog), TRUE);
	gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER_ALWAYS);
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
	{
		GCMXorg * xorg = gcm_xorg_new();
		gcm_xorg_stop(xorg);
		g_object_unref(xorg);
	}
	gtk_widget_destroy (dialog);
	g_free(buffer);	
}

static int 
parse(void)
{
	GCMXorgConfig * xorg_config;
 	GCMXorgTemplate * template;

	/* Create template */
	if ((template = gcm_xorg_template_new((gchar *)template_file, 
										  xgl, aiglx)) == NULL)
	{
		g_warning("Error on create template");
		return -1;
	}
	
    g_signal_connect(template, "depth_choice", G_CALLBACK(on_depth_choice), NULL);
	/* Parse template */
	gcm_xorg_template_parse(template);
	
	if (!gcm_xorg_template_get_parsed(template))
	{
		g_warning("Bad template");
		return -1;
	}
	if (!gcm_xorg_template_get_modified(template))
	{
		g_warning("Xorg already configured");
		return -1;
	}
	
	/* Template description */
	printf("Parsed with %s\n", gcm_xorg_template_get_name(template));
	printf("Author : %s\n", gcm_xorg_template_get_author(template));
	printf("EMail : %s\n", gcm_xorg_template_get_email(template));
	
	/* Write xorg config file */
	xorg_config = gcm_xorg_template_get_config(template);
	if (!gcm_xorg_config_write(xorg_config, 
							   (gchar *)xorg_conf_output, 
							   (gchar **)&xorg_conf_backup))
	{
		g_warning("Error on write output file !");
		return -1;
	}
	if (!xorg_conf_output) 
		xorg_conf_output = gcm_xorg_config_get_filename(xorg_config);
	
	restart_xorg();
	
	return 0;
}

int
main (int argc, char *argv[])
{
	
#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	gnome_program_init ("gcm-config-xorg", VERSION,
		      LIBGNOMEUI_MODULE, argc, argv,
		      GNOME_PARAM_POPT_FLAGS, POPT_CONTEXT_POSIXMEHARDER,
			  GNOME_PARAM_POPT_TABLE, options,
			  GNOME_PARAM_NONE);
	
	if (!template_file || (!aiglx && !xgl)) 
	{
		printf("usage: gcm-config-xorg --help\n");
		return -1;
	}
	
	if (parse())
		return 0;
	else
		return -1;
}
