/* vi:set ts=8 sts=0 sw=8:
 * $Id: prefs.c,v 1.4 2001/01/25 06:02:10 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include "main.h"
#include "misc.h"
#include "prefs.h"


/*** PRIVATE FUNCTION PROTOTYPES ***/
static void prefs_modified_cb(GtkWidget *wgt, gpointer cbdata);
static void prefs_prefs_save(win_t *w);
static void prefs_close_cb(GtkWidget *wgt, gpointer cbdata);
static void optmenu_modified_set(int omenu_id, int prefs_id);
static void prefs_apply_cb(GtkWidget *wgt, gpointer cbdata);


enum ef_prefs_id_e {
	Ef_prefs_dialog,

	Ef_prefs_main_tbstyle,	/* optionmenu */
	Ef_prefs_fops_tbstyle,	/* optionmenu */
	Ef_prefs_adv_tbstyle,	/* optionmenu */
	Ef_prefs_mfile_tbstyle,	/* optionmenu */
	Ef_prefs_mdir_tbstyle,	/* optionmenu */
	Ef_prefs_show_tabs,	/* checkbutton */
	Ef_prefs_tab_pos,	/* optionmenu */

	Ef_prefs_open_file,	/* entry */
	Ef_prefs_delete_file,	/* entry */
	Ef_prefs_copy_file,	/* entry */
	Ef_prefs_print_file,	/* entry */
	Ef_prefs_exec_file,	/* entry */
	Ef_prefs_shell_file,	/* entry */
	Ef_prefs_cmd_prompting,	/* checkbutton */

	Ef_prefs_save_mainwin,	/* checkbutton */
	Ef_prefs_save_multfile,	/* checkbutton */
	Ef_prefs_save_multdir,	/* checkbutton */
	Ef_prefs_save_results,	/* checkbutton */
	Ef_prefs_save_advopts,	/* checkbutton */
	Ef_prefs_save_prefs,	/* checkbutton */
	Ef_prefs_save_builtin,	/* checkbutton */

	Ef_prefs_archive_fname,	/* entry */
	Ef_prefs_archive_add,	/* entry */
	Ef_prefs_archive_remove,/* entry */
	Ef_prefs_archive_list,	/* entry */
	Ef_prefs_archive_builtin,/* checkbutton */

	Ef_prefs_tmpdir,	/* entry */
	Ef_prefs_shell,		/* entry */
	Ef_prefs_use_dnd,	/* checkbutton */

	Ef_prefs_MAX
};
typedef enum ef_prefs_id_e ef_prefs_id_t;


static ef_t ef_prefs_widgets[] = {
	{
		"Properties",
		Ef_prefs_dialog,
		Ef_wgt_dialog,
	},

	{
		"optmenu_main_tbstyle",
		Ef_prefs_main_tbstyle,
		Ef_wgt_optmenu,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"optmenu_fops_tbstyle",
		Ef_prefs_fops_tbstyle,
		Ef_wgt_optmenu,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"optmenu_adv_tbstyle",
		Ef_prefs_adv_tbstyle,
		Ef_wgt_optmenu,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"optmenu_mfile_tbstyle",
		Ef_prefs_mfile_tbstyle,
		Ef_wgt_optmenu,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"optmenu_mdir_tbstyle",
		Ef_prefs_mdir_tbstyle,
		Ef_wgt_optmenu,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"chkbut_show_tabs",
		Ef_prefs_show_tabs,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"optmenu_tab_pos",
		Ef_prefs_tab_pos,
		Ef_wgt_optmenu,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},

	{
		"entry_open_file",
		Ef_prefs_open_file,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"entry_delete_file",
		Ef_prefs_delete_file,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"entry_copy_file",
		Ef_prefs_copy_file,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"entry_print_file",
		Ef_prefs_print_file,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"entry_exec_file",
		Ef_prefs_exec_file,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"entry_shell_file",
		Ef_prefs_shell_file,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"chkbut_cmd_prompting",
		Ef_prefs_cmd_prompting,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},

	{
		"chkbut_save_main_win",
		Ef_prefs_save_mainwin,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"chkbut_save_multi_file",
		Ef_prefs_save_multfile,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"chkbut_save_multi_dir",
		Ef_prefs_save_multdir,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"chkbut_save_results",
		Ef_prefs_save_results,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"chkbut_save_adv",
		Ef_prefs_save_advopts,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"chkbut_save_prefs",
		Ef_prefs_save_prefs,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"chkbut_save_builtin",
		Ef_prefs_save_builtin,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},

	{
		"entry_archive_fname",
		Ef_prefs_archive_fname,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"entry_archive_add",
		Ef_prefs_archive_add,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"entry_archive_remove",
		Ef_prefs_archive_remove,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"entry_archive_list",
		Ef_prefs_archive_list,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"chkbut_archive_builtin_view",
		Ef_prefs_archive_builtin,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},

	{
		"entry_tmpdir",
		Ef_prefs_tmpdir,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"entry_shell",
		Ef_prefs_shell,
		Ef_wgt_entry,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},
	{
		"chkbut_dnd",
		Ef_prefs_use_dnd,
		Ef_wgt_checkbutton,
		GTK_SIGNAL_FUNC(prefs_modified_cb),
	},

	{ NULL }
}; /* ef_prefs_widgets[] */


enum data_e {
	PrefsBool,
	PrefsInt,
	PrefsString
};
typedef enum data_e data_t;


struct prefs_s {
	char *		name;		/* name of preference */
	data_t		datatype;	/* data type of preference */
	ef_prefs_id_t	wgt_id;		/* corresponding widget id */
	gpointer	data;		/* the actual value of this pref */
					/* initially set to the corresp */
					/* data_t value for sanity checks. */
};
typedef struct prefs_s prefs_t;


static prefs_t app_prefs[] = {
	{
		"main_tb_style=2",
		PrefsInt,
		Ef_prefs_main_tbstyle,
		GINT_TO_POINTER(Prefs_main_tbstyle)
	},
	{
		"fops_tb_style=0",
		PrefsInt,
		Ef_prefs_fops_tbstyle,
		GINT_TO_POINTER(Prefs_fops_tbstyle)
	},
	{
		"adv_tb_style=0",
		PrefsInt,
		Ef_prefs_adv_tbstyle,
		GINT_TO_POINTER(Prefs_adv_tbstyle)
	},
	{
		"mfile_tb_style=0",
		PrefsInt,
		Ef_prefs_mfile_tbstyle,
		GINT_TO_POINTER(Prefs_mfile_tbstyle)
	},
	{
		"mdir_tb_style=0",
		PrefsInt,
		Ef_prefs_mdir_tbstyle,
		GINT_TO_POINTER(Prefs_mdir_tbstyle)
	},
	{
		"main_nb_show_tabs=FALSE",
		PrefsBool,
		Ef_prefs_show_tabs,
		GINT_TO_POINTER(Prefs_show_tabs)
	},
	{
		"main_nb_tab_pos=3",
		PrefsInt,
		Ef_prefs_tab_pos,
		GINT_TO_POINTER(Prefs_tab_pos)
	},

	{
		"cmd_file_open=gnp $files &",
		PrefsString,
		Ef_prefs_open_file,
		GINT_TO_POINTER(Prefs_file_open)
	},
	{
		"cmd_file_delete=rm -f $files",
		PrefsString,
		Ef_prefs_delete_file,
		GINT_TO_POINTER(Prefs_file_delete)
	},
	{
		"cmd_file_copy=cp -f $files $dest",
		PrefsString,
		Ef_prefs_copy_file,
		GINT_TO_POINTER(Prefs_file_copy)
	},
	{
		"cmd_file_print=lpr $files",
		PrefsString,
		Ef_prefs_print_file,
		GINT_TO_POINTER(Prefs_file_print)
	},
	{
		"cmd_file_exec=exec $files",
		PrefsString,
		Ef_prefs_exec_file,
		GINT_TO_POINTER(Prefs_file_exec)
	},
	{
		"cmd_file_shell=",
		PrefsString,
		Ef_prefs_shell_file,
		GINT_TO_POINTER(Prefs_file_shell)
	},
	{
		"cmd_prompting=FALSE",
		PrefsBool,
		Ef_prefs_cmd_prompting,
		GINT_TO_POINTER(Prefs_cmd_prompting)
	},

	{
		"save_main_win=FALSE",
		PrefsBool,
		Ef_prefs_save_mainwin,
		GINT_TO_POINTER(Prefs_save_main_win)
	},
	{
		"save_mfile_win=TRUE",
		PrefsBool,
		Ef_prefs_save_multfile,
		GINT_TO_POINTER(Prefs_save_mfile_win)
	},
	{
		"save_mdir_win=TRUE",
		PrefsBool,
		Ef_prefs_save_multdir,
		GINT_TO_POINTER(Prefs_save_mdir_win)
	},
	{
		"save_results_win=TRUE",
		PrefsBool,
		Ef_prefs_save_results,
		GINT_TO_POINTER(Prefs_save_results_win)
	},
	{
		"save_adv_win=FALSE",
		PrefsBool,
		Ef_prefs_save_advopts,
		GINT_TO_POINTER(Prefs_save_advopts_win)
	},
	{
		"save_prefs_win=FALSE",
		PrefsBool,
		Ef_prefs_save_prefs,
		GINT_TO_POINTER(Prefs_save_prefs_win)
	},
	{
		"save_builtin_win=TRUE",
		PrefsBool,
		Ef_prefs_save_builtin,
		GINT_TO_POINTER(Prefs_save_builtin_win)
	},

	{
		"archive_fname=gnome-find-data.tar",
		PrefsString,
		Ef_prefs_archive_fname,
		GINT_TO_POINTER(Prefs_archive_fname)
	},
	{
		"cmd_archive_add=gtar -rf $archive $files",
		PrefsString,
		Ef_prefs_archive_add,
		GINT_TO_POINTER(Prefs_archive_add)
	},
	{
		"cmd_archive_remove=gtar --delete -f $archive $files",
		PrefsString,
		Ef_prefs_archive_remove,
		GINT_TO_POINTER(Prefs_archive_remove)
	},
	{
		"cmd_archive_list=gtar -tvf $archive",
		PrefsString,
		Ef_prefs_archive_list,
		GINT_TO_POINTER(Prefs_archive_list)
	},
	{
		"archive_builtin_view=TRUE",
		PrefsBool,
		Ef_prefs_archive_builtin,
		GINT_TO_POINTER(Prefs_archive_builtin)
	},

	{
		"tmpdir=/var/tmp",
		PrefsString,
		Ef_prefs_tmpdir,
		GINT_TO_POINTER(Prefs_tmpdir)
	},
	{
		"shell=/bin/sh",
		PrefsString,
		Ef_prefs_shell,
		GINT_TO_POINTER(Prefs_shell)
	},
	{
		"use_dnd=FALSE",
		PrefsBool,
		Ef_prefs_use_dnd,
		GINT_TO_POINTER(Prefs_dnd)
	},

	{
		"main_win_h",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_main_win_h)
	},
	{
		"main_win_w",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_main_win_w)
		
	},
	{
		"mfile_win_h",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_mfile_win_h)
	},
	{
		"mfile_win_w",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_mfile_win_w)
	},
	{
		"mdir_win_h",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_mdir_win_h)
	},
	{
		"mdir_win_w",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_mdir_win_w)
	},
	{
		"res_win_h",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_res_win_h)
	},
	{
		"res_win_w",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_res_win_w)
	},
	{
		"adv_win_h",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_adv_win_h)
	},
	{
		"adv_win_w",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_adv_win_w)
	},
	{
		"prefs_win_h",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_prefs_win_h)
	},
	{
		"prefs_win_w",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_prefs_win_w)
	},
	{
		"builtin_win_h",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_builtin_win_h)
	},
	{
		"builtin_win_w",
		PrefsInt,
		Ef_prefs_MAX,
		GINT_TO_POINTER(Prefs_builtin_win_w)
	},
	{ NULL }
}; /* app_prefs[] */


void
prefs_save(win_t *w)
{
	prefs_t *pap;

	gnome_config_push_prefix("/gnome-find/Prefs/");

	for (pap = app_prefs; pap->name; pap++) {
		switch (pap->datatype) {
		case PrefsBool:
			gnome_config_set_bool(pap->name, (gboolean)pap->data);
			APPDBG_PREFS(("prefs_save: %s is %s\n",
						pap->name,
						((gboolean)(pap->data)) ?
							"TRUE" : "FALSE"));
			break;
		case PrefsInt:
			gnome_config_set_int(pap->name,
					     GPOINTER_TO_INT(pap->data));
			APPDBG_PREFS(("prefs_save: %s is %d\n",
						pap->name,
						GPOINTER_TO_INT(pap->data)));
			break;
		case PrefsString:
			gnome_config_set_string(pap->name, (char *)pap->data);
			APPDBG_PREFS(("prefs_save: %s is \"%s\"\n",
						pap->name,
						(char *)(pap->data)));
			break;
		default:
			APPDBG_PREFS(("prefs_save: %s has unknown type (%d)\n",
						pap->name, pap->datatype));
			break;
		}
	}

	gnome_config_pop_prefix();
	gnome_config_sync();
} /* prefs_save */


void
prefs_cb(GtkWidget *wgt, gpointer cbdata)
{
	static GnomeHelpMenuEntry help = { NULL, "prefs" };
	GtkWidget **wgts;
	prefs_t *pap;
	int num;
	win_t *w = (win_t *)cbdata;

	/* setup the widgets in the prefs dialog */
	wgts = ef_prefs_wgts_get();
	num = (sizeof(ef_prefs_widgets) / sizeof(ef_prefs_widgets[0])) - 1;
	wgts = my_widgets_setup(w, wgts, num, ef_prefs_widgets, "Properties",
				Ef_prefs_dialog, Ef_prefs_MAX, FALSE);
	ef_prefs_wgts_set(wgts);
	gtk_window_set_title(GTK_WINDOW(wgts[Ef_prefs_dialog]),
			     PACKAGE" Properties...");
	gtk_window_set_position(GTK_WINDOW(wgts[Ef_prefs_dialog]),
				GLADE_POS_MOUSE);

	/* attach callbacks to the signals generated by the dialog buttons */
	gtk_signal_connect(GTK_OBJECT(wgts[Ef_prefs_dialog]), "delete_event",
			   GTK_SIGNAL_FUNC(gtk_true), NULL);
	gtk_signal_connect(GTK_OBJECT(wgts[Ef_prefs_dialog]), "apply",
			   GTK_SIGNAL_FUNC(prefs_apply_cb), w);
	gtk_signal_connect(GTK_OBJECT(wgts[Ef_prefs_dialog]), "close",
			   GTK_SIGNAL_FUNC(prefs_close_cb), NULL);
	help.name = gnome_app_id;
	gtk_signal_connect(GTK_OBJECT(wgts[Ef_prefs_dialog]), "help",
			   GTK_SIGNAL_FUNC(gnome_help_pbox_display), &help);

	/* Set the widget states according to the actual values. */
	for (pap = app_prefs; pap->name; pap++) {
		switch (pap->datatype) {
		case PrefsBool:
			g_assert(GTK_IS_TOGGLE_BUTTON(wgts[pap->wgt_id]));
			gtk_toggle_button_set_active(
					GTK_TOGGLE_BUTTON(wgts[pap->wgt_id]),
					(gboolean)(GPOINTER_TO_INT(pap->data)));
			break;
		case PrefsInt:
			g_assert(pap->wgt_id == Ef_prefs_MAX ||
				 pap->wgt_id == Ef_prefs_main_tbstyle ||
				 pap->wgt_id == Ef_prefs_fops_tbstyle ||
				 pap->wgt_id == Ef_prefs_adv_tbstyle ||
				 pap->wgt_id == Ef_prefs_mfile_tbstyle ||
				 pap->wgt_id == Ef_prefs_mdir_tbstyle ||
				 pap->wgt_id == Ef_prefs_tab_pos);

			if (pap->wgt_id != Ef_prefs_MAX) {
				gtk_option_menu_set_history(
					GTK_OPTION_MENU(wgts[pap->wgt_id]),
					GPOINTER_TO_INT(pap->data));
			}
			break;
		case PrefsString:
			g_assert(GTK_IS_ENTRY(wgts[pap->wgt_id]));
			gtk_entry_set_text(GTK_ENTRY(wgts[pap->wgt_id]),
					   (char *)pap->data);
			break;
		default:
			break;
		}
	}

	/* set prefs window size */
	my_prefs_window_set_size(wgts[Ef_prefs_dialog], Prefs_save_prefs_win,
				 Prefs_prefs_win_w, Prefs_prefs_win_h);

	/* initially, nothing has been modified, so make sure it shows */
	gnome_property_box_set_modified(
			GNOME_PROPERTY_BOX(wgts[Ef_prefs_dialog]), FALSE);

	gtk_widget_show(wgts[Ef_prefs_dialog]);
} /* prefs_cb */


void
prefs_init(void)
{
	prefs_t *pap;
#ifdef ENABLE_DEBUG
	int id = 0;
#endif
	gnome_config_push_prefix("/gnome-find/Prefs/");

	for (pap = app_prefs; pap->name; pap++) {
#ifdef ENABLE_DEBUG
		g_assert(GPOINTER_TO_INT(pap->data) == id++);
#endif
		switch (pap->datatype) {
		case PrefsBool:
			pap->data = (gpointer)gnome_config_get_bool(pap->name);
			APPDBG_PREFS(("prefs_init: %s is %s\n",
						pap->name,
						((gboolean)(pap->data)) ?
							"TRUE" : "FALSE"));
			break;
		case PrefsInt:
			pap->data = (gpointer)gnome_config_get_int(pap->name);
			APPDBG_PREFS(("prefs_init: %s is %d\n",
						pap->name,
						GPOINTER_TO_INT(pap->data)));
			break;
		case PrefsString:
			pap->data= (gpointer)gnome_config_get_string(pap->name);
			APPDBG_PREFS(("prefs_init: %s is \"%s\"\n",
						pap->name,
						(char *)(pap->data)));
			break;
		default:
			APPDBG_PREFS(("prefs_init: %s has unknown type (%d)\n",
						pap->name, pap->datatype));
			break;
		}
	}

	gnome_config_pop_prefix();
} /* prefs_init */


int
prefs_int_get(prefs_id_t pref_id)
{
	g_assert(pref_id >= 0 && pref_id < Prefs_MAX);
	g_assert(app_prefs[pref_id].datatype == PrefsInt);

	return GPOINTER_TO_INT(app_prefs[pref_id].data);
} /* prefs_int_get */


gboolean
prefs_bool_get(prefs_id_t pref_id)
{
	g_assert(pref_id >= 0 && pref_id < Prefs_MAX);
	g_assert(app_prefs[pref_id].datatype == PrefsBool);

	return (gboolean)(GPOINTER_TO_INT(app_prefs[pref_id].data));
} /* prefs_bool_get */


char *
prefs_string_get(prefs_id_t pref_id)
{
	g_assert(pref_id >= 0 && pref_id < Prefs_MAX);
	g_assert(app_prefs[pref_id].datatype == PrefsString);

	return (char *)app_prefs[pref_id].data;
} /* prefs_string_get */


void
prefs_int_set(prefs_id_t pref_id, const int num)
{
	g_assert(pref_id >= 0 && pref_id < Prefs_MAX);
	g_assert(app_prefs[pref_id].datatype == PrefsInt);

	app_prefs[pref_id].data = GINT_TO_POINTER(num);
} /* prefs_int_set */


void
prefs_bool_set(prefs_id_t pref_id, const gboolean bval)
{
	g_assert(pref_id >= 0 && pref_id < Prefs_MAX);
	g_assert(app_prefs[pref_id].datatype == PrefsBool);

	app_prefs[pref_id].data = GINT_TO_POINTER((int)bval);
} /* prefs_bool_set */


void
prefs_string_set(prefs_id_t pref_id, const char *str)
{
	g_assert(pref_id >= 0 && pref_id < Prefs_MAX);
	g_assert(app_prefs[pref_id].datatype == PrefsString);

	app_prefs[pref_id].data = (gpointer)str;
} /* prefs_string_set */


/*
 * Sets the toolbar style of various toolbars in a window.
 */
void
prefs_toolbar_set(int prefs_id, GtkWidget *tb)
{
	int val;

	if (!tb)
		return;

	val = prefs_int_get(prefs_id);
	if (val >= 0 && val < 3)
		gtk_toolbar_set_style(GTK_TOOLBAR(tb), (GtkToolbarStyle)val);
} /* prefs_toolbar_set */


/*** PRIVATE FUNCTION DEFINITIONS ***/
static void
prefs_modified_cb(GtkWidget *wgt, gpointer cbdata)
{
	GtkWidget **wgts;

	wgts = ef_prefs_wgts_get();
	gnome_property_box_changed(GNOME_PROPERTY_BOX(wgts[Ef_prefs_dialog]));
} /* prefs_modified_cb */


static void
prefs_prefs_save(win_t *w)
{
	int width, height;
	GtkWidget **wgts;

	if ((wgts = ef_prefs_wgts_get()) == NULL)
		return;

	gdk_window_get_size(wgts[Ef_prefs_dialog]->window, &width, &height);
	prefs_int_set(Prefs_prefs_win_w, width);
	prefs_int_set(Prefs_prefs_win_h, height);
	prefs_save(w);
	gnome_config_sync();
} /* prefs_prefs_save */


static void
prefs_close_cb(GtkWidget *wgt, gpointer cbdata)
{
	GtkWidget **wgts = ef_prefs_wgts_get();

	prefs_prefs_save(cbdata);
	if (wgts && GTK_IS_WIDGET(wgts[Ef_prefs_dialog]))
		gtk_widget_destroy(wgts[Ef_prefs_dialog]);
	ef_prefs_wgts_set(NULL);
} /* prefs_close_cb */


static void
optmenu_modified_set(int omenu_id, int prefs_id)
{
	GtkWidget **wgts, *child;
	GtkButton *button;
	char *str;

	wgts = ef_prefs_wgts_get();

	button = &(GTK_OPTION_MENU(wgts[omenu_id])->button);
	child = GTK_BIN(button)->child;
	g_assert(GTK_IS_LABEL(child));
	gtk_label_get(GTK_LABEL(child), &str);

	if (g_strcasecmp(str, "Icons only") == 0 ||
	    g_strcasecmp(str, "Left") == 0)
		app_prefs[prefs_id].data = GINT_TO_POINTER(0);
	else if (g_strcasecmp(str, "Text only") == 0 ||
		 g_strcasecmp(str, "Right") == 0)
		app_prefs[prefs_id].data = GINT_TO_POINTER(1);
	else if (g_strcasecmp(str, "Icons and Text") == 0 ||
		 g_strcasecmp(str, "Top") == 0)
		app_prefs[prefs_id].data = GINT_TO_POINTER(2);
	else if (g_strcasecmp(str, "Use GNOME settings") == 0) {
		/*
		 * Seems like GNOME doesn't allow text-only toolbar buttons.
		 */
		if (gnome_preferences_get_toolbar_labels())
			app_prefs[prefs_id].data = GINT_TO_POINTER(2);
		else
			app_prefs[prefs_id].data = GINT_TO_POINTER(0);
	} else {
		g_assert(g_strcasecmp(str, "Bottom") == 0);
		app_prefs[prefs_id].data = GINT_TO_POINTER(3);
	}
} /* optmenu_modified_set */


static void
prefs_apply_cb(GtkWidget *wgt, gpointer cbdata)
{
	GtkToggleButton *tog;
	GtkEntry *entry;
	GtkWidget **wgts;
	prefs_t *pap;
	char *txt;

	wgts = ef_prefs_wgts_get();
	gnome_config_push_prefix("/gnome-find/Prefs/");

	for (pap = app_prefs; pap->name; pap++) {
		switch (pap->datatype) {
		case PrefsBool:
			g_assert(GTK_IS_TOGGLE_BUTTON(wgts[pap->wgt_id]));
			tog = GTK_TOGGLE_BUTTON(wgts[pap->wgt_id]);
			pap->data = GINT_TO_POINTER(tog->active);
			gnome_config_set_bool(pap->name, tog->active);
			APPDBG_PREFS(("prefs_apply_cb: %s is %s\n",
				      pap->name,
				      tog->active ? "TRUE" : "FALSE"));
			break;
		case PrefsInt:
			g_assert(pap->wgt_id == Ef_prefs_MAX ||
				 pap->wgt_id == Ef_prefs_main_tbstyle ||
				 pap->wgt_id == Ef_prefs_fops_tbstyle ||
				 pap->wgt_id == Ef_prefs_adv_tbstyle ||
				 pap->wgt_id == Ef_prefs_mfile_tbstyle ||
				 pap->wgt_id == Ef_prefs_mdir_tbstyle ||
				 pap->wgt_id == Ef_prefs_tab_pos);

			optmenu_modified_set(Ef_prefs_main_tbstyle,
					     Prefs_main_tbstyle);
			optmenu_modified_set(Ef_prefs_fops_tbstyle,
					     Prefs_fops_tbstyle);
			optmenu_modified_set(Ef_prefs_adv_tbstyle,
					     Prefs_adv_tbstyle);
			optmenu_modified_set(Ef_prefs_mfile_tbstyle,
					     Prefs_mfile_tbstyle);
			optmenu_modified_set(Ef_prefs_mdir_tbstyle,
					     Prefs_mdir_tbstyle);
			optmenu_modified_set(Ef_prefs_tab_pos,
					     Prefs_tab_pos);

			gnome_config_set_int(pap->name,
					     GPOINTER_TO_INT(pap->data));
			APPDBG_PREFS(("prefs_apply_cb: %s is %d\n",
				      pap->name, GPOINTER_TO_INT(pap->data)));
			break;
		case PrefsString:
			g_assert(GTK_IS_ENTRY(wgts[pap->wgt_id]));
			entry = GTK_ENTRY(wgts[pap->wgt_id]);
			txt = gtk_entry_get_text(entry);
			g_free(pap->data);
			pap->data = (txt) ? (gpointer)g_strdup(txt) : NULL;
			gnome_config_set_string(pap->name, (char *)pap->data);
			APPDBG_PREFS(("prefs_apply_cb: %s is \"%s\"\n",
				      pap->name,
				      pap->data ? (char *)(pap->data) : NULL));
			break;
		default:
			APPDBG_PREFS(("prefs_apply_cb: %s unknown type (%d)\n",
				      pap->name, pap->datatype));
			break;
		}
	}

	gnome_config_pop_prefix();
	gnome_config_sync();

	ef_windows_update_all_prefs();
} /* prefs_apply_cb */


/* the end */
