--[[
	Hearts - hearts.lua
	Copyright 2006 Sander Marechal
	
	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
]]

-- define the globals
hand, trick, score, rules = nil, nil, nil, nil
suit, rank = 1, 2

-- define the suits, the card values and the players
clubs, diamonds, hearts, spades = 0, 1, 2, 3
ace, two, three, four, five, six, seven, eight, nine, ten, jack, queen, king, ace_high = 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14
north, east, south, west = 1, 2, 3, 4 -- one off from the C code because lua arrays start at 1

--[[ Some usefull functions for selecting subsets of card lists ]]

-- return true if card is in list
function have_card(card, list)
	if type(list) ~= "table" then return false end
	if type(card) ~= "table" then return false end
	
	for list_index, list_card in ipairs(list) do
		if card[suit] == list_card[suit] and card[rank] == list_card[rank] then
			return list_index
		end
	end
	
	return false
end

-- returns true if a card of suit is in list
function have_suit(card_suit, list)
	local curr_card = nil
	for _, curr_card in ipairs(list) do
		if curr_card[suit] == card_suit then
			return true
		end
	end
	return false
end

-- return a list of all cards that are valid to play
function get_valid_cards(list)
	local card, result = nil, {}
	if type(list) ~= "table" then return {} end
	for _, card in ipairs(list) do
		if is_valid_card(card) then
			table.insert(result, {card[suit], card[rank]})
		end
	end
	return result
end

-- return all cards from a given suit
function get_cards_of_suit(list, card_suit)
	local card, result = nil, {}
	if type(list) ~= "table" then return {} end
	for _, card in ipairs(list) do
		if card[suit] == card_suit then
			table.insert(result, {card[suit], card[rank]})
		end
	end
	return result
end

-- return all valid cards from a given suit
function get_valid_cards_of_suit(list, card_suit)
	local card, result = nil, {}
	if type(list) ~= "table" then return {} end
	for _, card in ipairs(list) do
		if card[suit] == card_suit and is_valid_card(card) then
			table.insert(result, {card[suit], card[rank]})
		end
	end
	return result
end

-- return all cards that are not points
function get_pointless_cards(list, include_bonus)
	local card, result = nil, {}
	if type(list) ~= "table" then return {} end
	for _, card in ipairs(list) do
		if card[suit] ~= hearts and not (card[suit] == spades and card[rank] == queen) then
			table.insert(result, {card[suit], card[rank]})
		end
	end
	return result
end

-- return all cards that are not points
function get_point_cards(list, include_bonus)
	local card, result = nil, {}
	if type(list) ~= "table" then return {} end
	for _, card in ipairs(list) do
		if card[suit] == hearts or (card[suit] == spades and card[rank] == queen) then
			table.insert(result, {card[suit], card[rank]})
		end
	end
	return result
end

-- return the highest card on the trick
function get_highest_card()
	local trump = trick_get_trump()
	local high_card = {trump, 1}
	if trump == nil then return nil end
	
	for index = 1, 4 do
		local card = trick[index]
		if card ~= nil and card[suit] == trump and card[rank] > high_card[rank] then
			high_card = card
		end
	end
	return high_card
end

--[[ Various callbacks for sorting lists of cards ]]

-- sort ascending, regardless of suit
function ascending(one, two)
	assert(type(one) == "table", "sort ascending: one is not a card")
	assert(type(two) == "table", "sort ascending: two is not a card")
	
	if one[rank] < two[rank] then
		return true
	end
	
	return false
end

-- sort descending, regardless of suit
function descending(one, two)
	assert(type(one) == "table", "sort descending: one is not a card")
	assert(type(two) == "table", "sort descending: two is not a card")
	
	if one[rank] > two[rank] then
		return true
	end
	
	return false
end

-- sort clubs, diamonds, spades, hearts ascending
function suit_ascending(one, two)
	assert(type(one) == "table", "sort suit_ascending: one is not a card")
	assert(type(two) == "table", "sort suit_ascending: two is not a card")

	local one_id = one[suit] * 13 + one[rank]	
	local two_id = two[suit] * 13 + two[rank]	
	
	if one[suit] == hearts then one_id = one_id + 26 end
	if two[suit] == hearts then two_id = two_id + 26 end
	
	return (one_id < suit_id)
end

-- sort hearts, spades, diamonds, clubs descending
function suit_descending(one, two)
	assert(type(one) == "table", "sort suit_descending: one is not a card")
	assert(type(two) == "table", "sort suit_descending: two is not a card")

	local one_id = one[suit] * 13 + one[rank]	
	local two_id = two[suit] * 13 + two[rank]	
	
	if one[suit] == hearts then one_id = one_id + 26 end
	if two[suit] == hearts then two_id = two_id + 26 end
	
	return (one_id > suit_id)
end

-- sort cards ascending by point value
function points_ascending(one, two)
	assert(type(one) == "table", "sort points_ascending: one is not a card")
	assert(type(two) == "table", "sort points_ascending: two is not a card")
	
	if one[suit] == spades and one[rank] == queen then return false end
	if two[suit] == spades and two[rank] == queen then return true end
	
	if one[suit] == hearts and two[suit] ~= hearts then return false end
	if one[suit] ~= hearts and two[suit] == hearts then return true end
	
	if one[suit] == hearts and two[suit] == hearts then return (one[rank] < two[rank]) end
	
	return false
end

-- sort cards descending by point value
function points_descending(one, two)
	assert(type(one) == "table", "sort points_descending: one is not a card")
	assert(type(two) == "table", "sort points_descending: two is not a card")
	
	if one[suit] == spades and one[rank] == queen then return true end
	if two[suit] == spades and two[rank] == queen then return false end
	
	if one[suit] == hearts and two[suit] ~= hearts then return true end
	if one[suit] ~= hearts and two[suit] == hearts then return false end
	
	if one[suit] == hearts and two[suit] == hearts then return (one[rank] > two[rank]) end
	
	return false
end
