/* GnomeScanUI - Widgets for scan dialogs
 *
 * gnomescanoptionwidget.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gnomescan.h>
#include "gnomescanui.h"
#include "gnomescanui-intl.h"

#define GNOME_SCAN_OPTION_WIDGET_ERROR			(g_type_qname (GNOME_TYPE_SCAN_OPTION_WIDGET))
#define	GNOME_SCAN_OPTION_WIDGET_PARENT_CLASS(klass)	(GTK_WIDGET_CLASS (g_type_class_peek_parent ((klass))))
#define GET_PRIVATE(obj)			(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCAN_OPTION_WIDGET, GnomeScanOptionWidgetPrivate))

typedef struct _GnomeScanOptionWidgetPrivate		GnomeScanOptionWidgetPrivate;

struct _GnomeScanOptionWidgetPrivate {
  gboolean		dispose_has_run;

  /* PROPERTIES */
  gboolean		show_label;
};

void				create_label						(GnomeScanOptionWidget *widget);


/********************************
 * 	      GOBJECT		*
 ********************************/

void				gnome_scan_option_widget_set_property 			(GObject *obj,
											 guint property_id,
											 const GValue *value,
											 GParamSpec *pspec);

void				gnome_scan_option_widget_get_property 			(GObject *obj,
											 guint property_id,
											 GValue *value,
											 GParamSpec *pspec);

void				gnome_scan_option_widget_finalize			(GObject *obj);

void				gnome_scan_option_widget_dispose			(GObject *obj);


enum {
  PROP_0,
  PROP_CONTEXT,
  PROP_SHOW_LABEL
};

enum {
  N_SIGNALS
};

static guint signals[N_SIGNALS];


G_DEFINE_TYPE (GnomeScanOptionWidget, gnome_scan_option_widget, GTK_TYPE_HBOX);


void
gnome_scan_option_widget_class_init (GnomeScanOptionWidgetClass *klass)
{
  GObjectClass* gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property	= gnome_scan_option_widget_set_property;
  gobject_class->get_property	= gnome_scan_option_widget_get_property;
  gobject_class->dispose 	= gnome_scan_option_widget_dispose;
  /*   gobject_class->finalize 	= gnome_scan_option_widget_finalize; */

  g_type_class_add_private (gobject_class,
			    sizeof (GnomeScanOptionWidgetPrivate));

  /* Properties */
  g_object_class_install_property (gobject_class,
				   PROP_CONTEXT,
				   g_param_spec_object ("context",
							"Context",
							"The GnomeScanContext the widget is connected to.",
							GNOME_TYPE_SCAN_CONTEXT,
							G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
				   PROP_SHOW_LABEL,
				   g_param_spec_boolean ("show-label",
							 "Show label",
							 "Determine whether a label is displayed.",
							 TRUE,
							 G_PARAM_READWRITE));

  /* Signals */
}

void
gnome_scan_option_widget_init (GnomeScanOptionWidget *widget)
{
  GnomeScanOptionWidgetPrivate *priv = GET_PRIVATE (widget);

  widget->context = NULL;
  widget->label = NULL;
  widget->child = NULL;

  priv->dispose_has_run = FALSE;
  priv->show_label = TRUE;
}

void
gnome_scan_option_widget_dispose (GObject *obj)
{
  GnomeScanOptionWidget *widget = GNOME_SCAN_OPTION_WIDGET (obj);
  GnomeScanOptionWidgetPrivate *priv = GET_PRIVATE (widget);
  GnomeScanOptionWidgetClass *b_klass = GNOME_SCAN_OPTION_WIDGET_GET_CLASS (obj);

  g_object_unref (widget->child);
  g_object_unref (widget->label);

  /* That would be nice if g_return_if_fail were noiseless. */
  if (priv->dispose_has_run == TRUE) {
    return;
  }

  /* unref devices */
  g_object_unref (widget->context);
  priv->dispose_has_run = TRUE;

  /* chain */
  /*   GNOME_SCAN_OPTION_WIDGET_PARENT_CLASS (b_klass)->dispose (obj); */
}

void
gnome_scan_option_widget_finalize (GObject *obj)
{
  GnomeScanOptionWidget *widget = GNOME_SCAN_OPTION_WIDGET (obj);
  GnomeScanOptionWidgetClass *b_klass = GNOME_SCAN_OPTION_WIDGET_GET_CLASS (obj);

  /*   GNOME_SCAN_OPTION_WIDGET_PARENT_CLASS (b_klass)->finalize (obj); */
}

void
gnome_scan_option_widget_set_property (GObject *obj,
				       guint property_id,
				       const GValue *value,
				       GParamSpec *pspec)
{
  GnomeScanOptionWidget *widget = GNOME_SCAN_OPTION_WIDGET (obj);
  GnomeScanOptionWidgetPrivate *priv = GET_PRIVATE (widget);

  switch (property_id) {
  case PROP_CONTEXT:
    widget->context = GNOME_SCAN_CONTEXT (g_value_dup_object (value));
    break;
  case PROP_SHOW_LABEL:
    priv->show_label = g_value_get_boolean (value);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}

void
gnome_scan_option_widget_get_property (GObject *obj,
				       guint property_id,
				       GValue *value,
				       GParamSpec *pspec)
{
  GnomeScanOptionWidget *widget = GNOME_SCAN_OPTION_WIDGET (obj);
  GnomeScanOptionWidgetPrivate *priv = GET_PRIVATE (widget);

  switch (property_id) {
  case PROP_CONTEXT:
    g_value_set_object (value, widget->context);
    break;
  case PROP_SHOW_LABEL:
    g_value_set_boolean (value, GTK_WIDGET_VISIBLE (widget->label));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID(obj,
				      property_id,
				      pspec);
    break;
  }
}




/********************************
 * 	      METHODS		*
 ********************************/

/**
 * gnome_scan_option_widget_get_context:
 * @widget: a #GnomeScanOptionWidget
 * 
 * Return the #GnomeScanContext the widget is connected to.
 * 
 * Return value: a #GnomeScanContext
 */
GnomeScanContext*
gnome_scan_option_widget_get_context		(GnomeScanOptionWidget *widget)
{
  return widget->context;
}

/**
 * gnome_scan_option_widget_show_label:
 * @widget: a #GnomeScanOptionWidget
 * 
 * Flag the label to be displayed.
 */
void
gnome_scan_option_widget_show_label		(GnomeScanOptionWidget *widget)
{
  create_label (widget);
  gtk_widget_show (widget->label);
  GET_PRIVATE (widget)->show_label = TRUE;
}

/**
 * gnome_scan_option_widget_hide_label:
 * @widget: a #GnomeScanOptionWidget
 * 
 * Reverse the effect of gnome_scan_option_widget_hide_label().
 */
void
gnome_scan_option_widget_hide_label		(GnomeScanOptionWidget *widget)
{
  create_label (widget);
  gtk_widget_hide (widget->label);
  gtk_widget_set_no_show_all (widget->label, TRUE);
  GET_PRIVATE (widget)->show_label = FALSE;
}

/**
 * gnome_scan_option_widget_set_label:
 * @widget: a #GnomeScanOptionWidget
 * @text: a text
 * 
 * Define a label for the option widget. The label must not contain
 * any pango markup.
 */
void
gnome_scan_option_widget_set_label		(GnomeScanOptionWidget *widget,
						 const gchar *text)
{
  gchar *markup;

  create_label (widget);

  /* translators, %s is the label of the option widget (e.g. resolution). */
  markup = g_markup_printf_escaped (_("%s:"), text);
  gtk_label_set_markup (GTK_LABEL (widget->label),
			markup);
  g_free (markup);
}

/**
 * gnome_scan_option_widget_pack_child:
 * @widget: a #GnomeScanOptionWidget
 * @child: a #GtkWidget
 * @expand: wether the child widget should expand
 * @fill: wether the child widget should fill the allocated space
 * 
 * Pack @child aside from the label.
 */
void
gnome_scan_option_widget_pack_child		(GnomeScanOptionWidget *widget,
						 GtkWidget *child,
						 gboolean expand,
						 gboolean fill)
{
  g_return_if_fail (GNOME_IS_SCAN_OPTION_WIDGET (widget));
  g_return_if_fail (widget->child == NULL);
  g_return_if_fail (GTK_IS_WIDGET (child));

  widget->child = child;

  g_object_ref (child);

  create_label (widget);

  gtk_widget_set_no_show_all (child, FALSE);

  gtk_box_pack_start (GTK_BOX (widget),
		    child,
		    expand,
		    fill,
		    0);
}

/* Ugly hack in order to create label when needed because we don't
   have gnome_scan_option_widget_new() */
void
create_label (GnomeScanOptionWidget *widget)
{
  if (!widget->label) {
    widget->label = gtk_label_new (NULL);

    gtk_box_pack_start (GTK_BOX (widget),
			widget->label,
			FALSE,
			TRUE,
			6);

    gtk_widget_set_no_show_all (widget->label,
				!GET_PRIVATE (widget)->show_label);
  }
}
