/**
 ** config.c - configuration data storage/retrieval
 **
 ** Copyright (C) 2000 Matthew Pratt <mattpratt@yahoo.com>
 **
 ** This software is licensed under the terms of the GNU General 
 ** Public License (GPL). Please see the file LICENSE for details.
 **
 **/

#include "gnomp3.h"
#include "config.h"
#include "utility.h"
#include "playlist.h"

/*
 * The config storage system has been designed to minimise the effort needed to add
 * new varibles. It is also tied in tightly with the preferences system.
 *
 * To add a new variable:
 *    Add the new widget to the gnomp3 interface (main window or prefernces box) in 
 * glade, and name the widget. Use the same name to create a storage variable for
 * its value in the global gnomp3 structure (defined in gnomp3.h).
 * Add a line to the below "config_params" structure with the type being approprate
 * according to the widget/storage type. Start using gnomp3.[var name] in the code
 * elsewhere.
 */

char *D(char *str){return str;}

struct _config config_params[] = {
    CONFIG_VAR(song_search,      C_CHECK, D, "1"),
    CONFIG_VAR(use_id3,          C_CHECK, D, "0"),
    CONFIG_VAR(use_xmms,         C_CHECK, D, "0"),
    CONFIG_VAR(tight_xmms,       C_CHECK, D, "0"),
    CONFIG_VAR(dynamic_list_size,C_SPIN, D, "20"),

    CONFIG_VAR(song_search_path,   C_PATH, D, "/"),
    CONFIG_VAR(playlist_loaded,    C_PATH, D, ""),
    CONFIG_VAR(song_list_file,     C_PATH, D, "/share/mp3_list"), 
    CONFIG_VAR(add_song_dir,       C_PATH, file_in_home, ""),
    CONFIG_VAR(playlist_dir,       C_PATH, file_in_home, ".gnomp3"),

    /* colours */
    CONFIG_VAR( playing_fg_color,     C_COLOR, D, "#b6b5b8"),
    CONFIG_VAR( playing_bg_color,     C_COLOR, D, "#5782b8"),
    CONFIG_VAR( played_fg_color,      C_COLOR, D, "#000062"),
    CONFIG_VAR( played_bg_color,      C_COLOR, D, "#3c6f8f"),
    CONFIG_VAR( in_playlist_fg_color, C_COLOR, D, "#b8b7e2"),
    CONFIG_VAR( in_playlist_bg_color, C_COLOR, D, "#374d76"),

    CONFIG_VAR( add_button,    C_BUTTON, D, "0"),
    CONFIG_VAR( random_button, C_BUTTON, D, "0"),
    CONFIG_VAR( repeat_button, C_BUTTON, D, "0"),
    CONFIG_VAR( sort_button,   C_BUTTON, D, "0"),

    {NULL, NULL, 0, NULL, NULL}
};

/*
 * Traverse the config_params structure and load the stored values if found (using
 * gnome_config_get_* ) or else load the default value.
 */
void config_load()
{
    int i;
    char buff[1024], *text;
    struct _config *cp = config_params;

    gnome_config_push_prefix("/gnomp3/settings/");

    for(i = 0; cp[i].value_ptr; i++){
	text = cp[i].get_default( cp[i].default_val );
	g_snprintf( buff, sizeof(buff), "%s=%s", cp[i].name, text );

	switch(cp[i].type)
	    {
	    case C_SPIN:
	    case C_CHECK:
		DEREF(int, cp[i]) = gnome_config_get_int(buff);
		break;
	    case C_PATH:
		text = gnome_config_get_string(buff);
		strncpy( (char *)cp[i].value_ptr, text, MAX_PATH);
		g_free(text);
		break;
	    case C_COLOR:
		text = gnome_config_get_string(buff);
		gdk_color_parse ( text, (GdkColor *)cp[i].value_ptr);
		g_free(text);
		break;
	    case C_BUTTON:
		gtk_toggle_button_set_active( DEREF(GtkToggleButton*, cp[i]),
					      gnome_config_get_int(cp[i].name));
		break;
	    }
    }

    /* widget sizes */
    gtk_paned_set_position (GTK_PANED(gnomp3.hpaned), 
			    gnome_config_get_int("play_vbox_width=-1"));

    playlist_scan();
    config_handle_upgrade();    

    gnome_config_pop_prefix();
}

/*
 * Traverse the config_params structure and save each value.
 */
void config_save()
{
    int i;
    struct _config *cp = config_params;
    char buffer[60];
    GdkColor color;

    gnome_config_push_prefix("/gnomp3/settings/");

    for(i = 0; cp[i].value_ptr; i++){
	switch(cp[i].type)
	    {
	    case C_SPIN:
	    case C_CHECK:
		gnome_config_set_int(cp[i].name, *((int *)cp[i].value_ptr));
		break;
	    case C_PATH:
		gnome_config_set_string(cp[i].name, (char *)cp[i].value_ptr);
		break;
	    case C_COLOR:
		color = DEREF( GdkColor, cp[i] );
		g_snprintf (buffer, sizeof(buffer), "#%02x%02x%02x",
			    color.red >> 8,
			    color.green >> 8,
			    color.blue >> 8);
		
		gnome_config_set_string(cp[i].name, buffer );
		break;
	    case C_BUTTON:
		gnome_config_set_int(cp[i].name, 
				     (DEREF(GtkToggleButton *, cp[i]))->active );
		break;
	    }
    }
    gnome_config_pop_prefix();
    gnome_config_sync();
}



/*
 * Check the version number stored in the settings. If it does not match the
 * compiled in version number then we know an upgrade has taken place. 
 */
void config_handle_upgrade()
{
    char *text;
    text = gnome_config_get_string("version=?");
    if( strcmp( text, VERSION) ){
	printf("Detected gnomp3 upgrade... removing old song cache\n");
	unlink( file_in_home(".gnomp3/.song_cache"));
	unlink( file_in_home(".gnomp3/.dir_list"));
    }
    g_free(text);
}

