/**
 ** gnomp3.c - main for gnomp3, mp3 player and playlist manager
 **
 ** Copyright (C) 2000 Matthew Pratt <mattpratt@yahoo.com>
 **
 ** This software is licensed under the terms of the GNU General 
 ** Public License (GPL). Please see the file LICENSE for details.
 **/ 

#include "gnomp3.h"
#include "menus.h"
#include "mp3_control.h"
#include "mpg123_play.h"
#include "config.h"
#include "mp3info.h"
#include "playlist.h"
#include "dnd.h"
#include "utility.h"
#include "mp3list.h"
#include "pixmaps/pause.xpm"
#include "xmms_play.h"

#define STORE_XML_WIDGET(w) gnomp3.w = glade_xml_get_widget(xml, #w); g_assert(gnomp3.w != NULL)

/*****************************************************************************/
/* global data */
gnomp3_t gnomp3;
/*****************************************************************************/

/* 
 * this callback is executed when a button press over a list occurs. widget 
 * holds a pointer to the list where the click originated from. 
 * DOUBLE CLICK -> song is added to playlist or song is played (playlist) 
 * RIGHT CLICK  -> popup menu is displayed
 */
void popup_menu(GtkWidget *widget, GdkEventButton *event, gpointer data) 
{
    GtkWidget *temp_menu;
    GtkCListRow *clist_row;
    int row;
    int selection;
    MP3 *mp3;
    
    /* DOUBLE CLICK */
    if (event->type == GDK_2BUTTON_PRESS && event->button == 1 ){
	selection = gtk_clist_get_selection_info ( GTK_CLIST(widget), event->x, event->y,
						   &row, NULL);
	if( !selection )return;
	if( widget == gnomp3.song_ctree || widget == gnomp3.search_clist 
	    || widget == gnomp3.all_clist || widget == gnomp3.dir_clist
	    || widget == gnomp3.time_clist ){
	    clist_row = g_list_nth( GTK_CLIST(widget)->row_list, row )->data;
	    mp3 = clist_row->data;
	    playlist_add_row( mp3);

	}else if (widget == gnomp3.play_clist ){
	    mp3_control_play_nth(row);
	}
	
    }
    
    /* RIGHT CLICK */
    if (event->button == 3) {
	temp_menu = gnome_popup_menu_new(data);
	gnome_popup_menu_do_popup_modal(temp_menu, NULL, NULL, event, widget);
	gtk_widget_destroy(temp_menu);
    }
}

/* 
 * callback that is executed when the 'add song' button on the toolbar is 
 * activated 
 */
void add_song_toggled(GtkWidget *w)
{
#define SMALL_WIDTH 370
#define SONG_LIST_WIDTH 400

  GtkAllocation *a;

  if (GTK_TOGGLE_BUTTON (gnomp3.add_button)->active){
    a = &GTK_WIDGET(gnomp3.gnomp3)->allocation;
    gdk_window_resize(gnomp3.gnomp3->window, SMALL_WIDTH + SONG_LIST_WIDTH, a->height );
    gtk_paned_set_position (GTK_PANED(gnomp3.hpaned), SMALL_WIDTH);
  }else{
    a = &GTK_WIDGET(gnomp3.gnomp3)->allocation;
    gdk_window_resize(gnomp3.gnomp3->window, SMALL_WIDTH, a->height );
    gtk_paned_set_position (GTK_PANED(gnomp3.hpaned), SMALL_WIDTH);
  }
}

/* callback to quit the program. Settings are saved and sub process killed  */
void gnomp3_exit(GtkWidget *w, gpointer data)
{
    if( gnomp3.playlist_loaded[0])
	playlist_save_cb(NULL, NULL);
    
    mp3list_save_to_file( file_in_home(".gnomp3/.song_cache"));
    config_save();

    if(!gnomp3.use_xmms || !gnomp3.tight_xmms)
	mp3_control_stop();
    
    gtk_main_quit();
}

void gnomp3_minimal()
{
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (gnomp3.add_button), FALSE);
    gtk_toolbar_set_style( GTK_TOOLBAR(gnomp3.gnomp3_toolbar),GTK_TOOLBAR_ICONS);
    gtk_toolbar_set_style( GTK_TOOLBAR(gnomp3.mp3_toolbar),GTK_TOOLBAR_ICONS);
     while(gtk_events_pending())
	 gtk_main_iteration();
    add_song_toggled(NULL);
}

void gnomp3_full()
{
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (gnomp3.add_button), TRUE);
    gtk_toolbar_set_style( GTK_TOOLBAR(gnomp3.gnomp3_toolbar),GTK_TOOLBAR_BOTH);
    gtk_toolbar_set_style( GTK_TOOLBAR(gnomp3.mp3_toolbar),GTK_TOOLBAR_BOTH);
}

/* 
 * callback to display the about dialog box 
 */
void create_about_box()
{
    GladeXML *xml;
    
    xml = glade_xml_new( gnomp3.glade_file, "about_box");
    if(!xml) {
	g_warning("We could not load the main_window interface!");
	return;
    }
    glade_xml_signal_autoconnect(xml);  
    gtk_object_unref(GTK_OBJECT(xml));
}

/* 
 * load the main gtk+ UI from the glade file and display. Pointers to important 
 * widgets are stored before the XML data is destroyed (since no longer needed) 
 * Signals for the pop menus are connected 
*/
void create_main_window()
{
    GladeXML *xml;
    
    /* load the main_window decsription */
    xml = glade_xml_new( gnomp3.glade_file, "gnomp3");
    if(!xml){
	xml = glade_xml_new( "gnomp3.glade", "gnomp3");
    }
    /* in case we can't load the interface, bail */
    if(!xml) {
	g_warning("We could not load the main_window interface!");
	return;
    }
    
    STORE_XML_WIDGET(gnomp3);
    STORE_XML_WIDGET(song_ctree);
    STORE_XML_WIDGET(play_clist);
    STORE_XML_WIDGET(sort_button);
    STORE_XML_WIDGET(playlist_combo);
    STORE_XML_WIDGET(hpaned);
    STORE_XML_WIDGET(mp3_hscale);
    STORE_XML_WIDGET(mp3_control_label);
    STORE_XML_WIDGET(random_button);
    STORE_XML_WIDGET(gnomp3_toolbar);
    STORE_XML_WIDGET(mp3_toolbar);
    STORE_XML_WIDGET(search_clist);
    STORE_XML_WIDGET(repeat_button);
    STORE_XML_WIDGET(play_vbox);
    STORE_XML_WIDGET(all_clist);
    STORE_XML_WIDGET(mp3_control_frame);
    STORE_XML_WIDGET(add_button);
    STORE_XML_WIDGET(time_clist);
    STORE_XML_WIDGET(dir_clist);
    STORE_XML_WIDGET(days_spin);
    STORE_XML_WIDGET(mp3_control_song_name);
    
     /* pause button and icon */
    gtk_toolbar_insert_item (GTK_TOOLBAR(glade_xml_get_widget(xml, "mp3_toolbar")), 
			     "Pause", NULL, NULL, 
			     gnome_pixmap_new_from_xpm_d(pause_xpm),
			     mp3_control_pause_cb, NULL, 2 );

    
    add_song_toggled( glade_xml_get_widget(xml, "togglebutton1" ) );
    
    /* autoconnect any signals */
    glade_xml_signal_autoconnect(xml);
    
    /* we don't need the GladeXML object any more, so unref it
       to save some memory */
    gtk_object_unref(GTK_OBJECT(xml));
    
    gtk_signal_connect( GTK_OBJECT(gnomp3.song_ctree), "button_press_event",
			GTK_SIGNAL_FUNC(popup_menu), 
			song_popup_menu);
    gtk_signal_connect( GTK_OBJECT(gnomp3.play_clist), "button_press_event",
			GTK_SIGNAL_FUNC(popup_menu), 
			playlist_popup_menu);    
    gtk_signal_connect( GTK_OBJECT(gnomp3.search_clist), "button_press_event",
			GTK_SIGNAL_FUNC(popup_menu), 
			search_popup_menu);    
    gtk_signal_connect( GTK_OBJECT(gnomp3.all_clist), "button_press_event",
			GTK_SIGNAL_FUNC(popup_menu), 
			all_list_popup_menu);
    gtk_signal_connect( GTK_OBJECT(gnomp3.dir_clist), "button_press_event",
			GTK_SIGNAL_FUNC(popup_menu), 
			dir_list_popup_menu);
    gtk_signal_connect( GTK_OBJECT(gnomp3.time_clist), "button_press_event",
			GTK_SIGNAL_FUNC(popup_menu), 
			time_list_popup_menu);
    
     /* setup the song_ctree to sort its entries */
    gtk_clist_set_sort_column(GTK_CLIST(gnomp3.song_ctree), 0);
    gtk_clist_set_auto_sort(GTK_CLIST(gnomp3.song_ctree), 1);
    
    gtk_clist_set_column_resizeable ( GTK_CLIST(gnomp3.song_ctree), 1, FALSE);
    gtk_clist_set_column_max_width  ( GTK_CLIST(gnomp3.song_ctree), 1, 40 );

    /* setup the playlist combo */
    gtk_signal_connect( GTK_OBJECT(GTK_COMBO(gnomp3.playlist_combo)->popwin),
			 "hide", GTK_SIGNAL_FUNC(playlist_combo_activated),
			 NULL );
			 
    /* drag and drop init */
    gnomp3_dnd_setup();
    
    /* add a pretty icon for the taskbar */
    gnome_window_icon_set_from_file(gnomp3.gnomp3, GNOMP3_TASK_ICON);
    
    return;
}
/*****************************************************************************/
int main(int argc, char *argv[])
{
    GtkWidget *error_dialog;
    
    /* initialize gnome */
    gnome_init("Gnomp3", VERSION, argc, argv);

    /* initialize glade for gnome */
    glade_gnome_init();
    
    check_and_create_dir(getenv("HOME"), ".gnomp3" );
    
    gtk_rc_parse ("./gtkrc");

    /* now assume we know here the glade file is */
    strncpy( gnomp3.glade_file, DEFAULT_GLADE_FILE, MAX_PATH );
    if( !file_exists (gnomp3.glade_file) ){
	error_dialog = gnome_error_dialog("Could not open gnomp3 glade file");
	gtk_signal_connect( GTK_OBJECT(error_dialog), "destroy", 
			    gtk_main_quit, NULL);
	gtk_main();
	exit(1);
    }

    /* sees the random number generator */
    srand(time(NULL));

    create_main_window(); /* see above procedure */
    pixmaps_load();
    config_load();
    mp3list_initial_load();

    mp3info_start();

    mpg123_play_init( );   /* see mpg123_play.c  */
    xmms_play_init();

    playlist_load(gnomp3.playlist_dir, gnomp3.playlist_loaded );

    srand(time(NULL)); /* init random number generator */

    /* run the main loop */
    gtk_main();
    
    return 0;

}
