/**
 ** utility.c - support/utility functions for gnomp3 
 **
 ** Copyright (C) 2000 Matthew Pratt <mattpratt@yahoo.com>
 **
 ** This software is licensed under the terms of the GNU General 
 ** Public License (GPL). Please see the file LICENSE for details.
 **/ 

#include "gnomp3.h"
#include <sys/stat.h>

/* breaks the given path into the filename and the dir directly above that */
void path_breakdown(char *path, char **dir, char **file)
{
    char *ptr;
    
    g_return_if_fail( path != NULL );
    g_return_if_fail( dir  != NULL );
    g_return_if_fail( file != NULL );
    
    ptr = path;
    *dir = *file = NULL;
    
    while( *ptr )ptr++; // find end of string
    
    /* get rid of .mp3 at end if exists */
    if( !strcasecmp( ptr - 4, ".mp3" )){
	ptr -= 4;
	*ptr = 0;
    }
    
    /* step back wards from the end of the string looking for the '/' */
    while( ptr != path ){
	if ( *ptr == '/' ){
	    /* file found */
	    if( *file == NULL ){
		*file = ptr + 1;
		*ptr = 0;
	    }else if ( *dir == NULL )
		*dir = ptr + 1;
	    else
		return;
	}
	ptr--;
    }   
}

void strip_underscores(char *string)
{
    while( *string ){
	if( *string == '_' )
	    *string = ' ';
	string++;
    }
}

/* creates the directory if it does not already exist */
void check_and_create_dir(char *prefix, char *dirname)
{
    struct stat st;
    char *path;
    
    if( prefix )
	path = g_strconcat( prefix, "/", dirname, NULL);
    else
	path = g_strdup (dirname);
    
    if (stat(path, &st) < 0)
	if (mkdir(path, 0755) < 0)
	    fprintf(stderr, "Could not create '%s' directory...functionality may be limited\n", path);
    
    g_free(path);
}

int is_directory(char *path)
{
    struct stat st;
    if (stat(path, &st) != -1){
	if( S_ISDIR(st.st_mode))
	    return TRUE;
    }
    return FALSE;
}

int file_exists(char *path)
{
    struct stat st;
    if (stat(path, &st) != -1){
	return TRUE;
    }
    return FALSE;
}

/*
 * Returns the full path of a file name that is assumed to be in the home
 * direcory of the caller, in a static buffer.
 */ 
char *file_in_home(char *filename)
{
    static char path[MAX_PATH];

    g_snprintf( path, sizeof(path), "%s/%s", getenv("HOME"), filename );
    return path;
}

GList *g_list_move_up(GList *list, GList *node)
{
    int pos;
    GList *nl;

    pos = g_list_position( list, node );
    if( pos == 0 )
	return list;

    nl = g_list_remove_link( list, node );
    nl = g_list_insert( nl, node->data, pos - 1 );
    //g_free(node);
    
    return nl;
}

GList *g_list_move_down(GList *list, GList *node)
{
    int pos;
    GList *nl;

    pos = g_list_position( list, node );
    nl = g_list_remove_link( list, node );
    nl = g_list_insert( nl, node->data, pos + 1 );
    //g_free(node);
    
    return nl;
}

char *itoa(int i)
{
    static char a[128];

    g_snprintf( a, sizeof(a), "%d", i );
    return a;
}

