/**
 ** xmms_play.c - routines for playing songs with xmms
 **
 ** Copyright (C) 2000 Matthew Pratt <mattpratt@yahoo.com>
 **
 ** This software is licensed under the terms of the GNU General 
 ** Public License (GPL). Please see the file LICENSE for details.
 **/ 

#include "gnomp3.h"
#include "mp3_control.h"
#include "playlist.h"
#include "xmms_play.h"

/* XMMS handle */
int xmms_session;
int xmms_init_done = FALSE;
int xmms_song_time = 1;
int xmms_timeout;
/*
 * Read and display the current status of the playling song.
 * Detect when the song changes and update the playlist to reflect.
 */
int xmms_play_poll_xmms(gpointer data)
{
    char text[128];
    int current;
    int pos, nch;
    static int oldpos = -1;

    if(!gnomp3.use_xmms)return FALSE;

    pos = xmms_remote_get_playlist_pos(xmms_session);
    if( oldpos < 0)oldpos = pos;
    if( pos != oldpos ){
	GList *ptr;
	
	ptr = g_list_nth(playlist, oldpos);
	if(ptr)
	    MP3_NODE(ptr)->playing = FALSE;
	ptr = g_list_nth(playlist, pos);
	if(ptr){
	    MP3_NODE(ptr)->playing = TRUE;
	    MP3_NODE(ptr)->played = TRUE;
	    MP3_NODE(ptr)->no_plays++;
	}
	playlist_update_display();
	oldpos = pos;
    }

    xmms_remote_get_info(xmms_session, &mp3_control_state.bitrate, 
			 &mp3_control_state.freq, &nch);
    mp3_control_state.bitrate =  mp3_control_state.bitrate / 1000;

    current = xmms_remote_get_output_time(xmms_session);
    mp3_control_state.progress = 100 *(double)current / (double)xmms_song_time;

    current = current / 1000;
    g_snprintf(text, sizeof(text), "%d:%02d", current/60, current%60);
    mp3_control_state.time = text;

    mp3_control_update_display();

    return TRUE;
}

void xmms_play_init()
{
    if(!xmms_remote_is_running(xmms_session) && gnomp3.use_xmms){
	system("xmms&");
	xmms_remote_playlist_clear(xmms_session);
    }

    if(gnomp3.use_xmms && !xmms_init_done){
	xmms_init_done = TRUE;
	xmms_play_list_load();
	xmms_play_list_sync();
	xmms_timeout = gtk_timeout_add(100, xmms_play_poll_xmms, NULL);
	xmms_play_set_random();	
	xmms_play_set_repeat();
    }
}

void xmms_clean_up()
{
    gtk_timeout_remove(xmms_timeout);
    xmms_init_done = FALSE;
}

void xmms_play_song(MP3 *mp3, double pos)
{
    xmms_play_init();
    xmms_song_time = 
	xmms_remote_get_playlist_time(xmms_session, mp3->row_playlist);
    xmms_remote_set_playlist_pos(xmms_session, mp3->row_playlist);
    xmms_remote_play(xmms_session);
    xmms_remote_jump_to_time(xmms_session, (int)(pos / 100 * xmms_song_time));
}

void xmms_play_stop()
{
    xmms_remote_stop(xmms_session);
}

void xmms_play_pause()
{
    xmms_remote_pause(xmms_session);
}
/*** xmms button sync ********************************************************/
void xmms_play_set_random()
{
    if(GTK_TOGGLE_BUTTON (gnomp3.random_button)->active 
       != xmms_remote_is_shuffle(xmms_session))
	xmms_remote_toggle_shuffle(xmms_session);    
}
void xmms_play_set_repeat()
{
    if(GTK_TOGGLE_BUTTON (gnomp3.repeat_button)->active 
       != xmms_remote_is_repeat(xmms_session))
	xmms_remote_toggle_repeat(xmms_session);
}
/*** xmms playlist manipulation **********************************************/
void xmms_play_add_song(MP3 *mp3)
{
    GList *l = NULL;

    if(!gnomp3.use_xmms)
	return;
    
    l = g_list_append(l, mp3->filename);
    xmms_remote_playlist_add(xmms_session, l);
}

void xmms_play_del_song(MP3 *mp3)
{
    if(!gnomp3.use_xmms)
	return;
    xmms_remote_playlist_delete(xmms_session, mp3->row_playlist);
}

void xmms_play_list_sync()
{
    GList *l = NULL;
    GList *ptr = playlist;

    for(ptr = playlist; ptr; ptr = ptr->next)
	l = g_list_append(l, MP3_NODE(ptr)->filename);

    xmms_remote_playlist_clear(xmms_session);
    xmms_remote_playlist_add(xmms_session, l);
}

/* xmms_play_list_load(): retrieves xmms' current playlist, if any,
 *   and adds it to the active playlist
 */
void xmms_play_list_load()
{
    int i,count;
    MP3 *mp3;
    char *fn;
    
    count = xmms_remote_get_playlist_length(xmms_session);
    if (count <= 0)
	return;

    for(i = 0; i < count; i++) {
	fn = xmms_remote_get_playlist_file(xmms_session,i);
	if(!fn)continue;
	mp3 = mp3list_build_entry(fn);
	playlist_add_song(mp3);
    }
}
