/******************************** -*- C -*- ****************************
 *
 *	Byte code array utility routines.
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988,89,90,91,92,94,95,99,2000,2001,2002,2003
 * Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 *
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.	 If not, write to the Free Software
 * Foundation, 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 ***********************************************************************/

#include "gstpriv.h"

#define BYTECODE_CHUNK_SIZE  64

/* The table of bytecode sizes. */
const int _gst_bytecode_size_table[256] = {
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 0 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 16 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 32 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 48 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 64 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 80 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 96 */
  1, 1, 1, 1, 2, 2, 1, 1, 1, 1, 1, 1, 1, 1, 3, 3,	/* 112 */
  2, 2, 2, 2, 3, 2, 3, 1, 1, 1, 3, 1, 1, 1, 2, 1,	/* 128 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 144 */
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,	/* 160 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 176 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 192 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 208 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 224 */
  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1	/* 240 */
};

/* Where the compiled bytecodes go. */
bytecodes _gst_cur_bytecodes;

/* Reallocate an array of bytecodes, leaving space for DELTA more
   bytes. */
static void realloc_bytecodes (bytecodes bytecodes,
			       int delta);


/* These are the selectors for special case send bytecodes that the compiler
 * generates.  These are used only to print out the generated byte codes. */
static const char *special_message_name[] = {
  "+",
  "-",
  "<",
  ">",
  "<=",
  ">=",
  "=",
  "~=",
  "*",
  "/",
  "\\",
  "@",
  "bitShift:",
  "//",
  "bitAnd:",
  "bitOr:",
  "at:",
  "at:put:",
  "size",
  "next",
  "nextPut:",
  "atEnd",
  "==",
  "class",
  "blockCopy:",
  "value",
  "value:",
  "do:",
  "new",
  "new:",
  "isNil",
  "notNil"
};



bytecodes
_gst_extract_bytecodes (OOP byteArrayOOP)
{
  bytecodes result;
  int len;
  gst_byte_array byteArray;

  byteArray = (gst_byte_array) OOP_TO_OBJ (byteArrayOOP);
  len = oop_num_fields (byteArrayOOP);
  result = (bytecodes) xmalloc (sizeof (struct bytecode_array));

  result->base = (gst_uchar *) xmalloc (len);
  result->ptr = result->base + len;
  result->maxLen = len;
  memcpy (result->base, byteArray->bytes, len);
  return (result);
}

void
_gst_line_number (int n, mst_Boolean force)
{
  static int line;
  if (n > 65535) n = 65535;

  if (n == line && !force)
    return;

  line = n;
  if (n < 0)
    return;

  _gst_compile_byte (LINE_NUMBER_BYTECODE);
  _gst_compile_byte (n >> 8);
  _gst_compile_byte (n & 255);
}

void
_gst_compile_byte (gst_uchar byte)
{
  assert (_gst_cur_bytecodes);

  if ((_gst_cur_bytecodes->ptr - _gst_cur_bytecodes->base) >=
	   _gst_cur_bytecodes->maxLen)
    realloc_bytecodes (_gst_cur_bytecodes, BYTECODE_CHUNK_SIZE);

  *_gst_cur_bytecodes->ptr++ = byte;
}

void
_gst_free_bytecodes (bytecodes bytecodes)
{
  if (bytecodes != NULL)
    {
      xfree (bytecodes->base);
      xfree (bytecodes);
    }
}

void
_gst_compile_and_free_bytecodes (bytecodes bytecodes)
{
  _gst_compile_bytecodes (bytecodes->base, bytecodes->ptr);

  /* First add the worst case, then leave the net effect.  */
  ADD_STACK_DEPTH (bytecodes->max_stack_depth);
  SUB_STACK_DEPTH (bytecodes->max_stack_depth - bytecodes->stack_depth);

  _gst_free_bytecodes (bytecodes);
}

bytecodes
_gst_get_bytecodes (void)
{
  bytecodes curBytecodes;

  curBytecodes = _gst_cur_bytecodes;
  _gst_cur_bytecodes = NULL;

  return (curBytecodes);
}


bytecodes
_gst_save_bytecode_array ()
{
  bytecodes curBytecodes;

  curBytecodes = _gst_cur_bytecodes;
  _gst_alloc_bytecodes ();

  return (curBytecodes);
}


void
_gst_restore_bytecode_array (bytecodes bytecodes)
{
  _gst_cur_bytecodes = bytecodes;
}

int
_gst_bytecode_length (bytecodes bytecodes)
{
  if (bytecodes == NULL)
    return (0);

  return (bytecodes->ptr - bytecodes->base);
}


int
_gst_current_bytecode_length (void)
{
  if (_gst_cur_bytecodes == NULL)
    return (0);

  return (_gst_cur_bytecodes->ptr - _gst_cur_bytecodes->base);
}

void
_gst_copy_bytecodes (gst_uchar * dest,
		      bytecodes bytecodes)
{
  memcpy (dest, bytecodes->base, _gst_bytecode_length (bytecodes));
}

void
_gst_truncate_bytecodes (gst_uchar * here,
			  bytecodes bytecodes)
{
  bytecodes->ptr = here;
}



void
_gst_print_bytecodes (bytecodes bytecodes,
		       OOP * literal_vec)
{
  gst_uchar *b;
  int ip;

  if (bytecodes == NULL)
    return;

  for (b = bytecodes->base; b < bytecodes->ptr; b += BYTECODE_SIZE (*b))
    {
      ip = b - bytecodes->base;
      printf ("%5d:\t", ip);
      _gst_print_bytecode_name (b, ip, literal_vec);
      printf ("\n");
    }
  printf ("\n");
}

void
_gst_print_indexed (gst_uchar byte,
		    OOP * literal_vec,
		    mst_Boolean constOk)
{
  switch (byte >> 6)
    {
    case 0:
      printf ("Instance Variable[%d]", byte & 63);
      break;
    case 1:
      printf ("Temporary[%d]", byte & 63);
      break;
    case 2:
      printf ("Literal[%d]", byte & 63);
      if (literal_vec)
	printf (" = %O", literal_vec[byte & 63]);

      if (!constOk)
	printf (" -- INVALID!");

      break;
    case 3:
      printf ("Global Variable[%d]", byte & 63);
      if (literal_vec)
	printf (" = %+O", literal_vec[byte & 63]);
      break;
    }
}

void
_gst_print_bytecode_name (gst_uchar * bp,
			   int ip,
			   OOP * literal_vec)
{
  int lit;

  switch (bp[0])
    {
    case 0: case 1: case 2: case 3:
    case 4: case 5: case 6: case 7:
    case 8: case 9: case 10: case 11:
    case 12: case 13: case 14: case 15:
      printf ("push Instance Variable[%d]", bp[0] & 15);
      break;

    case 16: case 17: case 18: case 19:
    case 20: case 21: case 22: case 23:
    case 24: case 25: case 26: case 27:
    case 28: case 29: case 30: case 31:
      printf ("push Temporary[%d]", bp[0] & 15);
      break;

    case 32: case 33: case 34: case 35:
    case 36: case 37: case 38: case 39:
    case 40: case 41: case 42: case 43:
    case 44: case 45: case 46: case 47:
    case 48: case 49: case 50: case 51:
    case 52: case 53: case 54: case 55:
    case 56: case 57: case 58: case 59:
    case 60: case 61: case 62: case 63:
      printf ("push Literal[%d]", bp[0] & 31);
      if (literal_vec)
	printf (" = %O", literal_vec[bp[0] & 31]);

      break;

    case 64: case 65: case 66: case 67:
    case 68: case 69: case 70: case 71:
    case 72: case 73: case 74: case 75:
    case 76: case 77: case 78: case 79:
    case 80: case 81: case 82: case 83:
    case 84: case 85: case 86: case 87:
    case 88: case 89: case 90: case 91:
    case 92: case 93: case 94: case 95:
      printf ("push Global Variable[%d]", bp[0] & 31);
      if (literal_vec)
	printf (" = %+O", literal_vec[bp[0] & 31]);
      break;

    case 96: case 97: case 98: case 99:
    case 100: case 101: case 102: case 103:
      printf ("pop and store Instance Variable[%d]", bp[0] & 7);
      break;

    case 104: case 105: case 106: case 107:
    case 108: case 109: case 110: case 111:
      printf ("pop and store Temporary[%d]", bp[0] & 7);
      break;

    case 112:
      printf ("push self");
      break;

    case 113:
      printf ("push true");
      break;

    case 114:
      printf ("push false");
      break;

    case 115:
      printf ("push nil");
      break;

    case 116:
      printf ("push signed 8-bit integer %hhd", bp[1]);
      break;

    case 117:
      printf ("push unsigned 8-bit integer %hhu", bp[1]);
      break;

    case 118:
      printf ("push 0");
      break;

    case 119:
      printf ("push 1");
      break;

    case 120:
      printf ("return self");
      break;

    case 121:
      printf ("return true");
      break;

    case 122:
      printf ("return false");
      break;

    case 123:
      printf ("return nil");
      break;

    case 124:
      printf ("return explicitly from method");
      break;

    case 125:
      printf ("return stack top");
      break;

    case 126:
      lit = (bp[1] & 63) * 256 + bp[2];
      switch (bp[1] >> 6)
	{
	case 0:
	  printf ("push Literal[%d]", lit);
	  break;
	case 1:
	  printf ("push Global Variable[%d]", lit);
	  break;
	case 2:
	  printf ("pop and store Global Variable[%d]", lit);
	  break;
	case 3:
	  printf ("store Global Variable[%d]", lit);
	  break;
	}

      if (literal_vec)
	{
	  if (bp[1] >> 6)
	    printf (" = %+O", literal_vec[lit]);
	  else
	    printf (" = %O", literal_vec[lit]);
	}
      break;

    case 127:
      lit = bp[1] * 256 + bp[2];
      printf ("source line %d", lit);
      break;

    case 128:
      printf ("push ");
      _gst_print_indexed (bp[1], literal_vec, true);
      break;

    case 129:
      printf ("store ");
      _gst_print_indexed (bp[1], literal_vec, false);
      break;

    case 130:
      printf ("pop and store ");
      _gst_print_indexed (bp[1], literal_vec, false);
      break;

    case 131:
      printf ("send selector %d, %d args", bp[1] & 31, bp[1] >> 5);
      if (literal_vec)
	printf (" = %O", literal_vec[bp[1] & 31]);

      break;

    case 132:
      lit = bp[2] + ((bp[1] & 192) << 2);
      printf ("send %sselector %d, %d args",
	      bp[1] & EXTENDED_SEND_SUPER_FLAG ? "to super " : "", lit,
	      bp[1] & 31);
      if (literal_vec)
	printf (" = %O", literal_vec[lit]);
      break;

    case 133:
      printf ("send to super selector %d, %d args", bp[1] & 31,
	      bp[1] >> 5);
      if (literal_vec)
	printf (" = %O", literal_vec[bp[1] & 31]);
      break;

    case 134:
      switch (bp[1] >> 6)
	{
	case 0:
	  printf ("pop and store");
	  break;
	case 1:
	  printf ("push");
	  break;
	case 2:
	  printf ("pop and store");
	  break;
	case 3:
	  printf ("store");
	  break;
	}

      printf (" Instance Variable[%d]", (bp[1] & 63) * 256 + bp[2]);
      if ((bp[1] >> 6) == 0)
	printf (" of new stack top");
      break;

    case 135:
      printf ("pop stack top");
      break;

    case 136:
      printf ("duplicate stack top");
      break;

    case 137:
      printf ("push current context");
      break;

    case 138:
      switch (bp[1] >> 6)
	{
	case 0:
	  printf ("(invalid)");
	  break;
	case 1:
	  printf ("push");
	  break;
	case 2:
	  printf ("pop and store");
	  break;
	case 3:
	  printf ("store");
	  break;
	}
      printf (" outer var scopes = %d varIndex = %d", bp[2],
	      bp[1] & 63);
      break;

    case 139:
      printf ("no operation");
      break;

    case 140:
      printf ("set stack top to self");
      break;

    case 141:
      printf ("set stack top to 1");
      break;

    case 142:
      printf ("set stack top to ");
      _gst_print_indexed (bp[1], literal_vec, true);
      break;

    case 143:
      printf ("terminate interpreter");
      break;

    case 144: case 145: case 146: case 147:
    case 148: case 149: case 150: case 151:
      printf ("jump to %d", (bp[0] & 7) + ip + 1 + 1);
      break;

    case 152: case 153: case 154: case 155:
    case 156: case 157: case 158: case 159:
      printf ("jump to %d if false", (bp[0] & 7) + ip + 1 + 1);
      break;

    case 160: case 161: case 162: case 163:
    case 164: case 165: case 166: case 167:
      printf ("jump to %d", ((bp[0] & 7) - 4) * 256 + bp[1] + ip + 2);
      break;

    case 168: case 169: case 170: case 171:
      printf ("pop and jump to %d if true",
	      (bp[0] & 3) * 256 + bp[1] + ip + 2);
      break;

    case 172: case 173: case 174: case 175:
      printf ("pop and jump to %d if false",
	      (bp[0] & 3) * 256 + bp[1] + ip + 2);
      break;

    case 176: case 177: case 178: case 179:
    case 180: case 181: case 182: case 183:
    case 184: case 185: case 186: case 187:
    case 188: case 189: case 190: case 191:
      printf ("send arithmetic message #%s",
	      special_message_name[bp[0] - 176]);
      break;

    case 192: case 193: case 194: case 195:
    case 196: case 197: case 198: case 199:
    case 200: case 201: case 202: case 203:
    case 204: case 205: case 206: case 207:
      printf ("send special message #%s",
	      special_message_name[bp[0] - 176]);
      break;

    case 208: case 209: case 210: case 211:
    case 212: case 213: case 214: case 215:
    case 216: case 217: case 218: case 219:
    case 220: case 221: case 222: case 223:
    case 224: case 225: case 226: case 227:
    case 228: case 229: case 230: case 231:
    case 232: case 233: case 234: case 235:
    case 236: case 237: case 238: case 239:
    case 240: case 241: case 242: case 243:
    case 244: case 245: case 246: case 247:
    case 248: case 249: case 250: case 251:
    case 252: case 253: case 254: case 255:
      printf ("send selector %d, %d args", bp[0] & 15, (bp[0] - 208) >> 4);
      if (literal_vec)
	printf (" = %O", literal_vec[bp[0] & 15]);
      break;

    default:
      printf ("INVALID BYTECODE %d", bp[0]);
      break;
    }
}


void
_gst_compile_bytecodes (gst_uchar * from,
			gst_uchar * to)
{
  int free;
  assert (_gst_cur_bytecodes);

  free = _gst_cur_bytecodes->maxLen -
    (_gst_cur_bytecodes->ptr - _gst_cur_bytecodes->base);

  if (free < (to - from))
    {
      memcpy (_gst_cur_bytecodes->ptr, from, free);
      _gst_cur_bytecodes->ptr += free;
      from += free;
      realloc_bytecodes (_gst_cur_bytecodes,
		         BYTECODE_CHUNK_SIZE + (to - from));
    }

  memcpy (_gst_cur_bytecodes->ptr, from, to - from);
  _gst_cur_bytecodes->ptr += to - from;
}

void
_gst_alloc_bytecodes ()
{
  bytecodes newBytecodes;

  newBytecodes = (bytecodes) xmalloc (sizeof (struct bytecode_array));
  newBytecodes->base = (gst_uchar *) xmalloc (BYTECODE_CHUNK_SIZE);
  newBytecodes->ptr = newBytecodes->base;
  newBytecodes->maxLen = BYTECODE_CHUNK_SIZE;

  newBytecodes->stack_depth = 0;
  newBytecodes->max_stack_depth = 0;

  _gst_cur_bytecodes = newBytecodes;
}

void
realloc_bytecodes (bytecodes bytecodes,
		    int delta)
{
#ifndef OPTIMIZE
  if (bytecodes->maxLen != (bytecodes->ptr - bytecodes->base))
    _gst_errorf
      ("realloc_bytecodes called with maxLen != byteCode len");
#endif

  bytecodes->base =
    (gst_uchar *) xrealloc (bytecodes->base, bytecodes->maxLen + delta);
  bytecodes->ptr = bytecodes->base + bytecodes->maxLen;
  bytecodes->maxLen += delta;
}
