# GNU Enterprise Application Server - Authentication Services
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: geasAuthentication.py 7917 2005-09-19 10:10:46Z johannes $

from gnue.common.apps import errors
from gnue.appserver import data

# =============================================================================
# Exceptions
# =============================================================================

class AuthError (errors.UserError):
  pass

# =============================================================================
# Basic Authentication Class ( abstract )
# =============================================================================

class geasAuthAgent:

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self):
    pass

  # ---------------------------------------------------------------------------
  # authenticate an user
  # user contains the user name
  # auth contains authentication information (like password, fingerprint...)
  # return 1 on access ok  (0=access denied)
  # ---------------------------------------------------------------------------

  def authenticate (self, session, user, auth):
    return True


  # ---------------------------------------------------------------------------
  # check if user x has access for class/table y
  # return 1 on access ok  (0=access denied)
  # ---------------------------------------------------------------------------

  def hasAccess (self, session, user, classname):
    return True

# =============================================================================
# Database Authentication Agent
# =============================================================================

class geasDBAuthAgent (geasAuthAgent):

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self, connectionManager, database):
    # creation and populating the user/password/tablelist list should be
    # moved here, when the management interface is implemented and an easy
    # way to update permissions is working

    self._connections = connectionManager
    self._database    = database

  # ---------------------------------------------------------------------------
  # authenticate an user
  # user contains the user name
  # auth contains authentication information (like password, fingerprint...)
  # return 1 on access ok  (0=access denied)
  # ---------------------------------------------------------------------------

  def authenticate (self, session, user, auth):

    conn = data.connection (self._connections, self._database)
    try:
      fields    = [u'gnue_username', u'gnue_password', u'gnue_accesslist']
      contents  = {None: (u'gnue_useraccess', None, None, fields)}
      resultSet = conn.query (contents, {u'gnue_username': user}, [])

      try:
        hits = resultSet.count ()

        if not hits:
          raise AuthError, u_("User '%s' does not exist.") % user

        elif hits > 1:
          raise AuthError, \
              u_("Internal Error: More than one (%(numrec)s) record " \
                 "for user '%(username)s'.") \
              % {"numrec"  : hits, "username": user}

        rec = resultSet.nextRecord ()

        if rec.getField (u'gnue_password') != auth ['password']:
          raise AuthError, u_("Invalid password for user '%s'") % user

        assert gDebug (1, "User '%s' logged in." % user)

        session.tablelist = rec.getField (u'gnue_accesslist').split (',')

      finally:
        resultSet.close ()

      return 1

    finally:
      conn.close ()

    return True


  # ---------------------------------------------------------------------------
  # check if user x has access for class/table y
  # ---------------------------------------------------------------------------

  def hasAccess (self, session, user, classname):
    # this should be changed to check for the user again.
    # it will only be secure if the protocol supports session-management too.
    tables = session.tablelist
    if 'all' in tables or classname in tables:
      return True
    else:
      assert gDebug (1, "User '%(username)s' has no access to class "
                        "%(classname)s." \
                        % {"username" : user, "classname": classname})
      return False

# =============================================================================
# PAM Authentication Agent
# =============================================================================

class geasPAMAuthAgent (geasAuthAgent):

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self):
    import PAM

  # ---------------------------------------------------------------------------
  # authenticate an user
  # user contains the user name
  # auth contains authentication information (like password, fingerprint...)
  # return 1 on access ok  (0=access denied)
  # ---------------------------------------------------------------------------

  def authenticate (self, session, user, auth):
    auth = PAM.pam ()
    auth.start (service)

    # TODO: add pam challenge invocation + parsing

    if 0:
      return 0 # = no access

    print u_("User '%s' logged in.") % user

    return 1 # = has access

  # ---------------------------------------------------------------------------
  # check if user x has access for class/table y
  # ---------------------------------------------------------------------------

  def hasAccess (self, session, user, classname):
    # this should be changed to check for the user again.
    # it will only be secure if the protocol supports session-management too.
    if hasattr (session, "tablelist") and (classname in session.tablelist):
      return 1
    else:
      return 0
