# GNU Enterprise Appserver - GNUe Language Definition - Import GLDs
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: readgld.py 7915 2005-09-19 09:43:52Z johannes $

import sys
import os

from gnue.common.apps import GClientApp, i18n, errors
from gnue.common.apps.i18n import translate as _        # for epydoc
from gnue.common.utils.FileUtils import openResource
from gnue.common.datasources import GDataSource, GConditions
from gnue.common.utils.uuid import UUID

from gnue.appserver import VERSION
from gnue.appserver import geasConfiguration
from gnue.appserver.gld import GLParser
from gnue.appserver import repository

from gnue.common.apps import GBaseApp

# =============================================================================
# Exceptions
# =============================================================================

class StartupError (errors.UserError):
  pass

class Error (errors.ApplicationError):
  def __init__ (self, message, files = None):
    errors.ApplicationError.__init__ (self, message)
    text = []

    if files:
      text.append (u_("In file(s) '%s':") % ', '.join (files))

    text.append (message)
    self.detail = os.linesep.join (text)

class ClassNotFoundError (Error):
  def __init__ (self, classname, filename = None):
    msg = u_("Class '%s' not found in class repository") % classname
    Error.__init__ (self, msg, [filename])

class DuplicateItemError (Error):
  def __init__ (self, language, itemname, files):
    msg = u_("Duplicate definition of element '%(itemname)s' "
             "for language '%(language)s'") \
          % {'itemname' : itemname, 'language' : language}
    Error.__init__ (self, msg, files)

class PropertyNotFoundError (Error):
  def __init__ (self, classname, pName, filename):
    msg = u_("Class '%(classname)s' has no property '%(property)s'") \
          % {'classname': classname, 'property' : pName}
    Error.__init__ (self, msg, [filename])

class ProcedureNotFoundError (Error):
  def __init__ (self, classname, pName, filename):
    msg = u_("Class '%(classname)s' has no procedure '%(procedure)s'") \
          % {'classname': classname, 'procedure': pName}
    Error.__init__ (self, msg, [filename])


# =============================================================================
# This class reads a list of gld files and updates the class repository
# =============================================================================

class gldClient (GClientApp.GClientApp):

  NAME    = "gnue-readgld"
  VERSION = VERSION
  COMMAND = "gnue-readgld"
  USAGE   = "%s %s" % (GClientApp.GClientApp.USAGE, "file file ...")
  SUMMARY = _("Read GNUe Language Definitions and update the class repository")

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, connections = None):

    self.addCommandOption ('connection', 'c', argument='connectionname',
        help = _("Use the connection <connectionname> for creating the schema"))

    self.addCommandOption ('username', 'u', argument="user",
        help = _("Set the username for the database. If the database is to be "
                 "created, this username will be it's owner."))

    self.addCommandOption ('password', 'p', argument="password",
        help = _("Set the password for the database."))

    ConfigOptions = geasConfiguration.ConfigOptions

    GClientApp.GClientApp.__init__ (self, connections, 'appserver',
        ConfigOptions)

    if self.OPTIONS ['connection']:
      self._connection = self.OPTIONS ['connection']
    else:
      # Compatibility for 'database' setting in gnue.conf
      dbName = gConfig ('database')
      coName = gConfig ('connection')

      if dbName and coName == 'gnue':
        self._connection = dbName
      else:
        self._connection = coName


  # ---------------------------------------------------------------------------
  # Main program
  # ---------------------------------------------------------------------------

  def run (self):
    """
    This is the main function of the whole process. It verifies the given
    options, loads all layout definitions and then logs into the connection to
    perform all actions requested.
    """

    reader = gldReader (self.connections,
                        self._connection,
                        [unicode (a, i18n.encoding) for a in self.ARGUMENTS])

    self._prepareConnection ()

    reader.run ()


  # ---------------------------------------------------------------------------
  # Prepare the connection
  # ---------------------------------------------------------------------------

  def _prepareConnection (self):
    """
    This function makes sure the connection will have the proper username and
    password set.
    """

    connection = self.connections.getConnection (self._connection)

    if not connection.parameters.has_key ('username'):
      connection.parameters ['username'] = 'gnue'
    if not connection.parameters.has_key ('password'):
      connection.parameters ['password'] = 'gnue'

    if self.OPTIONS ['username'] is not None:
      connection.parameters ['username'] = self.OPTIONS ['username']

    if self.OPTIONS ['password'] is not None:
      connection.parameters ['password'] = self.OPTIONS ['password']



# =============================================================================
# This class implements an integrator for GNUe Language Definition files
# =============================================================================

class gldReader:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, connections, database, files, moduleLookup = None,
      classLookup = None, propertyLookup = None, procedureLookup = None):
    """
    Create a new instance of a gld reader

    @param connections: GConnections instance to be used
    @param database: name of the connection to use (in connection.conf)
    @param files: sequence of filenames to integerate
    @param moduleLookup: lookup dictionary for 'gnue_modules'
    @param classLookup: lookup dictionary for 'gnue_class'
    @param propertyLookup: lookup dictionary for 'gnue_property'
    @param procedureLookup: lookup dictionary for 'gnue_procedure'
    """

    self._connections = connections
    self._database    = database
    self._filenames   = files
    self._files       = {}
    self._uuidtype    = gConfig ('uuidtype', section = 'appserver').lower ()

    self.lpModule    = moduleLookup
    self.lpClass     = classLookup
    self.lpProperty  = propertyLookup
    self.lpProcedure = procedureLookup

    self.labels   = {}
    self.messages = {}

    if not len (self._filenames):
      raise StartupError, u_("No input file specified.")

    if not self._database:
      raise StartupError, u_("No connection specified.")

    try:
      for filename in files:
        self._files [filename] = openResource (filename)

    except IOError, err:
      while self._files:
        (name, item) = self._files.popitem ()
        item.close ()

      raise StartupError, u_("Unable to open input file: %s") % \
          errors.getException () [2]


  # ---------------------------------------------------------------------------
  # Process the given GLD-files
  # ---------------------------------------------------------------------------

  def run (self):
    """
    This function loads all GLD files, builds the object dictionaries and
    updates the repository using these dictionaries.
    """

    self.labels.clear ()
    self.messages.clear ()

    print _("Loading GNUe language definitions")

    while self._files:
      self.__currentFile, stream = self._files.popitem ()

      try:
        assert gDebug (1, "Loading gld file '%s'" % self.__currentFile)

        self.__currentModule = None
        self.__currentClass  = None

        # Load xml file and process all it's objects
        schema = GLParser.loadFile (stream)
        try:
          schema.walk (self.__iterateObjects)

        finally:
          self.__currentModule = None
          self.__currentClass  = None

          schema = None

      finally:
        stream.close ()


    # Replace all references to modules, classes, properties and procedures
    self.__replaceReferences ()

    # and finally update the repository
    self.__backend = self._connections.getConnection (self._database, True)

    try:
      self.__updateLabels ()
      self.__updateMessages ()

    finally:
      self.__backend.close ()



  # ---------------------------------------------------------------------------
  # Iterate over all top level elements
  # ---------------------------------------------------------------------------

  def __iterateObjects (self, sObject):
    """
    This function processes all objects of a GLD tree.

    @param sObject: current GLD object to be processed
    """

    if sObject._type == 'GLModule':
      self.__currentModule = sObject

    elif sObject._type == 'GLClass':
      self.__currentClass = sObject

    elif sObject._type in ['GLProperty', 'GLProcedure']:
      self.__translateItem (sObject)

    elif sObject._type == 'GLMessage':
      self.__translateMessage (sObject)


  # ---------------------------------------------------------------------------
  # Process a class item (property or procedure)
  # ---------------------------------------------------------------------------

  def __translateItem (self, item):
    """
    This function adds an item (label) to the labels dictionary. If the labels
    is already listed, a DuplicateItemError will be raised.
    """

    # This key makes sure to have no clashes
    key = "%s::%s.%s" % (self.__currentModule.language,
                         self.__currentClass.fullName, item.fullName)
    key = key.lower ()

    if self.labels.has_key (key):
      raise DuplicateItemError, \
          (self.__currentModule.language,
           "%s.%s" % (self.__currentClass.fullName, item.fullName),
           [self.labels [key]['_file'], self.__currentFile])

    self.labels [key] = { \
        'gnue_id'       : None,
        'gnue_class'    : self.__currentClass.fullName,
        'gnue_property' : item._type == 'GLProperty' and item.fullName or None,
        'gnue_procedure': item._type == 'GLProcedure' and item.fullName or None,
        'gnue_language' : self.__currentModule.language,
        '_file'         : self.__currentFile}

    # Add the optional attributes of the XML-tag
    self.labels [key].update (self.__fetchTags (item,
                              ['page', 'label', 'position', 'search', 'info']))


  # ---------------------------------------------------------------------------
  # Translate a message into a dictionary for later class-repository update
  # ---------------------------------------------------------------------------

  def __translateMessage (self, item):
    """
    This function adds a message to the message-dictionary. If a message is
    already listed, a DuplicateItemError will be raised.
    """

    key = "%s::%s" % (item.language.lower (), item.fullName.lower ())
    if self.messages.has_key (key):
      raise DuplicateItemError, \
          (self.__currentModule.language, item.fullName,
           [self.messages [key]['_file'], self.__currentFile])

    self.messages [key] = { \
        'gnue_language': item.language,
        'gnue_name'    : item.name,
        'gnue_text'    : item.getChildrenAsContent (),
        'gnue_module'  : item.module,
        '_file'        : self.__currentFile}



  # ---------------------------------------------------------------------------
  # Get a dictionary with all keys listed in tags and values from sObject
  # ---------------------------------------------------------------------------

  def __fetchTags (self, sObject, tags):
    """
    This function creates a dictionary with all attributes from sObject listed
    in tags, where the keys are constructed by 'gnue_%s' % attributename.

    @param sObject: Schema object to retriev attributes from
    @param tags: list of all attributes to retrieve

    @return: dictionary with the attribute names as keys and their values
    """

    res = {}
    for item in tags:
      if hasattr (sObject, item):
        res ["gnue_%s" % item] = getattr (sObject, item)

    return res


  # ---------------------------------------------------------------------------
  # Replace references to properties and procedures
  # ---------------------------------------------------------------------------

  def __replaceReferences (self):
    """
    This function replaces all references to modules, classes, properties and
    procedures with the apropriate gnue_id's.
    """

    # Build lookup dictionaries
    if self.lpModule is None:
      self.lpModule = self.__getModuleLookup ()

    if self.lpClass is None:
      self.lpClass = self.__getClassLookup ()

    if self.lpProperty is None:
      self.lpProperty = self.__getPropertyLookup ()

    if self.lpProcedure is None:
      self.lpProcedure = self.__getProcedureLookup ()

    # Replace references for all labels
    for label in self.labels.values ():
      classname = label ['gnue_class'].lower ()
      classid   = self.lpClass.get (classname)
      if classid is None:
        raise ClassNotFoundError, (label ['gnue_class'], label ['_file'])

      del label ['gnue_class']

      if label ['gnue_property'] is not None:
        (module, name) = repository.splitName (label ['gnue_property'].lower ())
        key  = "%s.%s_%s" % (classid, self.lpModule [module], name)
        ckey = "%s.%s_get%s" % (classname, module, name)

        if self.lpProperty.has_key (key):
          label ['gnue_property']  = self.lpProperty [key]
          label ['gnue_procedure'] = None

        elif self.lpProcedure.has_key (ckey):
          label ['gnue_procedure'] = self.lpProcedure [ckey]
          label ['gnue_property']  = None

        else:
          raise PropertyNotFoundError, \
              (classname, label ['gnue_property'], label ['_file'])

      elif label ['gnue_procedure'] is not None:
        (module, name) = repository.splitName (label ['gnue_procedure'].lower())
        key  = "%s.%s_%s" % (classname, module, name)
        ckey = "%s.%s_get%s" % (classname, module, name)
        ref  = self.lpProcedure.get (key, self.lpProcedure.get (ckey))
        if ref is None:
          raise ProcedureNotFoundError, \
              (classname, label ['gnue_procedure'], label ['_file'])

        label ['gnue_procedure'] = ref

      del label ['_file']


    # Replace references for all messages
    for message in self.messages.values ():
      message ['gnue_module'] = self.lpModule [message ['gnue_module'].lower ()]
      del message ['_file']


  # ---------------------------------------------------------------------------
  # Create a lookup dictionary for modules
  # ---------------------------------------------------------------------------

  def __getModuleLookup (self):
    """
    This function creates a lookup dictionary for modules. It's a mapping for
    id to name and vice versa.

    @return: dictionary with id-to-name and name-to-id maps for modules
    """
    
    result = {}

    resultSet = self.__openSource ('gnue_module',
                                   ['gnue_id', 'gnue_name']).createResultSet ()
    try:
      rec = resultSet.firstRecord ()
      while rec is not None:
        name = rec.getField ('gnue_name')
        gid  = rec.getField ('gnue_id')

        result [gid]           = name
        result [name.lower ()] = gid

        rec = resultSet.nextRecord ()

      return result

    finally:
      resultSet.close ()


  # ---------------------------------------------------------------------------
  # Create a lookup dictionary for classes
  # ---------------------------------------------------------------------------

  def __getClassLookup (self):
    """
    This function creates a lookup dictionary for classes, where the full
    qualified name is the key and the gnue_id is the value.

    @return: dictionary with class mapping
    """

    result = {}

    resultSet = self.__openSource ('gnue_class',
                    ['gnue_id', 'gnue_module', 'gnue_name']).createResultSet ()
    try:
      rec = resultSet.firstRecord ()
      while rec is not None:
        module = self.lpModule.get (rec.getField ('gnue_module'))
        name   = rec.getField ('gnue_name')
        gid    = rec.getField ('gnue_id')
        key    = repository.createName (module, name)

        result [key.lower ()] = gid
        result [gid]          = key.lower ()

        rec = resultSet.nextRecord ()

      return result

    finally:
      resultSet.close ()


  # ---------------------------------------------------------------------------
  # Build a lookup-dictionary for properties
  # ---------------------------------------------------------------------------

  def __getPropertyLookup (self):
    """
    This function creates a lookup dictionary for properties where the key is
    constructed from "classid.moduleid_propertyname".
    """

    result = {}

    resultSet = self.__openSource ('gnue_property', ['gnue_id',
          'gnue_module', 'gnue_class', 'gnue_name', 'gnue_type', 'gnue_length',
          'gnue_scale', 'gnue_nullable', 'gnue_comment']).createResultSet ()

    try:
      rec = resultSet.firstRecord ()
      while rec is not None:
        mid  = rec.getField ('gnue_module')
        cid  = rec.getField ('gnue_class')
        name = rec.getField ('gnue_name')
        key  = "%s.%s_%s" % (cid, mid, name.lower ())

        result [key] = rec.getField ('gnue_id')

        rec  = resultSet.nextRecord ()

      return result

    finally:
      resultSet.close ()


  # ---------------------------------------------------------------------------
  # Build a lookup-dictionary for procedures
  # ---------------------------------------------------------------------------

  def __getProcedureLookup (self):
    """
    This function creates a procedure lookup dictionary where the key is built
    from "classname.modulename_procedurename".
    """

    result = {}

    resultSet = self.__openSource ('gnue_procedure', ['gnue_id',
        'gnue_module', 'gnue_class', 'gnue_name', 'gnue_type', 'gnue_length',
        'gnue_scale', 'gnue_nullable', 'gnue_comment', 'gnue_code',
        'gnue_language']).createResultSet ()

    try:
      rec = resultSet.firstRecord ()
      while rec is not None:
        mName = self.lpModule.get (rec.getField ('gnue_module'))
        cName = self.lpClass.get (rec.getField ('gnue_class'))
        pName = rec.getField ('gnue_name')
        key   = "%s.%s_%s" % (cName, mName, pName)

        result [key.lower ()] = rec.getField ('gnue_id')

        rec  = resultSet.nextRecord ()

      return result

    finally:
      resultSet.close ()


  # ---------------------------------------------------------------------------
  # Create a new datasource for a given class
  # ---------------------------------------------------------------------------

  def __openSource (self, classname, fieldList):
    """
    This function creates a new datasource for the given classname with the
    given fieldlist. The primary key is always 'gnue_id'

    @param classname: name of the table to create a datasource for
    @param fieldList: sequence of fieldnames to use
    @return: datasource instance
    """

    return GDataSource.DataSourceWrapper (self._connections,
                                          fieldList,
                                          {'name'      : "dts_%s" % classname,
                                           'database'  : self._database,
                                           'table'     : classname,
                                           'primarykey': 'gnue_id'})


  # ---------------------------------------------------------------------------
  # Update all labels in the repository
  # ---------------------------------------------------------------------------

  def __updateLabels (self):
    """
    Update the class repository with labels read from GLD files.
    """

    src = self.__openSource ('gnue_label', ['gnue_id', 'gnue_property',
        'gnue_language', 'gnue_position', 'gnue_page', 'gnue_label',
        'gnue_procedure', 'gnue_search', 'gnue_info'])

    # Load and update all labels. To identify a label without gnue_id, use
    # 'gnue_language', 'gnue_property' and 'gnue_procedure'.
    rSet = src.createResultSet ()
    try:
      cond = ['gnue_language', 'gnue_property', 'gnue_procedure']
      stat = self.__processResultSet (rSet, self.labels, cond)

      if stat [0] + stat [1]:
        self.__backend.commit ()

      msg = u_("  Labels  : %(ins)3d inserted, %(upd)3d updated, %(kept)3d "
               "unchanged.") \
            % {'ins': stat [0], 'upd': stat [1], 'kept': stat [2]}

      assert gDebug (1, msg)
      print o(msg)

    finally:
      rSet.close ()


  # ---------------------------------------------------------------------------
  # Update all messages in the repository
  # ---------------------------------------------------------------------------

  def __updateMessages (self):
    """
    Update the class repository with all messages read from the GLD file.
    """

    src = self.__openSource ('gnue_message',
        ['gnue_id', 'gnue_module', 'gnue_language', 'gnue_name', 'gnue_text'])

    # Load and update all messages. To identify a message without a gnue_id
    # 'gnue_language', 'gnue_module' and 'gnue_name' will be used.
    rSet = src.createResultSet ()
    try:
      cond = ['gnue_language', 'gnue_module', 'gnue_name']
      stat = self.__processResultSet (rSet, self.messages, cond)

      if stat [0] + stat [1]:
        self.__backend.commit ()

      msg = u_("  Messages: %(ins)3d inserted, %(upd)3d updated, %(kept)3d "
                  "unchanged.") \
            % {'ins': stat [0], 'upd': stat [1], 'kept': stat [2]}

      assert gDebug (1, msg)
      print o(msg)

    finally:
      rSet.close ()


  # ---------------------------------------------------------------------------
  # Update a given resultset using a dictionary of records
  # ---------------------------------------------------------------------------

  def __processResultSet (self, resultSet, items, condition):
    """
    This function iterates over the given resultset and updates all records
    listed in the given dictionary. All items of the dictionary not listed in
    the resultset will be inserted as new records.

    @param resultSet: the resultset to update
    @param items: dictionary of records to update the resultset with
    @param condition: sequence of key-fields used for comparison.
    @return: triple with number of (inserted, modified, unchanged) records
    """

    stat = [0, 0, 0]
    needPost = False

    # First we build a mapping using the condition as key for all items
    mapping = {}
    for item in items.values ():
      key = self.__getKey (item, condition)
      mapping [key] = item

    # Now run over all existing records and update the records if necessary
    rec = resultSet.firstRecord ()
    while rec is not None:
      key = self.__getKey (rec, condition)
      if mapping.has_key (key):
        mapping [key]['gnue_id'] = rec.getField ('gnue_id')
        post = self.doUpdate (resultSet, mapping [key], True)
        needPost |= post
        stat [1 + [1, 0][post]] += 1
        del mapping [key]

      rec = resultSet.nextRecord ()

    # If there are keys left in the mapping, they must be new records
    for item in mapping.values ():
      item ['gnue_id'] = self.__generateId ()
      resultSet.insertRecord ()
      self.doUpdate (resultSet, item, True)
      needPost = True
      stat [0] += 1

    if needPost:
      resultSet.post ()
      resultSet.requery (False)

    return stat


  # ---------------------------------------------------------------------------
  # Get a tuple of fields from a record
  # ---------------------------------------------------------------------------

  def __getKey (self, record, fields):
    """
    This function creates a tuple of fields from a record. If a field value has
    a 'lower' function the value will be lowered. This ensures to get a
    case-insensitive key.

    @param record: dict-like record to retrieve values from
    @param fields: sequence of fieldnames to retrieve

    @return: n-tuple with the (lowered) values of all fields
    """

    result = []

    for field in fields:
      value = record [field]
      if hasattr (value, 'lower'):
        value = value.lower ()

      result.append (value)

    return tuple (result)


  # ---------------------------------------------------------------------------
  # Perform an update on the given resultset using a given data dictionary
  # ---------------------------------------------------------------------------

  def doUpdate (self, resultSet, data, deferPost = False):
    """
    This function sets all fields in the current record of the resultset based
    on the key/values given by the data dictionary. It returns TRUE, if a field
    value has been changed, otherwise FALSE. If a field was changed, the record
    gets posted unless 'deferPost' is set to True.

    @param resultSet: resultset with the current record to be updated
    @param data: dictionary with keys and values used for updates
    @param deferPost: if True, this function does not post () the changes. The
        caller will do this later.
    @return: TRUE if a field has been changed, FALSE if no field has been
        changed.
    """
    doPost = False

    for key in data:
      if resultSet.current.getField (key) != data [key]:
        resultSet.current.setField (key, data [key])
        doPost = True

    if doPost and not deferPost:
      resultSet.post ()
      resultSet.requery (False)

    return doPost


  # ---------------------------------------------------------------------------
  # Generate a new object id
  # ---------------------------------------------------------------------------

  def __generateId (self):
    """
    This function generates a new gnue_id like it is done by appserver. Once
    this algorithm should be replace by a better one.
    """

    if self._uuidtype == 'time':
      return UUID.generateTimeBased ()
    else:
      return UUID.generateRandom ()


# =============================================================================
# Main program
# =============================================================================

if __name__ == '__main__':
  gldClient ().run ()
