# GNU Enterprise Common Library - GParser - Helper classes
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: GParserHelpers.py 7898 2005-09-12 14:31:40Z jamest $

from xml.sax import saxutils
from types import StringType


# =============================================================================
# Base class for GParser objects
# =============================================================================

class ParserObj:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, parent = None, type = '_NotSet_'):
    
    self._type          = type
    self._children      = []     # The objects contained by this object
    self._attributes    = {}
    self._inits         = []     # functions called during phaseInit stage
    self._xmlnamespace  = None   # If the object is namespace-qualified,
                                 # the namespace
    self._xmlnamespaces = {}     # If attributes are namespace-qualified, a map

    self.setParent (parent)
    if parent:
      parent.addChild (self)


  # ---------------------------------------------------------------------------
  # Return the parent instance of an object in a GObj tree
  # ---------------------------------------------------------------------------

  def getParent (self):
    """
    Returns the immediate parent of an object instance in a GObj tree.

    @return: The parent of the object in the GObj tree.
    @rtype: any
    """

    return self.__parent


  # ---------------------------------------------------------------------------
  # Set the parent of an object in a GObj tree
  # ---------------------------------------------------------------------------

  def setParent (self, newParent):
    """
    Set the immediate parent of an object instance in a GObj tree

    @param newParent: instance to be set as parent
    """

    self.__parent = newParent


  # ---------------------------------------------------------------------------
  # Return a compareable id of an object
  # ---------------------------------------------------------------------------

  def _id_ (self, maxIdLength = None):
    """
    Return a compareable and identifying id of an object within a tree. Usually
    this is it's name (if available) or it's object type (as given in the
    constructor).

    @param maxIdLength: if not None only return up to maxIdLength characters.
    @return: id for the instance
    """

    if hasattr (self, 'name'):
      result = self.name.lower ()
      if maxIdLength is not None:
        result = result [:maxIdLength]
    else:
      result = self._type

    return result


  # ---------------------------------------------------------------------------
  # Assign a given set of attributes from another instance
  # ---------------------------------------------------------------------------

  def assign (self, source, recursive = False):
    """
    """

    if self.__class__ != source.__class__:
      raise AssignmentTypeError, (self, source)

    # Assign everything except the parent and the children
    for (name, value) in source.__dict__.items ():
      if name in ['_ParserObj__parent', '_children']:
        continue

      self.__dict__ [name] = value

    if recursive:
      self._children = []

      for child in source._children:
        new = child.__class__ (None)
        new.assign (child, recursive)

        self.addChild (new)
        new.setParent (self)


  # ---------------------------------------------------------------------------
  # Merge another object tree with this tree
  # ---------------------------------------------------------------------------

  def merge (self, other, maxIdLength = None):
    """
    Incorporate all subtrees from the given object tree of this instances type.
    """

    # First find objects of the same type in the other tree
    candidates = other.findChildrenOfType (self._type, True, True)

    # We keep a mapping of all our children
    mine = {}
    for mc in self._children:
      mine [mc._id_ (maxIdLength)] = mc

    # Iterate over all elements in the other tree having the same type
    for item in candidates:
      # and observe all their children ...
      for otherChild in item._children:
        # ... wether we have to start a recursive merge ...
        if otherChild._id_ (maxIdLength) in mine:
          mine [otherChild._id_ (maxIdLength)].merge (otherChild)
        # ... or we can copy the subtree
        else:
          new = otherChild.__class__ (self)
          new.assign (otherChild, True)


  # ---------------------------------------------------------------------------
  # Build a difference-tree from two given object trees
  # ---------------------------------------------------------------------------

  def diff (self, goal, maxIdLength = None):
    """
    """

    result = None

    mine   = {}
    others = {}

    # Build a mapping of our children
    for child in self._children:
      mine [child._id_ (maxIdLength)] = child

    # find the counterpart of this instance
    buddy = goal.findChildOfType (self._type, True, True)

    if buddy is not None:
      for child in buddy._children:
        others [child._id_ (maxIdLength)] = child

    # Find out new and changed items
    for (key, item) in others.items ():
      childDiff = None

      if not key in mine:
        if item._children or isinstance (item, GLeafNode):
          childDiff = item.__class__ (None)
          childDiff.assign (item, True)
          childDiff.walk (self._diffActionWalker_, "add")

      else:
        # we need to find out wether the instances are changed then
        childDiff = mine [key].diff (item, maxIdLength)
        if childDiff is not None:
          childDiff._action = "change"

      if childDiff is not None:
        if result is None:
          result = self.__class__ (None)
          result.assign (self)

        result.addChild (childDiff)
        childDiff.setParent (result)

    # Finally find out all 'obsolete' ones
    for (key, item) in mine.items ():
      if not key in others:
        if result is None:
          result = self.__class__ (None)
          result.assign (self)

        child = item.__class__ (None)
        child.assign (item, True)
        child.walk (self._diffActionWalker_, "remove")

        result.addChild (child)
        child.setParent (result)

    if result is not None:
      result._action = "change"

    return result


  # ---------------------------------------------------------------------------
  # Set the diff-action for a given object
  # ---------------------------------------------------------------------------

  def _diffActionWalker_ (self, obj, action):

    obj._action = action


# =============================================================================
# Mixin-class for leaf node objects
# =============================================================================

class GLeafNode:
  """
  This is a I{mixin}-class for parser objects which are leaf nodes. This will
  be a relevant information on building difference-trees between two object
  trees. To add this class to another class do something like this::

    class Foobar (ParserObj, GLeafNode): ...
    class Barbaz (SomeOtherClass, GLeafNode): ...

  There is nothing more to do if you want to flag a class as leaf node.
  """

  pass


# =============================================================================
# Base class for xml content
# =============================================================================

class GContent (ParserObj):

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, parent, content = None):

    ParserObj.__init__ (self, parent, '_content_')
    self._content = content


  # ---------------------------------------------------------------------------
  # Return the escaped contents
  # ---------------------------------------------------------------------------

  def getEscapedContent (self):

    return saxutils.escape (self._content)


  # ---------------------------------------------------------------------------
  # Return the contents
  # ---------------------------------------------------------------------------

  def getContent (self):

    return self._content


  # ---------------------------------------------------------------------------
  # Return a xml representation of the object
  # ---------------------------------------------------------------------------

  def dumpXML (self, lookupDict, treeDump = None, gap = None,
               escape = 1, textEncoding = '<locale>', xmlnamespaces = {},
               stripPrefixes = None):

    if textEncoding == '<locale>':
      textEncoding = gConfig ('textEncoding')

    if type (self._content) == StringType:
      xmlString = '%s' % unicode (self._content, textEncoding)
    else:
      xmlString = self._content

    return escape and saxutils.escape (xmlString) or xmlString


  # ---------------------------------------------------------------------------
  # Show a contents element within an (indented) tree
  # ---------------------------------------------------------------------------

  def showTree (self, indent = 0):

    print ' ' * indent + 'GContent ' + `self._content`


  # ---------------------------------------------------------------------------
  # Return a useful description of this object (used by designer clients)
  # ---------------------------------------------------------------------------

  def getDescription (self):

    return "(Content)"


  # ---------------------------------------------------------------------------
  # Don't merge contents instances
  # ---------------------------------------------------------------------------

  def merge (self, other):
    return
