#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# Designer.py
#
# DESCRIPTION:
#
# NOTES:

import sys, os, string
from gnue.common.apps import RuntimeSettings

from StringIO import StringIO
#from wxPython.wx import *
from gnue.common.apps.GClientApp import GClientApp
from gnue.common.apps import GDebug
from gnue.common.utils.FileUtils import dyn_import
from gnue.designer import VERSION, PACKAGE
from base.ModuleSupport import SupportedModules
from base import TemplateParser, TemplateChooser, MRUManager
from gnue.forms.GFConfig import ConfigOptions
from base.Config import ConfigOptions as DesignerConfigOptions

import time

class Designer(GClientApp):

  # GClientApp stuff...
  NAME = PACKAGE
  VERSION = VERSION
  COMMAND = "gnue-designer"
  SUMMARY = _("A graphical tool for the rapid deployment of GNU Enterprise forms and reports.")
  COMMAND_OPTIONS = [
      [ 'new', 'n', 'new', 1, None, 'module',
          _('Starts up opening a new instance of <module>, where module can be forms, reports, schema, etc.') ],
      ]
  USAGE = GClientApp.USAGE + " [file] [file] ..."
  USE_DATABASE_OPTIONS = 1

  def __init__(self):
    GClientApp.__init__(self, application="forms",defaults=ConfigOptions)

    self._instances = []

    # Load the configuration information
    self.configurationManager.loadApplicationConfig(section="designer",defaults=DesignerConfigOptions)
    self.configurationManager.registerAlias('gConfigNav', 'navigator')
    self.configurationManager.registerAlias('gConfigForms', 'forms')
    self.configurationManager.registerAlias('gConfigReports', 'reports')

  # ==========================================================================
  # Startup functions
  # ==========================================================================
  def run(self):
    """
    Startup logic for Designer.
    """
    print "Running"

    # TODO: make a startup/GConfig parameter, and/or do auto-detection
    from gnue.designer.uidrivers.wx.Base import UIBase

    self.ui = UIBase(self)


    # Setup the Most Recently Used URL manager
    self.mru = MRUManager.MRUManager(self)

    # get list of available modules
    self.loadModuleInformation()

    self.ui.run()


  def init(self):
    """
    Called by the UI Driver after the graphics subsystem is ready.
    At this point, we can start loading/drawing.
    """

    print "Init'ing"
    # Init the splashscreen
    self.ui.createStartupSplash()
    gStartupStatus(_('Initializing Client Library'))

    # Load the specified file, a requested new instance, or the default empty instance
    if len(self.ARGUMENTS):
      for arg in self.ARGUMENTS:
        #self.SetTopWindow(self.load(arg))
        self.load(arg)

    elif self.OPTIONS['new']:
      try:
        #self.SetTopWindow(self.newInstance(self.OPTIONS['new']))
        self.newInstance(self.OPTIONS['new'])
      except ImportError:
        raise
        self.handleStartupError('"%s" is not a valid option for --new' % (
            self.OPTIONS['new']))

    else:
      # TODO: What to do when no object is specified on command line?
      # TODO: jcater thinks of some task selector like kword uses,
      # TODO: or, restore the session to the way it was (same files, etc)

      #self.SetTopWindow(self.newInstance('forms'))
      self.newInstance('forms')

    # Close startup screen
    self.ui.closeStartupSplash()
    return True

  def loadModuleInformation(self):
    """
    Loads all the available editor modules into memory and
    extracts from them a list of file types/extensions
    that they support
    """
    gStartupStatus(_('Loading available tool modules'))

    self.supportedModuleObjects = []  # A list of supported modules along with their "nickname"
    self.supportedOpenWildcard = ""   #
    self.moduleExtensionMapping = {}  # A dictionary of {file extension: designer module}

    wildcard = []
    alltypes = []

    # TODO: wxWidgets currently doesn't consistently support multiple file
    # TODO: extension wildcards (e.g., "All GNUe Form Files (*.gfd,*.gfl)").
    # TODO: However, the code is left commented out in case some day it does.


    for module in SupportedModules:
      self.supportedModuleObjects.append((module.properties.nickname, module))


    ##  if len(module.properties.fileExtensions.keys()) > 1:
    ##
    ##    wildcard += "%s Files (*.%s)|*.%s|" % \
    ##             ( module.properties.application,
    ##               string.join(module.properties.fileExtensions.keys(),',*.'),
    ##               string.join(module.properties.fileExtensions.keys(),';*.') )


      # Make the "default" file extension for a module
      # appear before the other extensions.
      wildcard += [
       ( module.properties.defaultFileExtension,
         module.properties.fileExtensions[module.properties.defaultFileExtension]) ]

      for type in module.properties.fileExtensions.keys():
        if type != module.properties.defaultFileExtension:
          wildcard += [
             ( type, module.properties.fileExtensions[type]) ]
        alltypes.append(type)

        # Keep a dict of Extension::Handler mappings
        self.moduleExtensionMapping[type] = module.properties.module


    ##  self.supportedOpenWildcard = "All Supported Files (*.%s)|*.%s|" % \
    ##              ( string.join(alltypes,',*.'),
    ##              string.join(alltypes,';*.') ) \
    ##            + wildcard \
    ##            + "All Files (*.*)|*.*"
      self.supportedOpenWildcard = wildcard + [('*',"All Files")]
  # ==========================================================================
  # Instance functions
  # ==========================================================================
  def load(self, file):
    """
    Loads the requested file into a new instance of the appropriate
    designer module if the extension is recognized.

    @type file: string
    @param file: The url of the file to open.
    @rtype: designer editor instance
    @return: An instance of the appropriate editor module or None if
             an error is encountered
    """
    gStartupStatus(_('Loading document'))

    extension = string.lower(os.path.splitext(file)[1][1:])

    if not self.moduleExtensionMapping.has_key(extension):

      # TODO: This should try to figure out what kind of file this is.
      # TODO: We maintain a list of opening xml elements in the
      # TODO: (gnue.designer.SupportedModules).properties object.
      # TODO: [gnue.designer.SupportedModules is a list]

      # TODO: Shouldn't this display graphically?
      print "I do not know what to do with a .%s file!" % extension
      return None

    return self.newInstance(self.moduleExtensionMapping[extension], file)


  def newInstance(self, module, *args, **parms):
    """
    Creates an instance of the appropriate designer editor module
    """
    c = dyn_import("gnue.designer.%s.Instance" % module).Instance(self, *args, **parms)
    return c

  def addInstance(self, instance):
    """
    Adds a newly created instance to the list of open instances
    Sets several event handlers in the instance to point back to
    this class so it can proxy the events for the instance.

    @type instance: Designer Editor Module
    @param instance: An instance of a designer module
    """
    self._instances.append(instance)
    instance.registerEventListeners({
                       # Proxy for the main app
                       'RequestNew'          : self.OnNew,
                       'RequestSaveAll'      : self.OnSaveAll,
                       'RequestAbout'        : self.OnAbout,
                       'RequestNewWizard'    : self.OnWizard,
                       'RequestOpen'         : self.OnOpen,
                       'RequestOpenRecent'   : self.OnOpenRecent,
                       'RequestExit'         : self.OnExit
                    })

  def removeInstance(self, instance):
    """
    Removes an instance from the list of open instances

    @type instance: Designer Editor Module
    @param instance: An instance of a designer module
    """
    self._instances.remove(instance)


  # ==========================================================================
  # Proxy event functions
  # ==========================================================================
  def OnWizard(self, event):
    """
    Handles the request to display the wizard selection dialog

    type event: GNUe Event
    param event: The event request
    """
    wizard = TemplateChooser.TemplateChooser(self).run()
    if wizard != None:
      self.RunWizard(wizard)

  def OnOpen(self, event):
    """
    Handles the request to open a file

    type event: GNUe Event
    param event: The event request
    """

    path = self.ui.dialogOpenFile(_("Open GNUe File..."),
                           wildcards = self.supportedOpenWildcard)

    if path:
      if self.isFileOpened(path):
        self.ui.dialogOk( _("File is already opened."), _("Notice"))
      else:
        self.load(path)


  def OnOpenRecent(self, event):
    """
    Handles the request to open a recent opened URL

    type event: GNUe Event
    param event: The event request
    """
    location = self.mru.mruMenuMap[event.GetId()]
    if not self.isFileOpened(location):
      self.load(location )


  def OnExit(self, event=None):
    """
    Handles the request for application exit

    type event: GNUe Event
    param event: The event request
    """
    for instance in self._instances[:]:
      instance.OnClose(event)
      #
      # If this Close() causes issues you can replace
      # with the following.
      #
      #RuntimeSettings.saveRuntimeSettings(instance)
      #instance.Destroy()

  def OnSaveAll(self, event):
    """
    Handles the request for all instances to save their files

    type event: GNUe Event
    param event: The event request
    """
    for instance in self._instances:
      if instance.isDirty():
        instance.OnSave(event)

  def OnAbout(self, event):
    """
    Handles the request for an about box

    type event: GNUe Event
    param event: The event request
    """

    imports = []
    for f in ('common','forms','reports','appserver','navigator'):
      try:
        v = {}
        exec 'from gnue.%s import __version__, PACKAGE' % f in v
        imports.append('%s Version %s' % (v['PACKAGE'], v['__version__']))
      except:
        pass

    from wxPython import __version__ as v
    imports.append('wxPython Version %s' % v)

    from sys import version as v
    imports.append('Python Version %s' % v.replace('\n','\n    '))


    self.ui.dialogOk(self.NAME + " " +
                     _("Version  ") + self.VERSION + "\n\n" +
                     _("Copyright 2001-2005 Free Software Foundation\n\n") +
                     _("Environment:\n  %s\n\n") % string.join(imports,'\n  ') +
                     _("For help, email info@gnue.org."),
                     _("About ") + self.NAME)

  def OnNew(self, event):
    """
    Handles the request when for a new instance of a designer editor module

    type event: GNUe Event
    param event: The event request
    """
    self.newInstance(event.type)


  # ==========================================================================
  # Support functions
  # ==========================================================================

  def isDirty(self):
    """
    Determines if all instances currently have unsaved edits.

    @rtype: boolean
    @return: True if any instance contains unsaved information
             False if all instances are saved
    """
    isdirty = False
    for instance in self._instances:
      if instance.isDirty():
        isdirty = true
        break
    return isdirty

  def isFileOpened(self, location):
    """
    Determines if an instance already has the requested file open.

    TODO: Currently this function is not implemented.  It always
          returns false.

    @rtype: boolean
    @return: True if any instance contains the requested file
             False if the file is not currently loaded
    """

    return False

  def RunWizard(self, templateInformation):
    """

    """
    product = templateInformation['Product']
    templateSupport = dyn_import('gnue.designer.%s.TemplateSupport' % product)

    try:
      style = templateInformation['ProductStyle']
    except:
      style = None

    instance = self.newInstance(product, style=style)

    if not TemplateParser.TemplateParser(instance, instance.rootObject,
          instance.uiinstance, templateInformation).run():
      instance.uiinstance.close()

#
# Workaround for Python 2.2's big unicode bug!!!
# TODO: I don't think this is working!
# TODO: Pop up a warning dialog
#
if sys.version[:5] == "2.2.0":
  from cStringIO import StringIO

if __name__ == '__main__':
  Designer().run()

